from __future__ import annotations

import logging
from typing import TYPE_CHECKING, Any, cast

from sift.tags.v2.tags_pb2 import (
    CreateTagRequest,
    CreateTagResponse,
    ListTagsRequest,
    ListTagsResponse,
)
from sift.tags.v2.tags_pb2_grpc import TagServiceStub

from sift_client._internal.low_level_wrappers.base import LowLevelClientBase
from sift_client.sift_types.tag import Tag
from sift_client.transport import WithGrpcClient

if TYPE_CHECKING:
    from sift_client.transport.grpc_transport import GrpcClient

# Configure logging
logger = logging.getLogger(__name__)


class TagsLowLevelClient(LowLevelClientBase, WithGrpcClient):
    """Low-level client for the TagsAPI.

    This class provides a thin wrapper around the autogenerated bindings for the TagsAPI.
    """

    def __init__(self, grpc_client: GrpcClient):
        """Initialize the TagsLowLevelClient.

        Args:
            grpc_client: The gRPC client to use for making API calls.
        """
        super().__init__(grpc_client)

    async def create_tag(self, name: str) -> Tag:
        """Create a new tag.

        Args:
            name: The name of the tag.

        Returns:
            The created Tag.

        Raises:
            ValueError: If name is not provided.
        """
        if not name:
            raise ValueError("name must be provided")

        request = CreateTagRequest(name=name)
        response = await self._grpc_client.get_stub(TagServiceStub).CreateTag(request)
        grpc_tag = cast("CreateTagResponse", response).tag
        return Tag._from_proto(grpc_tag)

    async def list_tags(
        self,
        *,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
    ) -> tuple[list[Tag], str]:
        """List tags with optional filtering and pagination.

        Args:
            page_size: The maximum number of tags to return.
            page_token: A page token for pagination.
            query_filter: A CEL filter string.
            order_by: How to order the retrieved tags.

        Returns:
            A tuple of (tags, next_page_token).
        """
        request_kwargs: dict[str, Any] = {}
        if page_size is not None:
            request_kwargs["page_size"] = page_size
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if order_by is not None:
            request_kwargs["order_by"] = order_by

        request = ListTagsRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(TagServiceStub).ListTags(request)
        response = cast("ListTagsResponse", response)

        tags = [Tag._from_proto(tag) for tag in response.tags]
        return tags, response.next_page_token

    async def list_all_tags(
        self,
        *,
        query_filter: str | None = None,
        order_by: str | None = None,
        max_results: int | None = None,
    ) -> list[Tag]:
        """List all tags with optional filtering.

        Args:
            query_filter: A CEL filter string.
            order_by: How to order the retrieved tags.
            max_results: Maximum number of results to return.

        Returns:
            A list of all matching tags.
        """
        return await self._handle_pagination(
            self.list_tags,
            kwargs={"query_filter": query_filter},
            order_by=order_by,
            max_results=max_results,
        )
