from __future__ import annotations

import logging
from typing import TYPE_CHECKING, Any, cast

from google.protobuf.timestamp_pb2 import Timestamp
from sift.test_reports.v1.test_reports_pb2 import (
    CreateTestMeasurementRequest,
    CreateTestMeasurementResponse,
    CreateTestMeasurementsRequest,
    CreateTestMeasurementsResponse,
    CreateTestReportRequest,
    CreateTestReportResponse,
    CreateTestStepRequest,
    CreateTestStepResponse,
    DeleteTestMeasurementRequest,
    DeleteTestReportRequest,
    DeleteTestStepRequest,
    GetTestReportRequest,
    GetTestReportResponse,
    ImportTestReportRequest,
    ImportTestReportResponse,
    ListTestMeasurementsRequest,
    ListTestMeasurementsResponse,
    ListTestReportsRequest,
    ListTestReportsResponse,
    ListTestStepsRequest,
    ListTestStepsResponse,
    UpdateTestMeasurementRequest,
    UpdateTestMeasurementResponse,
    UpdateTestReportRequest,
    UpdateTestReportResponse,
    UpdateTestStepRequest,
    UpdateTestStepResponse,
)
from sift.test_reports.v1.test_reports_pb2_grpc import TestReportServiceStub

from sift_client._internal.low_level_wrappers.base import LowLevelClientBase
from sift_client.sift_types.test_report import (
    TestMeasurement,
    TestMeasurementCreate,
    TestMeasurementUpdate,
    TestReport,
    TestReportCreate,
    TestReportUpdate,
    TestStatus,
    TestStep,
    TestStepCreate,
    TestStepUpdate,
)
from sift_client.transport import WithGrpcClient
from sift_client.util.metadata import metadata_dict_to_proto

if TYPE_CHECKING:
    from sift_client.transport.grpc_transport import GrpcClient

# Configure logging
logger = logging.getLogger(__name__)


class TestResultsLowLevelClient(LowLevelClientBase, WithGrpcClient):
    """Low-level client for the TestResultsAPI.

    This class provides a thin wrapper around the autogenerated bindings for the TestResultsAPI.
    """

    def __init__(self, grpc_client: GrpcClient):
        """Initialize the TestResultsLowLevelClient.

        Args:
            grpc_client: The gRPC client to use for making API calls.
        """
        super().__init__(grpc_client)

    async def import_test_report(self, remote_file_id: str) -> TestReport:
        """Import a test report from an already-uploaded file.

        Args:
            remote_file_id: The remote file ID containing the XML test data.

        Returns:
            The imported TestReport.

        Raises:
            ValueError: If remote_file_id is not provided.
        """
        if not remote_file_id:
            raise ValueError("remote_file_id must be provided")

        request = ImportTestReportRequest(remote_file_id=remote_file_id)
        response = await self._grpc_client.get_stub(TestReportServiceStub).ImportTestReport(request)
        grpc_test_report = cast("ImportTestReportResponse", response).test_report
        return TestReport._from_proto(grpc_test_report)

    async def create_test_report(
        self,
        *,
        test_report: TestReportCreate,
    ) -> TestReport:
        """Create a new test report.

        Args:
            test_report: The test report to create.

        Returns:
            The created TestReport.
        """
        request_kwargs: dict[str, Any] = {
            "status": test_report.status.value
            if isinstance(test_report.status, TestStatus)
            else test_report.status,
            "name": test_report.name,
            "test_system_name": test_report.test_system_name,
            "test_case": test_report.test_case,
        }

        # Handle timestamps
        start_ts = Timestamp()
        start_ts.FromDatetime(test_report.start_time)
        request_kwargs["start_time"] = start_ts

        end_ts = Timestamp()
        end_ts.FromDatetime(test_report.end_time)
        request_kwargs["end_time"] = end_ts

        if test_report.metadata is not None:
            request_kwargs["metadata"] = metadata_dict_to_proto(test_report.metadata)

        if test_report.serial_number is not None:
            request_kwargs["serial_number"] = test_report.serial_number

        if test_report.part_number is not None:
            request_kwargs["part_number"] = test_report.part_number

        if test_report.system_operator is not None:
            request_kwargs["system_operator"] = test_report.system_operator

        request = CreateTestReportRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(TestReportServiceStub).CreateTestReport(request)
        grpc_test_report = cast("CreateTestReportResponse", response).test_report
        return TestReport._from_proto(grpc_test_report)

    async def get_test_report(self, test_report_id: str) -> TestReport:
        """Get a test report by test_report_id.

        Args:
            test_report_id: The test report ID to get.

        Returns:
            The TestReport.

        Raises:
            ValueError: If test_report_id is not provided.
        """
        if not test_report_id:
            raise ValueError("test_report_id must be provided")

        request = GetTestReportRequest(test_report_id=test_report_id)
        response = await self._grpc_client.get_stub(TestReportServiceStub).GetTestReport(request)
        grpc_test_report = cast("GetTestReportResponse", response).test_report
        return TestReport._from_proto(grpc_test_report)

    async def list_test_reports(
        self,
        *,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
    ) -> tuple[list[TestReport], str]:
        """List test reports with optional filtering and pagination.

        Args:
            page_size: The maximum number of test reports to return.
            page_token: A page token for pagination.
            query_filter: A CEL filter string.
            order_by: How to order the retrieved test reports.

        Returns:
            A tuple of (test_reports, next_page_token).
        """
        request_kwargs: dict[str, Any] = {}
        if page_size is not None:
            request_kwargs["page_size"] = page_size
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if order_by is not None:
            request_kwargs["order_by"] = order_by

        request = ListTestReportsRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(TestReportServiceStub).ListTestReports(request)
        response = cast("ListTestReportsResponse", response)

        test_reports = [TestReport._from_proto(tr) for tr in response.test_reports]
        return test_reports, response.next_page_token

    async def list_all_test_reports(
        self,
        *,
        query_filter: str | None = None,
        order_by: str | None = None,
        max_results: int | None = None,
    ) -> list[TestReport]:
        """List all test reports with optional filtering.

        Args:
            query_filter: A CEL filter string.
            order_by: How to order the retrieved test reports.
            max_results: Maximum number of results to return.

        Returns:
            A list of all matching test reports.
        """
        return await self._handle_pagination(
            self.list_test_reports,
            kwargs={"query_filter": query_filter},
            order_by=order_by,
            max_results=max_results,
        )

    async def update_test_report(self, update: TestReportUpdate) -> TestReport:
        """Update an existing test report.

        Args:
            update: The updates to apply.

        Returns:
            The updated TestReport.
        """
        test_report_proto, field_mask = update.to_proto_with_mask()
        request = UpdateTestReportRequest(test_report=test_report_proto, update_mask=field_mask)
        response = await self._grpc_client.get_stub(TestReportServiceStub).UpdateTestReport(request)
        grpc_test_report = cast("UpdateTestReportResponse", response).test_report
        return TestReport._from_proto(grpc_test_report)

    async def delete_test_report(self, test_report_id: str) -> None:
        """Delete a test report.

        Args:
            test_report_id: The ID of the test report to delete.

        Raises:
            ValueError: If test_report_id is not provided.
        """
        if not test_report_id:
            raise ValueError("test_report_id must be provided")

        request = DeleteTestReportRequest(test_report_id=test_report_id)
        await self._grpc_client.get_stub(TestReportServiceStub).DeleteTestReport(request)

    # Test Steps

    async def create_test_step(self, test_step: TestStepCreate) -> TestStep:
        """Create a new test step.

        Args:
            test_step: The test step to create.

        Returns:
            The created TestStep.
        """
        request = CreateTestStepRequest(test_step=test_step.to_proto())
        response = await self._grpc_client.get_stub(TestReportServiceStub).CreateTestStep(request)
        grpc_test_step = cast("CreateTestStepResponse", response).test_step
        return TestStep._from_proto(grpc_test_step)

    async def list_test_steps(
        self,
        *,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
    ) -> tuple[list[TestStep], str]:
        """List test steps with optional filtering and pagination.

        Args:
            page_size: The maximum number of test steps to return.
            page_token: A page token for pagination.
            query_filter: A CEL filter string.
            order_by: How to order the retrieved test steps.

        Returns:
            A tuple of (test_steps, next_page_token).
        """
        request_kwargs: dict[str, Any] = {}
        if page_size is not None:
            request_kwargs["page_size"] = page_size
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if order_by is not None:
            request_kwargs["order_by"] = order_by

        request = ListTestStepsRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(TestReportServiceStub).ListTestSteps(request)
        response = cast("ListTestStepsResponse", response)

        test_steps = [TestStep._from_proto(ts) for ts in response.test_steps]
        return test_steps, response.next_page_token

    async def list_all_test_steps(
        self,
        *,
        query_filter: str | None = None,
        order_by: str | None = None,
        max_results: int | None = None,
    ) -> list[TestStep]:
        """List all test steps with optional filtering.

        Args:
            query_filter: A CEL filter string.
            order_by: How to order the retrieved test steps.
            max_results: Maximum number of results to return.

        Returns:
            A list of all matching test steps.
        """
        return await self._handle_pagination(
            self.list_test_steps,
            kwargs={"query_filter": query_filter},
            order_by=order_by,
            max_results=max_results,
        )

    async def update_test_step(self, update: TestStepUpdate) -> TestStep:
        """Update an existing test step.

        Args:
            update: The updates to apply.

        Returns:
            The updated TestStep.
        """
        test_step_proto, field_mask = update.to_proto_with_mask()

        request = UpdateTestStepRequest(test_step=test_step_proto, update_mask=field_mask)
        response = await self._grpc_client.get_stub(TestReportServiceStub).UpdateTestStep(request)
        grpc_test_step = cast("UpdateTestStepResponse", response).test_step
        return TestStep._from_proto(grpc_test_step)

    async def delete_test_step(self, test_step_id: str) -> None:
        """Delete a test step.

        Args:
            test_step_id: The ID of the test step to delete.

        Raises:
            ValueError: If test_step_id is not provided.
        """
        if not test_step_id:
            raise ValueError("test_step_id must be provided")

        request = DeleteTestStepRequest(test_step_id=test_step_id)
        await self._grpc_client.get_stub(TestReportServiceStub).DeleteTestStep(request)

    # Test Measurements

    async def create_test_measurement(
        self, test_measurement: TestMeasurementCreate
    ) -> TestMeasurement:
        """Create a new test measurement.

        Args:
            test_measurement: The test measurement to create.

        Returns:
            The created TestMeasurement.
        """
        request = CreateTestMeasurementRequest(test_measurement=test_measurement.to_proto())
        response = await self._grpc_client.get_stub(TestReportServiceStub).CreateTestMeasurement(
            request
        )
        grpc_test_measurement = cast("CreateTestMeasurementResponse", response).test_measurement
        return TestMeasurement._from_proto(grpc_test_measurement)

    async def create_test_measurements(
        self, test_measurements: list[TestMeasurementCreate]
    ) -> tuple[int, list[str]]:
        """Create multiple test measurements in a single request.

        Args:
            test_measurements: The test measurements to create.

        Returns:
            A tuple of (measurements_created_count, measurement_ids).
        """
        measurement_protos = [tm.to_proto() for tm in test_measurements]
        request = CreateTestMeasurementsRequest(test_measurements=measurement_protos)
        response = await self._grpc_client.get_stub(TestReportServiceStub).CreateTestMeasurements(
            request
        )
        response = cast("CreateTestMeasurementsResponse", response)
        return response.measurements_created_count, list(response.measurement_ids)

    async def list_test_measurements(
        self,
        *,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
    ) -> tuple[list[TestMeasurement], str]:
        """List test measurements with optional filtering and pagination.

        Args:
            page_size: The maximum number of test measurements to return.
            page_token: A page token for pagination.
            query_filter: A CEL filter string.
            order_by: How to order the retrieved test measurements.

        Returns:
            A tuple of (test_measurements, next_page_token).
        """
        request_kwargs: dict[str, Any] = {}
        if page_size is not None:
            request_kwargs["page_size"] = page_size
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if order_by is not None:
            request_kwargs["order_by"] = order_by

        request = ListTestMeasurementsRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(TestReportServiceStub).ListTestMeasurements(
            request
        )
        response = cast("ListTestMeasurementsResponse", response)

        test_measurements = [TestMeasurement._from_proto(tm) for tm in response.test_measurements]
        return test_measurements, response.next_page_token

    async def list_all_test_measurements(
        self,
        *,
        query_filter: str | None = None,
        order_by: str | None = None,
        max_results: int | None = None,
    ) -> list[TestMeasurement]:
        """List all test measurements with optional filtering.

        Args:
            query_filter: A CEL filter string.
            order_by: How to order the retrieved test measurements.
            max_results: Maximum number of results to return.

        Returns:
            A list of all matching test measurements.
        """
        return await self._handle_pagination(
            self.list_test_measurements,
            kwargs={"query_filter": query_filter},
            order_by=order_by,
            max_results=max_results,
        )

    async def update_test_measurement(self, update: TestMeasurementUpdate) -> TestMeasurement:
        """Update an existing test measurement.

        Args:
            update: The updates to apply.

        Returns:
            The updated TestMeasurement.
        """
        test_measurement_proto, field_mask = update.to_proto_with_mask()
        request = UpdateTestMeasurementRequest(
            test_measurement=test_measurement_proto, update_mask=field_mask
        )
        response = await self._grpc_client.get_stub(TestReportServiceStub).UpdateTestMeasurement(
            request
        )
        grpc_test_measurement = cast("UpdateTestMeasurementResponse", response).test_measurement
        return TestMeasurement._from_proto(grpc_test_measurement)

    async def delete_test_measurement(self, measurement_id: str) -> None:
        """Delete a test measurement.

        Args:
            measurement_id: The ID of the test measurement to delete.

        Raises:
            ValueError: If measurement_id is not provided.
        """
        if not measurement_id:
            raise ValueError("measurement_id must be provided")

        request = DeleteTestMeasurementRequest(measurement_id=measurement_id)
        await self._grpc_client.get_stub(TestReportServiceStub).DeleteTestMeasurement(request)
