"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const path = require("path");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const logs = require("aws-cdk-lib/aws-logs");
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, defaults.generateIntegStackName(__filename));
(0, core_1.SetConsistentFeatureFlags)(stack);
stack.templateOptions.description = 'Integration Test for aws-openapigateway-lambda';
const apiDefinitionAsset = new aws_s3_assets_1.Asset(stack, 'ApiDefinitionAsset', {
    path: path.join(__dirname, 'openapi/apiDefinition.yaml')
});
const messagesLambda = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/messages-lambda`),
    }
});
// We need a different log group, because using the function name
// causes CDK to create a log group with the same name (that is not destroyed)
// so there's a collision with the default log group
const replacementLogGroup = new logs.LogGroup(stack, 'our-log-group', {
    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
});
defaults.addCfnSuppressRules(replacementLogGroup, [
    {
        id: 'W84',
        reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
    }
]);
const photosLambda = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        functionName: 'PhotosExistingLambdaTestFromAsset',
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/photos-lambda`),
        logGroup: replacementLogGroup
    }
});
new lib_1.OpenApiGatewayToLambda(stack, 'OpenApiGatewayToLambda', {
    apiDefinitionAsset,
    apiIntegrations: [
        {
            id: 'MessagesHandler',
            existingLambdaObj: messagesLambda
        },
        {
            id: 'PhotosHandler',
            existingLambdaObj: photosLambda
        }
    ]
});
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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