"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommandStepSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const non_retriable_exception_1 = require("../domain/non-retriable-exception");
const platform_1 = require("../domain/platform");
const precondition_1 = require("../domain/precondition");
const response_code_1 = require("../domain/response-code");
const environment_1 = require("../interface/environment");
const ps_module_simulation_1 = require("./command/ps-module-simulation");
const run_powershell_script_simulation_1 = require("./command/run-powershell-script-simulation");
const run_shell_script_simulation_1 = require("./command/run-shell-script-simulation");
class CommandStepSimulation {
    constructor(step, props) {
        this.props = {
            environment: props.environment ?? new environment_1.LoggingEnvironment(),
            simulationPlatform: props.simulationPlatform ?? platform_1.Platform.LINUX,
        };
        this.step = step;
    }
    /**
     * Invokes the current step on the input and will return a SimulationResult.
     * @param inputs must contain all of the inputs declared by the current step.
     * @returns a SimulationResult with the step outputs in the case of success or stacktrace in the case of failure.
     */
    invoke(inputs) {
        console.log('Executing Step: ' + this.step.name);
        return this.invokeWithFallback(inputs);
    }
    /**
     * If fallback/retries are specified for this step, the retry or skip logic is handled in this function.
     */
    invokeWithFallback(allInputs) {
        try {
            const shouldExecute = this.shouldExecuteStep(allInputs);
            if (shouldExecute) {
                const filteredInputs = this.filterInputs(allInputs);
                this.step.inputObserver.accept(filteredInputs);
                this.tryExecute(filteredInputs);
                this.step.outputObserver.accept({});
            }
            else {
                console.log(`skipping step execution for ${this.step.name}`);
            }
            const chainPrefix = shouldExecute ? [this.step.name] : [];
            if (this.step.nextStep && !this.step.exitOnSuccess) {
                const nextStepRes = new CommandStepSimulation(this.step.nextStep, this.props).invoke(allInputs);
                return this.formatResult(nextStepRes);
            }
            else {
                const finallyStep = this.runFinally(allInputs);
                const finallyList = finallyStep ? [finallyStep] : [];
                return {
                    responseCode: response_code_1.ResponseCode.SUCCESS,
                    executedSteps: [...chainPrefix, ...finallyList],
                };
            }
        }
        catch (error) {
            if (this.step.exitOnFailure) {
                const finallyStep = this.runFinally(allInputs);
                const finallyList = finallyStep ? [finallyStep] : [];
                return {
                    responseCode: response_code_1.ResponseCode.FAILED,
                    stackTrace: error.message + '\n' + error.stack,
                    executedSteps: [this.step.name, ...finallyList],
                };
            }
            else if (this.step.markSuccessAndExitOnFailure) {
                const finallyStep = this.runFinally(allInputs);
                const finallyList = finallyStep ? [finallyStep] : [];
                return {
                    responseCode: response_code_1.ResponseCode.SUCCESS,
                    stackTrace: error.message + '\n' + error.stack,
                    executedSteps: [this.step.name, ...finallyList],
                };
            }
            if (this.step.nextStep) {
                const nextStepRes = new CommandStepSimulation(this.step.nextStep, this.props).invoke(allInputs);
                return this.formatResult(nextStepRes);
            }
            else {
                return {
                    responseCode: response_code_1.ResponseCode.FAILED,
                    stackTrace: error.message + '\n' + error.stack,
                    executedSteps: [this.step.name],
                };
            }
        }
    }
    runFinally(allInputs) {
        let last = this.step;
        while (last.nextStep != undefined) {
            last = last.nextStep;
        }
        if (last.finallyStep && last != this.step) {
            new CommandStepSimulation(last, this.props).invoke(allInputs);
            return last.name;
        }
        return undefined;
    }
    formatResult(nextStepRes) {
        if (nextStepRes.responseCode != response_code_1.ResponseCode.SUCCESS) {
            return {
                responseCode: nextStepRes.responseCode,
                outputs: {},
                stackTrace: nextStepRes.stackTrace,
                executedSteps: this.prependSelf(nextStepRes.executedSteps),
            };
        }
        else {
            return {
                responseCode: response_code_1.ResponseCode.SUCCESS,
                executedSteps: this.prependSelf(nextStepRes.executedSteps),
            };
        }
    }
    /**
     * This function receives all of the available inputs and returns the subset of inputs that were requested.
     * If any input requested in the current step is not available in the received inputs, this will throw.
     * @param inputs all available inputs for the currently executing step
     * @returns the subset of inputs which are requested by the current step
     */
    filterInputs(inputs) {
        const foundAll = this.step.listInputs().every(required => Object.keys(inputs).includes(required));
        if (!foundAll) {
            throw new non_retriable_exception_1.NonRetriableException(`Not all inputs required were provided. Required: ${this.step.listInputs()}. Provided: ${JSON.stringify(inputs)}`);
        }
        const filteredEntries = Object.entries(inputs).filter(entry => {
            const [key] = entry;
            return key.startsWith('global:') || key.startsWith('automation:') || this.step.listInputs().includes(key);
        });
        return Object.fromEntries(filteredEntries);
    }
    /**
     * Adds this step name to the list of executed steps.
     * Since the steps are invoked as a chain, the current step is prepended as invocation stack is popped.
     */
    prependSelf(futureSteps) {
        return [this.step.name, ...futureSteps];
    }
    tryExecute(inputs) {
        const start = Date.now();
        console.log(`Execute step ${this.step.name} start: ${start}`);
        this.mySimulation().executeStep(inputs);
        console.log(`Execute step ${this.step.name} end: ${Date.now()}`);
    }
    shouldExecuteStep(inputs) {
        const copiedInputs = Object.assign({}, inputs);
        copiedInputs[precondition_1.Precondition.INJECTED_PLAYFORM_TYPE_KEY] = this.props.simulationPlatform.toString();
        if (!this.step.precondition) {
            // This mimics the "Using the precondition parameter" section
            // of https://docs.aws.amazon.com/systems-manager/latest/userguide/document-schemas-features.html
            // "For documents that use schema version 2.2 or later, if precondition isn't specified, each plugin is either run or skipped based on the plugin’s"
            // "compatibility with the operating system. Plugin compatibility with the operating system is evaluated before the precondition. For"
            // "documents that use schema 2.0 or earlier, incompatible plugins throw an error."
            return this.step.platforms.some(platform => precondition_1.Precondition.newPlatformPrecondition(platform).evaluate(copiedInputs));
        }
        return this.step.precondition.evaluate(copiedInputs);
    }
    mySimulation() {
        switch (this.step.action) {
            case 'aws:runShellScript':
                return new run_shell_script_simulation_1.RunShellScriptSimulation(this.step, this.props);
            case 'aws:runPowerShellScript':
                return new run_powershell_script_simulation_1.RunPowerShellScriptSimulation(this.step, this.props);
            case 'aws:psModule':
                return new ps_module_simulation_1.PsModuleSimulation(this.step, this.props);
            default:
                throw new Error('No simulation available for action: ' + this.step.name);
        }
    }
}
exports.CommandStepSimulation = CommandStepSimulation;
_a = JSII_RTTI_SYMBOL_1;
CommandStepSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.CommandStepSimulation", version: "0.0.23" };
//# sourceMappingURL=data:application/json;base64,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