"""
peakrdl-python is a tool to generate Python Register Access Layer (RAL) from SystemRDL
Copyright (C) 2021 - 2025

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as
published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.

This package is intended to distributed as part of automatically generated code by the PeakRDL
Python tool. It provide the base class common to both the async and non-async versions
"""
import unittest
from abc import ABC, abstractmethod
from typing import Union, Optional
from itertools import product

from ..lib import FieldReadWrite, FieldReadOnly, FieldWriteOnly
from ..lib import FieldEnumReadWrite, FieldEnumReadOnly, FieldEnumWriteOnly
from ..lib import FieldAsyncReadOnly, FieldAsyncWriteOnly, FieldAsyncReadWrite
from ..lib import FieldEnumAsyncReadOnly, FieldEnumAsyncWriteOnly, FieldEnumAsyncReadWrite
from ..lib import RegReadOnly, RegReadWrite, RegWriteOnly
from ..lib import RegAsyncReadOnly, RegAsyncReadWrite, RegAsyncWriteOnly
from ..lib import AddressMap, AsyncAddressMap
from ..lib import RegFile, AsyncRegFile
from ..lib.memory import BaseMemory
from ..lib import MemoryReadOnly, MemoryReadOnlyLegacy
from ..lib import MemoryWriteOnly, MemoryWriteOnlyLegacy
from ..lib import MemoryReadWrite, MemoryReadWriteLegacy
from ..lib import MemoryAsyncReadOnly, MemoryAsyncReadOnlyLegacy
from ..lib import MemoryAsyncWriteOnly, MemoryAsyncWriteOnlyLegacy
from ..lib import MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy
from ..lib.base_register import BaseReg
from ..lib import Node
from ..lib import Base
from ..lib import SystemRDLEnum
from .utilities import get_field_bitmask_int, get_field_inv_bitmask
from ..sim_lib.simulator import BaseSimulator


class NodeIterators:
    """
    The Node Iterator class is intended to an efficient way to define the iterators of particular
    type that are present on a node
    """
    __slots__ = ['__node_descriptions']

    def __init__(self, *args: Union[str, tuple[str, list[int]]]):
        self.__node_descriptions = args

    @staticmethod
    def __rolled_item(item: Union[str, tuple[str, list[int]]]) -> str:
        if isinstance(item, tuple):
            return item[0]
        return item

    @property
    def rolled(self) -> set[str]:
        """
        name of all the rolled nodes in a set
        """
        return {self.__rolled_item(item) for item in self.__node_descriptions}

    @property
    def unrolled(self) -> set[str]:
        """
        name of all the unrolled nodes in a set
        """
        return_list = []
        for item in self.__node_descriptions:
            if isinstance(item, tuple):
                dim_set = list(product(*[range(dim) for dim in item[1]]))
                for dim in dim_set:
                    # to match the systemrdl compiler dimension put into the inst name of
                    # the array, the name must be item[x][y]
                    dim_str = ''.join([f'[{str(i)}]' for i in dim])
                    return_list.append(f'{item[0]}{dim_str}')
            else:
                return_list.append(item)
        return set(return_list)


class CommonTestBase(unittest.TestCase, ABC):
    """
    Base Test class for the autogenerated register test to be used for the async and
    non-async cases
    """

    @property
    @abstractmethod
    def simulator_instance(self) -> BaseSimulator:
        """
        Simulator configured for the DUT
        """

    @property
    @abstractmethod
    def legacy_block_access(self) -> bool:
        """
        Whether the register model has been configured for legacy block access or not
        """

    # pylint:disable-next=too-many-arguments
    def _single_field_property_test(self, *,
                                    fut: Union[FieldReadWrite,
                                    FieldReadOnly,
                                    FieldWriteOnly,
                                    FieldEnumReadWrite,
                                    FieldEnumReadOnly,
                                    FieldEnumWriteOnly,
                                    FieldAsyncReadOnly,
                                    FieldAsyncWriteOnly,
                                    FieldAsyncReadWrite,
                                    FieldEnumAsyncReadOnly,
                                    FieldEnumAsyncWriteOnly,
                                    FieldEnumAsyncReadWrite],
                                    lsb: int,
                                    msb: int,
                                    low: int,
                                    high: int,
                                    is_volatile: bool,
                                    default: Optional[int],
                                    rdl_name: Optional[str],
                                    rdl_desc: Optional[str],
                                    parent_full_inst_name: str,
                                    inst_name: str
                                    ) -> None:
        self.assertEqual(fut.lsb, lsb)
        self.assertEqual(fut.msb, msb)
        self.assertEqual(fut.low, low)
        self.assertEqual(fut.high, high)
        self.assertEqual(fut.bitmask, get_field_bitmask_int(fut))
        self.assertEqual(fut.inverse_bitmask, get_field_inv_bitmask(fut))
        width = (fut.high - fut.low) + 1
        self.assertEqual(fut.width, width)
        self.assertEqual(fut.max_value, (2 ** width) - 1)
        self.assertEqual(fut.is_volatile, is_volatile)

        if default is None:
            self.assertIsNone(fut.default)
        else:
            if isinstance(fut, (FieldEnumReadWrite,
                                FieldEnumReadOnly,
                                FieldEnumWriteOnly,
                                FieldEnumAsyncReadOnly,
                                FieldEnumAsyncWriteOnly,
                                FieldEnumAsyncReadWrite)):
                # pylint does not realise this is a class being returned rather than an object, so
                # is unhappy with the name
                # pylint:disable-next=invalid-name
                EnumCls = fut.enum_cls
                if default in [item.value for item in fut.enum_cls]:
                    self.assertEqual(fut.default, EnumCls(default))
                else:
                    # this is a special case if the default value for the field does not map
                    # to a legal value of the encoding
                    self.assertIsNone(fut.default)
            else:
                self.assertEqual(fut.default, default)

        self.__single_node_rdl_name_and_desc_test(dut=fut,
                                                  rdl_name=rdl_name,
                                                  rdl_desc=rdl_desc)

        self.__test_node_inst_name(dut=fut,
                                   parent_full_inst_name=parent_full_inst_name,
                                   inst_name=inst_name)

        self.__bad_attribute_test(dut=fut)

    # pylint:disable-next=too-many-arguments
    def _single_register_property_test(self, *,
                                       rut: BaseReg,
                                       address: int,
                                       width: int,
                                       accesswidth: Optional[int],
                                       size: int,
                                       rdl_name: Optional[str],
                                       rdl_desc: Optional[str],
                                       parent_full_inst_name: str,
                                       inst_name: str
                                       ) -> None:
        self.assertEqual(rut.address, address)
        self.assertEqual(rut.width, width)
        if accesswidth is not None:
            self.assertEqual(rut.accesswidth, accesswidth)
        else:
            self.assertEqual(rut.accesswidth, width)
        self.assertEqual(rut.size, size)

        self.__single_node_rdl_name_and_desc_test(dut=rut,
                                                  rdl_name=rdl_name,
                                                  rdl_desc=rdl_desc)

        self.__test_node_inst_name(dut=rut,
                                   parent_full_inst_name=parent_full_inst_name,
                                   inst_name=inst_name)

        self.__bad_attribute_test(dut=rut)

    # pylint:disable-next=too-many-arguments
    def _single_memory_property_test(self, *,
                                     mut: BaseMemory,
                                     address: int,
                                     width: int,
                                     entries: int,
                                     accesswidth: Optional[int],
                                     array_typecode: Optional[str],
                                     size: int,
                                     rdl_name: Optional[str],
                                     rdl_desc: Optional[str],
                                     parent_full_inst_name: str,
                                     inst_name: str
                                     ) -> None:
        self.assertEqual(mut.address, address)
        self.assertEqual(mut.width, width)
        self.assertEqual(mut.entries, entries)
        if accesswidth is not None:
            self.assertEqual(mut.accesswidth, accesswidth)
        else:
            self.assertEqual(mut.accesswidth, width)
        if self.legacy_block_access:
            self.assertEqual(mut.array_typecode, array_typecode)
        else:
            self.assertIsNone(array_typecode)
        self.assertEqual(mut.size, size)

        self.__single_node_rdl_name_and_desc_test(dut=mut,
                                                  rdl_name=rdl_name,
                                                  rdl_desc=rdl_desc)

        self.__test_node_inst_name(dut=mut,
                                   parent_full_inst_name=parent_full_inst_name,
                                   inst_name=inst_name)

        self.__bad_attribute_test(dut=mut)

    # pylint:disable-next=too-many-arguments
    def _single_addrmap_property_test(self, *,
                                      dut: Union[AddressMap, AsyncAddressMap],
                                      size: int,
                                      rdl_name: Optional[str],
                                      rdl_desc: Optional[str],
                                      parent_full_inst_name: Optional[str],
                                      inst_name: str
                                      ) -> None:

        self.assertEqual(dut.size, size)

        self.__single_node_rdl_name_and_desc_test(dut=dut,
                                                  rdl_name=rdl_name,
                                                  rdl_desc=rdl_desc)

        self.__test_node_inst_name(dut=dut,
                                   parent_full_inst_name=parent_full_inst_name,
                                   inst_name=inst_name)

        self.__bad_attribute_test(dut=dut)

    # pylint:disable-next=too-many-arguments
    def _single_regfile_property_test(self, *,
                                      dut: Union[RegFile, AsyncRegFile],
                                      size: int,
                                      rdl_name: Optional[str],
                                      rdl_desc: Optional[str],
                                      parent_full_inst_name: str,
                                      inst_name: str
                                      ) -> None:

        self.assertEqual(dut.size, size)

        self.__single_node_rdl_name_and_desc_test(dut=dut,
                                                  rdl_name=rdl_name,
                                                  rdl_desc=rdl_desc)

        self.__test_node_inst_name(dut=dut,
                                   parent_full_inst_name=parent_full_inst_name,
                                   inst_name=inst_name)

        self.__bad_attribute_test(dut=dut)

    def __single_node_rdl_name_and_desc_test(self,
                                             dut: Base,
                                             rdl_name: Optional[str],
                                             rdl_desc: Optional[str]) -> None:
        """
        Check the SystemRDL Name and Desc properties for a node
        """
        if rdl_name is None:
            self.assertIsNone(dut.rdl_name)
        else:
            self.assertEqual(dut.rdl_name, rdl_name)

        if rdl_desc is None:
            self.assertIsNone(dut.rdl_desc)
        else:
            self.assertEqual(dut.rdl_desc, rdl_desc)

    def __test_node_inst_name(self,
                              dut: Base,
                              parent_full_inst_name: Optional[str],
                              inst_name: str) -> None:
        """
        Test the `inst_name` and `full_inst_name` attributes of a node
        """
        self.assertEqual(dut.inst_name, inst_name)
        if parent_full_inst_name is None:
            # root node (which has no parent)
            self.assertEqual(dut.full_inst_name, inst_name)
        else:
            full_inst_name = parent_full_inst_name + '.' + inst_name
            self.assertEqual(dut.full_inst_name, full_inst_name)

    def __bad_attribute_test(self, dut: Base) -> None:
        """
        Check that adding an attribute fails, the __slots__ should prevent this

        The attribute name: cppkbrgmgeloagvfgjjeiiushygirh was randomly generated to be unlikely to
        every be a attribute name
        """
        with self.assertRaises(AttributeError):
            dut.cppkbrgmgeloagvfgjjeiiushygirh = 1  # type: ignore[attr-defined,union-attr]

    def __test_name_map(self, dut: Node, child_names: set[str]) -> None:
        """
        Test that the get_child_by_system_rdl_name and systemrdl_python_child_name_map are
        populated correctly
        """
        self.assertCountEqual(dut.systemrdl_python_child_name_map, child_names)
        self.assertEqual(set(dut.systemrdl_python_child_name_map.keys()), child_names)
        for child_name in child_names:
            self.assertEqual(dut.get_child_by_system_rdl_name(child_name).inst_name, child_name)

    def _test_field_iterators(self, *,
                              rut: Union[RegReadOnly,
                              RegReadWrite,
                              RegWriteOnly,
                              RegAsyncReadOnly,
                              RegAsyncReadWrite,
                              RegAsyncWriteOnly],
                              has_sw_readable: bool,
                              has_sw_writable: bool,
                              readable_fields: set[str],
                              writeable_fields: set[str]) -> None:
        if has_sw_readable:
            if not isinstance(rut, (RegReadOnly,
                                    RegReadWrite,
                                    RegAsyncReadOnly,
                                    RegAsyncReadWrite,
                                    )):
                raise TypeError(f'Register was expected to readable, got {type(rut)}')

            child_readable_field_names = {field.inst_name for field in rut.readable_fields}

            self.assertEqual(readable_fields, child_readable_field_names)
        else:
            self.assertFalse(hasattr(rut, 'readable_fields'))
            # check the readable_fields is empty
            self.assertFalse(readable_fields)

        if has_sw_writable:
            if not isinstance(rut, (RegWriteOnly,
                                    RegReadWrite,
                                    RegAsyncWriteOnly,
                                    RegAsyncReadWrite,
                                    )):
                raise TypeError(f'Register was expected to writable, got {type(rut)}')

            child_writeable_fields_names = {field.inst_name for field in rut.writable_fields}

            self.assertEqual(writeable_fields, child_writeable_fields_names)
        else:
            self.assertFalse(hasattr(rut, 'writeable_fields'))
            # check the writeable_fields is empty
            self.assertFalse(writeable_fields)

        child_field_names = {field.inst_name for field in rut.fields}
        self.assertEqual(readable_fields | writeable_fields, child_field_names)

        # Check the child name map
        self.__test_name_map(dut=rut, child_names=readable_fields | writeable_fields)

    def _test_register_iterators(self,
                                 dut: Union[AddressMap, AsyncAddressMap, RegFile, AsyncRegFile,
                                 MemoryReadOnly, MemoryReadOnlyLegacy,
                                 MemoryWriteOnly, MemoryWriteOnlyLegacy,
                                 MemoryReadWrite, MemoryReadWriteLegacy,
                                 MemoryAsyncReadOnly, MemoryAsyncReadOnlyLegacy,
                                 MemoryAsyncWriteOnly, MemoryAsyncWriteOnlyLegacy,
                                 MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy],
                                 readable_registers: NodeIterators,
                                 writeable_registers: NodeIterators) -> None:

        if isinstance(dut, (AddressMap, AsyncAddressMap, RegFile, AsyncRegFile,
                            MemoryReadOnly, MemoryReadOnlyLegacy,
                            MemoryReadWrite, MemoryReadWriteLegacy,
                            MemoryAsyncReadOnly, MemoryAsyncReadOnlyLegacy,
                            MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy)):
            child_readable_reg_names = {reg.inst_name for reg in
                                        dut.get_readable_registers(unroll=True)}
            self.assertEqual(readable_registers.unrolled, child_readable_reg_names)
            child_readable_reg_names = {reg.inst_name for reg in
                                        dut.get_readable_registers(unroll=False)}
            self.assertEqual(readable_registers.rolled, child_readable_reg_names)
        else:
            self.assertFalse(hasattr(dut, 'get_readable_registers'))

        if isinstance(dut, (AddressMap, AsyncAddressMap, RegFile, AsyncRegFile,
                            MemoryWriteOnly, MemoryWriteOnlyLegacy,
                            MemoryReadWrite, MemoryReadWriteLegacy,
                            MemoryAsyncWriteOnly, MemoryAsyncWriteOnlyLegacy,
                            MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy)):
            child_writable_reg_names = {reg.inst_name for reg in
                                        dut.get_writable_registers(unroll=True)}
            self.assertEqual(writeable_registers.unrolled, child_writable_reg_names)
            child_writable_reg_names = {reg.inst_name for reg in
                                        dut.get_writable_registers(unroll=False)}
            self.assertEqual(writeable_registers.rolled, child_writable_reg_names)
        else:
            self.assertFalse(hasattr(dut, 'get_writable_registers'))

        child_reg_names = {field.inst_name for field in dut.get_registers(unroll=True)}
        self.assertEqual(readable_registers.unrolled | writeable_registers.unrolled,
                         child_reg_names)
        child_reg_names = {field.inst_name for field in dut.get_registers(unroll=False)}
        self.assertEqual(readable_registers.rolled | writeable_registers.rolled,
                         child_reg_names)

        # The register file and addrmap have other items in their child map so it has to be
        # tested at the next level up, however, a memory only has child registers
        if isinstance(dut, (MemoryReadOnly, MemoryReadOnlyLegacy,
                            MemoryWriteOnly, MemoryWriteOnlyLegacy,
                            MemoryReadWrite, MemoryReadWriteLegacy,
                            MemoryAsyncReadOnly, MemoryAsyncReadOnlyLegacy,
                            MemoryAsyncWriteOnly, MemoryAsyncWriteOnlyLegacy,
                            MemoryAsyncReadWrite, MemoryAsyncReadWriteLegacy)):
            # Check the child name map
            self.__test_name_map(dut=dut,
                                 child_names=readable_registers.rolled |
                                             writeable_registers.rolled)

    def _test_memory_iterators(self,
                               dut: Union[AddressMap, AsyncAddressMap],
                               memories: NodeIterators) -> None:
        child_mem_names = {reg.inst_name for reg in dut.get_memories(unroll=True)}
        self.assertEqual(memories.unrolled, child_mem_names)
        child_mem_names = {reg.inst_name for reg in dut.get_memories(unroll=False)}
        self.assertEqual(memories.rolled, child_mem_names)

    def __test_section_iterators(self,
                                 dut: Union[AddressMap, AsyncAddressMap, RegFile, AsyncRegFile],
                                 sections: NodeIterators) -> None:
        child_section_names = {reg.inst_name for reg in dut.get_sections(unroll=True)}
        self.assertEqual(sections.unrolled, child_section_names)
        child_section_names = {reg.inst_name for reg in dut.get_sections(unroll=False)}
        self.assertEqual(sections.rolled, child_section_names)

    def _test_addrmap_iterators(self, *,
                                dut: Union[AddressMap, AsyncAddressMap],
                                memories: NodeIterators,
                                sections: NodeIterators,
                                readable_registers: NodeIterators,
                                writeable_registers: NodeIterators) -> None:
        self._test_register_iterators(dut=dut,
                                      readable_registers=readable_registers,
                                      writeable_registers=writeable_registers)
        self._test_memory_iterators(dut=dut,
                                    memories=memories)
        self.__test_section_iterators(dut=dut,
                                      sections=sections)

        # Check the child name map
        self.__test_name_map(dut=dut, child_names=memories.rolled | readable_registers.rolled |
                                                  writeable_registers.rolled | sections.rolled)

    def _test_regfile_iterators(self,
                                dut: Union[RegFile, AsyncRegFile],
                                sections: NodeIterators,
                                readable_registers: NodeIterators,
                                writeable_registers: NodeIterators) -> None:
        self._test_register_iterators(dut=dut,
                                      readable_registers=readable_registers,
                                      writeable_registers=writeable_registers)
        self.__test_section_iterators(dut=dut,
                                      sections=sections)
        self.assertFalse(hasattr(dut, 'get_memories'))

        # Check the child name map
        self.__test_name_map(dut=dut, child_names=readable_registers.rolled |
                                                  writeable_registers.rolled |
                                                  sections.rolled)

    def _full_to_reduced_enum_conversion(
            self,
            full_enum_def: dict[str, tuple[int, Optional[str], Optional[str]]]) -> dict[str, int]:
        return {key:value[0] for key,value in full_enum_def.items() }

    def _test_enum_def_rdl_name_desc_(
            self,
            fut: Union[FieldEnumReadOnly, FieldEnumReadOnly, FieldEnumWriteOnly,
                       FieldEnumAsyncReadOnly, FieldEnumAsyncReadOnly, FieldEnumAsyncWriteOnly],
            full_enum_def: dict[str, tuple[int, Optional[str], Optional[str]]]) -> None:
        """
        Check that the enumeration in the field matches the enumeration specifed in the
        systemRDL

        Args:
            fut: field node
            full_enum_def: definition of the enumeration a dictionary, with the of the
                           entry as a key and the value a tuple that has:
                           1. int value encoding the enumeration
                           2. system RDL name (or None)
                           3. system RDL name (or None)

        Returns: None

        """

        # pylint does not realise this is a class being returned rather than an object, so
        # is unhappy with the name
        # pylint:disable-next=invalid-name
        EnumCls = fut.enum_cls
        for name, value in full_enum_def.items():
            enum_inst = EnumCls[name]
            self.assertEqual(enum_inst.value, value[0])

            if issubclass(EnumCls, SystemRDLEnum):
                if value[1] is None:
                    self.assertIsNone(enum_inst.rdl_name)
                else:
                    self.assertEqual(enum_inst.rdl_name, value[1])

                if value[2] is None:
                    self.assertIsNone(enum_inst.rdl_desc)
                else:
                    self.assertEqual(enum_inst.rdl_desc, value[2])

            else:
                # if using a legacy enumeration, then the systemRDL name and desc must be None
                # as the legacy enum did not support these
                self.assertIsNone(value[1])
                self.assertIsNone(value[2])
                self.assertFalse(hasattr(enum_inst, 'rdl_name'))
                self.assertFalse(hasattr(enum_inst, 'rdl_desc'))
