from __future__ import annotations

import multiprocessing as mp
from pathlib import Path
from typing import Any
from unittest.mock import Mock, patch

import pytest

from kreuzberg import ExtractionConfig, ExtractionResult
from kreuzberg.extraction import (
    batch_extract_bytes,
    batch_extract_bytes_sync,
    batch_extract_file,
    batch_extract_file_sync,
)


@pytest.mark.anyio
async def test_batch_extract_file_empty_list() -> None:
    """Test batch_extract_file with empty file list."""
    result = await batch_extract_file([])
    assert result == []


@pytest.mark.anyio
async def test_batch_extract_file_single_file(tmp_path: Path) -> None:
    """Test batch_extract_file with single file."""
    test_file = tmp_path / "test.txt"
    test_file.write_text("Test content")

    result = await batch_extract_file([test_file])

    assert len(result) == 1
    assert result[0].content == "Test content"
    assert result[0].mime_type == "text/plain"


@pytest.mark.anyio
async def test_batch_extract_file_multiple_files(tmp_path: Path) -> None:
    """Test batch_extract_file with multiple files."""
    file1 = tmp_path / "test1.txt"
    file2 = tmp_path / "test2.txt"
    file1.write_text("Content 1")
    file2.write_text("Content 2")

    config = ExtractionConfig(use_cache=False)
    result = await batch_extract_file([file1, file2], config)

    assert len(result) == 2
    assert result[0].content == "Content 1"
    assert result[1].content == "Content 2"


@pytest.mark.anyio
async def test_batch_extract_file_with_error() -> None:
    """Test batch_extract_file handles extraction errors gracefully."""
    nonexistent_file = Path("/nonexistent/file.txt")

    result = await batch_extract_file([nonexistent_file])

    assert len(result) == 1
    assert "Error:" in result[0].content
    assert "ValidationError" in result[0].content
    assert result[0].mime_type == "text/plain"
    assert "error" in result[0].metadata
    assert "error_context" in result[0].metadata


@pytest.mark.anyio
async def test_batch_extract_file_mixed_success_and_error(tmp_path: Path) -> None:
    """Test batch_extract_file with mix of valid and invalid files."""
    valid_file = tmp_path / "valid.txt"
    valid_file.write_text("Valid content")
    invalid_file = Path("/nonexistent/invalid.txt")

    result = await batch_extract_file([valid_file, invalid_file])

    assert len(result) == 2
    # First file should succeed
    assert result[0].content == "Valid content"
    # Second file should fail gracefully
    assert "Error:" in result[1].content
    assert "error" in result[1].metadata


@pytest.mark.anyio
async def test_batch_extract_file_concurrency_limits() -> None:
    """Test batch_extract_file respects concurrency limits."""
    # Create many files to test concurrency
    files = [f"/tmp/file_{i}.txt" for i in range(50)]

    with patch("kreuzberg.extraction.extract_file") as mock_extract:
        # Mock extract_file to track concurrent calls
        call_count = 0
        max_concurrent = 0
        current_concurrent = 0

        async def mock_extract_func(*args: Any, **kwargs: Any) -> ExtractionResult:
            nonlocal call_count, max_concurrent, current_concurrent
            current_concurrent += 1
            max_concurrent = max(max_concurrent, current_concurrent)
            call_count += 1

            # Simulate some work
            import asyncio

            await asyncio.sleep(0.01)

            current_concurrent -= 1
            return ExtractionResult(content=f"Content {call_count}", mime_type="text/plain", metadata={}, chunks=[])

        mock_extract.side_effect = mock_extract_func

        result = await batch_extract_file(files)

    assert len(result) == 50
    assert call_count == 50
    # Should not exceed a reasonable limit - this can vary by system
    # so let's be more flexible with the assertion
    assert max_concurrent > 0
    assert max_concurrent <= 50  # Should not process more than we requested


@pytest.mark.anyio
async def test_batch_extract_bytes_empty_list() -> None:
    """Test batch_extract_bytes with empty content list."""
    result = await batch_extract_bytes([])
    assert result == []


@pytest.mark.anyio
async def test_batch_extract_bytes_single_content() -> None:
    """Test batch_extract_bytes with single content."""
    content = (b"Test content", "text/plain")

    result = await batch_extract_bytes([content])

    assert len(result) == 1
    assert result[0].content == "Test content"
    assert result[0].mime_type == "text/plain"


@pytest.mark.anyio
async def test_batch_extract_bytes_multiple_contents() -> None:
    """Test batch_extract_bytes with multiple contents."""
    contents = [(b"Content 1", "text/plain"), (b"Content 2", "text/plain"), (b"Content 3", "text/plain")]

    result = await batch_extract_bytes(contents)

    assert len(result) == 3
    assert result[0].content == "Content 1"
    assert result[1].content == "Content 2"
    assert result[2].content == "Content 3"


@pytest.mark.anyio
async def test_batch_extract_bytes_with_error() -> None:
    """Test batch_extract_bytes handles extraction errors gracefully."""
    # Cause an error by mocking extract_bytes to raise an exception
    with patch("kreuzberg.extraction.extract_bytes") as mock_extract:
        mock_extract.side_effect = ValueError("Test error")

        contents = [(b"Test content", "text/plain")]
        result = await batch_extract_bytes(contents)

    assert len(result) == 1
    assert "Error:" in result[0].content
    assert "ValueError" in result[0].content
    assert result[0].mime_type == "text/plain"
    assert "error" in result[0].metadata
    assert "error_context" in result[0].metadata


def test_batch_extract_file_sync_empty_list() -> None:
    """Test batch_extract_file_sync with empty file list."""
    result = batch_extract_file_sync([])
    assert result == []


def test_batch_extract_file_sync_single_file(tmp_path: Path) -> None:
    """Test batch_extract_file_sync with single file."""
    test_file = tmp_path / "test.txt"
    test_file.write_text("Test content")

    result = batch_extract_file_sync([test_file])

    assert len(result) == 1
    assert result[0].content == "Test content"
    assert result[0].mime_type == "text/plain"


def test_batch_extract_file_sync_single_file_no_parallelism(tmp_path: Path) -> None:
    """Test batch_extract_file_sync uses single-threaded approach for one file."""
    test_file = tmp_path / "test.txt"
    test_file.write_text("Test content")

    # Should not use ThreadPoolExecutor for single file
    with patch("kreuzberg.extraction.ThreadPoolExecutor") as mock_executor:
        result = batch_extract_file_sync([test_file])

    mock_executor.assert_not_called()
    assert len(result) == 1
    assert result[0].content == "Test content"


def test_batch_extract_file_sync_multiple_files(tmp_path: Path) -> None:
    """Test batch_extract_file_sync with multiple files uses parallelism."""
    file1 = tmp_path / "test1.txt"
    file2 = tmp_path / "test2.txt"
    file1.write_text("Content 1")
    file2.write_text("Content 2")

    config = ExtractionConfig(use_cache=False)
    result = batch_extract_file_sync([file1, file2], config)

    assert len(result) == 2
    assert result[0].content == "Content 1"
    assert result[1].content == "Content 2"


def test_batch_extract_file_sync_with_error() -> None:
    """Test batch_extract_file_sync handles extraction errors gracefully."""
    nonexistent_files = [Path("/nonexistent/file1.txt"), Path("/nonexistent/file2.txt")]

    result = batch_extract_file_sync(nonexistent_files)

    assert len(result) == 2
    for res in result:
        assert "Error:" in res.content
        assert "ValidationError" in res.content
        assert res.mime_type == "text/plain"
        assert "error" in res.metadata


def test_batch_extract_file_sync_worker_count() -> None:
    """Test batch_extract_file_sync uses appropriate worker count."""
    files = [f"/tmp/file_{i}.txt" for i in range(20)]

    with (
        patch("kreuzberg.extraction.ThreadPoolExecutor") as mock_executor,
        patch("kreuzberg.extraction.extract_file_sync"),
    ):
        # Mock ThreadPoolExecutor context manager
        mock_executor_instance = Mock()
        mock_executor.return_value.__enter__ = Mock(return_value=mock_executor_instance)
        mock_executor.return_value.__exit__ = Mock(return_value=None)

        # Mock submit method to return futures
        mock_future = Mock()
        mock_future.result.return_value = (
            0,
            ExtractionResult(content="Test", mime_type="text/plain", metadata={}, chunks=[]),
        )
        mock_executor_instance.submit.return_value = mock_future

        # Mock as_completed to return our mock future
        with patch("kreuzberg.extraction.as_completed", return_value=[mock_future]):
            batch_extract_file_sync(files)

    # Should create executor with min(file_count, cpu_count) workers
    expected_workers = min(len(files), mp.cpu_count())
    mock_executor.assert_called_once_with(max_workers=expected_workers)


def test_batch_extract_bytes_sync_empty_list() -> None:
    """Test batch_extract_bytes_sync with empty content list."""
    result = batch_extract_bytes_sync([])
    assert result == []


def test_batch_extract_bytes_sync_single_content() -> None:
    """Test batch_extract_bytes_sync with single content."""
    content = (b"Test content", "text/plain")

    result = batch_extract_bytes_sync([content])

    assert len(result) == 1
    assert result[0].content == "Test content"
    assert result[0].mime_type == "text/plain"


def test_batch_extract_bytes_sync_single_content_no_parallelism() -> None:
    """Test batch_extract_bytes_sync uses single-threaded approach for one content."""
    content = (b"Test content", "text/plain")

    # Should not use ThreadPoolExecutor for single content
    with patch("kreuzberg.extraction.ThreadPoolExecutor") as mock_executor:
        result = batch_extract_bytes_sync([content])

    mock_executor.assert_not_called()
    assert len(result) == 1
    assert result[0].content == "Test content"


def test_batch_extract_bytes_sync_multiple_contents() -> None:
    """Test batch_extract_bytes_sync with multiple contents uses parallelism."""
    contents = [(b"Content 1", "text/plain"), (b"Content 2", "text/plain"), (b"Content 3", "text/plain")]

    result = batch_extract_bytes_sync(contents)

    assert len(result) == 3
    assert result[0].content == "Content 1"
    assert result[1].content == "Content 2"
    assert result[2].content == "Content 3"


def test_batch_extract_bytes_sync_with_error() -> None:
    """Test batch_extract_bytes_sync handles extraction errors gracefully."""
    # Create invalid content that will cause an error
    with patch("kreuzberg.extraction.extract_bytes_sync") as mock_extract:
        mock_extract.side_effect = ValueError("Test error")

        contents = [(b"Test content", "text/plain"), (b"Another content", "text/plain")]
        result = batch_extract_bytes_sync(contents)

    assert len(result) == 2
    for res in result:
        assert "Error:" in res.content
        assert "ValueError" in res.content
        assert res.mime_type == "text/plain"
        assert "error" in res.metadata


def test_batch_extract_bytes_sync_worker_count() -> None:
    """Test batch_extract_bytes_sync uses appropriate worker count."""
    contents = [(b"Content", "text/plain") for _ in range(15)]

    with patch("kreuzberg.extraction.ThreadPoolExecutor") as mock_executor:
        # Mock ThreadPoolExecutor context manager
        mock_executor_instance = Mock()
        mock_executor.return_value.__enter__ = Mock(return_value=mock_executor_instance)
        mock_executor.return_value.__exit__ = Mock(return_value=None)

        # Mock submit method to return futures
        mock_future = Mock()
        mock_future.result.return_value = (
            0,
            ExtractionResult(content="Test", mime_type="text/plain", metadata={}, chunks=[]),
        )
        mock_executor_instance.submit.return_value = mock_future

        # Mock as_completed to return our mock future
        with patch("kreuzberg.extraction.as_completed", return_value=[mock_future]):
            batch_extract_bytes_sync(contents)

    # Should create executor with min(content_count, cpu_count) workers
    expected_workers = min(len(contents), mp.cpu_count())
    mock_executor.assert_called_once_with(max_workers=expected_workers)


@pytest.mark.anyio
async def test_batch_extract_bytes_error_context_includes_index() -> None:
    """Test batch_extract_bytes includes index in error context."""
    with patch("kreuzberg.extraction.extract_bytes") as mock_extract:
        mock_extract.side_effect = RuntimeError("Test extraction error")

        contents = [(b"Content 1", "text/plain"), (b"Content 2", "text/plain")]
        result = await batch_extract_bytes(contents)

    assert len(result) == 2

    # Check first error has index 0
    assert result[0].metadata["error_context"]["index"] == 0
    assert result[0].metadata["error_context"]["operation"] == "batch_extract_bytes"
    assert result[0].metadata["error_context"]["mime_type"] == "text/plain"
    assert result[0].metadata["error_context"]["content_size"] == 9
    # Check second error has index 1
    assert result[1].metadata["error_context"]["index"] == 1
    assert result[1].metadata["error_context"]["operation"] == "batch_extract_bytes"


@pytest.mark.anyio
async def test_batch_extract_file_error_context_includes_index() -> None:
    """Test batch_extract_file includes index in error context."""
    nonexistent_files = [Path("/nonexistent/file1.txt"), Path("/nonexistent/file2.txt")]

    result = await batch_extract_file(nonexistent_files)

    assert len(result) == 2

    # Check first error has index 0
    assert result[0].metadata["error_context"]["index"] == 0
    assert result[0].metadata["error_context"]["operation"] == "batch_extract_file"
    # File path might be stored differently depending on implementation
    # File path might be stored differently depending on implementation
    assert "file_path" in result[0].metadata["error_context"] or str(nonexistent_files[0]) in str(
        result[0].metadata["error_context"]
    )
    # Check second error has index 1
    assert result[1].metadata["error_context"]["index"] == 1
    assert result[1].metadata["error_context"]["operation"] == "batch_extract_file"
    # File path might be stored differently depending on implementation
    # File path might be stored differently depending on implementation
    assert "file_path" in result[1].metadata["error_context"] or str(nonexistent_files[1]) in str(
        result[1].metadata["error_context"]
    )


def test_batch_extract_bytes_sync_error_context_preserves_ordering() -> None:
    """Test batch_extract_bytes_sync preserves result ordering even with errors."""
    contents = [(b"Content 1", "text/plain"), (b"Content 2", "text/plain"), (b"Content 3", "text/plain")]

    with patch("kreuzberg.extraction.extract_bytes_sync") as mock_extract:

        def side_effect(content: Any, mime_type: Any, config: Any = None) -> ExtractionResult:
            content_str = content.decode("utf-8")
            if "Content 2" in content_str:
                raise ValueError("Error on content 2")
            return ExtractionResult(content=content_str, mime_type=mime_type, metadata={}, chunks=[])

        mock_extract.side_effect = side_effect
        result = batch_extract_bytes_sync(contents)

    assert len(result) == 3
    assert result[0].content == "Content 1"  # Success
    assert "Error:" in result[1].content
    assert "ValueError" in result[1].content
    assert result[2].content == "Content 3"  # Success


def test_batch_extract_file_sync_error_context_preserves_ordering() -> None:
    """Test batch_extract_file_sync preserves result ordering even with errors."""
    files = [
        Path("/file1.txt"),
        Path("/nonexistent.txt"),  # This will cause an error
        Path("/file3.txt"),
    ]

    with patch("kreuzberg.extraction.extract_file_sync") as mock_extract:

        def side_effect(file_path: Any, mime_type: Any = None, config: Any = None) -> ExtractionResult:
            if "nonexistent" in str(file_path):
                raise ValueError("File not found")
            return ExtractionResult(
                content=f"Content from {file_path.name}", mime_type="text/plain", metadata={}, chunks=[]
            )

        mock_extract.side_effect = side_effect
        result = batch_extract_file_sync(files)

    assert len(result) == 3
    assert "file1.txt" in result[0].content  # Success
    assert "Error:" in result[1].content
    assert "ValueError" in result[1].content
    assert "file3.txt" in result[2].content  # Success
