from __future__ import annotations

from pathlib import Path
from unittest.mock import AsyncMock, Mock, patch

import pytest

from kreuzberg import ExtractionConfig, ExtractionResult, LanguageDetectionConfig
from kreuzberg.exceptions import ValidationError
from kreuzberg.extraction import (
    DEFAULT_CONFIG,
    _handle_chunk_content,
    _validate_and_post_process_async,
    _validate_and_post_process_helper,
    _validate_and_post_process_sync,
    extract_bytes,
    extract_bytes_sync,
    extract_file,
    extract_file_sync,
)


def test_default_config_is_extraction_config() -> None:
    """Test that DEFAULT_CONFIG is an ExtractionConfig instance."""
    assert isinstance(DEFAULT_CONFIG, ExtractionConfig)
    assert DEFAULT_CONFIG.use_cache is True
    assert DEFAULT_CONFIG.chunk_content is False


@pytest.mark.anyio
async def test_extract_bytes_with_valid_mime_type() -> None:
    """Test extract_bytes with valid content and mime type."""
    content = b"Hello, World!"
    mime_type = "text/plain"

    result = await extract_bytes(content, mime_type)

    assert isinstance(result, ExtractionResult)
    assert result.content == "Hello, World!"
    assert result.mime_type == "text/plain"
    assert result.chunks == []
    assert result.metadata == {}


@pytest.mark.anyio
async def test_extract_bytes_with_unknown_mime_type() -> None:
    """Test extract_bytes falls back to safe_decode for unknown mime types."""
    content = b"Unknown content type"
    mime_type = "application/unknown"

    # Mock the validation to allow unknown mime type for testing
    with patch("kreuzberg.extraction.validate_mime_type", return_value=mime_type):
        result = await extract_bytes(content, mime_type)

    assert result.content == "Unknown content type"
    assert result.mime_type == "application/unknown"


def test_extract_bytes_sync_with_valid_mime_type() -> None:
    """Test extract_bytes_sync with valid content and mime type."""
    content = b"Hello, World!"
    mime_type = "text/plain"

    result = extract_bytes_sync(content, mime_type)

    assert isinstance(result, ExtractionResult)
    assert result.content == "Hello, World!"
    assert result.mime_type == "text/plain"


def test_extract_bytes_sync_with_unknown_mime_type() -> None:
    """Test extract_bytes_sync falls back to safe_decode for unknown mime types."""
    content = b"Unknown content type"
    mime_type = "application/unknown"

    # Mock the validation to allow unknown mime type for testing
    with patch("kreuzberg.extraction.validate_mime_type", return_value=mime_type):
        result = extract_bytes_sync(content, mime_type)

    assert result.content == "Unknown content type"
    assert result.mime_type == "application/unknown"


@pytest.mark.anyio
async def test_extract_file_nonexistent_file() -> None:
    """Test extract_file raises ValidationError for nonexistent file."""
    nonexistent_path = "/nonexistent/file.txt"

    with pytest.raises(ValidationError, match="The file does not exist"):
        await extract_file(nonexistent_path)


def test_extract_file_sync_nonexistent_file() -> None:
    """Test extract_file_sync raises ValidationError for nonexistent file."""
    nonexistent_path = "/nonexistent/file.txt"

    with pytest.raises(ValidationError, match="The file does not exist"):
        extract_file_sync(nonexistent_path)


@pytest.mark.anyio
async def test_extract_file_with_cache_disabled(tmp_path: Path) -> None:
    """Test extract_file with caching disabled."""
    test_file = tmp_path / "test.txt"
    test_file.write_text("Test content")

    config = ExtractionConfig(use_cache=False)
    result = await extract_file(test_file, config=config)

    assert result.content == "Test content"
    assert result.mime_type == "text/plain"


def test_extract_file_sync_with_cache_disabled(tmp_path: Path) -> None:
    """Test extract_file_sync with caching disabled."""
    test_file = tmp_path / "test.txt"
    test_file.write_text("Test content")

    config = ExtractionConfig(use_cache=False)
    result = extract_file_sync(test_file, config=config)

    assert result.content == "Test content"
    assert result.mime_type == "text/plain"


@pytest.mark.anyio
async def test_extract_file_with_unknown_mime_type(tmp_path: Path) -> None:
    """Test extract_file with unknown mime type falls back to reading as text."""
    test_file = tmp_path / "test.unknown"
    test_file.write_text("Unknown file type content")

    # Mock the validation to allow unknown mime type for testing
    with patch("kreuzberg.extraction.validate_mime_type", return_value="application/unknown"):
        result = await extract_file(test_file, mime_type="application/unknown")

    assert result.content == "Unknown file type content"
    assert result.mime_type == "application/unknown"


def test_extract_file_sync_with_unknown_mime_type(tmp_path: Path) -> None:
    """Test extract_file_sync with unknown mime type falls back to reading as text."""
    test_file = tmp_path / "test.unknown"
    test_file.write_text("Unknown file type content")

    # Mock the validation to allow unknown mime type for testing
    with patch("kreuzberg.extraction.validate_mime_type", return_value="application/unknown"):
        result = extract_file_sync(test_file, mime_type="application/unknown")

    assert result.content == "Unknown file type content"
    assert result.mime_type == "application/unknown"


def test_handle_chunk_content() -> None:
    """Test _handle_chunk_content function."""
    content = "This is a long text that should be chunked into smaller pieces for processing."
    config = ExtractionConfig(max_chars=20, max_overlap=5)

    chunks = _handle_chunk_content(
        mime_type="text/plain",
        config=config,
        content=content,
    )

    assert chunks is not None
    assert len(chunks) > 1  # Should be chunked


@pytest.mark.anyio
async def test_extract_bytes_with_chunking() -> None:
    """Test extract_bytes with chunking enabled."""
    content = b"This is a long text that should be chunked into smaller pieces for processing."
    mime_type = "text/plain"
    config = ExtractionConfig(chunk_content=True, max_chars=20, max_overlap=5)

    result = await extract_bytes(content, mime_type, config)

    assert result.chunks is not None
    assert len(result.chunks) > 1


def test_extract_bytes_sync_with_chunking() -> None:
    """Test extract_bytes_sync with chunking enabled."""
    content = b"This is a long text that should be chunked into smaller pieces for processing."
    mime_type = "text/plain"
    config = ExtractionConfig(chunk_content=True, max_chars=20, max_overlap=5)

    result = extract_bytes_sync(content, mime_type, config)

    assert result.chunks is not None
    assert len(result.chunks) > 1


@pytest.mark.anyio
async def test_extract_bytes_with_language_detection() -> None:
    """Test extract_bytes with language detection enabled."""
    content = b"This is some English text for language detection."
    mime_type = "text/plain"
    config = ExtractionConfig(auto_detect_language=True, language_detection_config=LanguageDetectionConfig())

    with patch("kreuzberg.extraction.detect_languages") as mock_detect:
        mock_detect.return_value = ["en"]
        result = await extract_bytes(content, mime_type, config)

    assert result.detected_languages == ["en"]
    mock_detect.assert_called_once()


def test_extract_bytes_sync_with_language_detection() -> None:
    """Test extract_bytes_sync with language detection enabled."""
    content = b"This is some English text for language detection."
    mime_type = "text/plain"
    config = ExtractionConfig(auto_detect_language=True, language_detection_config=LanguageDetectionConfig())

    with patch("kreuzberg.extraction.detect_languages") as mock_detect:
        mock_detect.return_value = ["en"]
        result = extract_bytes_sync(content, mime_type, config)

    assert result.detected_languages == ["en"]
    mock_detect.assert_called_once()


@pytest.mark.anyio
async def test_extract_bytes_with_entity_extraction_success() -> None:
    """Test extract_bytes with successful entity extraction."""
    content = b"John works at Apple Inc. in California."
    mime_type = "text/plain"
    config = ExtractionConfig(extract_entities=True)

    with patch("kreuzberg.extraction.extract_entities") as mock_extract:
        mock_entities = [{"text": "John", "label": "PERSON"}]
        mock_extract.return_value = mock_entities
        result = await extract_bytes(content, mime_type, config)

    assert result.entities == mock_entities  # type: ignore[comparison-overlap]
    mock_extract.assert_called_once()


@pytest.mark.anyio
async def test_extract_bytes_with_entity_extraction_runtime_error() -> None:
    """Test extract_bytes handles RuntimeError in entity extraction gracefully."""
    content = b"Some text for entity extraction."
    mime_type = "text/plain"
    config = ExtractionConfig(extract_entities=True)

    with patch("kreuzberg.extraction.extract_entities") as mock_extract:
        mock_extract.side_effect = RuntimeError("Entity extraction failed")
        result = await extract_bytes(content, mime_type, config)

    assert result.entities is None
    mock_extract.assert_called_once()


def test_extract_bytes_sync_with_entity_extraction_runtime_error() -> None:
    """Test extract_bytes_sync handles RuntimeError in entity extraction gracefully."""
    content = b"Some text for entity extraction."
    mime_type = "text/plain"
    config = ExtractionConfig(extract_entities=True)

    with patch("kreuzberg.extraction.extract_entities") as mock_extract:
        mock_extract.side_effect = RuntimeError("Entity extraction failed")
        result = extract_bytes_sync(content, mime_type, config)

    assert result.entities is None
    mock_extract.assert_called_once()


@pytest.mark.anyio
async def test_extract_bytes_with_keyword_extraction_success() -> None:
    """Test extract_bytes with successful keyword extraction."""
    content = b"Machine learning and artificial intelligence are important technologies."
    mime_type = "text/plain"
    config = ExtractionConfig(extract_keywords=True, keyword_count=5)

    with patch("kreuzberg.extraction.extract_keywords") as mock_extract:
        mock_keywords = ["machine", "learning", "artificial", "intelligence", "technologies"]
        mock_extract.return_value = mock_keywords
        result = await extract_bytes(content, mime_type, config)

    assert result.keywords == mock_keywords  # type: ignore[comparison-overlap]
    mock_extract.assert_called_once_with(result.content, keyword_count=5)


@pytest.mark.anyio
async def test_extract_bytes_with_keyword_extraction_runtime_error() -> None:
    """Test extract_bytes handles RuntimeError in keyword extraction gracefully."""
    content = b"Some text for keyword extraction."
    mime_type = "text/plain"
    config = ExtractionConfig(extract_keywords=True)

    with patch("kreuzberg.extraction.extract_keywords") as mock_extract:
        mock_extract.side_effect = RuntimeError("Keyword extraction failed")
        result = await extract_bytes(content, mime_type, config)

    assert result.keywords is None
    mock_extract.assert_called_once()


def test_extract_bytes_sync_with_keyword_extraction_runtime_error() -> None:
    """Test extract_bytes_sync handles RuntimeError in keyword extraction gracefully."""
    content = b"Some text for keyword extraction."
    mime_type = "text/plain"
    config = ExtractionConfig(extract_keywords=True)

    with patch("kreuzberg.extraction.extract_keywords") as mock_extract:
        mock_extract.side_effect = RuntimeError("Keyword extraction failed")
        result = extract_bytes_sync(content, mime_type, config)

    assert result.keywords is None
    mock_extract.assert_called_once()


@pytest.mark.anyio
async def test_extract_bytes_with_auto_detect_document_type() -> None:
    """Test extract_bytes with auto document type detection enabled."""
    content = b"This is a document with specific formatting."
    mime_type = "text/plain"
    config = ExtractionConfig(auto_detect_document_type=True)

    with patch("kreuzberg.extraction.auto_detect_document_type") as mock_detect:
        mock_result = ExtractionResult(
            content="This is a document with specific formatting.",
            mime_type="text/plain",
            metadata={"document_type": "article"},  # type: ignore[typeddict-unknown-key]
            chunks=[],
        )
        mock_detect.return_value = mock_result
        result = await extract_bytes(content, mime_type, config)

    assert result.metadata.get("document_type") == "article"
    mock_detect.assert_called_once()


@pytest.mark.anyio
async def test_validate_and_post_process_async_with_validators() -> None:
    """Test _validate_and_post_process_async with custom validators."""
    result = ExtractionResult(content="Test content", mime_type="text/plain", metadata={}, chunks=[])

    async_validator = AsyncMock()
    sync_validator = Mock()

    config = ExtractionConfig(validators=[async_validator, sync_validator])

    processed_result = await _validate_and_post_process_async(result, config)

    async_validator.assert_called_once_with(result)
    sync_validator.assert_called_once_with(result)
    assert processed_result.content == "Test content"


@pytest.mark.anyio
async def test_validate_and_post_process_async_with_post_processors() -> None:
    """Test _validate_and_post_process_async with post processing hooks."""
    result = ExtractionResult(content="Test content", mime_type="text/plain", metadata={}, chunks=[])

    modified_result = ExtractionResult(
        content="Modified content",
        mime_type="text/plain",
        metadata={"processed": True},  # type: ignore[typeddict-unknown-key]
        chunks=[],
    )

    async_processor = AsyncMock(return_value=modified_result)
    sync_processor = Mock(return_value=modified_result)

    config = ExtractionConfig(post_processing_hooks=[async_processor, sync_processor])

    # Mock _validate_and_post_process_helper to return the original result
    with patch("kreuzberg.extraction._validate_and_post_process_helper", return_value=result):
        processed_result = await _validate_and_post_process_async(result, config)

    # Only the last processor's result should be returned
    assert processed_result.content == "Modified content"
    assert processed_result.metadata.get("processed") is True


def test_validate_and_post_process_sync_with_validators() -> None:
    """Test _validate_and_post_process_sync with custom validators."""
    result = ExtractionResult(content="Test content", mime_type="text/plain", metadata={}, chunks=[])

    validator = Mock()
    config = ExtractionConfig(validators=[validator])

    processed_result = _validate_and_post_process_sync(result, config)

    validator.assert_called_once_with(result)
    assert processed_result.content == "Test content"


def test_validate_and_post_process_sync_with_post_processors() -> None:
    """Test _validate_and_post_process_sync with post processing hooks."""
    result = ExtractionResult(content="Test content", mime_type="text/plain", metadata={}, chunks=[])

    modified_result = ExtractionResult(
        content="Modified content",
        mime_type="text/plain",
        metadata={"processed": True},  # type: ignore[typeddict-unknown-key]
        chunks=[],
    )

    processor = Mock(return_value=modified_result)
    config = ExtractionConfig(post_processing_hooks=[processor])

    # Mock _validate_and_post_process_helper to return the original result
    with patch("kreuzberg.extraction._validate_and_post_process_helper", return_value=result):
        processed_result = _validate_and_post_process_sync(result, config)

    assert processed_result.content == "Modified content"
    assert processed_result.metadata.get("processed") is True
    processor.assert_called_once_with(result)


def test_validate_and_post_process_helper_with_all_features() -> None:
    """Test _validate_and_post_process_helper with all features enabled."""
    result = ExtractionResult(
        content="This is test content for processing.", mime_type="text/plain", metadata={}, chunks=[]
    )

    config = ExtractionConfig(
        chunk_content=True,
        extract_entities=True,
        extract_keywords=True,
        auto_detect_language=True,
        auto_detect_document_type=True,
        max_chars=10,
        max_overlap=2,
        keyword_count=3,
    )

    with (
        patch("kreuzberg.extraction.get_chunker") as mock_chunker,
        patch("kreuzberg.extraction.extract_entities") as mock_entities,
        patch("kreuzberg.extraction.extract_keywords") as mock_keywords,
        patch("kreuzberg.extraction.detect_languages") as mock_languages,
        patch("kreuzberg.extraction.auto_detect_document_type") as mock_doc_type,
    ):
        # Setup mocks
        mock_chunker_instance = Mock()
        mock_chunker_instance.chunks.return_value = ["chunk1", "chunk2"]
        mock_chunker.return_value = mock_chunker_instance

        mock_entities.return_value = [{"text": "test", "label": "MISC"}]
        mock_keywords.return_value = ["test", "content", "processing"]
        mock_languages.return_value = ["en"]
        mock_doc_type.return_value = result  # Return unchanged

        processed_result = _validate_and_post_process_helper(result, config, Path("/test/path.txt"))

    assert processed_result.chunks == ["chunk1", "chunk2"]
    assert processed_result.entities == [{"text": "test", "label": "MISC"}]  # type: ignore[comparison-overlap]
    assert processed_result.keywords == ["test", "content", "processing"]  # type: ignore[comparison-overlap]
    assert processed_result.detected_languages == ["en"]

    mock_chunker.assert_called_once_with(mime_type="text/plain", max_characters=10, overlap_characters=2)
    mock_entities.assert_called_once_with(result.content, custom_patterns=None)
    mock_keywords.assert_called_once_with(result.content, keyword_count=3)
    mock_languages.assert_called_once()
    mock_doc_type.assert_called_once_with(result, config, file_path=Path("/test/path.txt"))
