
import os
import sys
import unittest
import importlib.util
from types import ModuleType
from enum import Enum

from configparser import ConfigParser
from datetime import datetime, date

import pandas as pd
from rich.console import Console


class Submodule(Enum):
    EXPORT = 'metalarchivist.export', './src/metalarchivist/export/__init__.py'
    REPORT = 'metalarchivist.report', './src/metalarchivist/report/__init__.py'
    IFACE = 'metalarchivist.interface', './src/metalarchivist/interface/__init__.py'


def run_test_cases():
    unittest.main(argv=[''], verbosity=2)


def prepare_submodule(submodule: Submodule) -> ModuleType:
    submodule_name, submodule_path = submodule.value
    spec = importlib.util.spec_from_file_location(submodule_name, submodule_path)
    module = importlib.util.module_from_spec(spec)
    sys.modules[submodule_name] = module
    spec.loader.exec_module(module)

    return module


def load_submodules() -> tuple[ModuleType, ModuleType, ModuleType]:
    interface = prepare_submodule(Submodule.IFACE)
    export = prepare_submodule(Submodule.EXPORT)
    report = prepare_submodule(Submodule.REPORT)

    return interface, export, report


class TestMetalArchivesDirectory(unittest.TestCase):

    def test_releases_endpoint(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)
        
        self.assertIn('MetalArchivesDirectory', dir(export))

        range_start = datetime(1990, 1, 1).strftime('%Y-%m-%d')
        range_stop = datetime(1990, 12, 31).strftime('%Y-%m-%d')

        self.assertEqual(range_start, '1990-01-01')
        self.assertEqual(range_stop, '1990-12-31')

        expected_endpoint = ('https://www.metal-archives.com/release/ajax-upcoming/json/1'
                             '?sEcho=0&iDisplayStart=0&iDisplayLength=100'
                             '&fromDate=1990-01-01&toDate=1990-12-31')

        endpoint_query = dict(from_date=range_start, to_date=range_stop)
        actual_endpoint = export.MetalArchivesDirectory.upcoming_releases(**endpoint_query)

        self.assertEqual(expected_endpoint, actual_endpoint)


class TestAlbums(unittest.TestCase):

    def test_releases(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        
        self.assertIn('Albums', dir(export))

        upcoming_component_attributes = dir(export.Albums)
        self.assertIn('get_all', upcoming_component_attributes)
        self.assertIn('get_upcoming', upcoming_component_attributes)
        self.assertIn('get_range', upcoming_component_attributes)

    def test_release_fields(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)
        
        releases = export.Albums.get_upcoming(verbose=True)

        releases_attributes = dir(releases)
        self.assertIn('total_records', releases_attributes)
        self.assertIn('total_display_records', releases_attributes)
        self.assertIn('echo', releases_attributes)
        self.assertIn('data', releases_attributes)

        self.assertIsInstance(releases.total_records, int)
        self.assertIsInstance(releases.total_display_records, int)
        self.assertIsInstance(releases.echo, int)
        self.assertIsInstance(releases.data, list)

        self.assertEqual(releases.total_records, releases.total_display_records)
        self.assertEqual(releases.echo, 0)

    def test_upcoming(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        releases = export.Albums.get_upcoming(verbose=True)
        self.assertIsNotNone(releases)
        self.assertIsInstance(releases, interface.ReleasePage)

        data = releases.data
        self.assertIsInstance(data, list)
        self.assertGreaterEqual(len(data), releases.total_records)

        album_release = data.pop()
        self.assertIsInstance(album_release, interface.AlbumRelease)

    def test_range(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        self.assertIn('Albums', dir(export))

        releases = export.Albums.get_range(datetime(1990, 1, 1), datetime(1990, 12, 31), 
                                           verbose=True)
        self.assertIsNotNone(releases)
        self.assertIsInstance(releases, interface.ReleasePage)

        total_records = releases.total_records
        total_display_records = releases.total_display_records
        self.assertEqual(total_records, total_display_records)

        self.assertEqual(releases.echo, 0)

        data = releases.data
        self.assertIsInstance(data, list)
        self.assertGreaterEqual(len(data), total_records)

        album_release = data.pop()
        self.assertIsInstance(album_release, interface.AlbumRelease)

    def test_range_with_null_upper_bound(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        self.assertIn('Albums', dir(export))

        releases = export.Albums.get_range(datetime(2023, 8, 11), verbose=True)
        self.assertIsNotNone(releases)
        self.assertIsInstance(releases, interface.ReleasePage)

        total_records = releases.total_records
        total_display_records = releases.total_display_records
        self.assertEqual(total_records, total_display_records)

        self.assertEqual(releases.echo, 0)

        data = releases.data
        self.assertIsInstance(data, list)
        self.assertGreaterEqual(len(data), total_records)

        album_release = data.pop()
        self.assertIsInstance(album_release, interface.AlbumRelease)

    def test_album_release(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        self.assertIn('Albums', dir(export))

        releases = export.Albums.get_range(datetime(2023, 8, 11), verbose=True)

        data = releases.data
        self.assertIsInstance(data, list)

        # can be greater than total records due to split albums
        self.assertGreaterEqual(len(data), releases.total_records)

        album_release = data.pop()
        self.assertIsInstance(album_release, interface.AlbumRelease)

        self.assertIn('release_type', dir(album_release))
        self.assertIn('genres', dir(album_release))
        self.assertIn('release_date', dir(album_release))
        self.assertIn('added_date', dir(album_release))
        self.assertIn('band', dir(album_release))
        self.assertIn('album', dir(album_release))

        self.assertIsInstance(album_release.genres, interface.Genres)

        self.assertIsInstance(album_release.release_date, date)

        if album_release.added_date:
            self.assertIsInstance(album_release.added_date, date)

        self.assertIsInstance(album_release.band, interface.BandLink)
        self.assertIsInstance(album_release.album, interface.AlbumLink)

    def test_release_report(self):
        console = Console()

        config = ConfigParser({'unittests': {'OUTPUTDIR': './'}})
        config.read('metallum.cfg')

        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        report = prepare_submodule(Submodule.REPORT)
        self.assertIsNotNone(report)

        releases = report.get_albums(datetime(2023, 11, 22), datetime(2023, 11, 23), verbose=True)
        self.assertIsInstance(releases, pd.DataFrame)

        console.log(releases.info())

        output_path_csv = os.path.join(config['unittests']['OUTPUTDIR'], 'releases-upcoming.csv')
        releases.to_csv(output_path_csv, index=False)

        output_path_json = os.path.join(config['unittests']['OUTPUTDIR'], 'releases-upcoming.json')
        releases.to_json(output_path_json, orient='records')

    def test_album_profile(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        self.assertIn('Album', dir(export))

        album = export.Album.get_profile('https://www.metal-archives.com/albums/Urfaust/Untergang/1161736')
        self.assertIsNotNone(album)

    def test_album_profiles(self):  
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        self.assertIn('Album', dir(export))

        album = export.Album.get_profiles(['https://www.metal-archives.com/albums/Urfaust/Untergang/1161736',
                                           'https://www.metal-archives.com/albums/Furia/Huta_Luna/1166382',
                                           'https://www.metal-archives.com/albums/Hades_Almighty/...Again_Shall_Be/91367'])
        self.assertIsNotNone(album)

    def test_album_themes(self):
        interface = prepare_submodule(Submodule.IFACE)
        self.assertIsNotNone(interface)

        export = prepare_submodule(Submodule.EXPORT)
        self.assertIsNotNone(export)

        report = prepare_submodule(Submodule.REPORT)
        self.assertIsNotNone(report)

        albums = report.get_albums(verbose=True)


if __name__ == '__main__':
    run_test_cases()
