
"""
    DragonPy - Dragon 32 emulator in Python
    =======================================

    information from:
        http://dragon32.info/info/memmap.html - Dragon 32/64 Memory Map (with CoCo refs)
        http://dragon32.info/info/romref.html - Dragon 32/64 ROM Reference guide
    AUTHOR: Graham E. Kinns

    other useful information:
        http://www.onastick.clara.co.uk/dragonhw.htm
        http://www.onastick.clara.co.uk/sys1.htm
        http://www.onastick.clara.co.uk/sys2.htm
        http://www.onastick.clara.co.uk/sys3.htm
        http://www.onastick.clara.co.uk/sys4.htm
        http://www.onastick.clara.co.uk/sys5.htm
        http://www.onastick.clara.co.uk/sys6.htm

    :copyleft: 2013-2014 by the DragonPy team, see AUTHORS for more details.
    :license: GNU GPL v3 or above, see LICENSE for more details.
"""

from __future__ import absolute_import, division, print_function

import logging

log=logging.getLogger(__name__)
from MC6809.core.memory_info import BaseMemoryInfo
from dragonpy.utils.humanize import nice_hex


class DragonMemInfo(BaseMemoryInfo):
    MEM_INFO = (
        (0x0, 0x0, "BREAK message flag - if negative print BREAK"),
        (0x1, 0x1, """String delimiting char (0x22 '"')"""),
        (0x2, 0x2, """Another delimiting char (0x22 '"')"""),
        (0x3, 0x3, "General counter byte"),
        (0x4, 0x4, "Count of IFs looking for ELSE"),
        (0x5, 0x5, "DIM flag"),
        (0x6, 0x6, "%VALTYP% Variable type flag (0x00 numeric, Non-0x00 string)"),
        (0x7, 0x7, "Garbage collection flag"),
        (0x8, 0x8, "Subscript allowed flag"),
        (0x9, 0x9, "INPUT/READ flag"),
        (0xa, 0xa, "Arithmetic use"),
        (0xb, 0xc, "String ptr first free temporary"),
        (0xd, 0xe, "String ptr last free temporary"),
        (0xf, 0x18, "Temporary results"),
        (0x19, 0x1a, "Start address of BASIC program ($1e01, $2401 with DOS)"),
        (0x1b, 0x1c, "Start address of simple variables"),
        (0x1d, 0x1e, "Start address of array variables"),
        (0x1f, 0x20, "End of storage, Start of unused mem after BASIC program"),
        (0x21, 0x22, "Top of stack, growing down ($7e36)"),
        (0x23, 0x24, "Top of free string space ($7ffe)"),
        (0x25, 0x26, "Temp Ptr to string in string space"),
        (0x27, 0x28, "Top of Ram available to BASIC - returned by DOS HIMEM ($7ffe)"),
        (0x29, 0x2a, "Last/CONT line number"),
        (0x2b, 0x2c, "Temp/Input line number store"),
        (0x2d, 0x2e, "Ptr to next statement to be executed"),
        (0x2f, 0x30, "Direct mode command text pointer"),
        (0x31, 0x32, "Current DATA statement line number"),
        (0x33, 0x34, "Ptr to next item in current DATA statement"),
        (0x35, 0x36, "Ptr to keyboard input buffer"),
        (0x37, 0x38, "Ptr to variable last in use"),
        (0x37, 0x38, "ASCII codes of last variable used {1}"),
        (0x39, 0x3a, "VARPTR address of last variable used"),
        (0x3b, 0x4e, "Evaluation variables"),
        (0x41, 0x42, "High end destination addr for block"),
        (0x43, 0x44, "High end origin addr"),
        (0x45, 0x46, "Low end destination addr for block"),
        (0x47, 0x48, "Low end origin addr"),
        (0x4f, 0x54, "Floating Point Accumulator Num 1"),
        (0x4f, 0x4f, "Exponent"),
        (0x50, 0x53, "Mantissa"),
        (0x50, 0x51, "16 bit values in FAC stored here"),
        (0x52, 0x53, "VARPTR of variables is stored here {1}"),
        (0x54, 0x54, "Mantissa Sign (0x00 positive, 0xff negative)"),
        (0x55, 0x55, "Temp sign of FAC"),
        (0x56, 0x56, "String variable length"),
        (0x57, 0x5b, "String Descriptor temporaries"),
        (0x5c, 0x61, "Floating Point Accumulator Num 2"),
        (0x62, 0x62, "Sign comparison"),
        (0x62, 0x67, "Misc use"),
        (0x63, 0x63, "CoCo - Extended precision byte {1} - also Dragon ?"),
        (0x68, 0x69, "Current Line number (0xffff in direct mode)"),
        (0x6a, 0x6e, "Device Params used in PRINT"),
        (0x6a, 0x6a, "Device Comma field width (VDU - 0x10)"),
        (0x6b, 0x6b, "Device Last comma field"),
        (0x6c, 0x6c, "Device Current column num (VDU - 0x00-0x1f)"),
        (0x6d, 0x6d, "Device Line width - num chars per line (VDU 0x20)"),
        (0x6e, 0x6e, "Cassette I/O in progress flag - 0xff on input or output occurring  {9}"),
        (0x6f, 0x6f, "%DEVNUM% Current device number 0x00 VDU screen 0x01-0x04 DOS - DosPlus only - drive number. 0xfd serial port (Dragon 64 only) 0xfe printer 0xff tape"),
        (0x70, 0x70, "Cassette EOF flag - non-zero if EOF - used by EOF(-1) {9}"),
        (0x71, 0x71, "Restart flag - if not 0x55 cold start on reset, see $0072"),
        (0x72, 0x73, "Restart vector - Following a reset if $0072 pts to a NOP opcode and $0071 is 0x55 then a warm start is performed to this vector else a cold start. (0xb44f) (DOS SuperDosE6 $c706)"),
        (0x74, 0x75, "Physical end of Ram minus 1 (0x7ffe)"),
        (0x76, 0x77, "Unused"),
        (0x78, 0x78, "Cassette status 0x00 closed 0x01 input 0x02 output"),
        (0x79, 0x79, "Cassette I/O - Buffer size - bytes in block"),
        (0x7a, 0x7b, "Header buffer addr - ptr to filename block"),
        (0x7c, 0x7c, "%BLKTYP% Cassette block type 0x00 filename 0x01 data 0xff EOF block"),
        (0x7d, 0x7d, "%DBLEN% Cassette block length, number bytes read/to write"),
        (0x7e, 0x7f, "%DBADR% Cassette I/O Buffer address Contains 1 + End address of last program loaded"),
        (0x80, 0x80, "Cassette I/O - block checksum used internally"),
        (0x81, 0x81, "Cassette I/O - error code 0x00 none 0x01 CRC (checksum) error 0x02 attempt to load into ROM"),
        (0x82, 0x82, "Cassette I/O - Pulse width counter"),
        (0x83, 0x83, "Cassette I/O - Sync bits counter"),
        (0x84, 0x84, "Cassette I/O - Bit phase flag"),
        (0x85, 0x85, "Last sine wave value for output to DAC"),
        (0x86, 0x86, "Data for low res SET/RESET, POINT routines"),
        (0x87, 0x87, "ASCII code of last key pressed (cleared by Break check)"),
        (0x88, 0x89, "Current VDU cursor addr (typ 0x0400-0x05ff)"),
        (0x8a, 0x8b, "Gen purpose 16bit scratch pad / 16bit zero (0x0000) {2}"),
        (0x8a, 0x8b, "CoCo - Motor on delay"),
        (0x8c, 0x8c, "Sound pitch frequency"),
        (0x8d, 0x8e, "Gen purpose countdown (?sound timer {2})"),
        (0x8f, 0x8f, "Cursor flash counter (0x20)"),
        (0x90, 0x91, "Cassette leader byte count - number of 0x55 bytes written as sync leader (D32 - 0x0080, D64 - 0x0100)"),
        (0x92, 0x92, "Minimum cycle width of 1200Hz (0x12)"),
        (0x92, 0x93, "CoCo - Cassette leader byte count"),
        (0x93, 0x93, "Minimum pulse width of 1200Hz (0x0a)"),
        (0x94, 0x94, "Maximum pulse width of 1200Hz (0x12)"),
        (0x95, 0x96, "Motor on delay (0xda5c = approx 0.5s)"),
        (0x95, 0x96, "CoCo - Serial Baud rate constant (0x0057 = 600 baud)"),
        (0x97, 0x98, "Keyboard scan debounce delay constant (0x045e)"),
        (0x97, 0x98, "CoCo - Serial Line Printer End of Line delay (0x0001)"),
        (0x99, 0x99, "Printer comma field width (0x10 = 16)"),
        (0x9a, 0x9a, "Printer last comma field (0x74 = 116) (CoCo 0x70 = 112)"),
        (0x9b, 0x9b, "Printer line width dflt (0x84 = 132)"),
        (0x9c, 0x9c, "Printer head column posn == POS(-2) Updated by LPOUT ($800f) routine"),
        (0x9d, 0x9e, "EXEC default entry address (D32 - $8b8d = ?FC ERROR; D64 - $bf49 = Boot 64k mode)"),
        (0x9f, 0xaa, "%CHRGET% Self modifying routine to read next char"),
        (0x9f, 0xa0, "INC <$A7"),
        (0xa1, 0xa2, "BNE $00A5"),
        (0xa3, 0xa4, "INC <$A6"),
        (0xa5, 0xa7, "LDA >xxxx"),
        (0xa6, 0xa7, "Ptr to next character to read"),
        (0xa8, 0xaa, "JMP $BB26"),
        (0xab, 0xae, "Used by RND"),
        (0xaf, 0xaf, "TRON/TROFF trace flag - non zero for TRON"),
        (0xb0, 0xb1, "Ptr to start of USR table ($0134; DOS - $0683)"),
        (0xb2, 0xb2, "Current foreground colour (0x03)"),
        (0xb3, 0xb3, "Current background colour (0x00)"),
        (0xb4, 0xb4, "Temp/active colour in use"),
        (0xb5, 0xb5, "Byte value for current colour - ie bit pattern"),
        (0xb6, 0xb6, "Graphics PMODE number in use (0x00)"),
        (0xb7, 0xb8, "Ptr to last byte+1 of current graphics mode ($0c00 w/o Dos)"),
        (0xb9, 0xb9, "Number of bytes per line in current PMODE (0x10)"),
        (0xba, 0xbb, "Ptr to first byte of current graphics mode ($0600)"),
        (0xbc, 0xbc, "Msb of start of graphics pages (0x06 or 0x0c with Dos)"),
        (0xbd, 0xbe, "Current X cursor position (not user available ?{1})"),
        (0xbf, 0xc0, "Current Y cursor position (not user available ?{1})"),
        (0xc1, 0xc1, "Colour set currently in use (0x08 if colorset 1 {1})"),
        (0xc2, 0xc2, "Plot/Unplot flag: 0x00 reset, non zero set"),
        (0xc3, 0xc4, "Current horizontal pixel number"),
        (0xc5, 0xc6, "Current vertical pixel number"),
        (0xc7, 0xc8, "Current X cursor coord (0x0080)"),
        (0xc9, 0xca, "Current Y cursor coord (0x0060)"),
        (0xcb, 0xcc, "CIRCLE command X coood as if drawn in PMODE 4"),
        (0xcd, 0xce, "CIRCLE command Y coord as if drawn in PMODE 4"),
        (0xcf, 0xd0, "CIRCLE radius as if drawn in PMODE 4"),
        (0xcf, 0xd0, "RENUM increment value"),
        (0xd1, 0xd2, "RENUM start line"),
        (0xd3, 0xd4, "CLOADM 2's complement load offset"),
        (0xd5, 0xd6, "RENUM new start line"),
        (0xd7, 0xd7, "EDIT line length (not user available)"),
        (0xd7, 0xd7, "PLAY -"),
        (0xd8, 0xd8, "PLAY - bytes left in string"),
        (0xd9, 0xda, "PLAY - ptr to current char in string"),
        (0xd8, 0xdd, "Graphics use ? {2}"),
        (0xde, 0xde, "PLAY: Current octave in use (0-4) (0x02)"),
        (0xdf, 0xe0, "PLAY: Volume data for volume setting (D32 - 0xba42) (D64 - 0xb844)"),
        (0xe1, 0xe1, "PLAY: Current note length (0x04)"),
        (0xe2, 0xe2, "PLAY: Current tempo (0x02)"),
        (0xe3, 0xe4, "PLAY: Music duration count"),
        (0xe5, 0xe5, "PLAY: Music dotted note flag"),
        (0xe6, 0xff, "D32 - Unused in Dragon 32 w/o DOS"),
        (0xe6, 0xe6, "CoCo - baud rate constant"),
        (0xe7, 0xe7, "Coco - Input timeout constant"),
        (0xe8, 0xe8, "Current angle used in DRAW {1} (??)"),
        (0xe9, 0xe9, "Current scale used in DRAW {1} (??)"),
        (0xea, 0xf6, "DOS - Used by DragonDos"),
        (0xf8, 0xf8, "DOS - sector currently seeking {SuperDos Rom}"),
        (0x100, 0x102, "SWI3 Secondary vector (Uninitialised)"),
        (0x103, 0x105, "SWI2 Secondary vector (Uninitialised)"),
        (0x106, 0x108, "SWI Secondary vector (Uninitialised)"),
        (0x109, 0x10b, "NMI Secondary vector (Uninitialised) (CoCo DOS JMP $d7ae; SuperDos E6 JMP $c71e)"),
        (0x10c, 0x10e, "IRQ Secondary vector - JMP $9d3d (CoCo JMP $a9b3 or $894c (extended); CoCo DOS JMP $d7bc; SuperDos E6 JMP $c727)"),
        (0x10f, 0x111, "FIRQ Secondary vector - JMP $b469 (CoCo JMP $a0f6; SuperDos E6 JMP $c7da)"),
        (0x112, 0x113, "TIMER value"),
        (0x114, 0x114, "Unused"),
        (0x115, 0x119, "Random number seeds (0x80, 0x4f, 0xc7, 0x52, 0x59)"),
        (0x11a, 0x11f, "D32 - Unused"),
        (0x11a, 0x11a, "D64 - %FLAG64% checked on Reset from 64K mode if 0x55 then checksum at $011b is checked against current contents of Ram, if the same then a warm start is performed (64 mode) else a cold start (32 mode)"),
        (0x11a, 0x11a, "CoCo - Caps lock, 0x00 lower, non-0x00 upper"),
        (0x11b, 0x11c, "D64 - %CSUM64% 16bit sum of words of BASIC Rom-in-ram in 64K mode from $c000 to $feff"),
        (0x11b, 0x11c, "CoCo - Keyboard Delay constant"),
        (0x11d, 0x11f, "CoCo - JMP $8489 ? {1}"),
        (0x11d, 0x11d, "D64 - %LSTKEY% Last key code return by keybd poll routine"),
        (0x11e, 0x11e, "D64 - %CNTDWN% Auto repeat countdown"),
        (0x11f, 0x11f, "D64 - %REPDLY% Auto repeat inter-repeat delay value (0x05)"),
        (0x120, 0x120, "%STUB0% Stub 0 - Number of reserved words (0x4e)"),
        (0x121, 0x122, "Stub 0 - Ptr to reserved words table ($8033)"),
        (0x123, 0x124, "Stub 0 - Ptr to reserved words dispatch table ($8154)"),
        (0x125, 0x125, "Stub 0 - Number of functions (0x22)"),
        (0x126, 0x127, "Stub 0 - Ptr to reserved function words table ($81ca)"),
        (0x128, 0x129, "Stub 0 - Ptr to function words dispatch table ($8250)"),
        (0x12a, 0x12a, "%STUB1% Stub 1 - Number of reserved words (0x00) (DOS 0x1a)"),
        (0x12b, 0x12c, "Stub 1 - Ptr to reserved words table (0x0000) (DOS $ded4; SuperDosE6 $deda)"),
        (0x12d, 0x12e, "Stub 1 - Ptr to reserved words token processing routine ($89b4; DOS $c64c; SuperDosE6 $c670)"),
        (0x12f, 0x12f, "Stub 1 - Number of functions (0x00) (DOS 0x07)"),
        (0x130, 0x131, "Stub 1 - Ptr to function table (0x0000) (DOS $debb; SuperDosE6 $dec1)"),
        (0x132, 0x133, "Stub 1 - Ptr to function token processing routine ($89b4; DOS $c667; SuperDosE6 $c68b)"),
        (0x134, 0x134, "%STUB2% Stub 2 - acts as a stub terminator under DOS"),
        (0x134, 0x147, "USR address table, relocated by DOS (10 x 2 bytes) ($8b8d)"),
        (0x148, 0x148, "Auto line feed flag on buffer full - setting this to 0x00 causes a EOL sequence to be sent to printer when buffer reaches length in $009b (0xff)"),
        (0x149, 0x149, "Alpha Lock flag - 0x00 Lower case, 0xff Upper case (0xff)"),
        (0x14a, 0x150, "Line Printer End of line termination sequence"),
        (0x14a, 0x14a, "Number of bytes in EOL sequence 1-6 (0x01)"),
        (0x14b, 0x14b, "EOL chr 1 (0x0d CR)"),
        (0x14c, 0x14c, "EOL chr 2 (0x0a LF)"),
        (0x14d, 0x14d, "EOL chr 3 (D64 - 0x00; D32 - 0x20 ' ')"),
        (0x14e, 0x14e, "EOL chr 4 (D64 - 0x00; D32 - 0x44 'D' Duncan)"),
        (0x14f, 0x14f, "EOL chr 5 (D64 - 0x00; D32 - 0x4e 'N' N.)"),
        (0x150, 0x150, "EOL chr 6 (D64 - 0x00; D32 - 0x4f 'S' Smeed)"),
        (0x151, 0x159, "Keyboard matrix state table"),
        (0x152, 0x159, "CoCo - Keyboard roll-over table"),
        (0x15a, 0x15d, "%POTVAL% Joystick values (0-63)"),
        (0x15a, 0x15a, "Right Joystick, x value == JOYSTK(0)"),
        (0x15b, 0x15b, "Right Joystick, y value == JOYSTK(1)"),
        (0x15c, 0x15c, "Left Joystick, x value == JOYSTK(2)"),
        (0x15d, 0x15d, "Left Joystick, y value == JOYSTK(3)"),
        (0x15e, 0x1a8, "Ram hooks - each is called from ROM with a JSR before carrying out the specified function"),
        (0x15e, 0x160, "Device Open (DOS JMP $d902; SuperDosE6 $d8f4)"),
        (0x161, 0x163, "Verify Device Number (DOS SuperDosE6 JMP $d8ec)"),
        (0x164, 0x166, "Device Init (DOS SuperDosE6 JMP $c29c)"),
        (0x167, 0x169, "Output char in A to DEVN (DOS JMP $d8fa; SuperDosE6 $d90b)"),
        (0x167, 0x167, "Setting to 0xff disables keyboard ?!? {1} Setting to 0x39 (RTS) allows use of SCREEN 0,1 etc. ??{1}"),
        (0x16a, 0x16c, "Input char from DEVN to A (DOS SuperDosE6 JMP $c29c)"),
        (0x16d, 0x16f, "Input from DEVN using INPUT (DOS SuperDosE6 JMP $c29c)"),
        (0x170, 0x172, "Output to DEVN using PRINT (DOS SuperDosE6 JMP $c29c)"),
        (0x173, 0x175, "Close all files (DOS SuperDosE6 JMP $c29c)"),
        (0x176, 0x178, "Close file (DOS JMP $d917; SuperDosE6 $d6f5)"),
        (0x179, 0x17b, "Command Interpreter - interpret token in A as command (DOS SuperDosE6 JMP $c29c)"),
        (0x17c, 0x17e, "Re-request input from keyboard (DOS JMP $d960; SuperDosE6 $d954)"),
        (0x17f, 0x181, "Check keys - scan for BREAK, SHIFT+'@' (DOS SuperDosE6 JMP $c29c)"),
        (0x17f, 0x17f, "Setting this to 0x9e disables LIST/DIR {1}"),
        (0x182, 0x184, "Line input from DEVN using LINE INPUT (DOS JMP $d720; SuperDosE6 $dac5)"),
        (0x185, 0x187, "Close BASIC file read in and goto Command mode (DOS SuperDosE6 JMP $c29c)"),
        (0x188, 0x18a, "Check EOF on DEVN (DOS JMP $dd4d; SuperDosE6 $dd54)"),
        (0x18b, 0x18d, "Evaluate expression (DOS SuperDosE6 JMP $c29c)"),
        (0x18e, 0x190, "User error trap, called from $8344 (DOS SuperDosE6 JMP $c29c)"),
        (0x191, 0x193, "System error trap, called from $8344 (DOS JMP $c69e; SuperDosE6 $c6c5)"),
        (0x194, 0x196, "Run Link - used by DOS to RUN filename (DOS JMP $d490; SuperDosE6 $d4b7)"),
        (0x197, 0x199, "Reset Basic Memory, editing or entering BASIC lines"),
        (0x19a, 0x19c, "Get next command - reading in next command to be executed"),
        (0x19d, 0x19f, "Assign string variable"),
        (0x1a0, 0x1a2, "Screen access - CLS, GET, PUT"),
        (0x1a3, 0x1a5, "Tokenise line"),
        (0x1a6, 0x1a8, "De-Tokenise line"),
        (0x1a9, 0x1d0, "String buffer area"),
        (0x1d1, 0x1d1, "Cassette filename length in range 0-8"),
        (0x1d2, 0x1d9, "Cassette filename to search for or write out"),
        (0x1da, 0x2d8, "Cassette I/O default data buffer - 255 bytes"),
        (0x1da, 0x268, "D64 - 64K mode bootstrap routine is copied here to run"),
        (0x1da, 0x1e1, "Cassette buffer - filename of file read"),
        (0x1e2, 0x1e2, "Cassette buffer - filetype 0x00 BASIC program 0x01 Data 0x02 Machine code"),
        (0x1e3, 0x1e3, "Cassette buffer - ASCII flag 0x00 Binary 0xff ASCII flag"),
        (0x1e4, 0x1e4, "Cassette buffer - gap flag 0x00 Continous 0xff Gapped file"),
        (0x1e5, 0x1e6, "Cassette buffer - Entry (Exec) addr of m/c file"),
        (0x1e7, 0x1e8, "Cassette buffer - Load address for ungapped m/c file"),
        (0x2d9, 0x2dc, "BASIC line input buffer preamble"),
        (0x2dd, 0x3d8, "BASIC line input buffer - used for de-/tokenising data"),
        (0x2dd, 0x3dc, "CoCo - 255 byte keyboard buffer {1}"),
        (0x2e1, 0x33b, "CoCo - 90 byte screen buffer {1}"),
        (0x3d9, 0x3ea, "Buffer space"),
        (0x3eb, 0x3fc, "Unused"),
        (0x3fd, 0x3ff, "D32 - Unused in Dragon 32"),
        (0x3fd, 0x3fe, "D64 - Printer end of line delay in milliseconds (0x0000)"),
        (0x3ff, 0x3ff, "D64 - %PRNSEL% selects default printer port 0x00 Parallel, non-0x00 Serial (0x00)"),
        (0x400, 0x5ff, "Default Text screen"),
        (0x600, 0x1dff, "Available graphics pages w/o DOS"),
        (0x600, 0xbff, "DOS - workspace area see also $00ea-$00f6"),
        (0x600, 0xdff, "CoCo DOS workspace area (no more info)"),
        (0xc00, 0x23ff, "DOS - Available graphics pages"),
        (0x8000, 0xbfff, "BASIC ROM in 32K mode"),
        (0xbff0, 0xbfff, "These addresses mapped from ROM to $fff0-$ffff by the SAM"),
        (0xc000, 0xdfff, "DOS - Dos ROM"),
        (0xc000, 0xfeff, "DOS - Cumana DOS ROM only"),
        (0xc000, 0xfeff, "Available address range to cartridge expansion port 32K mode"),
        (0xc000, 0xfeff, "D64 - 64K mode - copy of BASIC ROM 2 exists in RAM here"),
        (0xff00, 0xff00, "PIA 0 A side Data reg. PA7"),
        (0xff01, 0xff01, "PIA 0 A side Control reg. CA1"),
        (0xff02, 0xff02, "PIA 0 B side Data reg. PB7"),
        (0xff03, 0xff03, "PIA 0 B side Control reg. CB1"),
        (0xff04, 0xff04, "D64 - ACIA serial port read/write data reg."),
        (0xff05, 0xff05, "D64 - ACIA serial port status (R)/ reset (W) reg."),
        (0xff06, 0xff06, "D64 - ACIA serial port command reg."),
        (0xff07, 0xff07, "D64 - ACIA serial port control reg."),
        (0xff20, 0xff20, "PIA 1 A side Data reg. PA7"),
        (0xff21, 0xff21, "PIA 1 A side Control reg. CA1"),
        (0xff22, 0xff22, "PIA 1 B side Data reg. PB7"),
        (0xff23, 0xff23, "PIA 1 B side Control reg. CB1"),
        (0xff40, 0xff40, "DOS - Disk Controller command/status reg."),
        (0xff41, 0xff41, "DOS - Disk Controller track reg."),
        (0xff42, 0xff42, "DOS - Disk Controller sector reg."),
        (0xff43, 0xff43, "DOS - Disk Controller data reg."),
        (0xff48, 0xff48, "DOS - Disk Controller hardware control reg."),
        (0xffc0, 0xffdf, "SAM (Synchronous Address Multiplexer) register bits - use even address to clear, odd address to set"),
        (0xffc0, 0xffc5, "SAM VDG Mode registers V0-V2"),
        (0xffc0, 0xffc1, "SAM VDG Reg V0"),
        (0xffc2, 0xffc3, "SAM VDG Reg V1"),
        (0xffc3, 0xffc5, "SAM VDG Reg V2"),
        (0xffc6, 0xffd3, "SAM Display offset in 512 byte pages F0-F6"),
        (0xffc6, 0xffc7, "SAM Display Offset bit F0"),
        (0xffc8, 0xffc9, "SAM Display Offset bit F1"),
        (0xffca, 0xffcb, "SAM Display Offset bit F2"),
        (0xffcc, 0xffcd, "SAM Display Offset bit F3"),
        (0xffce, 0xffcf, "SAM Display Offset bit F4"),
        (0xffd0, 0xffc1, "SAM Display Offset bit F5"),
        (0xffd2, 0xffc3, "SAM Display Offset bit F6"),
        (0xffd4, 0xffd5, "SAM Page #1 bit - in D64 maps upper 32K Ram to $0000 to $7fff"),
        (0xffd6, 0xffd9, "SAM MPU Rate R0-R1"),
        (0xffd6, 0xffd7, "SAM MPU Rate bit R0"),
        (0xffd8, 0xffd9, "SAM MPU Rate bit R1"),
        (0xffda, 0xffdd, "SAM Memory Size select M0-M1"),
        (0xffda, 0xffdb, "SAM Memory Size select bit M0"),
        (0xffdc, 0xffdd, "SAM Memory Size select bit M1"),
        (0xffde, 0xffdf, "SAM Map Type - in D64 switches in upper 32K RAM $8000-$feff"),
        (0xffec, 0xffef, "PC-Dragon - Used by Paul Burgin's emulator to provide enhanced services"),
        (0xfff0, 0xffff, "6809 interrupt vectors mapped from $bff0-$bfff by SAM"),
        (0xfff0, 0xfff1, "Reserved ($0000; D64 64K mode 0x3634 '64')"),
        (0xfff2, 0xfff3, "SWI3  ($0100)"),
        (0xfff4, 0xfff5, "SWI2  ($0103)"),
        (0xfff6, 0xfff7, "FIRQ  ($010f)"),
        (0xfff8, 0xfff9, "IRQ  ($010c)"),
        (0xfffa, 0xfffb, "SWI  ($0106)"),
        (0xfffc, 0xfffd, "NMI  ($0109)"),
        (0xfffe, 0xffff, "RESET  ($b3b4; D64 64K mode $c000 - never accessed)"),
        (0x8000, 0x8002, "Hardware Initialisation (JMP $bb40; D64 JMP $bb3c)"),
        (0x8003, 0x8005, "Software Initialisation (JMP $bb88)"),
        (0x8006, 0x8008, "%POLCAT% Scans keyboard, rets value in A (JMP $bbe5) Called indirectly from [$a000]"),
        (0x8009, 0x800b, "%CBLINK% Cursor Blink (JMP $bbb5)"),
        (0x800c, 0x800e, "%CHROUT% Write chr in A to screen (JMP $bcab)"),
        (0x800f, 0x8011, "%LPOUT% Line printer character output in A reg., forces end-of-line  sequences, outputs extra spaces to cause LFs etc. See: $0099-009c, $0148 (JMP $bd1a)"),
        (0x8012, 0x8014, "Reads Joystick (JMP $bd52) Called indirectly from [$a00a]"),
        (0x8015, 0x8017, "Cassette relay on (JMP $bdcf)"),
        (0x8018, 0x801a, "Cassette relay off (JMP $bddc)"),
        (0x801b, 0x801d, "Write leader to cassette (JMP $be68) Called indirectly from [$a00c]"),
        (0x801e, 0x8020, "Output byte in A to cassette (JMP $be12)"),
        (0x8021, 0x8023, "Cassette on, init for reading (JMP $bde7) Called indirectly from [$a004]"),
        (0x8024, 0x8026, "Input byte from cassette into A (JMP $bdad)"),
        (0x8027, 0x8029, "Get one bit from cassette to CC.C (JMP $bda5)"),
        (0x802a, 0x802c, "Reads a byte from serial port (JMP $be7b)"),
        (0x802d, 0x802f, "Sends a byte to serial port (D32 JMP $be7c; D64 JMP $be98)"),
        (0x8030, 0x8032, "Set serial port baud rate (D32 JMP $be7d; D64 JMP $bea6)"),
        (0x8033, 0x8153, "Reserved words table"),
        (0x8154, 0x81c9, "Reserved words dispatch table (Tokens 0x80-0xba only)"),
        (0x81ca, 0x824f, "Function words table"),
        (0x8250, 0x8293, "Function words dispatch table"),
        (0x8294, 0x82a8, "Another dispatch table: 0x79, $910e   + 0x79, $9105 - 0x7b, $9275 * 0x7b, $933c / 0x7f, $96a0 ^ 0x50, $8a12 AND 0x46, $8a11 OR The related functions come from {2, p.347}"),
        (0x82a9, 0x82de, "Error code table"),
        (0x82df, 0x82e5, "' ERROR', 0x00"),
        (0x82e6, 0x82ea, "' IN ', 0x00"),
        (0x82eb, 0x82ef, "0x0d, 'OK', 0x0d, 0x00"),
        (0x82f0, 0x82f6, "0x0d, 'BREAK', 0x00"),
        (0x8344, 0x8344, "System Error - generates error message according to B (CoCo $ac46)"),
        (0x8371, 0x8371, "Prints OK prompt and returns to Command mode (CoCo $ac73)"),
        (0x83ed, 0x83ed, "BasVect2 - completes init once BASIC program loaded (CoCo $acef)"),
        (0x8415, 0x8415, "NEW token dispatch address"),
        (0x8417, 0x8417, "Erases program in memory same as NEW (CoCo $ad19) {1} Called from BASIC init routine ($b3ba)"),
        (0x841f, 0x841f, "BasVect1 - sets up various vectors after a BASIC program loaded (CoCo $ad21)"),
        (0x8434, 0x8434, "Resets stack - all entries are lost (CoCo $ad33)"),
        (0x8448, 0x8448, "FOR token dispatch address"),
        (0x849f, 0x849f, "RUN BASIC - runs BASIC program in mem used by Autorun programs (CoCo $ad9e)"),
        (0x8514, 0x8514, "RESTORE token dispatch address"),
        (0x852b, 0x852b, "Waits for key press, returns key in A (CoCo $adfb)"),
        (0x8532, 0x8532, "END token dispatch address"),
        (0x8539, 0x8539, "STOP token dispatch address"),
        (0x8560, 0x8560, "CONT token dispatch address"),
        (0x8571, 0x8571, "CLEAR token dispatch address"),
        (0x85a5, 0x85a5, "RUN token dispatch address"),
        (0x85b9, 0x85b9, "GO token dispatch address"),
        (0x85f3, 0x85f3, "RETURN token dispatch address"),
        (0x8613, 0x8613, "DATA token dispatch address"),
        (0x8616, 0x8616, "REM, ', ELSE token dispatch address"),
        (0x8647, 0x8647, "IF token dispatch address"),
        (0x8675, 0x8675, "ON token dispatch address"),
        (0x86bc, 0x86bc, "LET token dispatch address"),
        (0x8705, 0x870b, "'?REDO', 0x0d, 0x00"),
        (0x872b, 0x872b, "INPUT token dispatch address"),
        (0x8777, 0x8777, "READ token dispatch address"),
        (0x8829, 0x8829, "NEXT token dispatch address"),
        (0x8877, 0x8877, "Get expression: evaluates next expression and puts VARPTR addr in $0052:0053 (CoCo $b146)"),
        (0x8887, 0x8887, "Get string: compiles a string and puts it in free string space (CoCo $b156)"),
        (0x89a4, 0x89a4, "CkClBrak: checks cmd line for closed bracket (CoCo $b267)"),
        (0x89a7, 0x89a7, "CkOpBrak: checks cmd line for open bracket (CoCo $b26a)"),
        (0x89aa, 0x89aa, "CkComa: checks next cmd line chr for a comma, if not a Syntax error is generated (CoCo $b26d)"),
        (0x89ac, 0x89ac, "CkChar: as for CkComa but checks for chr in B (CoCo $b26f)"),
        (0x89b4, 0x89b4, "Both reserved word and function dispatch tables of Stub 1 at $012a point here"),
        (0x8a11, 0x8a11, "OR operator dispatch address (see $8294)"),
        (0x8a12, 0x8a12, "AND operator dispatch address (see $8294)"),
        (0x8a8b, 0x8a8b, "DIM token dispatch address"),
        (0x8a94, 0x8a94, "%GETVAR% gets VARPTR addr of following variables name (CoCo $b357)"),
        (0x8b29, 0x8b29, "%GETUSR% returns value of arg in USR fn in D (CoCo $b3e9)"),
        (0x8b2d, 0x8b2d, "%INTCNV%"),
        (0x8b30, 0x8b30, "Passes parameters to machine code routine {1}"),
        (0x8b39, 0x8b39, "Used to pass back values from m/c to BASIC {1}"),
        (0x8b8d, 0x8b91, "Jump to this causes ?FC ERROR"),
        (0x8c31, 0x8c31, "MEM function token dispatch address"),
        (0x8c35, 0x8c3f, "Assign16Bit: assigns D to numeric variable (CoCo $b4f2) ?? 8c35:8c36 is a BRN"),
        (0x8c36, 0x8c3f, "Assign8Bit: assigns B to numeric variable (CoCo $b4f3) clears A, stores D in FAC"),
        (0x8c37, 0x8c3f, "%GIVABF% returns D as 16bit val in FAC"),
        (0x8c40, 0x8c40, "STR$ function token dispatch address"),
        (0x8cd7, 0x8cd7, "Forces String space Garbage collection (CoCo $b591)"),
        (0x8d5d, 0x8d5d, "Negate FPA0 Mantissa"),
        (0x8d9a, 0x8d9a, "Sub -"),
        (0x8d9f, 0x8d9f, "%DELVAR% frees space taken by variable (CoCo $b659)"),
        (0x8dc7, 0x8dc7, "LEN function token dispatch address"),
        (0x8dd2, 0x8dd2, "CHR$ function token dispatch address"),
        (0x8de6, 0x8de6, "ASC function token dispatch address"),
        (0x8df1, 0x8df1, "LEFT$ function token dispatch address"),
        (0x8e0e, 0x8e0e, "RIGHT$ function token dispatch address"),
        (0x8e15, 0x8e15, "MID$ function token dispatch address"),
        (0x8e51, 0x8e51, "Get8Bit: returns value of following number in B (CoCo $b70b)"),
        (0x8e5c, 0x8e5c, "VAL function token dispatch address"),
        (0x8e83, 0x8e83, "Get16Bit: returns value of following number in X (CoCo $b73d)"),
        (0x8e96, 0x8e96, "PEEK function token dispatch address"),
        (0x8e9d, 0x8e9d, "POKE token dispatch address"),
        (0x8ea4, 0x8ea4, "LLIST token dispatch address"),
        (0x8eaa, 0x8eaa, "LIST token dispatch address, List Basic to DEVN (CoCo $b764)"),
        (0x903d, 0x903d, "PRINT token dispatch address"),
        (0x90a1, 0x90a1, "Print CR/LF: moves cursor posn to start of new line (CoCo $b958)"),
        (0x90e5, 0x90e5, "Out String: Prints ASCIIZ string ptd to by X to DEVN (CoCo $b99c)"),
        (0x9105, 0x9105, "- operator dispatch address (see $8294)"),
        (0x910e, 0x910e, "+ operator dispatch address (see $8294)"),
        (0x91c2, 0x91c2, "Toggle sign of FPA0 mantissa"),
        (0x9201, 0x9201, "Shift FPA0 Mantissa and sub byte one bit to the right"),
        (0x923c, 0x923c, "LOG function token dispatch address"),
        (0x9275, 0x9275, "* operator dispatch address (see $8294)"),
        (0x933c, 0x933c, "/ operator dispatch address (see $8294)"),
        (0x93bf, 0x93d4, "%MOVFM% Converts a 5-byte BASIC Real variable structure pointed to by X to FAC at $004f {2}"),
        (0x9425, 0x9425, "SGN function token dispatch address"),
        (0x942d, 0x942d, "Jumped to from GIVABF routine at 8c37-8c3f - returns 16 bit value in FAC to BASIC ??"),
        (0x943e, 0x943e, "ABS function token dispatch address"),
        (0x9499, 0x9499, "INT function token dispatch address"),
        (0x9564, 0x9568, "5 byte floating point number"),
        (0x9569, 0x956d, "5 byte floating point number"),
        (0x956e, 0x9572, "5 byte floating point number"),
        (0x957a, 0x957a, "Print Number: outputs D as number to DEVN (CoCo $bdcc)"),
        (0x9697, 0x9697, "SQR function token dispatch address"),
        (0x96a0, 0x96a0, "^ operator dispatch address (see $8294)"),
        (0x9713, 0x9713, "EXP function token dispatch address"),
        (0x9772, 0x9772, "RND function token dispatch address"),
        (0x978e, 0x978e, "Random Number: generates 8bit random number in $0116"),
        (0x97cb, 0x97cb, "COS function token dispatch address"),
        (0x97d1, 0x97d1, "SIN function token dispatch address"),
        (0x9816, 0x9816, "TAN function token dispatch address"),
        (0x9877, 0x9877, "ATN function token dispatch address"),
        (0x98e3, 0x98e3, "Called from BASIC init routine ($b3ba-)"),
        (0x9956, 0x9956, "FIX function token dispatch address"),
        (0x9965, 0x9965, "EDIT token dispatch address"),
        (0x9ad9, 0x9ad9, "TRON token dispatch address"),
        (0x9ada, 0x9ada, "TROFF token dispatch address"),
        (0x9ade, 0x9ade, "POS function token dispatch address"),
        (0x9af4, 0x9af4, "VARPTR function token dispatch address"),
        (0x9b84, 0x9b84, "STRING$ function token dispatch address"),
        (0x9bb4, 0x9bb4, "INSTR function token dispatch address"),
        (0x9c3e, 0x9c3e, "Assign16BitB: assigns value in $0052:0053 to a variable (CoCo $880e)"),
        (0x9c81, 0x9c81, "DEF token dispatch address"),
        (0x9d00, 0x9d0c, "Sub"),
        (0x9d0d, 0x9d1c, "Sub which returns a ptr in X to the address in the USR table (ref'd by $00b0:00b1) of the USR function to be called"),
        (0x9d1d, 0x9d1d, "USR function token dispatch address"),
        (0x9d35, 0x9d3c, "Sub which checks for '=' token, then gets value of next expression by JMPing to $8e83"),
        (0x9d3d, 0x9d3d, "IRQ interrupt service routine"),
        (0x9d59, 0x9d59, "TIMER function token dispatch address"),
        (0x9d61, 0x9d61, "DEL token dispatch address"),
        (0x9dfa, 0x9dfa, "RENUM token dispatch address"),
        (0x9fca, 0x9fcd, "3 bytes Unused ? 'UL '"),
        (0x9fce, 0x9fff, "D32 - Unused 0x00"),
        (0x9fd0, 0x9fd8, "D64 - Patch to support VARPTRs >32K"),
        (0x9fd9, 0x9fff, "D64 - Unused 0x00"),
        (0xa000, 0xa00d, "Indirect jump table"),
        (0xa000, 0xa001, "Keyboard input ($8006)"),
        (0xa002, 0xa003, "Character output in A to DEVNUM ($b54a)"),
        (0xa004, 0xa005, "Cassette on for reading ($8021)"),
        (0xa006, 0xa007, "Block input from tape ($b93e)"),
        (0xa008, 0xa009, "Block output to tape ($b999)"),
        (0xa00a, 0xa00b, "Joystick input ($8012)"),
        (0xa00c, 0xa00d, "Write leader to cassette ($801b)"),
        (0xa00e, 0xa00e, "HEX$ function token dispatch address"),
        (0xa049, 0xa049, "DLOAD token dispatch address"),
        (0xa0e0, 0xa0e0, "Can be used to wait for 2 keypresses {1}"),
        (0xa0ea, 0xa0ea, "Waits for keypress with flashing cursor (CoCo $8cc6)"),
        (0xa0f4, 0xa0f4, "Sub: called from DLOAD routine"),
        (0xa438, 0xa440, "Sub - returns CC.C set if A is not char '0'-'9'"),
        (0xa6ef, 0xa6ef, "PSET token dispatch address"),
        (0xa6f3, 0xa6f3, "PRESET token dispatch address"),
        (0xa6c7, 0xa6c7, "PPOINT function token dispatch address"),
        (0xa749, 0xa749, "LINE token dispatch address"),
        (0xa8c0, 0xa8c0, "PCLS token dispatch address"),
        (0xa8c7, 0xa8c7, "Clear Graphics: Clears current graphics screen with data in B (CoCo $9539)"),
        (0xa8d4, 0xa8d4, "COLOR token dispatch address"),
        (0xa928, 0xa928, "Set Colours: sets up locations $00b4, $00b5 (CoCo $959a)"),
        (0xa938, 0xa938, "Select Display: Selects Text (CC.Z=1) or Graphics (CC.Z=0) mode (CoCo $95aa)"),
        (0xa93a, 0xa93a, "Reset VDU: resets default VDU mode (CoCo $95ac)"),
        (0xa989, 0xa989, "Set VDG Mode given in A (CoCo $95fb)"),
        (0xa99d, 0xa99d, "Set VDG Offset for graphics mode (CoCo $960f)"),
        (0xa9a4, 0xa9a4, "Set VDG Colour set from $00c1 (CoCo $9616)"),
        (0xa9af, 0xa9af, "PMODE token dispatch address"),
        (0xa9e1, 0xa9e1, "Set Page to B (CoCo $9653)"),
        (0xa9fe, 0xa9fe, "SCREEN token dispatch address"),
        (0xaa10, 0xaa10, "Select Colour set 0 or 1 from B (CoCo $9682)"),
        (0xaa19, 0xaa19, "PCLEAR token dispatch address"),
        (0xaa23, 0xaa23, "Reserve HiRes Graphics RAM, moves BASIC if nec. (CoCo $9695)"),
        (0xaa81, 0xaa81, "Called from BASIC init routine ($b3ba-)"),
        (0xaabe, 0xaabe, "PCOPY token dispatch address"),
        (0xaada, 0xaaef, "Sub - reads and checks PCOPY page numbers"),
        (0xaaf0, 0xaaf0, "GET token dispatch address"),
        (0xaaf3, 0xaaf3, "PUT token dispatch address"),
        (0xabd4, 0xabec, "5 entry table of form Word, Word, Token for matching PUT mode"),
        (0xac87, 0xac87, "PAINT token dispatch address"),
        (0xadbd, 0xadbd, "PLAY token dispatch address"),
        (0xae9a, 0xae9a, "Play Note: A contains ASCII code (CoCo $9ad7)"),
        (0xaf33, 0xaf33, "Sub - gets next PLAY cmd in A"),
        (0xafd9, 0xafd9, "PLAY IRQ routine - branches to $bb02 if $00e3 == 0x0000"),
        (0xaff6, 0xaffc, "Table maps PLAY notes A-G to tone number"),
        (0xaffd, 0xb014, "Table"),
        (0xb015, 0xb050, "Table"),
        (0xb051, 0xb051, "DRAW token dispatch address (CoCo $9cb6)"),
        (0xb238, 0xb238, "CIRCLE token dispatch address"),
        (0xb39b, 0xb3b3, "Called after Hardware init routine, following a RESET Inits stack, checks for Cold/warm start.  If $0071 contains 0x55 and $0072:0073 points to NOP then execution is passed to [$0072:0073] otherwise branches to $b3ba"),
        (0xb3b4, 0xb3b9, "RESET interrupt service routine (CoCo $a027) Loads Y with $b39b and JMPs $8000"),
        (0xb3ba, 0xb3ba, "Cold start routine - clears lo mem, inits BASIC"),
        (0xb400, 0xb400, "Boot Basic: Restarts BASIC interpreter (CoCo $a0b6)"),
        (0xb432, 0xb43b, "Checks for $c000:c001 = 0x444b ('DK') and branches to $c002 if found"),
        (0xb44f, 0xb44f, "Warm start RESET jumps here"),
        (0xb469, 0xb46e, "FIRQ interrupt service routine - branches to $b46f if source was cartridge else RTI"),
        (0xb46f, 0xb47a, "Calls $b480 twice, then JMPs to $8000 with Y = $b47b"),
        (0xb47b, 0xb47f, "CLRs $0071, then JMPs $c000 to start cartridge"),
        (0xb480, 0xb486, "Loads X from $008a (0x0000) and decrements to 0x0000 Gives 524,298 cycle, approx. 590ms delay"),
        (0xb487, 0xb494, "14 bytes copied to $009d-$00aa at startup"),
        (0xb495, 0xb4b2, "30 bytes copied to $010c-$0129 at startup"),
        (0xb4b3, 0xb4cc, "'(C) 1982 DRAGON DATA LTD ', 0x0d"),
        (0xb4cd, 0xb4ec, "'16K BASIC INTERPRETER 1.0      ', 0x0d"),
        (0xb4ed, 0xb504, "'(C) 1982 BY MICROSOFT', 0x0d, 0x0d, 0x00"),
        (0xb505, 0xb505, "Waits for key with cursor ? Called from SuperDos E6 ROM"),
        (0xb54a, 0xb54a, "%OUTCHR% Sends char in A to DEVN (CoCo $a282) Called indirectly from [$a002]"),
        (0xb64c, 0xb64c, "CLOSE token dispatch address"),
        (0xb65f, 0xb65f, "Close files: closes an open tape stream (CoCo $a42d)"),
        (0xb682, 0xb682, "CSAVE token dispatch address"),
        (0xb6d4, 0xb6d4, "CLOAD token dispatch address"),
        (0xb6a5, 0xb6a5, "Write BASIC: writes BASIC program to cassette (CoCo $a469)"),
        (0xb714, 0xb714, "Loads a headerless program from cassette {10}"),
        (0xb748, 0xb748, "Read Binary file from tape (CoCo $a511)"),
        (0xb770, 0xb770, "EXEC token dispatch address"),
        (0xb796, 0xb796, "INKEY$ function token dispatch address"),
        (0xb7cc, 0xb7cc, "Copy B bytes from X to U"),
        (0xb800, 0xb800, "EOF function token dispatch address"),
        (0xb81e, 0xb81e, "SKIPF token dispatch address"),
        (0xb828, 0xb828, "OPEN token dispatch address"),
        (0xb8b3, 0xb8b3, "Find File: searches tape for matching filename (CoCo $a681)"),
        (0xb933, 0xb933, "Read 1st Block: reads filename block to cas. buf (CoCo $a701)"),
        (0xb93e, 0xb93e, "%BLKIN% reads a block of data into cas. buffer (CoCo $a70b) Called indirectly from [$a006]"),
        (0xb981, 0xb981, "MOTOR token dispatch address"),
        (0xb991, 0xb991, "Write 1st Block (CoCo $a7e5)"),
        (0xb999, 0xb9d1, "%BLKOUT% writes a block of data to cassette (CoCo $a7f4) Called indirectly from [$a008]"),
        (0xb9cd, 0xb9d1, "Write byte 0x55 to cassette by JMPing to $801e"),
        (0xb9cf, 0xb9d1, "JMPs to $801e to write byte in A"),
        (0xb9d2, 0xb9d2, "SET token dispatch address"),
        (0xb9df, 0xb9df, "Set LoRes Pixel: X contains screen addr, B the colour and $00b8 the OR data (CoCo $a88d)"),
        (0xba03, 0xba03, "RESET token dispatch address"),
        (0xba07, 0xba07, "Reset LoRes Pixel: X contains screen addr, $00b8 OR data (CoCo $a8b5)"),
        (0xba28, 0xba28, "Calc LoRes Pixel Pos: Stack contains vert and horiz coords (CoCo $a8d9)"),
        (0xba44, 0xba44, "POINT function token dispatch address"),
        (0xba5f, 0xba5f, "CLS token dispatch address"),
        (0xba77, 0xba85, "Clears text screen (CoCo $a928)"),
        (0xba79, 0xba85, "Clears text screen with value in B (CoCo $a92a)"),
        (0xba86, 0xba8d, "Clears screen, prints '(C) 1982 BY MICROSOFT' msg at $b4ec"),
        (0xba9a, 0xba9a, "SOUND token dispatch address"),
        (0xbaa0, 0xbaa0, "Produces Beep of length B, pitch in $008c (CoCo $a951)"),
        (0xbac3, 0xbac3, "Audio off: Disables sound (CoCo $a974)"),
        (0xbac5, 0xbad3, "Enables sound (CoCo $a976)"),
        (0xbad4, 0xbade, "Reset DAC - writes (D32 0x7e; D64 0x7c) to D/A (CoCo $a985)"),
        (0xbad6, 0xbade, "Writes A to DAC - bits 0,1 of A must be 0 (CoCo $a987)"),
        (0xbadf, 0xbadf, "AUDIO token dispatch address"),
        (0xbaec, 0xbaec, "AUDIO ON: B must be 0x00 on entry (CoCo $a99d)"),
        (0xbaf1, 0xbb01, "Sub to init PIA for sound source output specified by the lower 2 bits of B - 00 = DAC, 01 = Tape input, 10 = External Cart source, 11 = none This code duplicates the function of $bd41-bd51, the only dif being an ASRB replacing a RORB"),
        (0xbb02, 0xbb0c, "Interrupt service subroutine to decrement $008d:008e, returns with RTI"),
        (0xbb0d, 0xbb0d, "JOYSTK function token dispatch address"),
        (0xbb26, 0xbb34, "Jumped to from selfmodifying CHRGET routine at $009f"),
        (0xbb3c, 0xbb87, "D64 - Hardware init, JMPd to from $8000, Y should contain return address"),
        (0xbb40, 0xbb87, "D32 - Hardware init, JMPd to from $8000, Y should contain return address"),
        (0xbb80, 0xbb82, "D64 - Boots 64K mode with JMP $bf49"),
        (0xbb88, 0xbb9e, "Software init, JMPd to from $8003, inits $008f-009b, $0148-0150 with ROM defaults"),
        (0xbb92, 0xbb94, "D64 - patched to JSR $BEFF (CLR $03fd-03ff; LDU #$0148)"),
        (0xbb97, 0xbb9e, "CopyXtoU - copies B bytes from X to U"),
        (0xbb9f, 0xbbab, "default values for $008f-009b"),
        (0xbbac, 0xbbb4, "default values for $0148-0150"),
        (0xbbb5, 0xbbcc, "Flashes cursor, JMPd to from $8009 (CoCo $a199)"),
        (0xbbc5, 0xbbcc, "Decrements X from 0x045e, gives 1ms delay"),
        (0xbbc8, 0xbbcc, "Decrements X until 0 then returns"),
        (0xbbcd, 0xbbd9, "subroutine"),
        (0xbbe5, 0xbbea, "%INCH% Scans keyboard, returns char in A, JMPd to from $8006 (CoCo $a1c1)"),
        (0xbbec, 0xbbec, "Keyboard polling sub, returns key code in A"),
        (0xbca0, 0xbca0, "Clear VDU line from cursor posn (CoCo $a323)"),
        (0xbcab, 0xbcab, "%OUTCH% Put char in A to screen, JMPd to from $800c (CoCo $a30a)"),
        (0xbcf5, 0xbd09, "%TXLPCH%  Low level routine to print char in A (CoCo $)"),
        (0xbd0a, 0xbd0a, "PCRLF: moves print head to next line"),
        (0xbd1a, 0xbd1a, "Send char in A to printer, JMPd to from $800f (CoCo $a2bf) D64 - This routine uses Y as well and is source of a bug in the Clear-J fn of Visitext Word-Pro."),
        (0xbd41, 0xbd51, "%SNDSEL%  Selects Analogue Mux channel from B.  Used to read Joysticks or set sound channel (CoCo $a9a2) 0x00 DAC       / Right Joystick, x-axis 0x01 Cassette  / Right Joystick, y-axis 0x02 Cartridge / Left Joystick, x-axis 0x03 Unused    / Left Joystick, y-axis See also $baf1-bb01"),
        (0xbd52, 0xbd52, "%JOYIN% Reads Joysticks updates $015a-015d, JMPd to from $8012 (CoCo $a9de)"),
        (0xbd8a, 0xbd90, "Increments $0082, samples Cassette input bit into CC.C"),
        (0xbd91, 0xbd9d, "Sub"),
        (0xbd97, 0xbd9d, "Sub"),
        (0xbd99, 0xbd9d, "Sub"),
        (0xbd9e, 0xbd9e, "Sub"),
        (0xbda0, 0xbda0, "Sub"),
        (0xbda5, 0xbda5, "%BITIN% Get one bit from cassette to CC.C, JMPd to from $8027 (CoCo $a755)"),
        (0xbdad, 0xbdad, "%CBIN% Input byte from cassette into A, JMPd to from $8024 (CoCo $a749)"),
        (0xbdb9, 0xbdb9, "Sub"),
        (0xbdbf, 0xbdbf, "Sub"),
        (0xbdcf, 0xbddb, "%CASON% Motor on - Turns on relay, loads X with delay value from $0095:0096 (0xda5c) then LBRAs to $bbc8, JMPd to from $8015 (CoCo $a7ca)"),
        (0xbddc, 0xbde6, "%CASOFF% Motor off - turns relay off, re-enables FIRQ, IRQ, JMPd to from $8018 (CoCo $a7eb)"),
        (0xbde7, 0xbde7, "%CSRDON% Read Leader - Disables FIRQ, IRQ, Cassette on, init for reading, JMPd to from $8021 (CoCo $a77c)"),
        (0xbe12, 0xbe43, "%CBOUT% Output byte in A to cassette, JMPd to from $801e (CoCo $a82a)"),
        (0xbe44, 0xbe67, "36 bytes of Sine Wave data in upper 5 bits of each byte."),
        (0xbe68, 0xbe7a, "%WRTLDR% disables FIRQ, IRQ, turns on cassette, writes number of leader bytes = 0x55 according to $0090:0091 (d32 0x0080), JMPd to from $801b Pre: B,Y Des: CC,A,X Ret: X=0x0000,A=0x55"),
        (0xbe7b, 0xbe7b, "D32 - JMPd to from $802a (RTS)"),
        (0xbe7b, 0xbe97, "D64 - Read byte from serial port to A, JMPd to from $802a"),
        (0xbe7c, 0xbe7c, "D32 - JMPd to from $802d (RTS)"),
        (0xbe7d, 0xbe7e, "D32 - JMPd to from $8030 (COMB; RTS)"),
        (0xbe7f, 0xbfef, "D32 - Unused area, all 0x00"),
        (0xbe98, 0xbea5, "D64 - Output byte in A to serial port, JMPd to from $802d"),
        (0xbea6, 0xbebd, "D64 - Set baud rate in B, JMPd to from $8030"),
        (0xbebe, 0xbec4, "D64 - Lookup table to map B to ACIA baud rate constant"),
        (0xbec5, 0xbeda, "D64 - Key scan which pauses printer during printing {8}"),
        (0xbeff, 0xbf0b, "D64 - Patched into Software init to CLR $03fd-03ff"),
        (0xbf0c, 0xbf1f, "D64 - handles serial printer end of line delay"),
        (0xbf20, 0xbf48, "D64 - IRQ routine which can be hooked in D32 mode to provide 64 mode style auto-repeat and serial support"),
        (0xbf49, 0xbf59, "D64 - Copies 64 mode bootstrap code to $01da and JMPs EXEC points here at startup."),
        (0xbf5a, 0xbfe8, "D64 - Bootstrap routine code copied to $01da-0268"),
        (0xbf9c, 0xbf9d, "D64 - '64' Can be used to check for D64 32k mode ROM"),
        (0xbfe9, 0xbfef, "D64 - Unused 0x00"),
        (0xbff0, 0xbfff, "6809 interrupt vectors mapped to $fff0-$ffff by SAM"),
        (0xbff0, 0xfff1, "Reserved     ($0000)"),
        (0xbff2, 0xbff3, "SWI3 vector  ($0100)"),
        (0xbff4, 0xbff5, "SWI2 vector  ($0103)"),
        (0xbff6, 0xbff7, "FIRQ vector  ($010f)"),
        (0xbff8, 0xbff9, "IRQ vector   ($010c)"),
        (0xbffa, 0xbffb, "SWI vector   ($0106)"),
        (0xbffc, 0xbffd, "NMI vector   ($0109)"),
        (0xbffe, 0xbfff, "RESET vector ($b3b4)"),
    )


def print_out(txt):
    print(txt)


def get_dragon_meminfo():
    return DragonMemInfo(log.debug)


if __name__ == "__main__":
#     mem_info = DragonMemInfo(print_out)
#
#     # 0xaf-0xaf - TRON/TROFF trace flag - non zero for TRON
#     mem_info(0xaf)
#     mem_info(0xaf, shortest=False)
#     print
#
#     # 5x entries
#     mem_info(0xbff0)
#     mem_info(0xbff0, shortest=False)
#     print
#
#     # 0xf-0x18 - Temporary results
#     mem_info(0xf)
#     mem_info(0xf, shortest=False)
#     print
#     mem_info(0x10)
#     mem_info(0x10, shortest=False)
#     print
#     mem_info(0x18)
#     mem_info(0x18, shortest=False)
#     print
#
#     print "\n --- END --- \n"
    for s, e, txt in DragonMemInfo.MEM_INFO:
        if s == e:
            addr = nice_hex(s)
        else:
            addr = "%s-%s" % (nice_hex(s), nice_hex(e))

        print("%-11s ; %s" % (addr, txt))
