"""InnerLoop CLI."""

from __future__ import annotations

import typer

from ..home import set_home
from .agent import agent_app
from .run import run_command
from .schema import schema_app
from .server import server_app

app = typer.Typer(
    name="innerloop",
    help="Python SDK for headless OpenCode invocation",
    no_args_is_help=True,
)

# Add subcommands
app.add_typer(server_app, name="server")
app.add_typer(schema_app, name="schema")
app.add_typer(agent_app, name="agent")


@app.callback()
def main(
    home: str | None = typer.Option(
        None,
        "--home",
        help="Override home directory",
        envvar="BOTASSEMBLY_HOME",
    ),
) -> None:
    """InnerLoop - Headless OpenCode invocation with structured outputs."""
    if home:
        set_home(home)


# Default command (run)
@app.command(name="run")
def run(
    prompt: str = typer.Argument(
        ..., help="Prompt to execute (or read from stdin with -)"
    ),
    model: str = typer.Option(
        "anthropic/claude-sonnet-4-5", "--model", "-m", help="Model to use"
    ),
    struct: str | None = typer.Option(
        None, "--struct", help="Structured output schema name"
    ),
    session: str | None = typer.Option(
        None, "--session", "-s", help="Continue existing session"
    ),
    timeout: float | None = typer.Option(
        None, "--timeout", "-t", help="Idle timeout in seconds"
    ),
    stream: bool = typer.Option(False, "--stream", help="Stream events"),
    format: str = typer.Option(
        "text",
        "--format",
        "-f",
        help="Output format: text, json, events",
    ),
    server_url: str | None = typer.Option(
        None, "--server", help="OpenCode server URL (enables server mode)"
    ),
) -> None:
    """Execute a prompt via OpenCode (subprocess or server mode)."""
    run_command(
        prompt=prompt,
        model=model,
        struct=struct,
        session=session,
        timeout=timeout,
        stream=stream,
        format=format,
        server_url=server_url,
    )


def cli() -> None:
    """CLI entry point."""
    app()


if __name__ == "__main__":
    cli()


__all__ = ["app", "cli"]
