"""Server management commands."""

from __future__ import annotations

import typer

from ..server.manager import get_status, start_server, stop_server

server_app = typer.Typer(name="server", help="Manage OpenCode server")


@server_app.command()
def start(
    port: int = typer.Option(4096, "--port", "-p", help="Server port"),
    hostname: str = typer.Option(
        "127.0.0.1", "--hostname", "-h", help="Server hostname"
    ),
) -> None:
    """Start OpenCode server in background."""
    typer.echo(f"Starting OpenCode server on {hostname}:{port}...")

    proc = start_server(port=port, hostname=hostname, background=True)

    if proc:
        typer.echo(f"✓ Server started (PID: {proc.pid})")
        typer.echo(f"  URL: http://{hostname}:{port}")
    else:
        typer.echo("✗ Failed to start server", err=True)
        raise typer.Exit(1)


@server_app.command()
def stop(
    url: str = typer.Option(
        "http://127.0.0.1:4096", "--url", help="Server URL"
    ),
) -> None:
    """Stop running OpenCode server."""
    typer.echo(f"Stopping OpenCode server at {url}...")

    if stop_server(url):
        typer.echo("✓ Server stopped")
    else:
        typer.echo("✗ Server not running", err=True)
        raise typer.Exit(1)


@server_app.command()
def status(
    url: str = typer.Option(
        "http://127.0.0.1:4096", "--url", help="Server URL"
    ),
) -> None:
    """Check OpenCode server status."""
    stat = get_status(url)

    if stat["running"]:
        typer.echo(f"✓ Server running at {stat['url']}")
        if stat["agents"]:
            typer.echo(f"  Agents: {len(stat['agents'])}")
            for agent in stat["agents"]:
                typer.echo(f"    - {agent.get('name')}")
    else:
        typer.echo(f"✗ Server not running at {stat['url']}")
        raise typer.Exit(1)


@server_app.command()
def restart(
    port: int = typer.Option(4096, "--port", "-p", help="Server port"),
    hostname: str = typer.Option(
        "127.0.0.1", "--hostname", "-h", help="Server hostname"
    ),
    url: str = typer.Option(
        "http://127.0.0.1:4096", "--url", help="Current server URL"
    ),
) -> None:
    """Restart OpenCode server (for config changes)."""
    typer.echo("Restarting OpenCode server...")

    # Stop if running
    if stop_server(url):
        typer.echo("  Stopped old server")

    # Start new
    proc = start_server(port=port, hostname=hostname, background=True)

    if proc:
        typer.echo(f"✓ Server restarted (PID: {proc.pid})")
        typer.echo(f"  URL: http://{hostname}:{port}")
    else:
        typer.echo("✗ Failed to restart server", err=True)
        raise typer.Exit(1)


__all__ = ["server_app"]
