"""SlimSchema management for InnerLoop.

Stores schemas as .ss.yaml files (SlimSchema YAML format).
Location: ~/.local/botassembly/innerloop/schemas/
"""

from __future__ import annotations

from pathlib import Path
from typing import Any

from pydantic import BaseModel, TypeAdapter
from slimschema import to_schema  # type: ignore[import-untyped]

from .home import get_home


def get_schemas_dir() -> Path:
    """Get slimschema directory path.

    Returns:
        Path to slimschema directory (does not create)
    """
    return get_home() / "slimschema"


def save_schema(
    name: str, response_format: type[BaseModel] | TypeAdapter[Any]
) -> Path:
    """Save Pydantic model as SlimSchema YAML file.

    Args:
        name: Schema name (without extension)
        response_format: Pydantic BaseModel class or TypeAdapter

    Returns:
        Path to saved schema file

    Example:
        >>> from pydantic import BaseModel
        >>> class Person(BaseModel):
        ...     name: str
        ...     age: int
        >>> save_schema("person", Person)
        Path('~/.local/botassembly/innerloop/schemas/person.ss.yaml')
    """
    schemas_dir = get_schemas_dir()
    schemas_dir.mkdir(parents=True, exist_ok=True)

    schema = to_schema(response_format)
    schema_file = schemas_dir / f"{name}.ss.yaml"
    schema_file.write_text(str(schema))

    return schema_file


def load_schema(name: str) -> str:
    """Load SlimSchema YAML file by name.

    Args:
        name: Schema name (without extension)

    Returns:
        Schema content as YAML string

    Raises:
        FileNotFoundError: If schema doesn't exist
    """
    schema_file = get_schemas_dir() / f"{name}.ss.yaml"

    if not schema_file.exists():
        raise FileNotFoundError(f"Schema not found: {name}")

    return schema_file.read_text()


def list_schemas() -> list[str]:
    """List all available schema names.

    Returns:
        List of schema names (without .ss.yaml extension)
    """
    schemas_dir = get_schemas_dir()

    if not schemas_dir.exists():
        return []

    schema_files = schemas_dir.glob("*.ss.yaml")
    # Remove .ss.yaml suffix (f.stem only removes .yaml, leaving .ss)
    return sorted(f.name.removesuffix(".ss.yaml") for f in schema_files)


def delete_schema(name: str) -> bool:
    """Delete a schema file.

    Args:
        name: Schema name (without extension)

    Returns:
        True if deleted, False if didn't exist
    """
    schema_file = get_schemas_dir() / f"{name}.ss.yaml"

    if not schema_file.exists():
        return False

    schema_file.unlink()
    return True


__all__ = [
    "get_schemas_dir",
    "save_schema",
    "load_schema",
    "list_schemas",
    "delete_schema",
]
