"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.contentFingerprint = exports.fingerprint = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const ignore_1 = require("./ignore");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
const CR = '\r';
const LF = '\n';
const CRLF = `${CR}${LF}`;
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * Line endings are converted from CRLF to LF.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    const rootDirectory = fs.statSync(fileOrDirectory).isDirectory()
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const ignoreMode = options.ignoreMode || options_1.IgnoreMode.GLOB;
    if (ignoreMode != options_1.IgnoreMode.GLOB) {
        _hashField(hash, 'options.ignoreMode', ignoreMode);
    }
    const ignoreStrategy = ignore_1.IgnoreStrategy.fromCopyOptions(options, fileOrDirectory);
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (!isRootDir && ignoreStrategy.ignores(symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, contentFingerprint(realPath));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function contentFingerprint(file) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // eslint-disable-next-line no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    let size = 0;
    let isBinary = false;
    let lastStr = '';
    let read = 0;
    try {
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            const slicedBuffer = buffer.slice(0, read);
            // Detect if file is binary by checking the first 8k bytes for the
            // null character (git like implementation)
            if (size === 0) {
                isBinary = slicedBuffer.indexOf(0) !== -1;
            }
            let dataBuffer = slicedBuffer;
            if (!isBinary) { // Line endings normalization (CRLF -> LF)
                const str = buffer.slice(0, read).toString();
                // We are going to normalize line endings to LF. So if the current
                // buffer ends with CR, it could be that the next one starts with
                // LF so we need to save it for later use.
                if (new RegExp(`${CR}$`).test(str)) {
                    lastStr += str;
                    continue;
                }
                const data = lastStr + str;
                const normalizedData = data.replace(new RegExp(CRLF, 'g'), LF);
                dataBuffer = Buffer.from(normalizedData);
                lastStr = '';
            }
            size += dataBuffer.length;
            hash.update(dataBuffer);
        }
        if (lastStr) {
            hash.update(Buffer.from(lastStr));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${size}:${hash.digest('hex')}`;
}
exports.contentFingerprint = contentFingerprint;
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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