"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VERSION_LOCKED = exports.trimFromStart = exports.calculateFunctionHash = void 0;
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const cx_api_1 = require("@aws-cdk/cx-api");
const function_1 = require("./function");
function calculateFunctionHash(fn) {
    const stack = core_1.Stack.of(fn);
    const functionResource = fn.node.defaultChild;
    // render the cloudformation resource from this function
    const config = stack.resolve(functionResource._toCloudFormation());
    // config is of the shape: { Resources: { LogicalId: { Type: 'Function', Properties: { ... } }}}
    const resources = config.Resources;
    const resourceKeys = Object.keys(resources);
    if (resourceKeys.length !== 1) {
        throw new Error(`Expected one rendered CloudFormation resource but found ${resourceKeys.length}`);
    }
    const logicalId = resourceKeys[0];
    const properties = resources[logicalId].Properties;
    let stringifiedConfig;
    if (core_1.FeatureFlags.of(fn).isEnabled(cx_api_1.LAMBDA_RECOGNIZE_VERSION_PROPS)) {
        const updatedProps = sortProperties(filterUsefulKeys(properties));
        stringifiedConfig = JSON.stringify(updatedProps);
    }
    else {
        const sorted = sortProperties(properties);
        config.Resources[logicalId].Properties = sorted;
        stringifiedConfig = JSON.stringify(config);
    }
    if (core_1.FeatureFlags.of(fn).isEnabled(cx_api_1.LAMBDA_RECOGNIZE_LAYER_VERSION)) {
        stringifiedConfig = stringifiedConfig + calculateLayersHash(fn._layers);
    }
    const hash = crypto.createHash('md5');
    hash.update(stringifiedConfig);
    return hash.digest('hex');
}
exports.calculateFunctionHash = calculateFunctionHash;
function trimFromStart(s, maxLength) {
    const desiredLength = Math.min(maxLength, s.length);
    const newStart = s.length - desiredLength;
    return s.substring(newStart);
}
exports.trimFromStart = trimFromStart;
/*
 * The list of properties found in CfnFunction (or AWS::Lambda::Function).
 * They are classified as "locked" to a Function Version or not.
 * When a property is locked, any change to that property will not take effect on previously created Versions.
 * Instead, a new Version must be generated for the change to take effect.
 * Similarly, if a property that's not locked to a Version is modified, a new Version
 * must not be generated.
 *
 * Adding a new property to this list - If the property is part of the UpdateFunctionConfiguration
 * API or UpdateFunctionCode API, then it must be classified as true, otherwise false.
 * See https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html and
 * https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html
 */
exports.VERSION_LOCKED = {
    // locked to the version
    Architectures: true,
    Code: true,
    DeadLetterConfig: true,
    Description: true,
    Environment: true,
    EphemeralStorage: true,
    FileSystemConfigs: true,
    FunctionName: true,
    Handler: true,
    ImageConfig: true,
    KmsKeyArn: true,
    Layers: true,
    MemorySize: true,
    PackageType: true,
    Role: true,
    Runtime: true,
    SnapStart: true,
    Timeout: true,
    TracingConfig: true,
    VpcConfig: true,
    // not locked to the version
    CodeSigningConfigArn: false,
    ReservedConcurrentExecutions: false,
    Tags: false,
};
function filterUsefulKeys(properties) {
    const versionProps = { ...exports.VERSION_LOCKED, ...function_1.Function._VER_PROPS };
    const unclassified = Object.entries(properties)
        .filter(([k, v]) => v != null && !Object.keys(versionProps).includes(k))
        .map(([k, _]) => k);
    if (unclassified.length > 0) {
        throw new Error(`The following properties are not recognized as version properties: [${unclassified}].`
            + ' See the README of the aws-lambda module to learn more about this and to fix it.');
    }
    const notLocked = Object.entries(versionProps).filter(([_, v]) => !v).map(([k, _]) => k);
    notLocked.forEach(p => delete properties[p]);
    const ret = {};
    Object.entries(properties).filter(([k, _]) => versionProps[k]).forEach(([k, v]) => ret[k] = v);
    return ret;
}
function sortProperties(properties) {
    const ret = {};
    // We take all required properties in the order that they were historically,
    // to make sure the hash we calculate is stable.
    // There cannot be more required properties added in the future,
    // as that would be a backwards-incompatible change.
    const requiredProperties = ['Code', 'Handler', 'Role', 'Runtime'];
    for (const requiredProperty of requiredProperties) {
        ret[requiredProperty] = properties[requiredProperty];
    }
    // then, add all of the non-required properties,
    // in the original order
    for (const property of Object.keys(properties)) {
        if (requiredProperties.indexOf(property) === -1) {
            ret[property] = properties[property];
        }
    }
    return ret;
}
function calculateLayersHash(layers) {
    const layerConfig = {};
    for (const layer of layers) {
        const stack = core_1.Stack.of(layer);
        const layerResource = layer.node.defaultChild;
        // if there is no layer resource, then the layer was imported
        // and we will include the layer arn and runtimes in the hash
        if (layerResource === undefined) {
            layerConfig[layer.layerVersionArn] = layer.compatibleRuntimes;
            continue;
        }
        const config = stack.resolve(layerResource._toCloudFormation());
        const resources = config.Resources;
        const resourceKeys = Object.keys(resources);
        if (resourceKeys.length !== 1) {
            throw new Error(`Expected one rendered CloudFormation resource but found ${resourceKeys.length}`);
        }
        const logicalId = resourceKeys[0];
        const properties = resources[logicalId].Properties;
        // all properties require replacement, so they are all version locked.
        layerConfig[layer.node.id] = properties;
    }
    const hash = crypto.createHash('md5');
    hash.update(JSON.stringify(layerConfig));
    return hash.digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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