"""
Auto-instrumentation for popular AI/ML libraries
"""

import functools
import logging
from typing import Any, Dict, Optional, Callable, TYPE_CHECKING
import asyncio

from ..client.models import EventType
from ..utils.exceptions import InstrumentationError

if TYPE_CHECKING:
    from ..client.telemetry_client import TelemetryClient


class AutoInstrumentation:
    """Automatic instrumentation for popular AI/ML libraries"""
    
    def __init__(self, client: 'TelemetryClient'):
        self.client = client
        self.original_methods: Dict[str, Dict[str, Any]] = {}
        self.patched_modules = set()
        self._logger = logging.getLogger(__name__)

    def instrument_all(self) -> None:
        """Instrument all supported libraries"""
        instrumentation_methods = [
            self.instrument_openai,
            self.instrument_anthropic,
            self.instrument_langchain,
            self.instrument_llamaindex,
            self.instrument_requests,
            self.instrument_httpx,
        ]
        
        for method in instrumentation_methods:
            try:
                method()
            except Exception as e:
                self._logger.debug(f"Failed to instrument with {method.__name__}: {e}")

    def uninstrument_all(self) -> None:
        """Remove all instrumentation"""
        for module_name, methods in self.original_methods.items():
            try:
                module = __import__(module_name, fromlist=[''])
                for method_path, original_method in methods.items():
                    self._set_nested_attr(module, method_path, original_method)
            except ImportError:
                continue
            except Exception as e:
                self._logger.error(f"Failed to uninstrument {module_name}: {e}")
        
        self.original_methods.clear()
        self.patched_modules.clear()

    def _patch_method(self, module_name: str, method_path: str, wrapper_func: Callable) -> bool:
        """Generic method patching utility"""
        try:
            module = __import__(module_name, fromlist=[''])
            original_method = self._get_nested_attr(module, method_path)
            
            if original_method is None:
                return False
            
            # Store original method
            if module_name not in self.original_methods:
                self.original_methods[module_name] = {}
            self.original_methods[module_name][method_path] = original_method
            
            # Apply wrapper
            wrapped_method = wrapper_func(original_method)
            self._set_nested_attr(module, method_path, wrapped_method)
            
            self.patched_modules.add(module_name)
            self._logger.debug(f"Successfully patched {module_name}.{method_path}")
            return True
            
        except ImportError:
            self._logger.debug(f"Module {module_name} not available for instrumentation")
            return False
        except Exception as e:
            self._logger.error(f"Failed to patch {module_name}.{method_path}: {e}")
            return False

    def _get_nested_attr(self, obj: Any, path: str) -> Any:
        """Get nested attribute using dot notation"""
        for attr in path.split('.'):
            obj = getattr(obj, attr, None)
            if obj is None:
                return None
        return obj

    def _set_nested_attr(self, obj: Any, path: str, value: Any) -> None:
        """Set nested attribute using dot notation"""
        attrs = path.split('.')
        for attr in attrs[:-1]:
            obj = getattr(obj, attr)
        setattr(obj, attrs[-1], value)

    def instrument_openai(self) -> None:
        """Instrument OpenAI client"""
        
        def create_openai_wrapper(original_method):
            if asyncio.iscoroutinefunction(original_method):
                @functools.wraps(original_method)
                async def async_wrapper(*args, **kwargs):
                    # Extract model and messages from args/kwargs
                    model = kwargs.get('model', 'unknown')
                    messages = kwargs.get('messages', [])
                    
                    input_text = ""
                    if messages and isinstance(messages, list) and len(messages) > 0:
                        last_message = messages[-1]
                        if isinstance(last_message, dict):
                            input_text = str(last_message.get('content', ''))
                        else:
                            input_text = str(last_message)
                    
                    async with self.client.trace_model_call(
                        provider="openai",
                        model=model,
                        source_component="openai_auto"
                    ) as span:
                        span.set_input(input_text[:1000])
                        
                        try:
                            result = await original_method(*args, **kwargs)
                            
                            # Extract response details based on response structure
                            if hasattr(result, 'choices') and result.choices:
                                choice = result.choices[0]
                                if hasattr(choice, 'message') and hasattr(choice.message, 'content'):
                                    span.set_output(choice.message.content[:1000])
                                elif hasattr(choice, 'text'):
                                    span.set_output(choice.text[:1000])
                                    
                                if hasattr(choice, 'finish_reason'):
                                    span.set_metadata("finish_reason", choice.finish_reason)
                            
                            if hasattr(result, 'usage'):
                                usage = result.usage
                                if hasattr(usage, 'total_tokens'):
                                    span.set_tokens(usage.total_tokens)
                                
                                # Estimate cost
                                cost = self._estimate_openai_cost(model, usage.total_tokens)
                                if cost:
                                    span.set_cost(cost)
                            
                            return result
                            
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                            
                return async_wrapper
            else:
                @functools.wraps(original_method)
                def sync_wrapper(*args, **kwargs):
                    # For sync methods, we'll use the sync tracing
                    model = kwargs.get('model', 'unknown')
                    messages = kwargs.get('messages', [])
                    
                    input_text = ""
                    if messages and isinstance(messages, list) and len(messages) > 0:
                        last_message = messages[-1]
                        if isinstance(last_message, dict):
                            input_text = str(last_message.get('content', ''))
                    
                    with self.client.trace_model_call_sync(
                        provider="openai",
                        model=model,
                        source_component="openai_auto"
                    ) as span:
                        span.set_input(input_text[:1000])
                        
                        try:
                            result = original_method(*args, **kwargs)
                            
                            # Extract response details
                            if hasattr(result, 'choices') and result.choices:
                                choice = result.choices[0]
                                if hasattr(choice, 'message') and hasattr(choice.message, 'content'):
                                    span.set_output(choice.message.content[:1000])
                                elif hasattr(choice, 'text'):
                                    span.set_output(choice.text[:1000])
                            
                            if hasattr(result, 'usage') and hasattr(result.usage, 'total_tokens'):
                                span.set_tokens(result.usage.total_tokens)
                            
                            return result
                            
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                
                return sync_wrapper

        # Patch different OpenAI client versions and methods
        patterns = [
            ("openai", "ChatCompletion.acreate"),
            ("openai", "ChatCompletion.create"),
            ("openai", "Completion.acreate"), 
            ("openai", "Completion.create"),
            ("openai.resources.chat.completions", "Completions.create"),
            ("openai.resources.completions", "Completions.create"),
        ]
        
        for module_name, method_path in patterns:
            self._patch_method(module_name, method_path, create_openai_wrapper)

    def instrument_anthropic(self) -> None:
        """Instrument Anthropic Claude client"""
        
        def create_anthropic_wrapper(original_method):
            if asyncio.iscoroutinefunction(original_method):
                @functools.wraps(original_method)
                async def async_wrapper(*args, **kwargs):
                    model = kwargs.get('model', 'claude')
                    messages = kwargs.get('messages', [])
                    
                    input_text = ""
                    if messages and isinstance(messages, list) and len(messages) > 0:
                        last_message = messages[-1]
                        if isinstance(last_message, dict):
                            input_text = str(last_message.get('content', ''))
                    
                    async with self.client.trace_model_call(
                        provider="anthropic",
                        model=model,
                        source_component="anthropic_auto"
                    ) as span:
                        span.set_input(input_text[:1000])
                        
                        try:
                            result = await original_method(*args, **kwargs)
                            
                            if hasattr(result, 'content') and result.content:
                                if isinstance(result.content, list) and len(result.content) > 0:
                                    content = result.content[0]
                                    if hasattr(content, 'text'):
                                        span.set_output(content.text[:1000])
                                else:
                                    span.set_output(str(result.content)[:1000])
                            
                            if hasattr(result, 'usage'):
                                usage = result.usage
                                total_tokens = 0
                                if hasattr(usage, 'input_tokens'):
                                    total_tokens += usage.input_tokens
                                if hasattr(usage, 'output_tokens'):
                                    total_tokens += usage.output_tokens
                                if total_tokens > 0:
                                    span.set_tokens(total_tokens)
                            
                            return result
                            
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                            
                return async_wrapper
            else:
                @functools.wraps(original_method)
                def sync_wrapper(*args, **kwargs):
                    return original_method(*args, **kwargs)
                return sync_wrapper

        patterns = [
            ("anthropic", "Anthropic.messages.create"),
            ("anthropic", "AsyncAnthropic.messages.create"),
            ("anthropic.resources.messages", "Messages.create"),
        ]
        
        for module_name, method_path in patterns:
            self._patch_method(module_name, method_path, create_anthropic_wrapper)

    def instrument_langchain(self) -> None:
        """Instrument LangChain components"""
        
        def create_langchain_wrapper(original_method):
            if asyncio.iscoroutinefunction(original_method):
                @functools.wraps(original_method)
                async def async_wrapper(self_obj, *args, **kwargs):
                    component_name = self_obj.__class__.__name__
                    
                    # Determine event type based on component
                    if any(x in component_name for x in ['LLM', 'ChatModel', 'Chat']):
                        event_type = EventType.MODEL_CALL
                        context_method = self.client.trace_model_call
                        context_kwargs = {'source_component': f'langchain_{component_name}'}
                    elif 'Tool' in component_name:
                        event_type = EventType.TOOL_EXECUTION  
                        context_method = self.client.trace_tool_execution
                        context_kwargs = {'tool_name': component_name, 'source_component': f'langchain_{component_name}'}
                    else:
                        event_type = EventType.AGENT_ACTION
                        context_method = self.client.trace_agent_action
                        context_kwargs = {'action_type': 'chain_execution', 'source_component': f'langchain_{component_name}'}
                    
                    async with context_method(**context_kwargs) as span:
                        input_data = str(args[0]) if args else str(kwargs)
                        span.set_input(input_data[:1000])
                        
                        try:
                            result = await original_method(self_obj, *args, **kwargs)
                            span.set_output(str(result)[:1000])
                            return result
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                            
                return async_wrapper
            else:
                @functools.wraps(original_method)
                def sync_wrapper(self_obj, *args, **kwargs):
                    component_name = self_obj.__class__.__name__
                    
                    # Use sync context managers
                    if any(x in component_name for x in ['LLM', 'ChatModel', 'Chat']):
                        context_manager = self.client.trace_model_call_sync(source_component=f'langchain_{component_name}')
                    elif 'Tool' in component_name:
                        context_manager = self.client.trace_tool_execution_sync(tool_name=component_name, source_component=f'langchain_{component_name}')
                    else:
                        context_manager = self.client.trace_agent_action_sync(action_type='chain_execution', source_component=f'langchain_{component_name}')
                    
                    with context_manager as span:
                        input_data = str(args[0]) if args else str(kwargs)
                        span.set_input(input_data[:1000])
                        
                        try:
                            result = original_method(self_obj, *args, **kwargs)
                            span.set_output(str(result)[:1000])
                            return result
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                
                return sync_wrapper

        patterns = [
            ("langchain.llms.base", "BaseLLM._call"),
            ("langchain.llms.base", "BaseLLM._acall"),
            ("langchain.chat_models.base", "BaseChatModel._call"),
            ("langchain.chat_models.base", "BaseChatModel._acall"),
            ("langchain.tools.base", "BaseTool._run"),
            ("langchain.tools.base", "BaseTool._arun"),
            ("langchain.chains.base", "Chain._call"),
            ("langchain.chains.base", "Chain._acall"),
        ]
        
        for module_name, method_path in patterns:
            self._patch_method(module_name, method_path, create_langchain_wrapper)

    def instrument_llamaindex(self) -> None:
        """Instrument LlamaIndex components"""
        
        def create_llamaindex_wrapper(original_method):
            if asyncio.iscoroutinefunction(original_method):
                @functools.wraps(original_method)
                async def async_wrapper(self_obj, *args, **kwargs):
                    component_name = self_obj.__class__.__name__
                    
                    async with self.client.trace_agent_action(
                        action_type="index_query",
                        source_component=f'llamaindex_{component_name}'
                    ) as span:
                        # Extract query if available
                        query = args[0] if args and isinstance(args[0], str) else str(kwargs.get('query', ''))
                        span.set_input(query[:1000])
                        
                        try:
                            result = await original_method(self_obj, *args, **kwargs)
                            
                            # Extract response text
                            if hasattr(result, 'response'):
                                span.set_output(str(result.response)[:1000])
                            elif isinstance(result, str):
                                span.set_output(result[:1000])
                            
                            # Extract metadata
                            if hasattr(result, 'source_nodes'):
                                span.set_metadata("source_nodes_count", len(result.source_nodes))
                            elif hasattr(result, 'metadata'):
                                source_nodes = result.metadata.get('source_nodes', [])
                                span.set_metadata("source_nodes_count", len(source_nodes))
                            
                            return result
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                            
                return async_wrapper
            else:
                @functools.wraps(original_method)
                def sync_wrapper(self_obj, *args, **kwargs):
                    component_name = self_obj.__class__.__name__
                    
                    with self.client.trace_agent_action_sync(
                        action_type="index_query",
                        source_component=f'llamaindex_{component_name}'
                    ) as span:
                        query = args[0] if args and isinstance(args[0], str) else str(kwargs.get('query', ''))
                        span.set_input(query[:1000])
                        
                        try:
                            result = original_method(self_obj, *args, **kwargs)
                            
                            if hasattr(result, 'response'):
                                span.set_output(str(result.response)[:1000])
                            elif isinstance(result, str):
                                span.set_output(result[:1000])
                            
                            return result
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                
                return sync_wrapper

        patterns = [
            ("llama_index.query_engine.base", "BaseQueryEngine.query"),
            ("llama_index.query_engine.base", "BaseQueryEngine.aquery"),
            ("llama_index.indices.base", "BaseIndex.query"),
            ("llama_index.indices.base", "BaseIndex.aquery"),
            ("llama_index.chat_engine.base", "BaseChatEngine.chat"),
            ("llama_index.chat_engine.base", "BaseChatEngine.achat"),
        ]
        
        for module_name, method_path in patterns:
            self._patch_method(module_name, method_path, create_llamaindex_wrapper)

    def instrument_requests(self) -> None:
        """Instrument requests library for HTTP tool executions"""
        
        def create_requests_wrapper(original_method):
            @functools.wraps(original_method)
            def sync_wrapper(*args, **kwargs):
                # Extract URL and method
                url = args[0] if args else kwargs.get('url', 'unknown')
                method = original_method.__name__.upper()
                
                with self.client.trace_tool_execution_sync(
                    tool_name="http_request",
                    source_component="requests_auto"
                ) as span:
                    span.set_metadata("url", str(url))
                    span.set_metadata("method", method)
                    
                    # Add request data if present
                    if 'json' in kwargs:
                        span.set_metadata("has_json_payload", True)
                    if 'data' in kwargs:
                        span.set_metadata("has_data_payload", True)
                    
                    try:
                        result = original_method(*args, **kwargs)
                        
                        # Extract response details
                        span.set_metadata("status_code", result.status_code)
                        span.set_metadata("response_size", len(result.content) if hasattr(result, 'content') else 0)
                        
                        return result
                        
                    except Exception as e:
                        span.set_metadata("error", str(e))
                        raise
                        
            return sync_wrapper

        patterns = [
            ("requests", "get"),
            ("requests", "post"),
            ("requests", "put"),
            ("requests", "delete"),
            ("requests", "patch"),
            ("requests", "head"),
            ("requests", "options"),
        ]
        
        for module_name, method_path in patterns:
            self._patch_method(module_name, method_path, create_requests_wrapper)

    def instrument_httpx(self) -> None:
        """Instrument httpx library for async HTTP tool executions"""
        
        def create_httpx_wrapper(original_method):
            if asyncio.iscoroutinefunction(original_method):
                @functools.wraps(original_method)
                async def async_wrapper(*args, **kwargs):
                    # Extract URL and method from args/kwargs
                    if len(args) >= 2:
                        method = args[0]
                        url = args[1]
                    else:
                        method = kwargs.get('method', 'GET')
                        url = kwargs.get('url', 'unknown')
                    
                    async with self.client.trace_tool_execution(
                        tool_name="http_request",
                        source_component="httpx_auto"
                    ) as span:
                        span.set_metadata("url", str(url))
                        span.set_metadata("method", str(method))
                        
                        try:
                            result = await original_method(*args, **kwargs)
                            
                            # Extract response details
                            if hasattr(result, 'status_code'):
                                span.set_metadata("status_code", result.status_code)
                            if hasattr(result, 'content'):
                                span.set_metadata("response_size", len(result.content))
                            
                            return result
                            
                        except Exception as e:
                            span.set_metadata("error", str(e))
                            raise
                            
                return async_wrapper
            else:
                @functools.wraps(original_method)
                def sync_wrapper(*args, **kwargs):
                    return original_method(*args, **kwargs)
                return sync_wrapper

        patterns = [
            ("httpx", "AsyncClient.request"),
            ("httpx", "Client.request"),
            ("httpx", "request"),
        ]
        
        for module_name, method_path in patterns:
            self._patch_method(module_name, method_path, create_httpx_wrapper)

    def _estimate_openai_cost(self, model: str, total_tokens: int) -> Optional[float]:
        """Estimate cost for OpenAI models (approximate rates as of 2024)"""
        # These are example rates - should be updated regularly
        rates_per_1k = {
            'gpt-4': 0.03,
            'gpt-4-32k': 0.06,
            'gpt-4-turbo': 0.01,
            'gpt-4o': 0.005,
            'gpt-3.5-turbo': 0.002,
            'gpt-3.5-turbo-16k': 0.004,
            'text-davinci-003': 0.02,
            'text-davinci-002': 0.02,
            'text-curie-001': 0.002,
            'text-babbage-001': 0.0005,
            'text-ada-001': 0.0004,
            'davinci': 0.02,
            'curie': 0.002,
            'babbage': 0.0005,
            'ada': 0.0004,
        }
        
        model_lower = model.lower()
        for model_prefix, rate in rates_per_1k.items():
            if model_lower.startswith(model_prefix.lower()):
                return (total_tokens / 1000) * rate
        
        return None

    def get_instrumentation_status(self) -> Dict[str, bool]:
        """Get status of all instrumentation attempts"""
        status = {}
        
        libraries = [
            'openai',
            'anthropic', 
            'langchain',
            'llama_index',
            'requests',
            'httpx',
        ]
        
        for lib in libraries:
            status[lib] = lib in self.patched_modules
        
        return status

    def is_instrumented(self, library: str) -> bool:
        """Check if a specific library is instrumented"""
        return library in self.patched_modules