# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

# coding: utf-8
# pylint: disable=wildcard-import
"""Binomial distribution class."""
__all__ = ['Binomial']

from .distribution import Distribution
from .utils import prob2logit, logit2prob, cached_property, sample_n_shape_converter
from .utils import gammaln
from .constraint import Interval, Real, NonNegativeInteger
from .... import np, npx


class Binomial(Distribution):
    r"""Create a binomial distribution object.

    Parameters
    ----------
    n : scalar
        Non-negative interger of Bernoulli trials to stop.
    prob : Tensor or scalar, default None
        Probability of sampling `1`.
    logit : Tensor or scalar, default None
        The log-odds of sampling `1`.
    """
    # pylint: disable=abstract-method

    support = NonNegativeInteger()
    arg_constraints = {'prob': Interval(0, 1),
                       'logit': Real()}

    def __init__(self, n=1, prob=None, logit=None, validate_args=None):
        if (n < 0) or (n % 1 != 0):
            raise ValueError(
                "Expect `n` to be non-negative integer, received n={}".format(n))
        if (prob is None) == (logit is None):
            raise ValueError(
                "Either `prob` or `logit` must be specified, but not both. " +
                "Received prob={}, logit={}".format(prob, logit))

        if prob is not None:
            self.prob = prob
        else:
            self.logit = logit
        self.n = n
        super(Binomial, self).__init__(
            event_dim=0, validate_args=validate_args)

    @cached_property
    def prob(self):
        """Get the probability of sampling `1`.

        Returns
        -------
        Tensor
            Parameter tensor.
        """
        # pylint: disable=method-hidden
        return logit2prob(self.logit, True)

    @cached_property
    def logit(self):
        """Get the log-odds of sampling `1`.

        Returns
        -------
        Tensor
            Parameter tensor.
        """
        # pylint: disable=method-hidden
        return prob2logit(self.prob, True)

    @property
    def mean(self):
        return self.n * self.prob

    @property
    def variance(self):
        p = self.prob
        return self.n * p * (1 - p)

    def broadcast_to(self, batch_shape):
        new_instance = self.__new__(type(self))
        if 'prob' in self.__dict__:
            new_instance.prob = np.broadcast_to(self.prob, batch_shape)
        else:
            new_instance.logit = np.broadcast_to(self.logit, batch_shape)
        new_instance.n = self.n
        super(Binomial, new_instance).__init__(event_dim=self.event_dim,
                                               validate_args=False)
        new_instance._validate_args = self._validate_args
        return new_instance

    def log_prob(self, value):
        if self._validate_args:
            self._validate_samples(value)
        lgamma = gammaln()
        binomal_coef = lgamma(self.n + 1) - lgamma(1 +
                                                   value) - lgamma(self.n - value + 1)
        # log(prob) may have numerical issue.
        unnormalized_log_prob = (value * np.log(self.prob) +
                                 (self.n - value) * np.log1p(-self.prob))
        return binomal_coef + unnormalized_log_prob

    def sample(self, size=None):
        if size is not None:
            logit = np.broadcast_to(self.logit, size)
        else:
            logit = self.logit
        expanded_logit = np.repeat(
            np.expand_dims(logit, -1), int(self.n), -1)
        return npx.random.bernoulli(logit=expanded_logit).sum(-1)

    def sample_n(self, size=None):
        logit = self.logit
        expanded_logit = np.repeat(
            np.expand_dims(logit, -1), int(self.n), -1)
        return npx.random.bernoulli(
            logit=expanded_logit,
            size=sample_n_shape_converter(size)
        ).sum(-1)
