"""
A simple, file-based API mocking server built with Flask.
Designed to help developers quickly simulate API endpoints for testing and development.
"""
from flask import Flask, jsonify, request, Response
from flask_cors import CORS # Import CORS
import json
import re
import time
import argparse
import os
from watchdog.observers import Observer
from watchdog.events import FileSystemEventHandler
import logging
from collections import deque
import signal
import sys
import base64
import threading
import jsonschema # Import jsonschema
from .config_parser import load_and_validate_config, ValidationError # Import config loader and ValidationError
from .core.auth import check_authentication
from .core.rate_limiter import handle_rate_limiting
from .core.response import prepare_response, validate_request_body
from .core.metrics import track_request, generate_metrics

# Global variable for the observer, initialized to None
observer = None

def signal_handler(sig, frame):
    """
    Handles termination signals (SIGINT, SIGTERM) to gracefully shut down the server.
    """
    global observer
    logger.info(f"Signal {sig} received. Shutting down server gracefully.")
    if observer and observer.is_alive():
        observer.stop()
        observer.join()
    sys.exit(0)

# Configure logging
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

# Store server start time for uptime calculation
SERVER_START_TIME = time.time()

def health_check(): # Make it a regular function
    """Returns the health status and uptime of the server."""
    uptime = time.time() - SERVER_START_TIME
    response = jsonify({"status": "ok", "uptime_seconds": round(uptime, 2)})
    response.headers["Content-Type"] = "application/json"
    return response, 200

def handle_404_error(e):
    """Returns a JSON 404 error response."""
    logger.warning(f"Unknown route accessed: {request.path}")
    return jsonify({"error": "Not Found", "message": f"The requested URL {request.path} was not found on the server."}), 404





def _handle_delay(response_config):
    """Handles response delay."""
    delay = response_config.get('delay', 0)
    if delay > 0:
        logger.info(f"Delaying response for {delay} seconds.")
        time.sleep(delay)

def generate_openapi_spec(routes_config, host, port):
    """Generates an OpenAPI 3.0 specification from the routes configuration."""
    spec = {
        "openapi": "3.0.0",
        "info": {
            "title": "Simple Mock API",
            "version": "1.0.0",
            "description": "A mock API generated by simple-api-mock-server"
        },
        "servers": [
            {"url": f"http://{host}:{port}"}
        ],
        "paths": {},
        "components": {
            "securitySchemes": {
                "apiKey": {
                    "type": "apiKey",
                    "in": "header",
                    "name": "X-API-Key"
                },
                "basicAuth": {
                    "type": "http",
                    "scheme": "basic"
                },
                "bearerAuth": {
                    "type": "http",
                    "scheme": "bearer"
                }
            }
        }
    }

    for route in routes_config:
        path = route['path']
        path_item = {}

        # Add path parameters
        parameters = []
        path_params = re.findall(r'{(\w+)}', path)
        for param in path_params:
            parameters.append({
                "name": param,
                "in": "path",
                "required": True,
                "schema": {"type": "string"}
            })

        for method in route['methods']:
            method_lower = method.lower()
            operation = {
                "summary": route.get('description', 'No description'),
                "tags": route.get('tags', []),
                "parameters": parameters,
                "responses": {}
            }

            # Add query parameters
            if 'query_params' in route:
                for param in route['query_params']:
                    operation['parameters'].append({
                        "name": param['name'],
                        "in": "query",
                        "required": param.get('required', False),
                        "schema": {"type": param.get('type', 'string')}
                    })

            response_config = route.get('response', {})
            status_code = str(response_config.get('code', 200))

            operation["responses"][status_code] = {
                "description": "Mocked response",
                "content": {
                    "application/json": {
                        "schema": {
                            "type": "object",
                            "example": response_config.get('data', {})
                        }
                    }
                }
            }

            if route.get('request_body'):
                operation['requestBody'] = {
                    "content": {
                        "application/json": {
                            "schema": {
                                "type": "object",
                                "example": route.get('request_body')
                            }
                        }
                    }
                }
            
            # Add security schemes
            if 'auth' in route:
                auth = route['auth']
                if 'api_key' in auth:
                    operation['security'] = [{"apiKey": []}]
                elif 'basic_auth' in auth:
                    operation['security'] = [{"basicAuth": []}]
                elif 'bearer_token' in auth:
                    operation['security'] = [{"bearerAuth": []}]

            path_item[method_lower] = operation
        
        spec["paths"][path] = path_item

    return spec

def make_endpoint_function(responses, allowed_methods, route_template_path):
    """Factory function to create a unique endpoint function for each route."""
    def endpoint(**kwargs):
        if request.path != '/metrics':
            track_request(request.path, request.method)
        logger.info(f"Incoming request: {request.method} {request.path}")
        if logger.isEnabledFor(logging.DEBUG):
            logger.debug(f"Request headers: {dict(request.headers)}")
            if request.is_json:
                logger.debug(f"Request body: {request.get_json(force=True, silent=True)}")
        
        response_config = responses.get(request.method)
        if not response_config:
            logger.warning(f"Method Not Allowed: {request.method} {request.path}")
            resp = jsonify({'error': 'Method Not Allowed', 'allowed_methods': allowed_methods})
            resp.status_code = 405
            resp.headers['Allow'] = ', '.join(allowed_methods)
            return resp

        # Handle rate limiting
        rate_limit_response, endpoint_key = handle_rate_limiting(response_config, request, route_template_path)
        if rate_limit_response:
            return rate_limit_response

        # Handle authentication
        auth_response = check_authentication(response_config, request, route_template_path)
        if auth_response:
            return auth_response

        # Handle delay
        _handle_delay(response_config)

        # Validate request body and get parsed parameters
        request_body_params, validation_error_response = validate_request_body(response_config, request)
        if validation_error_response:
            return validation_error_response
        # Prepare and return the response
        return prepare_response(response_config, kwargs, request, endpoint_key, request_body_params)
    return endpoint

def create_mock_server(config_path='api.json', static_folder_path=None, host='127.0.0.1', port=5001):
    """Loads API configuration and registers routes with the Flask app."""
    app = Flask(__name__, static_folder=static_folder_path) # Initialize app here, use provided static folder path
    CORS(app) # Enable CORS for all routes
    
    # Register health check
    app.add_url_rule("/health", "health_check", health_check, methods=["GET"])
    
    # Register 404 error handler
    app.register_error_handler(404, handle_404_error)

    logger.info(f"Loading API configuration from {config_path}")
    try:
        routes_config = load_and_validate_config(config_path)
        logger.info("API configuration validated successfully.")
    except ValidationError as e: # ValidationError is now from jsonschema directly
        logger.exception(f"Invalid api.json: {e.message}") # Use logger.exception
        raise Exception(f"Invalid api.json: {e.message}") from e

    # Add OpenAPI spec endpoint
    @app.route('/openapi.json')
    def openapi_spec():
        spec = generate_openapi_spec(routes_config, host, port)
        return jsonify(spec)

    @app.route('/metrics')
    def metrics():
        return Response(generate_metrics(), mimetype='text/plain')

    registered_routes = set()
    routes_by_path = {}
    for route in routes_config:
        path = route.get('path')
        if not path:
            logger.warning(f"Skipping route with missing path: {route}")
            continue

        # Convert {param} to <param> for Flask
        flask_path = re.sub(r'{(\w+)}', r'<\1>', path)

        methods = route.get('methods', ['GET'])
        for method in methods:
            if (flask_path, method) in registered_routes:
                logger.error(f"Route conflict: {method} {flask_path} is already defined.")
                raise Exception(f"Route conflict: {method} {flask_path} is already defined.")
            registered_routes.add((flask_path, method))

        if flask_path not in routes_by_path:
            # Sanitize path for endpoint name
            endpoint_name = re.sub(r'[^a-zA-Z0-9_]', '_', flask_path)
            routes_by_path[flask_path] = {'methods': [], 'responses': {}, 'endpoint_name': endpoint_name}
        
        routes_by_path[flask_path]['methods'].extend(methods)
        for method in methods:
            method_response_config = route.get('response', {})
            routes_by_path[flask_path]['responses'][method] = {
                'data': method_response_config.get('data', {}),
                'code': method_response_config.get('code', 200),
                'delay': method_response_config.get('delay', 0),
                'headers': method_response_config.get('headers', {}),
                'auth': route.get('auth', {}),
                'rate_limit': route.get('rate_limit', {})
            }
    for path, route_data in routes_by_path.items():
        allowed_methods_list = list(set(route_data['methods']))
        logger.info(f"Registering route: {path} with methods {allowed_methods_list}")

        view_func = make_endpoint_function(
            route_data['responses'], 
            allowed_methods_list, 
            route_template_path=path
        )

        app.add_url_rule(
            path,
            endpoint=route_data['endpoint_name'],
            view_func=view_func,
            methods=allowed_methods_list
        )

    return app

class ConfigChangeHandler(FileSystemEventHandler):
    """Handles changes to the configuration file to trigger server reload."""
    def __init__(self, config_path, main_script_path, debounce_delay=0.5):
        super().__init__()
        self.config_path = config_path
        self.main_script_path = main_script_path
        self.debounce_delay = debounce_delay
        self.last_modified = 0

    def _trigger_reload(self, event_path):
        current_time = time.time()
        if (current_time - self.last_modified) > self.debounce_delay:
            logger.info(f"Configuration file {event_path} modified or created. Restarting server...")
            # Touch the main script file to trigger Flask's reloader
            os.utime(self.main_script_path, None)
            self.last_modified = current_time

    def on_modified(self, event):
        if os.path.abspath(event.src_path) == os.path.abspath(self.config_path):
            self._trigger_reload(event.src_path)

    def on_created(self, event):
        if os.path.abspath(event.src_path) == os.path.abspath(self.config_path):
            self._trigger_reload(event.src_path)

def main():
    """Main function to parse arguments and run the mock server."""
    parser = argparse.ArgumentParser(description="Run a simple API mock server.")
    parser.add_argument(
        "--config",
        type=str,
        default="api.json",
        help="Path to the API configuration JSON file."
    )
    parser.add_argument(
        "--port",
        type=int,
        default=5001,
        help="Port to run the server on."
    )
    parser.add_argument(
        "--host",
        type=str,
        default="127.0.0.1",
        help="Host to bind the server to."
    )
    parser.add_argument(
        "--debug",
        action="store_true",
        help="Enable Flask debug mode."
    )
    parser.add_argument(
        "--verbose",
        action="store_true",
        help="Enable verbose logging."
    )
    parser.add_argument(
        "--static-folder",
        type=str,
        help="Path to a static folder to serve files from (e.g., for UI assets)."
    )
    args = parser.parse_args()

    if args.verbose:
        logger.setLevel(logging.DEBUG)

    # Set up file watcher for hot reloading
    if args.debug: # Only set up hot reloading if debug mode is enabled
        config_dir = os.path.dirname(os.path.abspath(args.config))
        event_handler = ConfigChangeHandler(os.path.abspath(args.config), os.path.abspath(__file__))
        global observer # Declare observer as global
        observer = Observer()
        observer.schedule(event_handler, config_dir, recursive=False)
        observer.start()
    else:
        logger.info("Hot reloading is disabled. Run with --debug to enable.")

    # Register signal handlers
    signal.signal(signal.SIGINT, signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    try:
        logger.info("Starting mock server...")
        app = create_mock_server(config_path=args.config, static_folder_path=args.static_folder, host=args.host, port=args.port)
        if args.static_folder:
            logger.info(f"Serving static files from '{args.static_folder}' at /static/<filename>")
        app.run(debug=args.debug, port=args.port, host=args.host)
    except KeyboardInterrupt:
        logger.info("Server stopped by user (KeyboardInterrupt).")
    except OSError as e:
        if "Address already in use" in str(e):
            logger.error(f"Port {args.port} is already in use. Please choose a different port or stop the process using it.")
        else:
            logger.exception(f"Server failed to start due to an OS error: {e}")
    except Exception as e:
        logger.exception(f"Server failed to start: {e}") # Use logger.exception
    finally:
        if observer and observer.is_alive():
            observer.stop()
        if observer: # Only join if observer was created
            observer.join()

if __name__ == '__main__':
    main()