/************************************************************\
 * Copyright 2018 Lawrence Livermore National Security, LLC
 * (c.f. AUTHORS, NOTICE.LLNS, COPYING)
 *
 * This file is part of the Flux resource manager framework.
 * For details, see https://github.com/flux-framework.
 *
 * SPDX-License-Identifier: LGPL-3.0
\************************************************************/

#ifndef _FLUX_SECURITY_SIGN_H
#define _FLUX_SECURITY_SIGN_H

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************\
 * Copyright 2018 Lawrence Livermore National Security, LLC
 * (c.f. AUTHORS, NOTICE.LLNS, COPYING)
 *
 * This file is part of the Flux resource manager framework.
 * For details, see https://github.com/flux-framework.
 *
 * SPDX-License-Identifier: LGPL-3.0
\************************************************************/

#ifndef _FLUX_SECURITY_CONTEXT_H
#define _FLUX_SECURITY_CONTEXT_H

#ifdef __cplusplus
extern "C" {
#endif

enum {
    FLUX_SECURITY_DISABLE_PATH_PARANOIA     = 0x1,
    FLUX_SECURITY_FORCE_PATH_PARANOIA       = 0x2,
};

typedef struct flux_security flux_security_t;

typedef void (*flux_security_free_f)(void *arg);

flux_security_t *flux_security_create (int flags);
void flux_security_destroy (flux_security_t *ctx);

const char *flux_security_last_error (flux_security_t *ctx);
int flux_security_last_errnum (flux_security_t *ctx);

int flux_security_configure (flux_security_t *ctx, const char *pattern);

int flux_security_aux_set (flux_security_t *ctx, const char *name,
		           void *data, flux_security_free_f freefun);

void *flux_security_aux_get (flux_security_t *ctx, const char *name);

#ifdef __cplusplus
}
#endif

#endif /* !_FLUX_SECURITY_CONTEXT_H */

/* Overview:
 * These are generic functions for signing and verifying a payload.
 * They use flux_security_t context to provide preparsed
 * configuration, as a buffer for error messages/codes, and to manage
 * a local context for the sign functions.
 *
 * The output of flux_sign_wrap() is a NULL-terminated string of the
 * form HEADER.PAYLOAD.SIGNATURE, resembling JSC Compact Serialization,
 * described in Internet RFC 7515:
 *
 * HEADER
 *   Contains the userid and signing mechanism in an internal key-value
 *   format, base64 encoded.
 *
 * PAYLOAD
 *   User data, base64 encoded
 *
 * SIGNATURE
 *   Mechanism dependent signature over HEADER.PAYLOAD, a string that
 *   must not contain "." (delimiter).
 *
 * The actual signing mechanism used is determined by configuration.
 */

/* Required configuration:
 *
 * [sign]
 * default-type = "none"            # mechanism name for wrap
 * allowed-types = [ "none" ]       # array of mechanism names for unwrap
 * max-ttl = 259200                 # signature maximum TTL in seconds
 */

enum {
    FLUX_SIGN_NOVERIFY = 1,   // flux_sign_unwrap() need not verify signature
};

/* Sign payload/payloadsz, returning a NULL terminated string
 * suitable for feeding into flux_sign_unwrap().  The returned string
 * remains valid until the next call to flux_sign_wrap() or 'ctx'
 * is destroyed.  'flags' currently must be set to 0.
 * If 'mech_type' is NULL, use the configured 'default-type'.
 * On error, NULL is returned and context error state is updated.
 */
const char *flux_sign_wrap (flux_security_t *ctx,
                            const void *payload, int payloadsz,
                            const char *mech_type,
                            int flags);

/* Same as flux_sign_wrap(), but allow userid to to explicitly set.
 * Mainly useful for testing, the result will likely fail verification
 * by flux_sign_unwrap().
 */
const char *flux_sign_wrap_as (flux_security_t *ctx,
                               int64_t userid,
                               const void *payload, int payloadsz,
                               const char *mech_type,
                               int flags);


/* Given a NULL-terminated 'input' string generated by flux_sign_wrap(),
 * decode its contents and verify the signature.  If payload/payloadsz are
 * non-NULL, a pointer to the original payload and size is provided.
 * The payload remains valid until the next call to flux_sign_unwrap()
 * or 'ctx' is destroyed.  If 'userid' is non-NULL, the userid that
 * signed 'input' is returned.  'flags' may be set to 0, or if signature
 * validation is not required, it may be set to FLUX_SIGN_NOVERIFY.
 * On success, 0 is returned; on error, -1 is returned and context error
 * state is updated.
 */
int flux_sign_unwrap (flux_security_t *ctx, const char *input,
                      const void **payload, int *payloadsz,
                      int64_t *userid, int flags);

/* Same as above, but don't limit mech type to only the ones
 * configured in 'allowed-types'.  Set 'mech_type' to the mechanism
 * type used.
 */
int flux_sign_unwrap_anymech (flux_security_t *ctx, const char *input,
                              const void **payload, int *payloadsz,
                              const char **mech_type,
                              int64_t *userid, int flags);

#ifdef __cplusplus
}
#endif

#endif /* !_FLUX_SECURITY_SIGN_H */

/*
 * vi:tabstop=4 shiftwidth=4 expandtab
 */
