#!/usr/bin/env python3
"""Precommit handler.

Installs and removes a Git `pre-commit` hook that runs `bash2gitlab compile`.
Respects environment/TOML config, Git worktrees, and `core.hooksPath`.
"""

from __future__ import annotations

import configparser
import hashlib
import logging
import os
import stat
from pathlib import Path

from bash2gitlab.config import config
from bash2gitlab.errors.exceptions import Bash2GitlabError

logger = logging.getLogger(__name__)

__all__ = ["install", "uninstall", "PrecommitHookError", "hook_hash", "hook_path"]


class PrecommitHookError(Bash2GitlabError):
    """Raised when pre-commit hook installation or removal fails."""


HOOK_NAME = "pre-commit"

# POSIX sh (not bash). Avoid `pipefail` and bashisms.
# We do not use `set -e` because we intentionally fall through runners.
HOOK_CONTENT = """#!/bin/sh
# Auto-generated by bash2gitlab: run `bash2gitlab compile` before committing.
set -u
export BASH2GITLAB_HIDE_CORE_HELP=1
say() { printf '%s\\n' "$*"; }

say "[pre-commit] Running bash2gitlab compile..."

try_run() {
  # $* is the command string
  if sh -c "$*"; then
    say "[pre-commit] OK: $*"
    exit 0
  fi
  return 1
}

# 1) direct
if command -v bash2gitlab >/dev/null 2>&1; then
  try_run "bash2gitlab compile"
fi

# 2) common project runners
for tool in uv poetry pipenv pdm hatch rye; do
  if command -v "$tool" >/dev/null 2>&1; then
    case "$tool" in
      hatch) try_run "hatch run bash2gitlab compile" || true ;;
      *)     try_run "$tool run bash2gitlab compile" || true ;;
    esac
  fi
done

# 3) python module fallback
if command -v python >/dev/null 2>&1; then
  try_run "python -m bash2gitlab compile" || true
fi

say "[pre-commit] ERROR: could not run bash2gitlab. Install it or use uv/poetry/pipenv/pdm/hatch/rye."
exit 1
"""

CONFIG_SECTION = "tool.bash2gitlab"
REQUIRED_ENV_VARS = {"BASH2GITLAB_INPUT_DIR", "BASH2GITLAB_OUTPUT_DIR"}


def hook_hash(content: str) -> str:
    """Return a stable hash of hook content for conflict detection.

    Args:
        content: Hook script content.

    Returns:
        Hex digest string (sha256).
    """
    return hashlib.sha256(content.encode("utf-8")).hexdigest()


def resolve_git_dir(repo_root: Path) -> Path:
    """Resolve the actual .git directory, supporting worktrees (gitdir file).

    Args:
        repo_root: Candidate repo root.

    Returns:
        Path to the real .git directory.

    Raises:
        PrecommitHookError: If a git directory cannot be located.
    """
    git_path = repo_root / ".git"
    if git_path.is_dir():
        return git_path
    if git_path.is_file():
        try:
            data = git_path.read_text(encoding="utf-8").strip()
        except OSError as exc:
            raise PrecommitHookError(f"Unable to read {git_path}: {exc}") from exc
        if data.lower().startswith("gitdir:"):
            target = data.split(":", 1)[1].strip()
            return (repo_root / target).resolve()
    raise PrecommitHookError(f"Not inside a Git repository: {repo_root}")


def read_hooks_path(git_dir: Path, repo_root: Path) -> Path | None:
    """Read core.hooksPath from .git/config, if present.

    Args:
        git_dir: Path to .git directory (resolved).
        repo_root: Repo root.

    Returns:
        Path to hooks dir or None if not configured.
    """
    cfg = configparser.ConfigParser()
    cfg_path = git_dir / "config"
    if not cfg_path.is_file():
        return None
    try:
        cfg.read(cfg_path, encoding="utf-8")
    except Exception:  # best-effort
        return None
    if cfg.has_section("core") and cfg.has_option("core", "hooksPath"):
        hooks_path = Path(cfg.get("core", "hooksPath"))
        if not hooks_path.is_absolute():
            hooks_path = (repo_root / hooks_path).resolve()
        return hooks_path
    return None


def hooks_dir(repo_root: Path) -> Path:
    """Compute the correct hooks directory, honoring `core.hooksPath`."""
    git_dir = resolve_git_dir(repo_root)
    alt = read_hooks_path(git_dir, repo_root)
    return alt if alt else (git_dir / "hooks")


def hook_path(repo_root: Path) -> Path:
    """Return full path to the pre-commit hook file."""
    return hooks_dir(repo_root) / HOOK_NAME


def has_required_config() -> bool:
    """Check whether compile input/output are configured via env or TOML.

    Returns:
        True if `input_dir` and `output_dir` are present; otherwise False.
    """
    # Env wins
    if REQUIRED_ENV_VARS.issubset(os.environ):
        return True

    # TOML via singleton config
    if config.input_dir and config.output_dir:
        return True

    return False


def install(repo_root: Path | None = None, *, force: bool = False) -> None:
    """Install the bash2gitlab `pre-commit` hook.

    The hook will try multiple runners (direct, uv, poetry, pipenv, pdm, hatch, rye,
    then `python -m`) to invoke `bash2gitlab compile`.

    Args:
        repo_root: Directory considered as the repository root (defaults to CWD).
        force: Overwrite an existing non-matching hook if True.

    Raises:
        PrecommitHookError: If outside a Git repo, config missing, or conflict.
    """
    repo_root = repo_root or Path.cwd()

    # Validate repo and config
    _git_dir = resolve_git_dir(repo_root)  # raises if not a repo
    if not has_required_config():
        raise PrecommitHookError(
            "Missing bash2gitlab input/output configuration. Run `bash2gitlab init` to create TOML, or set BASH2GITLAB_INPUT_DIR and BASH2GITLAB_OUTPUT_DIR."
        )

    # Ensure hooks dir exists
    hdir = hooks_dir(repo_root)
    hdir.mkdir(parents=True, exist_ok=True)

    dest = hdir / HOOK_NAME
    new_hash = hook_hash(HOOK_CONTENT)

    if dest.exists():
        existing = dest.read_text(encoding="utf-8")
        if hook_hash(existing) == new_hash and not force:
            logger.info("Pre-commit hook is already up to date at %s", dest.relative_to(repo_root))
            return
        if hook_hash(existing) != new_hash and not force:
            raise PrecommitHookError(f"A different pre-commit hook exists at {dest}. Use force=True to overwrite.")

    dest.write_text(HOOK_CONTENT, encoding="utf-8")

    # Executable on POSIX; Git for Windows runs sh hooks regardless of mode.
    if os.name == "posix":
        st_mode = dest.stat().st_mode
        dest.chmod(st_mode | stat.S_IXUSR | stat.S_IXGRP | stat.S_IXOTH)

    logger.info("Installed pre-commit hook at %s", dest.relative_to(repo_root))


def uninstall(repo_root: Path | None = None, *, force: bool = False) -> None:
    """Remove the bash2gitlab `pre-commit` hook.

    Args:
        repo_root: Directory considered as the repository root (defaults to CWD).
        force: Remove even if the hook content does not match our generated one.

    Raises:
        PrecommitHookError: If outside a Git repo or conflict without `force`.
    """
    repo_root = repo_root or Path.cwd()
    _git_dir = resolve_git_dir(repo_root)  # raises if not a repo

    dest = hook_path(repo_root)
    if not dest.exists():
        logger.warning("No pre-commit hook to uninstall at %s", dest)
        return

    content = dest.read_text(encoding="utf-8")
    if hook_hash(content) != hook_hash(HOOK_CONTENT) and not force:
        raise PrecommitHookError(
            f"Pre-commit hook at {dest} does not match bash2gitlab's. Use force=True to remove anyway."
        )

    dest.unlink()
    logger.info("Removed pre-commit hook at %s", dest.relative_to(repo_root))
