"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property`
     * (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about
     * alternatives.
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${this.scope.node.path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        this.scope.node.addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
//# sourceMappingURL=data:application/json;base64,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