"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 * @experimental
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    // won't always return a string; if the input can't be resolved to a string,
    // the input will be returned.
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // CloudFormation treats numbers and strings interchangeably;
        // so, if we get a number here, convert it to a string
        if (typeof value === 'number') {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    // won't always return a number; if the input can't be parsed to a number,
    // the input will be returned.
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // return a number, if the input can be parsed as one
        let parsedValue;
        if (typeof value === 'string') {
            parsedValue = parseFloat(value);
            if (!isNaN(parsedValue)) {
                return parsedValue;
            }
        }
        // otherwise return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            return value.map(mapper);
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            const ret = {};
            for (const [key, val] of Object.entries(value)) {
                ret[key] = mapper(val);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return value;
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
    /** We're currently parsing the 'Rules' section. */
    CfnParsingContext[CfnParsingContext["RULES"] = 1] = "RULES";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = this.finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = this.finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: policy.EnableVersionUpgrade,
            useOnlineResharding: policy.UseOnlineResharding,
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: p.WaitOnResourceSignals,
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic !== undefined) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    get finder() {
        return this.options.finder;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = this.specialCaseRefs(refTarget);
                if (specialRef !== undefined) {
                    return specialRef;
                }
                else {
                    const refElement = this.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                // Fn::GetAtt takes a 2-element list as its argument
                const value = object[key];
                const target = this.finder.findResource(value[0]);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${value[0]}' not found`);
                }
                return target.getAtt(value[1]);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.listValue({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                // the first argument to FindInMap is the mapping name
                const mapping = this.finder.findMapping(value[0]);
                if (!mapping) {
                    throw new Error(`Mapping used in FindInMap expression with name '${value[0]}' was not found in the template`);
                }
                return cfn_fn_1.Fn.findInMap(mapping.logicalId, value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return cfn_fn_1.Fn.sub(this.parseFnSubString(fnSubString, map), map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                if (this.options.context === CfnParsingContext.RULES) {
                    return this.handleRulesIntrinsic(key, object);
                }
                else {
                    throw new Error(`Unsupported CloudFormation function '${key}'`);
                }
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(value, map = {}) {
        const leftBrace = value.indexOf('${');
        const rightBrace = value.indexOf('}') + 1;
        // don't include left and right braces when searching for the target of the reference
        if (leftBrace === -1 || leftBrace >= rightBrace) {
            return value;
        }
        const leftHalf = value.substring(0, leftBrace);
        const rightHalf = value.substring(rightBrace);
        const refTarget = value.substring(leftBrace + 2, rightBrace - 1).trim();
        if (refTarget[0] === '!') {
            return value.substring(0, rightBrace) + this.parseFnSubString(rightHalf, map);
        }
        // lookup in map
        if (refTarget in map) {
            return leftHalf + '${' + refTarget + '}' + this.parseFnSubString(rightHalf, map);
        }
        // since it's not in the map, check if it's a pseudo parameter
        const specialRef = this.specialCaseSubRefs(refTarget);
        if (specialRef !== undefined) {
            return leftHalf + specialRef + this.parseFnSubString(rightHalf, map);
        }
        const dotIndex = refTarget.indexOf('.');
        const isRef = dotIndex === -1;
        if (isRef) {
            const refElement = this.finder.findRefTarget(refTarget);
            if (!refElement) {
                throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
            }
            return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', true).toString() + this.parseFnSubString(rightHalf, map);
        }
        else {
            const targetId = refTarget.substring(0, dotIndex);
            const refResource = this.finder.findResource(targetId);
            if (!refResource) {
                throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
            }
            const attribute = refTarget.substring(dotIndex + 1);
            return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, true).toString() + this.parseFnSubString(rightHalf, map);
        }
    }
    handleRulesIntrinsic(key, object) {
        // Rules have their own set of intrinsics:
        // https://docs.aws.amazon.com/servicecatalog/latest/adminguide/intrinsic-function-reference-rules.html
        switch (key) {
            case 'Fn::ValueOf': {
                // ValueOf is special,
                // as it takes the name of a Parameter as its first argument
                const value = this.parseValue(object[key]);
                const parameterName = value[0];
                if (parameterName in this.parameters) {
                    // since ValueOf returns the value of a specific attribute,
                    // fail here - this substitution is not allowed
                    throw new Error(`Cannot substitute parameter '${parameterName}' used in Fn::ValueOf expression with attribute '${value[1]}'`);
                }
                const param = this.finder.findRefTarget(parameterName);
                if (!param) {
                    throw new Error(`Rule references parameter '${parameterName}' which was not found in the template`);
                }
                // create an explicit IResolvable,
                // as Fn.valueOf() returns a string,
                // which is incorrect
                // (Fn::ValueOf can also return an array)
                return lazy_1.Lazy.anyValue({ produce: () => ({ 'Fn::ValueOf': [param.logicalId, value[1]] }) });
            }
            default:
                // I don't want to hard-code the list of supported Rules-specific intrinsics in this function;
                // so, just return undefined here,
                // and they will be treated as a regular JSON object
                return undefined;
        }
    }
    specialCaseRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        switch (value) {
            case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
            case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
            case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
            case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
            case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
            case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
            case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
            case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
            default: return undefined;
        }
    }
    specialCaseSubRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
    }
    get parameters() {
        return this.options.parameters || {};
    }
}
exports.CfnParser = CfnParser;
function undefinedIfAllValuesAreEmpty(object) {
    return Object.values(object).some(v => v !== undefined) ? object : undefined;
}
//# sourceMappingURL=data:application/json;base64,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