"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lazy = void 0;
const stack_trace_1 = require("./stack-trace");
const token_1 = require("./token");
/**
 * Lazily produce a value
 *
 * Can be used to return a string, list or numeric value whose actual value
 * will only be calculated later, during synthesis.
 */
class Lazy {
    static stringValue(producer, options = {}) {
        return token_1.Token.asString(new LazyString(producer), options);
    }
    static numberValue(producer) {
        return token_1.Token.asNumber(new LazyNumber(producer));
    }
    static listValue(producer, options = {}) {
        return token_1.Token.asList(new LazyList(producer, options), options);
    }
    static anyValue(producer, options = {}) {
        return new LazyAny(producer, options);
    }
    constructor() {
    }
}
exports.Lazy = Lazy;
class LazyBase {
    constructor() {
        this.creationStack = stack_trace_1.captureStackTrace();
    }
    toString() {
        return token_1.Token.asString(this);
    }
    /**
     * Turn this Token into JSON
     *
     * Called automatically when JSON.stringify() is called on a Token.
     */
    toJSON() {
        return '<unresolved-lazy>';
    }
}
class LazyString extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyNumber extends LazyBase {
    constructor(producer) {
        super();
        this.producer = producer;
    }
    resolve(context) {
        return this.producer.produce(context);
    }
}
class LazyList extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (ret !== undefined && ret.length === 0 && this.options.omitEmpty) {
            return undefined;
        }
        return ret;
    }
}
class LazyAny extends LazyBase {
    constructor(producer, options = {}) {
        super();
        this.producer = producer;
        this.options = options;
    }
    resolve(context) {
        const ret = this.producer.produce(context);
        if (Array.isArray(ret) && ret.length === 0 && this.options.omitEmptyArray) {
            return undefined;
        }
        return ret;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGF6eS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImxhenkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0EsK0NBQWtEO0FBQ2xELG1DQUFnQztBQTRGaEM7Ozs7O0dBS0c7QUFDSCxNQUFhLElBQUk7SUFDUixNQUFNLENBQUMsV0FBVyxDQUFDLFFBQXlCLEVBQUUsVUFBa0MsRUFBRTtRQUN2RixPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxVQUFVLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUVNLE1BQU0sQ0FBQyxXQUFXLENBQUMsUUFBeUI7UUFDakQsT0FBTyxhQUFLLENBQUMsUUFBUSxDQUFDLElBQUksVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7SUFDbEQsQ0FBQztJQUVNLE1BQU0sQ0FBQyxTQUFTLENBQUMsUUFBdUIsRUFBRSxVQUFnQyxFQUFFO1FBQ2pGLE9BQU8sYUFBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLFFBQVEsQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDaEUsQ0FBQztJQUVNLE1BQU0sQ0FBQyxRQUFRLENBQUMsUUFBc0IsRUFBRSxVQUErQixFQUFFO1FBQzlFLE9BQU8sSUFBSSxPQUFPLENBQUMsUUFBUSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDtJQUNBLENBQUM7Q0FDRjtBQW5CRCxvQkFtQkM7QUFFRCxNQUFlLFFBQVE7SUFHckI7UUFDRSxJQUFJLENBQUMsYUFBYSxHQUFHLCtCQUFpQixFQUFFLENBQUM7SUFDM0MsQ0FBQztJQUdNLFFBQVE7UUFDYixPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNO1FBQ1gsT0FBTyxtQkFBbUIsQ0FBQztJQUM3QixDQUFDO0NBRUY7QUFFRCxNQUFNLFVBQVcsU0FBUSxRQUFRO0lBQy9CLFlBQTZCLFFBQXlCO1FBQ3BELEtBQUssRUFBRSxDQUFDO1FBRG1CLGFBQVEsR0FBUixRQUFRLENBQWlCO0lBRXRELENBQUM7SUFFTSxPQUFPLENBQUMsT0FBd0I7UUFDckMsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN4QyxDQUFDO0NBQ0Y7QUFFRCxNQUFNLFVBQVcsU0FBUSxRQUFRO0lBQy9CLFlBQTZCLFFBQXlCO1FBQ3BELEtBQUssRUFBRSxDQUFDO1FBRG1CLGFBQVEsR0FBUixRQUFRLENBQWlCO0lBRXRELENBQUM7SUFFTSxPQUFPLENBQUMsT0FBd0I7UUFDckMsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN4QyxDQUFDO0NBQ0Y7QUFFRCxNQUFNLFFBQVMsU0FBUSxRQUFRO0lBQzdCLFlBQTZCLFFBQXVCLEVBQW1CLFVBQWdDLEVBQUU7UUFDdkcsS0FBSyxFQUFFLENBQUM7UUFEbUIsYUFBUSxHQUFSLFFBQVEsQ0FBZTtRQUFtQixZQUFPLEdBQVAsT0FBTyxDQUEyQjtJQUV6RyxDQUFDO0lBRU0sT0FBTyxDQUFDLE9BQXdCO1FBQ3JDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzNDLElBQUksR0FBRyxLQUFLLFNBQVMsSUFBSSxHQUFHLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRTtZQUNuRSxPQUFPLFNBQVMsQ0FBQztTQUNsQjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztDQUNGO0FBRUQsTUFBTSxPQUFRLFNBQVEsUUFBUTtJQUM1QixZQUE2QixRQUFzQixFQUFtQixVQUErQixFQUFFO1FBQ3JHLEtBQUssRUFBRSxDQUFDO1FBRG1CLGFBQVEsR0FBUixRQUFRLENBQWM7UUFBbUIsWUFBTyxHQUFQLE9BQU8sQ0FBMEI7SUFFdkcsQ0FBQztJQUVNLE9BQU8sQ0FBQyxPQUF3QjtRQUNyQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUMzQyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLElBQUksR0FBRyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUU7WUFDekUsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElSZXNvbHZhYmxlLCBJUmVzb2x2ZUNvbnRleHQgfSBmcm9tICcuL3Jlc29sdmFibGUnO1xuaW1wb3J0IHsgY2FwdHVyZVN0YWNrVHJhY2UgfSBmcm9tICcuL3N0YWNrLXRyYWNlJztcbmltcG9ydCB7IFRva2VuIH0gZnJvbSAnLi90b2tlbic7XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBsYXp5IHN0cmluZyBwcm9kdWNlcnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJU3RyaW5nUHJvZHVjZXIge1xuICAvKipcbiAgICogUHJvZHVjZSB0aGUgc3RyaW5nIHZhbHVlXG4gICAqL1xuICBwcm9kdWNlKGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCk6IHN0cmluZyB8IHVuZGVmaW5lZDtcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgZm9yIGxhenkgbGlzdCBwcm9kdWNlcnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJTGlzdFByb2R1Y2VyIHtcbiAgLyoqXG4gICAqIFByb2R1Y2UgdGhlIGxpc3QgdmFsdWVcbiAgICovXG4gIHByb2R1Y2UoY29udGV4dDogSVJlc29sdmVDb250ZXh0KTogc3RyaW5nW10gfCB1bmRlZmluZWQ7XG59XG5cbi8qKlxuICogSW50ZXJmYWNlIGZvciBsYXp5IG51bWJlciBwcm9kdWNlcnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJTnVtYmVyUHJvZHVjZXIge1xuICAvKipcbiAgICogUHJvZHVjZSB0aGUgbnVtYmVyIHZhbHVlXG4gICAqL1xuICBwcm9kdWNlKGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCk6IG51bWJlciB8IHVuZGVmaW5lZDtcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgZm9yIGxhenkgdW50eXBlZCB2YWx1ZSBwcm9kdWNlcnNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQW55UHJvZHVjZXIge1xuICAvKipcbiAgICogUHJvZHVjZSB0aGUgdmFsdWVcbiAgICovXG4gIHByb2R1Y2UoY29udGV4dDogSVJlc29sdmVDb250ZXh0KTogYW55O1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGNyZWF0aW5nIGEgbGF6eSBzdHJpbmcgdG9rZW5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYXp5U3RyaW5nVmFsdWVPcHRpb25zIHtcbiAgLyoqXG4gICAqIFVzZSB0aGUgZ2l2ZW4gbmFtZSBhcyBhIGRpc3BsYXkgaGludFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGhpbnRcbiAgICovXG4gIHJlYWRvbmx5IGRpc3BsYXlIaW50Pzogc3RyaW5nO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGNyZWF0aW5nIGEgbGF6eSBsaXN0IHRva2VuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGF6eUxpc3RWYWx1ZU9wdGlvbnMge1xuICAvKipcbiAgICogVXNlIHRoZSBnaXZlbiBuYW1lIGFzIGEgZGlzcGxheSBoaW50XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gaGludFxuICAgKi9cbiAgcmVhZG9ubHkgZGlzcGxheUhpbnQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIElmIHRoZSBwcm9kdWNlZCBsaXN0IGlzIGVtcHR5LCByZXR1cm4gJ3VuZGVmaW5lZCcgaW5zdGVhZFxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgb21pdEVtcHR5PzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBjcmVhdGluZyBsYXp5IHVudHlwZWQgdG9rZW5zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGF6eUFueVZhbHVlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBVc2UgdGhlIGdpdmVuIG5hbWUgYXMgYSBkaXNwbGF5IGhpbnRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBoaW50XG4gICAqL1xuICByZWFkb25seSBkaXNwbGF5SGludD86IHN0cmluZztcblxuICAvKipcbiAgICogSWYgdGhlIHByb2R1Y2VkIHZhbHVlIGlzIGFuIGFycmF5IGFuZCBpdCBpcyBlbXB0eSwgcmV0dXJuICd1bmRlZmluZWQnIGluc3RlYWRcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG9taXRFbXB0eUFycmF5PzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBMYXppbHkgcHJvZHVjZSBhIHZhbHVlXG4gKlxuICogQ2FuIGJlIHVzZWQgdG8gcmV0dXJuIGEgc3RyaW5nLCBsaXN0IG9yIG51bWVyaWMgdmFsdWUgd2hvc2UgYWN0dWFsIHZhbHVlXG4gKiB3aWxsIG9ubHkgYmUgY2FsY3VsYXRlZCBsYXRlciwgZHVyaW5nIHN5bnRoZXNpcy5cbiAqL1xuZXhwb3J0IGNsYXNzIExhenkge1xuICBwdWJsaWMgc3RhdGljIHN0cmluZ1ZhbHVlKHByb2R1Y2VyOiBJU3RyaW5nUHJvZHVjZXIsIG9wdGlvbnM6IExhenlTdHJpbmdWYWx1ZU9wdGlvbnMgPSB7fSkge1xuICAgIHJldHVybiBUb2tlbi5hc1N0cmluZyhuZXcgTGF6eVN0cmluZyhwcm9kdWNlciksIG9wdGlvbnMpO1xuICB9XG5cbiAgcHVibGljIHN0YXRpYyBudW1iZXJWYWx1ZShwcm9kdWNlcjogSU51bWJlclByb2R1Y2VyKSB7XG4gICAgcmV0dXJuIFRva2VuLmFzTnVtYmVyKG5ldyBMYXp5TnVtYmVyKHByb2R1Y2VyKSk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGxpc3RWYWx1ZShwcm9kdWNlcjogSUxpc3RQcm9kdWNlciwgb3B0aW9uczogTGF6eUxpc3RWYWx1ZU9wdGlvbnMgPSB7fSkge1xuICAgIHJldHVybiBUb2tlbi5hc0xpc3QobmV3IExhenlMaXN0KHByb2R1Y2VyLCBvcHRpb25zKSwgb3B0aW9ucyk7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGFueVZhbHVlKHByb2R1Y2VyOiBJQW55UHJvZHVjZXIsIG9wdGlvbnM6IExhenlBbnlWYWx1ZU9wdGlvbnMgPSB7fSk6IElSZXNvbHZhYmxlIHtcbiAgICByZXR1cm4gbmV3IExhenlBbnkocHJvZHVjZXIsIG9wdGlvbnMpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcigpIHtcbiAgfVxufVxuXG5hYnN0cmFjdCBjbGFzcyBMYXp5QmFzZSBpbXBsZW1lbnRzIElSZXNvbHZhYmxlIHtcbiAgcHVibGljIHJlYWRvbmx5IGNyZWF0aW9uU3RhY2s6IHN0cmluZ1tdO1xuXG4gIGNvbnN0cnVjdG9yKCkge1xuICAgIHRoaXMuY3JlYXRpb25TdGFjayA9IGNhcHR1cmVTdGFja1RyYWNlKCk7XG4gIH1cblxuICBwdWJsaWMgYWJzdHJhY3QgcmVzb2x2ZShjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpOiBhbnk7XG4gIHB1YmxpYyB0b1N0cmluZygpIHtcbiAgICByZXR1cm4gVG9rZW4uYXNTdHJpbmcodGhpcyk7XG4gIH1cblxuICAvKipcbiAgICogVHVybiB0aGlzIFRva2VuIGludG8gSlNPTlxuICAgKlxuICAgKiBDYWxsZWQgYXV0b21hdGljYWxseSB3aGVuIEpTT04uc3RyaW5naWZ5KCkgaXMgY2FsbGVkIG9uIGEgVG9rZW4uXG4gICAqL1xuICBwdWJsaWMgdG9KU09OKCk6IGFueSB7XG4gICAgcmV0dXJuICc8dW5yZXNvbHZlZC1sYXp5Pic7XG4gIH1cblxufVxuXG5jbGFzcyBMYXp5U3RyaW5nIGV4dGVuZHMgTGF6eUJhc2Uge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb2R1Y2VyOiBJU3RyaW5nUHJvZHVjZXIpIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIHJlc29sdmUoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSB7XG4gICAgcmV0dXJuIHRoaXMucHJvZHVjZXIucHJvZHVjZShjb250ZXh0KTtcbiAgfVxufVxuXG5jbGFzcyBMYXp5TnVtYmVyIGV4dGVuZHMgTGF6eUJhc2Uge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb2R1Y2VyOiBJTnVtYmVyUHJvZHVjZXIpIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIHJlc29sdmUoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSB7XG4gICAgcmV0dXJuIHRoaXMucHJvZHVjZXIucHJvZHVjZShjb250ZXh0KTtcbiAgfVxufVxuXG5jbGFzcyBMYXp5TGlzdCBleHRlbmRzIExhenlCYXNlIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9kdWNlcjogSUxpc3RQcm9kdWNlciwgcHJpdmF0ZSByZWFkb25seSBvcHRpb25zOiBMYXp5TGlzdFZhbHVlT3B0aW9ucyA9IHt9KSB7XG4gICAgc3VwZXIoKTtcbiAgfVxuXG4gIHB1YmxpYyByZXNvbHZlKGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCkge1xuICAgIGNvbnN0IHJldCA9IHRoaXMucHJvZHVjZXIucHJvZHVjZShjb250ZXh0KTtcbiAgICBpZiAocmV0ICE9PSB1bmRlZmluZWQgJiYgcmV0Lmxlbmd0aCA9PT0gMCAmJiB0aGlzLm9wdGlvbnMub21pdEVtcHR5KSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICByZXR1cm4gcmV0O1xuICB9XG59XG5cbmNsYXNzIExhenlBbnkgZXh0ZW5kcyBMYXp5QmFzZSB7XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcHJvZHVjZXI6IElBbnlQcm9kdWNlciwgcHJpdmF0ZSByZWFkb25seSBvcHRpb25zOiBMYXp5QW55VmFsdWVPcHRpb25zID0ge30pIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIHJlc29sdmUoY29udGV4dDogSVJlc29sdmVDb250ZXh0KSB7XG4gICAgY29uc3QgcmV0ID0gdGhpcy5wcm9kdWNlci5wcm9kdWNlKGNvbnRleHQpO1xuICAgIGlmIChBcnJheS5pc0FycmF5KHJldCkgJiYgcmV0Lmxlbmd0aCA9PT0gMCAmJiB0aGlzLm9wdGlvbnMub21pdEVtcHR5QXJyYXkpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG4gIH1cbn0iXX0=