"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionedStackManagerCreateAndUpdater = exports.onEvent = void 0;
const client_cloudformation_1 = require("@aws-sdk/client-cloudformation");
const types_1 = require("./types");
const cloudformation = new client_cloudformation_1.CloudFormationClient();
const onEvent = async (event) => {
    console.debug("Event Received:", JSON.stringify(event));
    try {
        switch (event.RequestType) {
            case "Create":
                return await onCreateOrUpdate(event);
            case "Update":
                return await onCreateOrUpdate(event);
            case "Delete":
                return await onDelete(event);
            default:
                throw new Error(`Unknown event type in event ${event}`);
        }
    }
    catch (err) {
        console.error(err);
        throw new Error("Failed");
    }
};
exports.onEvent = onEvent;
const onCreateOrUpdate = async (event) => {
    const resourceProperties = event.ResourceProperties;
    const createOrUpdater = new VersionedStackManagerCreateAndUpdater(resourceProperties);
    const deletedStackNames = await createOrUpdater.createOrUpdate();
    return {
        Status: "SUCCESS",
        RequestId: event.RequestId,
        StackId: event.StackId,
        LogicalResourceId: event.LogicalResourceId,
        PhysicalResourceId: deletedStackNames?.join(",") ?? "None",
    };
};
const NON_DELETED_STACK_STATUSES = [
    client_cloudformation_1.StackStatus.CREATE_COMPLETE,
    client_cloudformation_1.StackStatus.CREATE_FAILED,
    client_cloudformation_1.StackStatus.CREATE_IN_PROGRESS,
    client_cloudformation_1.StackStatus.DELETE_FAILED,
    client_cloudformation_1.StackStatus.IMPORT_COMPLETE,
    client_cloudformation_1.StackStatus.IMPORT_IN_PROGRESS,
    client_cloudformation_1.StackStatus.IMPORT_ROLLBACK_COMPLETE,
    client_cloudformation_1.StackStatus.IMPORT_ROLLBACK_FAILED,
    client_cloudformation_1.StackStatus.IMPORT_ROLLBACK_IN_PROGRESS,
    client_cloudformation_1.StackStatus.REVIEW_IN_PROGRESS,
    client_cloudformation_1.StackStatus.ROLLBACK_COMPLETE,
    client_cloudformation_1.StackStatus.ROLLBACK_FAILED,
    client_cloudformation_1.StackStatus.ROLLBACK_IN_PROGRESS,
    client_cloudformation_1.StackStatus.UPDATE_COMPLETE,
    client_cloudformation_1.StackStatus.UPDATE_COMPLETE_CLEANUP_IN_PROGRESS,
    client_cloudformation_1.StackStatus.UPDATE_IN_PROGRESS,
    client_cloudformation_1.StackStatus.UPDATE_ROLLBACK_COMPLETE,
    client_cloudformation_1.StackStatus.UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS,
    client_cloudformation_1.StackStatus.UPDATE_ROLLBACK_FAILED,
    client_cloudformation_1.StackStatus.UPDATE_ROLLBACK_IN_PROGRESS,
];
const sortStacksAscending = (stackA, stackB) => {
    return (stackA?.StackName ?? "ZZZZZ") < (stackB?.StackName ?? "ZZZZZ")
        ? -1
        : 1;
};
const sortStacksDescending = (stackA, stackB) => {
    return (stackA?.StackName ?? "AAAAA") < (stackB?.StackName ?? "AAAAA")
        ? 1
        : -1;
};
class VersionedStackManagerCreateAndUpdater {
    constructor(resourceProperties) {
        this.resourceProperties = resourceProperties;
        (0, types_1.validateProps)(resourceProperties);
    }
    async createOrUpdate() {
        if (this.resourceProperties.numberOfStacksToKeep < 1) {
            return;
        }
        const allStacks = await this.listAllNonDeletedStacks();
        const relevantStacks = allStacks.filter((stack) => stack?.StackName?.startsWith(this.resourceProperties.stackNamePrefix));
        const stacksToDelete = relevantStacks
            .sort((stackA, stackB) => {
            if (this.resourceProperties.sortDirection === "ASCENDING") {
                return sortStacksAscending(stackA, stackB);
            }
            return sortStacksDescending(stackA, stackB);
        })
            .slice(this.resourceProperties.numberOfStacksToKeep);
        console.debug(`Deleting ${stacksToDelete.length} Stacks (dryRun: ${this.resourceProperties.dryRun ?? false}): `, stacksToDelete.map((s) => s.StackName).join(","));
        if (!this.resourceProperties.dryRun) {
            for (const stack of stacksToDelete) {
                await cloudformation.send(new client_cloudformation_1.DeleteStackCommand({
                    StackName: stack.StackName,
                }));
            }
        }
        return stacksToDelete.map((s) => s.StackName);
    }
    async listAllNonDeletedStacks() {
        const stacks = new Array();
        let cursor = undefined;
        do {
            const response = await cloudformation.send(new client_cloudformation_1.ListStacksCommand({
                StackStatusFilter: NON_DELETED_STACK_STATUSES,
                NextToken: cursor,
            }));
            cursor = response.NextToken;
            stacks.push(...(response.StackSummaries ?? []));
        } while (cursor);
        return stacks;
    }
}
exports.VersionedStackManagerCreateAndUpdater = VersionedStackManagerCreateAndUpdater;
const onDelete = async (event) => {
    // Doing nothing, perhaps someday allow for deletion of all resources created using a retention policy prop.
    return {
        Status: "SUCCESS",
        RequestId: event.RequestId,
        StackId: event.StackId,
        LogicalResourceId: event.LogicalResourceId,
        PhysicalResourceId: event.PhysicalResourceId,
    };
};
//# sourceMappingURL=data:application/json;base64,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