import requests
import uuid
import os
from datetime import datetime


class SentinelAnalyticsRealmPlugin:
    """
    Client for plugging an app into the Sentinel Analytics Realm.
    Supports Option A (HTTP Ingestion) with the canonical event taxonomy.
    """

    def __init__(self, project_id=None, app_id=None, env=None, access_token=None, base_url=None):
        self.project_id = project_id or os.environ.get("SENTINEL_ANALYTICS_REALM_PROJECT_ID")
        self.app_id = app_id or os.environ.get("SENTINEL_ANALYTICS_REALM_APP_ID")
        self.env = env or os.environ.get("SENTINEL_ANALYTICS_REALM_ENV", "prod")
        self.access_token = access_token
        self.base_url = (base_url or 
                         os.environ.get("SENTINEL_ANALYTICS_REALM_URL", "https://api.sentinelanalyticsrealm.com")).rstrip('/')

    def track(self, name, visitor_id=None, user_id=None, session_id=None, 
              category=None, action=None, label=None, value=None,
              properties=None, context=None, origin='real'):
        """
        Track a single event.
        """
        url = f"{self.base_url}/api/v1/ingest/events"
        headers = {"Content-Type": "application/json"}
        if self.access_token:
            headers["Authorization"] = f"Bearer {self.access_token}"

        payload = self._build_event_payload(
            name=name,
            visitor_id=visitor_id,
            user_id=user_id,
            session_id=session_id,
            category=category,
            action=action,
            label=label,
            value=value,
            properties=properties,
            context=context,
            origin=origin
        )

        try:
            response = requests.post(url, json=payload, headers=headers, timeout=5)
            response.raise_for_status()
            return response.json()
        except Exception as e:
            print(f"Sentinel Analytics Realm tracking failed: {e}")
            return None

    def track_batch(self, events):
        """
        Track multiple events in one call.
        'events' should be a list of dicts, each containing 'name' and optional fields.
        """
        url = f"{self.base_url}/api/v1/ingest/events/batch"
        headers = {"Content-Type": "application/json"}
        if self.access_token:
            headers["Authorization"] = f"Bearer {self.access_token}"

        batch_payload = {
            "project_id": str(self.project_id),
            "app_id": str(self.app_id),
            "env": self.env,
            "events": [self._build_event_payload(**e) for e in events]
        }

        try:
            response = requests.post(url, json=batch_payload, headers=headers, timeout=10)
            response.raise_for_status()
            return response.json()
        except Exception as e:
            print(f"Sentinel Analytics Realm batch tracking failed: {e}")
            return None

    def identify(self, user_id, visitor_id=None, traits=None):
        """
        Convenience method for identity linking.
        """
        return self.track(
            name="identify",
            user_id=user_id,
            visitor_id=visitor_id,
            properties=traits
        )

    def _build_event_payload(self, name, visitor_id=None, user_id=None, session_id=None,
                             category=None, action=None, label=None, value=None,
                             properties=None, context=None, origin='real'):
        return {
            "project_id": str(self.project_id),
            "app_id": str(self.app_id),
            "env": self.env,
            "event_id": str(uuid.uuid4()),
            "occurred_at": datetime.utcnow().isoformat() + "Z",
            "sent_at": datetime.utcnow().isoformat() + "Z",
            "origin": origin,
            "identity": {
                "visitor_id": visitor_id,
                "user_id": str(user_id) if user_id else None,
                "session_id": session_id
            },
            "event": {
                "name": name,
                "category": category or "",
                "action": action or "",
                "label": label or "",
                "value": value
            },
            "context": context or {},
            "properties": properties or {}
        }
