"""
Content Generators for Pipeline Processing

Provides generators for creating new Logseq content based on analysis results,
including summary pages, insight blocks, and structured content.
"""

from abc import ABC, abstractmethod
from typing import Dict, Any, Optional, List, Union
import logging
from datetime import datetime
from collections import defaultdict
import json

from .core import ProcessingContext
from ..builders import LogseqBuilder, PageBuilder, BlockBuilder


class ContentGenerator(ABC):
    """Abstract base class for content generators."""
    
    def __init__(self, name: str):
        self.name = name
        self.logger = logging.getLogger(f"generator.{name}")
    
    @abstractmethod
    def generate(self, context: ProcessingContext) -> Optional[Dict[str, Any]]:
        """Generate content based on processing context."""
        pass
    
    def can_generate(self, context: ProcessingContext) -> bool:
        """Check if this generator can produce content from the context."""
        return bool(context.blocks or context.analysis_results or context.extracted_content)


class SummaryPageGenerator(ContentGenerator):
    """Generate summary pages from analysis results."""
    
    def __init__(self):
        super().__init__("summary_page")
        self.template = """# Processing Summary - {date}

## Overview
- **Total Blocks Processed**: {block_count}
- **Processing Session**: {session_id}
- **Generated**: {timestamp}

{content_sections}

## Statistics
{statistics}

---
*Generated by Logseq Pipeline System*
"""
    
    def generate(self, context: ProcessingContext) -> Optional[Dict[str, Any]]:
        """Generate a comprehensive summary page."""
        if not self.can_generate(context):
            return None
        
        # Gather summary data
        summary_data = self._gather_summary_data(context)
        
        # Generate content sections
        content_sections = self._generate_content_sections(context)
        
        # Generate statistics
        statistics = self._generate_statistics(context)
        
        # Format the page
        page_content = self.template.format(
            date=datetime.now().strftime("%Y-%m-%d"),
            block_count=len(context.blocks),
            session_id=context.session_id[:8],
            timestamp=datetime.now().isoformat(),
            content_sections=content_sections,
            statistics=statistics
        )
        
        # Create page builder
        page_builder = (PageBuilder(f"Pipeline Summary - {datetime.now().strftime('%Y-%m-%d')}")
                       .add_content(page_content)
                       .add_property("generated_by", "logseq_pipeline")
                       .add_property("session_id", context.session_id)
                       .add_property("pipeline_id", context.pipeline_id)
                       .add_property("generation_date", datetime.now().isoformat()))
        
        return {
            'type': 'page',
            'title': page_builder.name,
            'content': page_content,
            'builder': page_builder,
            'metadata': summary_data
        }
    
    def _gather_summary_data(self, context: ProcessingContext) -> Dict[str, Any]:
        """Gather data for the summary."""
        return {
            'total_blocks': len(context.blocks),
            'processed_items': context.processed_items,
            'errors': len(context.errors),
            'extracted_items': context.extracted_content.get('count', 0),
            'analysis_results': len(context.analysis_results),
            'generated_content': len(context.generated_content)
        }
    
    def _generate_content_sections(self, context: ProcessingContext) -> str:
        """Generate content sections for the summary."""
        sections = []
        
        # Extraction results section
        if context.extracted_content:
            sections.append(self._generate_extraction_section(context))
        
        # Analysis results section
        if context.analysis_results:
            sections.append(self._generate_analysis_section(context))
        
        # Error section
        if context.errors:
            sections.append(self._generate_error_section(context))
        
        return "\n\n".join(sections)
    
    def _generate_extraction_section(self, context: ProcessingContext) -> str:
        """Generate extraction results section."""
        section = ["## Content Extraction Results"]
        
        extracted_count = context.extracted_content.get('count', 0)
        section.append(f"**Total Items Extracted**: {extracted_count}")
        
        # Group by extractor type
        if 'items' in context.extracted_content:
            extractor_counts = defaultdict(int)
            for item in context.extracted_content['items']:
                extractor_counts[item.get('extractor', 'unknown')] += 1
            
            section.append("\n**By Extractor Type**:")
            for extractor, count in extractor_counts.items():
                section.append(f"- {extractor}: {count} items")
        
        return "\n".join(section)
    
    def _generate_analysis_section(self, context: ProcessingContext) -> str:
        """Generate analysis results section."""
        section = ["## Analysis Results"]
        
        total_analyses = sum(
            result.get('count', 0) 
            for result in context.analysis_results.values() 
            if isinstance(result, dict)
        )
        section.append(f"**Total Analyses**: {total_analyses}")
        
        # Analysis by type
        section.append("\n**By Analysis Type**:")
        for analyzer, results in context.analysis_results.items():
            if isinstance(results, dict):
                count = results.get('count', 0)
                section.append(f"- {analyzer}: {count} analyses")
                
                # Add specific insights if available
                if analyzer == 'sentiment' and 'results' in results:
                    sentiment_stats = self._analyze_sentiment_results(results['results'])
                    if sentiment_stats:
                        section.append(f"  - Positive: {sentiment_stats.get('positive', 0)}")
                        section.append(f"  - Negative: {sentiment_stats.get('negative', 0)}")
                        section.append(f"  - Neutral: {sentiment_stats.get('neutral', 0)}")
        
        return "\n".join(section)
    
    def _generate_error_section(self, context: ProcessingContext) -> str:
        """Generate error section."""
        section = ["## Errors"]
        section.append(f"**Total Errors**: {len(context.errors)}")
        
        if context.errors:
            # Group errors by type
            error_types = defaultdict(int)
            for error in context.errors:
                error_type = error.get('type', 'Unknown')
                error_types[error_type] += 1
            
            section.append("\n**By Error Type**:")
            for error_type, count in error_types.items():
                section.append(f"- {error_type}: {count}")
        
        return "\n".join(section)
    
    def _generate_statistics(self, context: ProcessingContext) -> str:
        """Generate statistics section."""
        stats = []
        
        # Processing time
        elapsed = (datetime.now() - context.start_time).total_seconds()
        stats.append(f"**Processing Time**: {elapsed:.1f} seconds")
        
        # Processing rate
        if elapsed > 0 and context.total_items > 0:
            rate = context.processed_items / elapsed
            stats.append(f"**Processing Rate**: {rate:.1f} items/second")
        
        # Success rate
        if context.total_items > 0:
            success_rate = (context.processed_items / context.total_items) * 100
            stats.append(f"**Success Rate**: {success_rate:.1f}%")
        
        return "\n".join(stats)
    
    def _analyze_sentiment_results(self, sentiment_results: List[Dict[str, Any]]) -> Dict[str, int]:
        """Analyze sentiment results for statistics."""
        sentiment_counts = defaultdict(int)
        
        for result in sentiment_results:
            analysis = result.get('analysis', {})
            sentiment = analysis.get('sentiment', 'neutral')
            sentiment_counts[sentiment] += 1
        
        return dict(sentiment_counts)


class InsightsBlockGenerator(ContentGenerator):
    """Generate insight blocks from analysis results."""
    
    def __init__(self):
        super().__init__("insights_blocks")
    
    def generate(self, context: ProcessingContext) -> Optional[Dict[str, Any]]:
        """Generate insight blocks."""
        if not context.analysis_results:
            return None
        
        insights = []
        
        # Generate sentiment insights
        if 'sentiment' in context.analysis_results:
            sentiment_insight = self._generate_sentiment_insight(context.analysis_results['sentiment'])
            if sentiment_insight:
                insights.append(sentiment_insight)
        
        # Generate topic insights  
        if 'topics' in context.analysis_results:
            topic_insight = self._generate_topic_insight(context.analysis_results['topics'])
            if topic_insight:
                insights.append(topic_insight)
        
        # Generate summary insights
        if 'summary' in context.analysis_results:
            summary_insight = self._generate_summary_insight(context.analysis_results['summary'])
            if summary_insight:
                insights.append(summary_insight)
        
        if not insights:
            return None
        
        return {
            'type': 'blocks',
            'insights': insights,
            'count': len(insights),
            'generated_at': datetime.now().isoformat()
        }
    
    def _generate_sentiment_insight(self, sentiment_results: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Generate sentiment insight block."""
        if not sentiment_results.get('results'):
            return None
        
        # Analyze sentiment distribution
        sentiment_counts = defaultdict(int)
        total_polarity = 0
        count = 0
        
        for result in sentiment_results['results']:
            analysis = result.get('analysis', {})
            sentiment = analysis.get('sentiment', 'neutral')
            polarity = analysis.get('polarity', 0)
            
            sentiment_counts[sentiment] += 1
            total_polarity += polarity
            count += 1
        
        avg_polarity = total_polarity / count if count > 0 else 0
        
        # Create insight content
        content = f"""## Sentiment Analysis Insight

**Overall Sentiment Distribution:**
- Positive: {sentiment_counts['positive']}
- Neutral: {sentiment_counts['neutral']} 
- Negative: {sentiment_counts['negative']}

**Average Polarity**: {avg_polarity:.2f} (Range: -1.0 to 1.0)

*Analysis based on {count} content items*"""
        
        block_builder = (BlockBuilder()
                        .add_content(content)
                        .add_property("insight_type", "sentiment")
                        .add_property("generated_by", "logseq_pipeline")
                        .add_property("analysis_count", count))
        
        return {
            'type': 'sentiment_insight',
            'content': content,
            'builder': block_builder,
            'statistics': dict(sentiment_counts)
        }
    
    def _generate_topic_insight(self, topic_results: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Generate topic insight block."""
        if not topic_results.get('results'):
            return None
        
        # Collect all topics
        all_topics = defaultdict(int)
        all_keywords = defaultdict(int)
        
        for result in topic_results['results']:
            analysis = result.get('analysis', {})
            
            # Count topics
            for topic in analysis.get('topics', []):
                topic_name = topic.get('topic')
                topic_score = topic.get('score', 0)
                if topic_name:
                    all_topics[topic_name] += topic_score
            
            # Count keywords
            for keyword in analysis.get('keywords', []):
                word = keyword.get('word')
                count = keyword.get('count', 0)
                if word:
                    all_keywords[word] += count
        
        # Get top topics and keywords
        top_topics = sorted(all_topics.items(), key=lambda x: x[1], reverse=True)[:5]
        top_keywords = sorted(all_keywords.items(), key=lambda x: x[1], reverse=True)[:10]
        
        # Create insight content
        content_parts = ["## Topic Analysis Insight"]
        
        if top_topics:
            content_parts.append("\n**Top Topics:**")
            for topic, score in top_topics:
                content_parts.append(f"- {topic}: {score}")
        
        if top_keywords:
            content_parts.append("\n**Top Keywords:**")
            for keyword, count in top_keywords:
                content_parts.append(f"- {keyword}: {count}")
        
        content = "\n".join(content_parts)
        
        block_builder = (BlockBuilder()
                        .add_content(content)
                        .add_property("insight_type", "topics")
                        .add_property("generated_by", "logseq_pipeline")
                        .add_property("topic_count", len(all_topics))
                        .add_property("keyword_count", len(all_keywords)))
        
        return {
            'type': 'topic_insight',
            'content': content,
            'builder': block_builder,
            'top_topics': top_topics,
            'top_keywords': top_keywords
        }
    
    def _generate_summary_insight(self, summary_results: Dict[str, Any]) -> Optional[Dict[str, Any]]:
        """Generate summary insight block."""
        if not summary_results.get('results'):
            return None
        
        # Collect summaries
        summaries = []
        total_compression = 0
        count = 0
        
        for result in summary_results['results']:
            analysis = result.get('analysis', {})
            summary = analysis.get('summary')
            compression_ratio = analysis.get('compression_ratio', 1.0)
            
            if summary:
                summaries.append(summary)
                total_compression += compression_ratio
                count += 1
        
        if not summaries:
            return None
        
        avg_compression = total_compression / count if count > 0 else 1.0
        
        # Create insight content - combine key summaries
        content = f"""## Content Summary Insight

**Key Content Summaries:**

"""
        
        # Add top summaries (limit to 3)
        for i, summary in enumerate(summaries[:3], 1):
            content += f"{i}. {summary}\n\n"
        
        content += f"""**Summary Statistics:**
- Total summaries: {len(summaries)}
- Average compression: {avg_compression:.1%}
- Content reduction: {(1-avg_compression)*100:.1f}%"""
        
        block_builder = (BlockBuilder()
                        .add_content(content)
                        .add_property("insight_type", "summary")
                        .add_property("generated_by", "logseq_pipeline")
                        .add_property("summary_count", len(summaries))
                        .add_property("avg_compression", avg_compression))
        
        return {
            'type': 'summary_insight',
            'content': content,
            'builder': block_builder,
            'summaries': summaries
        }


class TaskAnalysisGenerator(ContentGenerator):
    """Generate task analysis reports."""
    
    def __init__(self):
        super().__init__("task_analysis")
    
    def generate(self, context: ProcessingContext) -> Optional[Dict[str, Any]]:
        """Generate task analysis content."""
        # Extract task-related blocks
        task_blocks = []
        for block in context.blocks:
            if block.content and any(marker in block.content for marker in ['TODO', 'DOING', 'DONE']):
                task_blocks.append(block)
        
        if not task_blocks:
            return None
        
        # Analyze tasks
        task_stats = self._analyze_tasks(task_blocks)
        
        # Create task analysis content
        content = self._create_task_analysis_content(task_stats)
        
        page_builder = (PageBuilder("Task Analysis Report")
                       .add_content(content)
                       .add_property("generated_by", "logseq_pipeline")
                       .add_property("task_count", len(task_blocks))
                       .add_property("generation_date", datetime.now().isoformat()))
        
        return {
            'type': 'task_analysis',
            'content': content,
            'builder': page_builder,
            'statistics': task_stats
        }
    
    def _analyze_tasks(self, task_blocks: List) -> Dict[str, Any]:
        """Analyze task blocks."""
        stats = {
            'total_tasks': len(task_blocks),
            'todo_count': 0,
            'doing_count': 0,
            'done_count': 0,
            'completion_rate': 0.0
        }
        
        for block in task_blocks:
            content = block.content.upper()
            if 'TODO' in content:
                stats['todo_count'] += 1
            elif 'DOING' in content:
                stats['doing_count'] += 1
            elif 'DONE' in content:
                stats['done_count'] += 1
        
        if stats['total_tasks'] > 0:
            stats['completion_rate'] = stats['done_count'] / stats['total_tasks']
        
        return stats
    
    def _create_task_analysis_content(self, stats: Dict[str, Any]) -> str:
        """Create task analysis content."""
        return f"""# Task Analysis Report

## Overview
- **Total Tasks**: {stats['total_tasks']}
- **Completion Rate**: {stats['completion_rate']:.1%}

## Task Breakdown
- **TODO**: {stats['todo_count']}
- **DOING**: {stats['doing_count']}
- **DONE**: {stats['done_count']}

## Progress Visualization
```
TODO:  {'█' * (stats['todo_count'] // 2)}
DOING: {'█' * (stats['doing_count'] // 2)}
DONE:  {'█' * (stats['done_count'] // 2)}
```

*Generated: {datetime.now().strftime('%Y-%m-%d %H:%M')}*
"""


class GeneratorRegistry:
    """Registry for managing content generators."""
    
    def __init__(self):
        self.generators: Dict[str, ContentGenerator] = {}
        self.logger = logging.getLogger("generator.registry")
        
        # Register default generators
        self.register_defaults()
    
    def register(self, generator: ContentGenerator):
        """Register a generator."""
        self.generators[generator.name] = generator
        self.logger.info(f"Registered generator: {generator.name}")
    
    def get(self, name: str) -> Optional[ContentGenerator]:
        """Get generator by name."""
        return self.generators.get(name)
    
    def get_all(self) -> Dict[str, ContentGenerator]:
        """Get all registered generators."""
        return self.generators.copy()
    
    def generate_content(self, context: ProcessingContext, generator_names: List[str] = None) -> Dict[str, Any]:
        """Generate content using specified generators."""
        results = {}
        
        if generator_names:
            generators = [self.get(name) for name in generator_names]
            generators = [g for g in generators if g is not None]
        else:
            generators = list(self.generators.values())
        
        for generator in generators:
            try:
                if generator.can_generate(context):
                    result = generator.generate(context)
                    if result:
                        results[generator.name] = result
            except Exception as e:
                results[generator.name] = {'error': str(e)}
        
        return results
    
    def register_defaults(self):
        """Register default generators."""
        self.register(SummaryPageGenerator())
        self.register(InsightsBlockGenerator())
        self.register(TaskAnalysisGenerator())


# Global registry instance
generator_registry = GeneratorRegistry()


# Convenience functions
def get_generator(name: str) -> Optional[ContentGenerator]:
    """Get generator by name from global registry."""
    return generator_registry.get(name)


def get_all_generators() -> Dict[str, ContentGenerator]:
    """Get all generators from global registry."""
    return generator_registry.get_all()


def generate_content(context: ProcessingContext, generator_names: List[str] = None) -> Dict[str, Any]:
    """Generate content using specified generators."""
    return generator_registry.generate_content(context, generator_names)