"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToKendra = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const defaults = require("@aws-solutions-constructs/core");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToKendra class.
 */
class LambdaToKendra extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToKendra class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToKendraProps} props - user provided props for the construct.
     * @since 1.120.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        if (props.kendraIndexProps && props.existingKendraIndexObj) {
            throw new Error('You may not provide both kendraIndexProps and existingKendraIndexObj');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.KENDRA);
        }
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        this.kendraIndex = defaults.buildKendraIndex(this, id, {
            kendraIndexProps: props.kendraIndexProps,
            existingIndexObj: props.existingKendraIndexObj
        });
        this.kendraDataSources = defaults.AddMultipleKendraDataSources(this, id, this.kendraIndex, props.kendraDataSourcesProps);
        // Update Lambda function IAM policy with correct privileges to Kendra index
        const normalizedPermissions = props.indexPermissions ? defaults.normalizeKendraPermissions(props.indexPermissions) : undefined;
        if (!normalizedPermissions || normalizedPermissions.includes("READ")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Query",
                    "kendra:Describe*",
                    "kendra:Get*",
                    "kendra:BatchGet*",
                    "kendra:List*",
                    "kendra:Retrieve"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (!normalizedPermissions || normalizedPermissions.includes("SUBMITFEEDBACK")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:SubmitFeedback"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (normalizedPermissions?.includes("WRITE")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Associate*",
                    "kendra:BatchPut*",
                    "kendra:Clear",
                    "kendra:Create*",
                    "kendra:Delete*",
                    "kendra:Disassociate*",
                    "kendra:Put*",
                    "kendra:Update*",
                    "kendra:Start*",
                    "kendra:Submit*",
                    "kendra:Stop*",
                    "kendra:TagResource",
                    "kendra:UntagResource"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        // Configure environment variables
        const indexIdEnvironmentVariableName = props.indexIdEnvironmentVariableName || 'KENDRA_INDEX_ID';
        this.lambdaFunction.addEnvironment(indexIdEnvironmentVariableName, this.kendraIndex.attrId);
    }
}
exports.LambdaToKendra = LambdaToKendra;
_a = JSII_RTTI_SYMBOL_1;
LambdaToKendra[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-kendra.LambdaToKendra", version: "2.97.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBOzs7Ozs7Ozs7OztHQVdHO0FBRUgsVUFBVTtBQUNWLDJEQUEyRDtBQUUzRCwyQ0FBMkM7QUFHM0MsMkNBQXVDO0FBb0V2Qzs7R0FFRztBQUNILE1BQWEsY0FBZSxTQUFRLHNCQUFTO0lBTTNDOzs7Ozs7O09BT0c7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTBCO1FBQ2xFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsUUFBUSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM5QixRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFakMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLElBQUksS0FBSyxDQUFDLHNCQUFzQixFQUFFLENBQUM7WUFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRUFBc0UsQ0FBQyxDQUFDO1FBQzFGLENBQUM7UUFFRCxJQUFJLEtBQUssQ0FBQyxTQUFTLElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBQ3pDLElBQUksQ0FBQyxHQUFHLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUU7Z0JBQ2xDLGVBQWUsRUFBRSxRQUFRLENBQUMsdUJBQXVCLEVBQUU7Z0JBQ25ELFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDOUIsWUFBWSxFQUFFLEtBQUssQ0FBQyxRQUFRO2dCQUM1QixpQkFBaUIsRUFBRTtvQkFDakIsa0JBQWtCLEVBQUUsSUFBSTtvQkFDeEIsZ0JBQWdCLEVBQUUsSUFBSTtpQkFDdkI7YUFDRixDQUFDLENBQUM7WUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLG9CQUFvQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3hGLENBQUM7UUFFRCw0QkFBNEI7UUFDNUIsSUFBSSxDQUFDLGNBQWMsR0FBRyxRQUFRLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFO1lBQ3ZELGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7WUFDMUMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtZQUM5QyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7U0FDZCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsV0FBVyxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3JELGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtTQUMvQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsUUFBUSxDQUFDLDRCQUE0QixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUV6SCw0RUFBNEU7UUFDNUUsTUFBTSxxQkFBcUIsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQywwQkFBMEIsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQy9ILElBQUksQ0FBQyxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQztZQUNyRSxvQ0FBb0M7WUFDcEMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxlQUFlLENBQ2pDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsT0FBTyxFQUFFO29CQUNQLGNBQWM7b0JBQ2Qsa0JBQWtCO29CQUNsQixhQUFhO29CQUNiLGtCQUFrQjtvQkFDbEIsY0FBYztvQkFDZCxpQkFBaUI7aUJBQUM7Z0JBQ3BCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDO2FBQ3RDLENBQUMsQ0FDSCxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksQ0FBQyxxQkFBcUIsSUFBSSxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxDQUFDO1lBQy9FLG9DQUFvQztZQUNwQyxJQUFJLENBQUMsY0FBYyxDQUFDLGVBQWUsQ0FDakMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUU7b0JBQ1AsdUJBQXVCO2lCQUFDO2dCQUMxQixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQzthQUN0QyxDQUFDLENBQ0gsQ0FBQztRQUNKLENBQUM7UUFFRCxJQUFJLHFCQUFxQixFQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdDLG9DQUFvQztZQUNwQyxJQUFJLENBQUMsY0FBYyxDQUFDLGVBQWUsQ0FDakMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUU7b0JBQ1AsbUJBQW1CO29CQUNuQixrQkFBa0I7b0JBQ2xCLGNBQWM7b0JBQ2QsZ0JBQWdCO29CQUNoQixnQkFBZ0I7b0JBQ2hCLHNCQUFzQjtvQkFDdEIsYUFBYTtvQkFDYixnQkFBZ0I7b0JBQ2hCLGVBQWU7b0JBQ2YsZ0JBQWdCO29CQUNoQixjQUFjO29CQUNkLG9CQUFvQjtvQkFDcEIsc0JBQXNCO2lCQUN2QjtnQkFDRCxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE9BQU8sQ0FBQzthQUN0QyxDQUFDLENBQ0gsQ0FBQztRQUNKLENBQUM7UUFFRCxrQ0FBa0M7UUFDbEMsTUFBTSw4QkFBOEIsR0FBRyxLQUFLLENBQUMsOEJBQThCLElBQUksaUJBQWlCLENBQUM7UUFDakcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxjQUFjLENBQUMsOEJBQThCLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUU5RixDQUFDOztBQTVHSCx3Q0E4R0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG4vLyBJbXBvcnRzXG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tIFwiQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlXCI7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSBcImF3cy1jZGstbGliL2F3cy1sYW1iZGFcIjtcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWlhbVwiO1xuaW1wb3J0ICogYXMga2VuZHJhIGZyb20gXCJhd3MtY2RrLWxpYi9hd3Mta2VuZHJhXCI7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSBcImF3cy1jZGstbGliL2F3cy1lYzJcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5cbi8qKlxuICogQHN1bW1hcnkgVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBMYW1iZGFUb0tlbmRyYSBjbGFzcy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFUb0tlbmRyYVByb3BzIHtcbiAgLyoqXG4gICAqXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gT3B0aW9uYWwgdXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcyBmb3IgdGhlIEtlbmRyYSBpbmRleC4gSXMgdGhpcyByZXF1aXJlZD9cbiAgICovXG4gIHJlYWRvbmx5IGtlbmRyYUluZGV4UHJvcHM/OiBrZW5kcmEuQ2ZuSW5kZXhQcm9wcyB8IGFueTtcbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBkYXRhIHNvdXJjZXMgdGhhdCB3aWxsIHByb3ZpZGUgZGF0YSB0byB0aGUgS2VuZHJhIGluZGV4LiA/QXQgbGVhc3QgMSBtdXN0IGJlIHNwZWNpZmllZC4gV2Ugd2lsbCBkbyBtYWpvcml0eSBvZlxuICAgKiBwcm9jZXNzaW5nIGZvciBzb21lIGRhdGEgc291cmNlcyAoUzMgY3Jhd2xlciBpbml0aWFsbHkpLCBidXQgZm9yIG90aGVycyB0aGUgcHJvcHMgbXVzdCBiZSBjb21wbGV0ZSAoZS5nLiBwcm9wZXIgcm9sZUFybiwgZXRjLilcbiAgICpcbiAgICogQGRlZmF1bHQgLSBlbXB0eSBsaXN0IChubyBkYXRhIHNvdXJjZXMpXG4gICAqL1xuICByZWFkb25seSBrZW5kcmFEYXRhU291cmNlc1Byb3BzOiAoa2VuZHJhLkNmbkRhdGFTb3VyY2VQcm9wcyB8IGFueSlbXTtcbiAgLyoqXG4gICAqIE9wdGlvbmFsIC0gaW5kZXggcGVybWlzc2lvbnMgdG8gZ3JhbnQgdG8gdGhlIExhbWJkYSBmdW5jdGlvbi4gT25lIG9yIG1vcmUgb2YgdGhlIGZvbGxvd2luZ1xuICAgKiBtYXkgYmUgc3BlY2lmaWVkOsKgYFJlYWRgLCBgU3VibWl0RmVlZGJhY2tgIGFuZCBgV3JpdGVgLiBEZWZhdWx0IGlzIGBbXCJSZWFkXCIsIFwiU3VibWl0RmVlZGJhY2tcIl1gLiBSZWFkIGlzXG4gICAqIGFsbCB0aGUgb3BlcmF0aW9ucyBJQU0gZGVmaW5lcyBhcyBSZWFkIGFuZCBMaXN0LiBTdWJtaXRGZWVkYmFjayBpcyBvbmx5IHRoZSBTdWJtaXRGZWVkYmFjayBhY3Rpb24uIFdyaXRlIGlzIGFsbCB0aGVcbiAgICogb3BlcmF0aW9ucyBJQU0gZGVmaW5lcyBhcyBXcml0ZSBhbmQgVGFnLiBUaGlzIGZ1bmN0aW9uYWxpdHkgbWF5IGJlIG92ZXJyaWRkZW4gYnkgcHJvdmlkaW5nIGEgc3BlY2lmaWMgcm9sZSBhcm4gaW4gbGFtYmRhRnVuY3Rpb25Qcm9wc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIFtcIlJlYWRcIiwgXCJTdWJtaXRGZWVkYmFja1wiXVxuICAgKi9cbiAgcmVhZG9ubHkgaW5kZXhQZXJtaXNzaW9ucz86IHN0cmluZ1tdO1xuICAvKipcbiAgICogRXhpc3RpbmcgaW5zdGFuY2Ugb2YgYSBLZW5kcmEgSW5kZXguIFByb3ZpZGluZyBib3RoIHRoaXMgYW5kIGtlbmRyYUluZGV4UHJvcHMgd2lsbCBjYXVzZSBhbiBlcnJvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ0tlbmRyYUluZGV4T2JqPzoga2VuZHJhLkNmbkluZGV4O1xuICAvKipcbiAgICogT3B0aW9uYWwgLSBpbnN0YW5jZSBvZiBhbiBleGlzdGluZyBMYW1iZGEgRnVuY3Rpb24gb2JqZWN0LCBwcm92aWRpbmcgYm90aCB0aGlzIGFuZCBgbGFtYmRhRnVuY3Rpb25Qcm9wc2Agd2lsbCBjYXVzZSBhbiBlcnJvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ0xhbWJkYU9iaj86IGxhbWJkYS5GdW5jdGlvbjtcbiAgLyoqXG4gICAqIE9wdGlvbmFsIC0gdXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcyBmb3IgdGhlIExhbWJkYSBmdW5jdGlvbi4gUHJvdmlkaW5nIGJvdGggdGhpcyBhbmQgYGV4aXN0aW5nTGFtYmRhT2JqYCBpcyBhbiBlcnJvci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BlcnRpZXMgYXJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBsYW1iZGFGdW5jdGlvblByb3BzPzogbGFtYmRhLkZ1bmN0aW9uUHJvcHM7XG4gIC8qKlxuICAgKiBBbiBleGlzdGluZyBWUEMgZm9yIHRoZSBjb25zdHJ1Y3QgdG8gdXNlIChjb25zdHJ1Y3Qgd2lsbCBOT1QgY3JlYXRlIGEgbmV3IFZQQyBpbiB0aGlzIGNhc2UpXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ1ZwYz86IGVjMi5JVnBjO1xuICAvKipcbiAgICogUHJvcGVydGllcyB0byBvdmVycmlkZSBkZWZhdWx0IHByb3BlcnRpZXMgaWYgZGVwbG95VnBjIGlzIHRydWVcbiAgICovXG4gIHJlYWRvbmx5IHZwY1Byb3BzPzogZWMyLlZwY1Byb3BzO1xuICAvKipcbiAgICogV2hldGhlciB0byBkZXBsb3kgYSBuZXcgVlBDXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGRlcGxveVZwYz86IGJvb2xlYW47XG4gIC8qKlxuICAgKiBPcHRpb25hbCBOYW1lIGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uIGVudmlyb25tZW50IHZhcmlhYmxlIHNldCB0byB0aGUgaW5kZXggaWQgZm9yIHRoZSBLZW5kcmEgaW5kZXguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gS0VORFJBX0lOREVYX0lEXG4gICAqL1xuICByZWFkb25seSBpbmRleElkRW52aXJvbm1lbnRWYXJpYWJsZU5hbWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQHN1bW1hcnkgVGhlIExhbWJkYVRvS2VuZHJhIGNsYXNzLlxuICovXG5leHBvcnQgY2xhc3MgTGFtYmRhVG9LZW5kcmEgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBwdWJsaWMgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb246IGxhbWJkYS5GdW5jdGlvbjtcbiAgcHVibGljIHJlYWRvbmx5IHZwYz86IGVjMi5JVnBjO1xuICBwdWJsaWMgcmVhZG9ubHkga2VuZHJhSW5kZXg6IGtlbmRyYS5DZm5JbmRleDtcbiAgcHVibGljIHJlYWRvbmx5IGtlbmRyYURhdGFTb3VyY2VzOiBrZW5kcmEuQ2ZuRGF0YVNvdXJjZVtdO1xuXG4gIC8qKlxuICAgKiBAc3VtbWFyeSBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBMYW1iZGFUb0tlbmRyYSBjbGFzcy5cbiAgICogQHBhcmFtIHtjZGsuQXBwfSBzY29wZSAtIHJlcHJlc2VudHMgdGhlIHNjb3BlIGZvciBhbGwgdGhlIHJlc291cmNlcy5cbiAgICogQHBhcmFtIHtzdHJpbmd9IGlkIC0gdGhpcyBpcyBhIGEgc2NvcGUtdW5pcXVlIGlkLlxuICAgKiBAcGFyYW0ge0xhbWJkYVRvS2VuZHJhUHJvcHN9IHByb3BzIC0gdXNlciBwcm92aWRlZCBwcm9wcyBmb3IgdGhlIGNvbnN0cnVjdC5cbiAgICogQHNpbmNlIDEuMTIwLjBcbiAgICogQGFjY2VzcyBwdWJsaWNcbiAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBMYW1iZGFUb0tlbmRyYVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICBkZWZhdWx0cy5DaGVja1ZwY1Byb3BzKHByb3BzKTtcbiAgICBkZWZhdWx0cy5DaGVja0xhbWJkYVByb3BzKHByb3BzKTtcblxuICAgIGlmIChwcm9wcy5rZW5kcmFJbmRleFByb3BzICYmIHByb3BzLmV4aXN0aW5nS2VuZHJhSW5kZXhPYmopIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignWW91IG1heSBub3QgcHJvdmlkZSBib3RoIGtlbmRyYUluZGV4UHJvcHMgYW5kIGV4aXN0aW5nS2VuZHJhSW5kZXhPYmonKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuZGVwbG95VnBjIHx8IHByb3BzLmV4aXN0aW5nVnBjKSB7XG4gICAgICB0aGlzLnZwYyA9IGRlZmF1bHRzLmJ1aWxkVnBjKHNjb3BlLCB7XG4gICAgICAgIGRlZmF1bHRWcGNQcm9wczogZGVmYXVsdHMuRGVmYXVsdElzb2xhdGVkVnBjUHJvcHMoKSxcbiAgICAgICAgZXhpc3RpbmdWcGM6IHByb3BzLmV4aXN0aW5nVnBjLFxuICAgICAgICB1c2VyVnBjUHJvcHM6IHByb3BzLnZwY1Byb3BzLFxuICAgICAgICBjb25zdHJ1Y3RWcGNQcm9wczoge1xuICAgICAgICAgIGVuYWJsZURuc0hvc3RuYW1lczogdHJ1ZSxcbiAgICAgICAgICBlbmFibGVEbnNTdXBwb3J0OiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgfSk7XG5cbiAgICAgIGRlZmF1bHRzLkFkZEF3c1NlcnZpY2VFbmRwb2ludChzY29wZSwgdGhpcy52cGMsIGRlZmF1bHRzLlNlcnZpY2VFbmRwb2ludFR5cGVzLktFTkRSQSk7XG4gICAgfVxuXG4gICAgLy8gU2V0dXAgdGhlIExhbWJkYSBmdW5jdGlvblxuICAgIHRoaXMubGFtYmRhRnVuY3Rpb24gPSBkZWZhdWx0cy5idWlsZExhbWJkYUZ1bmN0aW9uKHRoaXMsIHtcbiAgICAgIGV4aXN0aW5nTGFtYmRhT2JqOiBwcm9wcy5leGlzdGluZ0xhbWJkYU9iaixcbiAgICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHByb3BzLmxhbWJkYUZ1bmN0aW9uUHJvcHMsXG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgIH0pO1xuXG4gICAgdGhpcy5rZW5kcmFJbmRleCA9IGRlZmF1bHRzLmJ1aWxkS2VuZHJhSW5kZXgodGhpcywgaWQsIHtcbiAgICAgIGtlbmRyYUluZGV4UHJvcHM6IHByb3BzLmtlbmRyYUluZGV4UHJvcHMsXG4gICAgICBleGlzdGluZ0luZGV4T2JqOiBwcm9wcy5leGlzdGluZ0tlbmRyYUluZGV4T2JqXG4gICAgfSk7XG5cbiAgICB0aGlzLmtlbmRyYURhdGFTb3VyY2VzID0gZGVmYXVsdHMuQWRkTXVsdGlwbGVLZW5kcmFEYXRhU291cmNlcyh0aGlzLCBpZCwgdGhpcy5rZW5kcmFJbmRleCwgcHJvcHMua2VuZHJhRGF0YVNvdXJjZXNQcm9wcyk7XG5cbiAgICAvLyBVcGRhdGUgTGFtYmRhIGZ1bmN0aW9uIElBTSBwb2xpY3kgd2l0aCBjb3JyZWN0IHByaXZpbGVnZXMgdG8gS2VuZHJhIGluZGV4XG4gICAgY29uc3Qgbm9ybWFsaXplZFBlcm1pc3Npb25zID0gcHJvcHMuaW5kZXhQZXJtaXNzaW9ucyA/IGRlZmF1bHRzLm5vcm1hbGl6ZUtlbmRyYVBlcm1pc3Npb25zKHByb3BzLmluZGV4UGVybWlzc2lvbnMpIDogdW5kZWZpbmVkO1xuICAgIGlmICghbm9ybWFsaXplZFBlcm1pc3Npb25zIHx8IG5vcm1hbGl6ZWRQZXJtaXNzaW9ucy5pbmNsdWRlcyhcIlJFQURcIikpIHtcbiAgICAgIC8vIEFkZCBwb2xpY3kgd2l0aCBxdWVyeSBwZXJtaXNzaW9uc1xuICAgICAgdGhpcy5sYW1iZGFGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3koXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICBcImtlbmRyYTpRdWVyeVwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6RGVzY3JpYmUqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpHZXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpCYXRjaEdldCpcIixcbiAgICAgICAgICAgIFwia2VuZHJhOkxpc3QqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpSZXRyaWV2ZVwiXSxcbiAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmtlbmRyYUluZGV4LmF0dHJBcm5dXG4gICAgICAgIH0pXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmICghbm9ybWFsaXplZFBlcm1pc3Npb25zIHx8IG5vcm1hbGl6ZWRQZXJtaXNzaW9ucy5pbmNsdWRlcyhcIlNVQk1JVEZFRURCQUNLXCIpKSB7XG4gICAgICAvLyBBZGQgcG9saWN5IHdpdGggcXVlcnkgcGVybWlzc2lvbnNcbiAgICAgIHRoaXMubGFtYmRhRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgXCJrZW5kcmE6U3VibWl0RmVlZGJhY2tcIl0sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5rZW5kcmFJbmRleC5hdHRyQXJuXVxuICAgICAgICB9KVxuICAgICAgKTtcbiAgICB9XG5cbiAgICBpZiAobm9ybWFsaXplZFBlcm1pc3Npb25zPy5pbmNsdWRlcyhcIldSSVRFXCIpKSB7XG4gICAgICAvLyBBZGQgcG9saWN5IHdpdGggcXVlcnkgcGVybWlzc2lvbnNcbiAgICAgIHRoaXMubGFtYmRhRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICAgXCJrZW5kcmE6QXNzb2NpYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6QmF0Y2hQdXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpDbGVhclwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6Q3JlYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6RGVsZXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6RGlzYXNzb2NpYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6UHV0KlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6VXBkYXRlKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6U3RhcnQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpTdWJtaXQqXCIsXG4gICAgICAgICAgICBcImtlbmRyYTpTdG9wKlwiLFxuICAgICAgICAgICAgXCJrZW5kcmE6VGFnUmVzb3VyY2VcIixcbiAgICAgICAgICAgIFwia2VuZHJhOlVudGFnUmVzb3VyY2VcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5rZW5kcmFJbmRleC5hdHRyQXJuXVxuICAgICAgICB9KVxuICAgICAgKTtcbiAgICB9XG5cbiAgICAvLyBDb25maWd1cmUgZW52aXJvbm1lbnQgdmFyaWFibGVzXG4gICAgY29uc3QgaW5kZXhJZEVudmlyb25tZW50VmFyaWFibGVOYW1lID0gcHJvcHMuaW5kZXhJZEVudmlyb25tZW50VmFyaWFibGVOYW1lIHx8ICdLRU5EUkFfSU5ERVhfSUQnO1xuICAgIHRoaXMubGFtYmRhRnVuY3Rpb24uYWRkRW52aXJvbm1lbnQoaW5kZXhJZEVudmlyb25tZW50VmFyaWFibGVOYW1lLCB0aGlzLmtlbmRyYUluZGV4LmF0dHJJZCk7XG5cbiAgfVxuXG59Il19