"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExecuteScriptStep = exports.PythonVersion = exports.ScriptLanguage = exports.FileScriptCode = exports.InlineScriptCode = exports.ScriptCode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const python_script_handler_1 = require("../../script/python-script-handler");
const automation_step_1 = require("../automation-step");
// eslint-disable-next-line
const path = require('path');
// eslint-disable-next-line
const os = require('os');
/**
 * The code to run for the execution.
 * See "script" parameter here:
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html
 * Attachments are not yet supported.
 */
class ScriptCode {
    /**
     * Inline code to be executed. String will be used to produce function in yaml/json.
     * Simulation will execute the function in this string using the language specified.
     */
    static inline(code) {
        return new InlineScriptCode(code);
    }
    /**
     * Full path to the code to execute. File is parsed to produce yaml/json.
     * Simulation will execute this file using the language specified.
     * (Attachments not yet supported)
     */
    static fromFile(fullPath) {
        return new FileScriptCode(fullPath);
    }
}
exports.ScriptCode = ScriptCode;
_a = JSII_RTTI_SYMBOL_1;
ScriptCode[_a] = { fqn: "@cdklabs/cdk-ssm-documents.ScriptCode", version: "0.0.31" };
class InlineScriptCode extends ScriptCode {
    constructor(inlineCode) {
        super();
        this.inlineCode = inlineCode;
    }
    static toFile(data, suffix) {
        const tempDir = path.join(os.tmpdir(), 'tmp' + new Date().getSeconds() + Math.floor(Math.random() * 100000));
        fs.mkdirSync(tempDir);
        const tempFile = path.join(tempDir, 'execution' + suffix);
        fs.writeFileSync(tempFile, data);
        console.log(`Successfully wrote file to ${tempFile}`);
        return tempFile;
    }
    codeAsString() {
        return this.inlineCode;
    }
    createOrGetFile(suffix) {
        return InlineScriptCode.toFile(this.inlineCode, suffix);
    }
}
exports.InlineScriptCode = InlineScriptCode;
_b = JSII_RTTI_SYMBOL_1;
InlineScriptCode[_b] = { fqn: "@cdklabs/cdk-ssm-documents.InlineScriptCode", version: "0.0.31" };
class FileScriptCode extends ScriptCode {
    constructor(fullPath) {
        super();
        this.fullPath = fullPath;
    }
    codeAsString() {
        return fs.readFileSync(this.fullPath, 'utf8');
    }
    createOrGetFile(_suffix) {
        return this.fullPath;
    }
}
exports.FileScriptCode = FileScriptCode;
_c = JSII_RTTI_SYMBOL_1;
FileScriptCode[_c] = { fqn: "@cdklabs/cdk-ssm-documents.FileScriptCode", version: "0.0.31" };
/**
 * Specifies the script language as described in the "Runtime" argument here:
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html
 */
class ScriptLanguage {
    /**
     * Create a new ScriptLanguage for python execution.
     * @param version is the pythonVersion to use when writing the document (for simulation will not matter).
     * @param handlerName is the function name in code as entry point for script handler.
     */
    static python(version, handlerName) {
        return new PythonScript(version, handlerName);
    }
    /**
     * Creates a ScriptLanguage based on the provided runtime.
     * Prefer one of the other static constructors if possible.
     * @param runtime is the runtime name (such as "python3.6").
     * @param handlerName to be provided for python executions.
     */
    static fromRuntime(runtime, handlerName) {
        if (runtime.startsWith('python')) {
            return ScriptLanguage.python(ScriptLanguage.pythonVersion(runtime), handlerName);
        }
        throw new Error('Unrecognized runtime version ' + runtime);
    }
    static pythonVersion(runtime) {
        switch (runtime) {
            case 'python3.6':
                return PythonVersion.VERSION_3_6;
            case 'python3.7':
                return PythonVersion.VERSION_3_7;
            case 'python3.8':
                return PythonVersion.VERSION_3_8;
            default:
                throw new Error('Unrecognized python version ' + runtime);
        }
    }
}
exports.ScriptLanguage = ScriptLanguage;
_d = JSII_RTTI_SYMBOL_1;
ScriptLanguage[_d] = { fqn: "@cdklabs/cdk-ssm-documents.ScriptLanguage", version: "0.0.31" };
class PythonScript extends ScriptLanguage {
    constructor(version, handlerName) {
        super();
        this.version = version;
        this.handlerName = handlerName;
    }
    runtime() {
        switch (this.version) {
            case PythonVersion.VERSION_3_6:
                return 'python3.6';
            case PythonVersion.VERSION_3_7:
                return 'python3.7';
            case PythonVersion.VERSION_3_8:
                return 'python3.8';
        }
    }
    fileSuffix() {
        return '.py';
    }
    ssmInputs() {
        return {
            Handler: this.handlerName,
            Runtime: this.runtime(),
        };
    }
    simulate(code, inputs) {
        const pyHandler = new python_script_handler_1.PythonScriptHandler();
        return pyHandler.run(code.createOrGetFile(this.fileSuffix()), this.handlerName, inputs);
    }
}
/**
 * Python runtime to use when writing SSM Document.
 * Simulation will use local python version.
 */
var PythonVersion;
(function (PythonVersion) {
    PythonVersion[PythonVersion["VERSION_3_6"] = 0] = "VERSION_3_6";
    PythonVersion[PythonVersion["VERSION_3_7"] = 1] = "VERSION_3_7";
    PythonVersion[PythonVersion["VERSION_3_8"] = 2] = "VERSION_3_8";
})(PythonVersion = exports.PythonVersion || (exports.PythonVersion = {}));
/**
 * AutomationStep implementation for aws:executeScript
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html
 */
class ExecuteScriptStep extends automation_step_1.AutomationStep {
    constructor(stage, id, props) {
        super(stage, id, props);
        this.action = 'aws:executeScript';
        this.language = props.language;
        this.code = props.code;
        this.outputs = props.outputs ?? [];
        this.inputs = props.inputPayload;
        const nonPayloadOutputs = this.outputs.filter(out => !out.selector.startsWith('$.Payload'));
        if (nonPayloadOutputs.length != 0) {
            throw new Error('All script outputs must specify selector starting with "$.Payload", ' +
                `but these did not: ${JSON.stringify(nonPayloadOutputs)}`);
        }
    }
    listUserOutputs() {
        return this.outputs;
    }
    listInputs() {
        return Object.values(this.inputs).flatMap(i => i?.requiredInputs() ?? []);
    }
    toSsmEntry() {
        return this.prepareSsmEntry({
            ...this.language.ssmInputs(),
            Script: this.code.codeAsString(),
            InputPayload: this.inputs,
        });
    }
}
exports.ExecuteScriptStep = ExecuteScriptStep;
_e = JSII_RTTI_SYMBOL_1;
ExecuteScriptStep[_e] = { fqn: "@cdklabs/cdk-ssm-documents.ExecuteScriptStep", version: "0.0.31" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXhlY3V0ZS1zY3JpcHQtc3RlcC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9wYXJlbnQtc3RlcHMvYXV0b21hdGlvbi9leGVjdXRlLXNjcmlwdC1zdGVwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEseUJBQXlCO0FBSXpCLDhFQUF5RTtBQUN6RSx3REFBeUU7QUFDekUsMkJBQTJCO0FBQzNCLE1BQU0sSUFBSSxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUM3QiwyQkFBMkI7QUFDM0IsTUFBTSxFQUFFLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0FBRXpCOzs7OztHQUtHO0FBQ0gsTUFBc0IsVUFBVTtJQUM5Qjs7O09BR0c7SUFDSCxNQUFNLENBQUMsTUFBTSxDQUFDLElBQVk7UUFDeEIsT0FBTyxJQUFJLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFDRDs7OztPQUlHO0lBQ0gsTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFnQjtRQUM5QixPQUFPLElBQUksY0FBYyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3RDLENBQUM7O0FBZkgsZ0NBNEJDOzs7QUFFRCxNQUFhLGdCQUFpQixTQUFRLFVBQVU7SUFXOUMsWUFBWSxVQUFrQjtRQUM1QixLQUFLLEVBQUUsQ0FBQztRQUNSLElBQUksQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDO0lBQy9CLENBQUM7SUFiTyxNQUFNLENBQUMsTUFBTSxDQUFDLElBQVksRUFBRSxNQUFjO1FBQ2hELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxFQUFFLEtBQUssR0FBRyxJQUFJLElBQUksRUFBRSxDQUFDLFVBQVUsRUFBRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxHQUFHLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDN0csRUFBRSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN0QixNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxXQUFXLEdBQUcsTUFBTSxDQUFDLENBQUM7UUFDMUQsRUFBRSxDQUFDLGFBQWEsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDakMsT0FBTyxDQUFDLEdBQUcsQ0FBQyw4QkFBOEIsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUN0RCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBUUQsWUFBWTtRQUNWLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQztJQUN6QixDQUFDO0lBRUQsZUFBZSxDQUFDLE1BQWM7UUFDNUIsT0FBTyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUMxRCxDQUFDOztBQXRCSCw0Q0F1QkM7OztBQUVELE1BQWEsY0FBZSxTQUFRLFVBQVU7SUFFNUMsWUFBWSxRQUFnQjtRQUMxQixLQUFLLEVBQUUsQ0FBQztRQUNSLElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO0lBQzNCLENBQUM7SUFDRCxZQUFZO1FBQ1YsT0FBTyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVELGVBQWUsQ0FBQyxPQUFlO1FBQzdCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUN2QixDQUFDOztBQVpILHdDQWFDOzs7QUFFRDs7O0dBR0c7QUFDSCxNQUFzQixjQUFjO0lBRWxDOzs7O09BSUc7SUFDSCxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQXNCLEVBQUUsV0FBbUI7UUFDdkQsT0FBTyxJQUFJLFlBQVksQ0FBQyxPQUFPLEVBQUUsV0FBVyxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0gsTUFBTSxDQUFDLFdBQVcsQ0FBQyxPQUFlLEVBQUUsV0FBb0I7UUFDdEQsSUFBSSxPQUFPLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ2hDLE9BQU8sY0FBYyxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxFQUFFLFdBQVksQ0FBQyxDQUFDO1NBQ25GO1FBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQywrQkFBK0IsR0FBRyxPQUFPLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRU8sTUFBTSxDQUFDLGFBQWEsQ0FBQyxPQUFlO1FBQzFDLFFBQVEsT0FBTyxFQUFFO1lBQ2YsS0FBSyxXQUFXO2dCQUNkLE9BQU8sYUFBYSxDQUFDLFdBQVcsQ0FBQztZQUNuQyxLQUFLLFdBQVc7Z0JBQ2QsT0FBTyxhQUFhLENBQUMsV0FBVyxDQUFDO1lBQ25DLEtBQUssV0FBVztnQkFDZCxPQUFPLGFBQWEsQ0FBQyxXQUFXLENBQUM7WUFDbkM7Z0JBQ0UsTUFBTSxJQUFJLEtBQUssQ0FBQyw4QkFBOEIsR0FBRyxPQUFPLENBQUMsQ0FBQztTQUM3RDtJQUNILENBQUM7O0FBbkNILHdDQXlEQzs7O0FBRUQsTUFBTSxZQUFhLFNBQVEsY0FBYztJQUd2QyxZQUFZLE9BQXNCLEVBQUUsV0FBbUI7UUFDckQsS0FBSyxFQUFFLENBQUM7UUFDUixJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUN2QixJQUFJLENBQUMsV0FBVyxHQUFHLFdBQVcsQ0FBQztJQUNqQyxDQUFDO0lBRUQsT0FBTztRQUNMLFFBQVEsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNwQixLQUFLLGFBQWEsQ0FBQyxXQUFXO2dCQUM1QixPQUFPLFdBQVcsQ0FBQztZQUNyQixLQUFLLGFBQWEsQ0FBQyxXQUFXO2dCQUM1QixPQUFPLFdBQVcsQ0FBQztZQUNyQixLQUFLLGFBQWEsQ0FBQyxXQUFXO2dCQUM1QixPQUFPLFdBQVcsQ0FBQztTQUN0QjtJQUNILENBQUM7SUFFRCxVQUFVO1FBQ1IsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRUQsU0FBUztRQUNQLE9BQU87WUFDTCxPQUFPLEVBQUUsSUFBSSxDQUFDLFdBQVc7WUFDekIsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPLEVBQUU7U0FDeEIsQ0FBQztJQUNKLENBQUM7SUFFRCxRQUFRLENBQUMsSUFBZ0IsRUFBRSxNQUErQjtRQUN4RCxNQUFNLFNBQVMsR0FBRyxJQUFJLDJDQUFtQixFQUFFLENBQUM7UUFDNUMsT0FBTyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUMxRixDQUFDO0NBQ0Y7QUFFRDs7O0dBR0c7QUFDSCxJQUFZLGFBSVg7QUFKRCxXQUFZLGFBQWE7SUFDdkIsK0RBQVcsQ0FBQTtJQUNYLCtEQUFXLENBQUE7SUFDWCwrREFBVyxDQUFBO0FBQ2IsQ0FBQyxFQUpXLGFBQWEsR0FBYixxQkFBYSxLQUFiLHFCQUFhLFFBSXhCO0FBa0NEOzs7R0FHRztBQUNILE1BQWEsaUJBQWtCLFNBQVEsZ0NBQWM7SUFRbkQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUNyRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUhqQixXQUFNLEdBQVcsbUJBQW1CLENBQUM7UUFJNUMsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1FBQy9CLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztRQUN2QixJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDO1FBQ25DLElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUNqQyxNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1FBQzVGLElBQUksaUJBQWlCLENBQUMsTUFBTSxJQUFJLENBQUMsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLHNFQUFzRTtnQkFDcEYsc0JBQXNCLElBQUksQ0FBQyxTQUFTLENBQUMsaUJBQWlCLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDOUQ7SUFDSCxDQUFDO0lBRU0sZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUM7SUFDdEIsQ0FBQztJQUVNLFVBQVU7UUFDZixPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUM1RSxDQUFDO0lBRU0sVUFBVTtRQUNmLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQztZQUMxQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFO1lBQzVCLE1BQU0sRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRTtZQUNoQyxZQUFZLEVBQUUsSUFBSSxDQUFDLE1BQU07U0FDMUIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUFuQ0gsOENBcUNDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBPdXRwdXQgfSBmcm9tICcuLi8uLi9kb21haW4vb3V0cHV0JztcbmltcG9ydCB7IElHZW5lcmljVmFyaWFibGUgfSBmcm9tICcuLi8uLi9pbnRlcmZhY2UvdmFyaWFibGVzL3ZhcmlhYmxlJztcbmltcG9ydCB7IFB5dGhvblNjcmlwdEhhbmRsZXIgfSBmcm9tICcuLi8uLi9zY3JpcHQvcHl0aG9uLXNjcmlwdC1oYW5kbGVyJztcbmltcG9ydCB7IEF1dG9tYXRpb25TdGVwLCBBdXRvbWF0aW9uU3RlcFByb3BzIH0gZnJvbSAnLi4vYXV0b21hdGlvbi1zdGVwJztcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZVxuY29uc3QgcGF0aCA9IHJlcXVpcmUoJ3BhdGgnKTtcbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZVxuY29uc3Qgb3MgPSByZXF1aXJlKCdvcycpO1xuXG4vKipcbiAqIFRoZSBjb2RlIHRvIHJ1biBmb3IgdGhlIGV4ZWN1dGlvbi5cbiAqIFNlZSBcInNjcmlwdFwiIHBhcmFtZXRlciBoZXJlOlxuICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N5c3RlbXMtbWFuYWdlci9sYXRlc3QvdXNlcmd1aWRlL2F1dG9tYXRpb24tYWN0aW9uLWV4ZWN1dGVTY3JpcHQuaHRtbFxuICogQXR0YWNobWVudHMgYXJlIG5vdCB5ZXQgc3VwcG9ydGVkLlxuICovXG5leHBvcnQgYWJzdHJhY3QgY2xhc3MgU2NyaXB0Q29kZSB7XG4gIC8qKlxuICAgKiBJbmxpbmUgY29kZSB0byBiZSBleGVjdXRlZC4gU3RyaW5nIHdpbGwgYmUgdXNlZCB0byBwcm9kdWNlIGZ1bmN0aW9uIGluIHlhbWwvanNvbi5cbiAgICogU2ltdWxhdGlvbiB3aWxsIGV4ZWN1dGUgdGhlIGZ1bmN0aW9uIGluIHRoaXMgc3RyaW5nIHVzaW5nIHRoZSBsYW5ndWFnZSBzcGVjaWZpZWQuXG4gICAqL1xuICBzdGF0aWMgaW5saW5lKGNvZGU6IHN0cmluZykge1xuICAgIHJldHVybiBuZXcgSW5saW5lU2NyaXB0Q29kZShjb2RlKTtcbiAgfVxuICAvKipcbiAgICogRnVsbCBwYXRoIHRvIHRoZSBjb2RlIHRvIGV4ZWN1dGUuIEZpbGUgaXMgcGFyc2VkIHRvIHByb2R1Y2UgeWFtbC9qc29uLlxuICAgKiBTaW11bGF0aW9uIHdpbGwgZXhlY3V0ZSB0aGlzIGZpbGUgdXNpbmcgdGhlIGxhbmd1YWdlIHNwZWNpZmllZC5cbiAgICogKEF0dGFjaG1lbnRzIG5vdCB5ZXQgc3VwcG9ydGVkKVxuICAgKi9cbiAgc3RhdGljIGZyb21GaWxlKGZ1bGxQYXRoOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gbmV3IEZpbGVTY3JpcHRDb2RlKGZ1bGxQYXRoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcmV0dXJucyBjb2RlIGFzIGEgc3RyaW5nXG4gICAqL1xuICBhYnN0cmFjdCBjb2RlQXNTdHJpbmcoKTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJZiB0aGVyZSBpcyBhIGZpbGUgZm9yIHRoaXMgY29kZSwgcmV0dXJuIGl0LlxuICAgKiBPdGhlcndpc2UsIGNyZWF0ZSBhIGZpbGUgd2l0aCB0aGUgc3BlY2lmaWVkIHN1ZmZpeC5cbiAgICogQHBhcmFtIHN1ZmZpeCBvZiB0aGUgZmlsZSB0byBjcmVhdGUgKHN1Y2ggYXMgXCIucHlcIilcbiAgICovXG4gIGFic3RyYWN0IGNyZWF0ZU9yR2V0RmlsZShzdWZmaXg6IHN0cmluZyk6IHN0cmluZztcbn1cblxuZXhwb3J0IGNsYXNzIElubGluZVNjcmlwdENvZGUgZXh0ZW5kcyBTY3JpcHRDb2RlIHtcbiAgcHJpdmF0ZSBzdGF0aWMgdG9GaWxlKGRhdGE6IHN0cmluZywgc3VmZml4OiBzdHJpbmcpIHtcbiAgICBjb25zdCB0ZW1wRGlyID0gcGF0aC5qb2luKG9zLnRtcGRpcigpLCAndG1wJyArIG5ldyBEYXRlKCkuZ2V0U2Vjb25kcygpICsgTWF0aC5mbG9vcihNYXRoLnJhbmRvbSgpICogMTAwMDAwKSk7XG4gICAgZnMubWtkaXJTeW5jKHRlbXBEaXIpO1xuICAgIGNvbnN0IHRlbXBGaWxlID0gcGF0aC5qb2luKHRlbXBEaXIsICdleGVjdXRpb24nICsgc3VmZml4KTtcbiAgICBmcy53cml0ZUZpbGVTeW5jKHRlbXBGaWxlLCBkYXRhKTtcbiAgICBjb25zb2xlLmxvZyhgU3VjY2Vzc2Z1bGx5IHdyb3RlIGZpbGUgdG8gJHt0ZW1wRmlsZX1gKTtcbiAgICByZXR1cm4gdGVtcEZpbGU7XG4gIH1cblxuICByZWFkb25seSBpbmxpbmVDb2RlOiBzdHJpbmc7XG4gIGNvbnN0cnVjdG9yKGlubGluZUNvZGU6IHN0cmluZykge1xuICAgIHN1cGVyKCk7XG4gICAgdGhpcy5pbmxpbmVDb2RlID0gaW5saW5lQ29kZTtcbiAgfVxuXG4gIGNvZGVBc1N0cmluZygpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLmlubGluZUNvZGU7XG4gIH1cblxuICBjcmVhdGVPckdldEZpbGUoc3VmZml4OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIHJldHVybiBJbmxpbmVTY3JpcHRDb2RlLnRvRmlsZSh0aGlzLmlubGluZUNvZGUsIHN1ZmZpeCk7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIEZpbGVTY3JpcHRDb2RlIGV4dGVuZHMgU2NyaXB0Q29kZSB7XG4gIHJlYWRvbmx5IGZ1bGxQYXRoOiBzdHJpbmc7XG4gIGNvbnN0cnVjdG9yKGZ1bGxQYXRoOiBzdHJpbmcpIHtcbiAgICBzdXBlcigpO1xuICAgIHRoaXMuZnVsbFBhdGggPSBmdWxsUGF0aDtcbiAgfVxuICBjb2RlQXNTdHJpbmcoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gZnMucmVhZEZpbGVTeW5jKHRoaXMuZnVsbFBhdGgsICd1dGY4Jyk7XG4gIH1cblxuICBjcmVhdGVPckdldEZpbGUoX3N1ZmZpeDogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5mdWxsUGF0aDtcbiAgfVxufVxuXG4vKipcbiAqIFNwZWNpZmllcyB0aGUgc2NyaXB0IGxhbmd1YWdlIGFzIGRlc2NyaWJlZCBpbiB0aGUgXCJSdW50aW1lXCIgYXJndW1lbnQgaGVyZTpcbiAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zeXN0ZW1zLW1hbmFnZXIvbGF0ZXN0L3VzZXJndWlkZS9hdXRvbWF0aW9uLWFjdGlvbi1leGVjdXRlU2NyaXB0Lmh0bWxcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFNjcmlwdExhbmd1YWdlIHtcblxuICAvKipcbiAgICogQ3JlYXRlIGEgbmV3IFNjcmlwdExhbmd1YWdlIGZvciBweXRob24gZXhlY3V0aW9uLlxuICAgKiBAcGFyYW0gdmVyc2lvbiBpcyB0aGUgcHl0aG9uVmVyc2lvbiB0byB1c2Ugd2hlbiB3cml0aW5nIHRoZSBkb2N1bWVudCAoZm9yIHNpbXVsYXRpb24gd2lsbCBub3QgbWF0dGVyKS5cbiAgICogQHBhcmFtIGhhbmRsZXJOYW1lIGlzIHRoZSBmdW5jdGlvbiBuYW1lIGluIGNvZGUgYXMgZW50cnkgcG9pbnQgZm9yIHNjcmlwdCBoYW5kbGVyLlxuICAgKi9cbiAgc3RhdGljIHB5dGhvbih2ZXJzaW9uOiBQeXRob25WZXJzaW9uLCBoYW5kbGVyTmFtZTogc3RyaW5nKTogU2NyaXB0TGFuZ3VhZ2Uge1xuICAgIHJldHVybiBuZXcgUHl0aG9uU2NyaXB0KHZlcnNpb24sIGhhbmRsZXJOYW1lKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgU2NyaXB0TGFuZ3VhZ2UgYmFzZWQgb24gdGhlIHByb3ZpZGVkIHJ1bnRpbWUuXG4gICAqIFByZWZlciBvbmUgb2YgdGhlIG90aGVyIHN0YXRpYyBjb25zdHJ1Y3RvcnMgaWYgcG9zc2libGUuXG4gICAqIEBwYXJhbSBydW50aW1lIGlzIHRoZSBydW50aW1lIG5hbWUgKHN1Y2ggYXMgXCJweXRob24zLjZcIikuXG4gICAqIEBwYXJhbSBoYW5kbGVyTmFtZSB0byBiZSBwcm92aWRlZCBmb3IgcHl0aG9uIGV4ZWN1dGlvbnMuXG4gICAqL1xuICBzdGF0aWMgZnJvbVJ1bnRpbWUocnVudGltZTogc3RyaW5nLCBoYW5kbGVyTmFtZT86IHN0cmluZyk6IFNjcmlwdExhbmd1YWdlIHtcbiAgICBpZiAocnVudGltZS5zdGFydHNXaXRoKCdweXRob24nKSkge1xuICAgICAgcmV0dXJuIFNjcmlwdExhbmd1YWdlLnB5dGhvbihTY3JpcHRMYW5ndWFnZS5weXRob25WZXJzaW9uKHJ1bnRpbWUpLCBoYW5kbGVyTmFtZSEpO1xuICAgIH1cbiAgICB0aHJvdyBuZXcgRXJyb3IoJ1VucmVjb2duaXplZCBydW50aW1lIHZlcnNpb24gJyArIHJ1bnRpbWUpO1xuICB9XG5cbiAgcHJpdmF0ZSBzdGF0aWMgcHl0aG9uVmVyc2lvbihydW50aW1lOiBzdHJpbmcpIHtcbiAgICBzd2l0Y2ggKHJ1bnRpbWUpIHtcbiAgICAgIGNhc2UgJ3B5dGhvbjMuNic6XG4gICAgICAgIHJldHVybiBQeXRob25WZXJzaW9uLlZFUlNJT05fM182O1xuICAgICAgY2FzZSAncHl0aG9uMy43JzpcbiAgICAgICAgcmV0dXJuIFB5dGhvblZlcnNpb24uVkVSU0lPTl8zXzc7XG4gICAgICBjYXNlICdweXRob24zLjgnOlxuICAgICAgICByZXR1cm4gUHl0aG9uVmVyc2lvbi5WRVJTSU9OXzNfODtcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignVW5yZWNvZ25pemVkIHB5dGhvbiB2ZXJzaW9uICcgKyBydW50aW1lKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQnVpbGRzIHRoZSBzc20gaW5wdXRzLlxuICAgKi9cbiAgYWJzdHJhY3Qgc3NtSW5wdXRzKCk6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBUaGUgYXNzb2NpYXRlZCBydW50aW1lIG9mIHRoaXMgU2NyaXB0TGFuZ3VhZ2UuXG4gICAqL1xuICBhYnN0cmFjdCBydW50aW1lKCk6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHN1ZmZpeCB0byBhcHBseSB0byBmaWxlIG5hbWVzIG9mIHRoaXMgdHlwZSBvZiBleGVjdXRpb24uXG4gICAqL1xuICBhYnN0cmFjdCBmaWxlU3VmZml4KCk6IHN0cmluZztcblxuICAvKipcbiAgICogU2ltdWxhdGUgYW4gZXhlY3V0aW9uIG9mIHRoaXMgU2NyaXB0TGFuZ3VhZ2UuXG4gICAqIFByb3ZpZGUgdGhlIGlucHV0cyBhZnRlciByZXBsYWNlZCB3aXRoIHRoZSBhY3R1YWwgdmFsdWVzIChub3QgdmFyaWFibGVzKS5cbiAgICovXG4gIGFic3RyYWN0IHNpbXVsYXRlKGNvZGU6IFNjcmlwdENvZGUsIGlucHV0czogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH0gKTogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH07XG59XG5cbmNsYXNzIFB5dGhvblNjcmlwdCBleHRlbmRzIFNjcmlwdExhbmd1YWdlIHtcbiAgcmVhZG9ubHkgdmVyc2lvbjogUHl0aG9uVmVyc2lvbjtcbiAgcmVhZG9ubHkgaGFuZGxlck5hbWU6IHN0cmluZztcbiAgY29uc3RydWN0b3IodmVyc2lvbjogUHl0aG9uVmVyc2lvbiwgaGFuZGxlck5hbWU6IHN0cmluZykge1xuICAgIHN1cGVyKCk7XG4gICAgdGhpcy52ZXJzaW9uID0gdmVyc2lvbjtcbiAgICB0aGlzLmhhbmRsZXJOYW1lID0gaGFuZGxlck5hbWU7XG4gIH1cblxuICBydW50aW1lKCk6IHN0cmluZyB7XG4gICAgc3dpdGNoICh0aGlzLnZlcnNpb24pIHtcbiAgICAgIGNhc2UgUHl0aG9uVmVyc2lvbi5WRVJTSU9OXzNfNjpcbiAgICAgICAgcmV0dXJuICdweXRob24zLjYnO1xuICAgICAgY2FzZSBQeXRob25WZXJzaW9uLlZFUlNJT05fM183OlxuICAgICAgICByZXR1cm4gJ3B5dGhvbjMuNyc7XG4gICAgICBjYXNlIFB5dGhvblZlcnNpb24uVkVSU0lPTl8zXzg6XG4gICAgICAgIHJldHVybiAncHl0aG9uMy44JztcbiAgICB9XG4gIH1cblxuICBmaWxlU3VmZml4KCk6IHN0cmluZyB7XG4gICAgcmV0dXJuICcucHknO1xuICB9XG5cbiAgc3NtSW5wdXRzKCk6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9IHtcbiAgICByZXR1cm4ge1xuICAgICAgSGFuZGxlcjogdGhpcy5oYW5kbGVyTmFtZSxcbiAgICAgIFJ1bnRpbWU6IHRoaXMucnVudGltZSgpLFxuICAgIH07XG4gIH1cblxuICBzaW11bGF0ZShjb2RlOiBTY3JpcHRDb2RlLCBpbnB1dHM6IHsgW3A6IHN0cmluZ106IHN0cmluZyB9KTogeyBbcDogc3RyaW5nXTogc3RyaW5nIH0ge1xuICAgIGNvbnN0IHB5SGFuZGxlciA9IG5ldyBQeXRob25TY3JpcHRIYW5kbGVyKCk7XG4gICAgcmV0dXJuIHB5SGFuZGxlci5ydW4oY29kZS5jcmVhdGVPckdldEZpbGUodGhpcy5maWxlU3VmZml4KCkpLCB0aGlzLmhhbmRsZXJOYW1lLCBpbnB1dHMpO1xuICB9XG59XG5cbi8qKlxuICogUHl0aG9uIHJ1bnRpbWUgdG8gdXNlIHdoZW4gd3JpdGluZyBTU00gRG9jdW1lbnQuXG4gKiBTaW11bGF0aW9uIHdpbGwgdXNlIGxvY2FsIHB5dGhvbiB2ZXJzaW9uLlxuICovXG5leHBvcnQgZW51bSBQeXRob25WZXJzaW9uIHtcbiAgVkVSU0lPTl8zXzYsXG4gIFZFUlNJT05fM183LFxuICBWRVJTSU9OXzNfOCxcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBFeGVjdXRlU2NyaXB0U3RlcFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEV4ZWN1dGVTY3JpcHRTdGVwUHJvcHMgZXh0ZW5kcyBBdXRvbWF0aW9uU3RlcFByb3BzIHtcblxuICAvKipcbiAgICogKFJlcXVpcmVkKSBMYW5ndWFnZSB1c2VkIHRvIGV4ZWN1dGUgdGhlIHNjcmlwdC5cbiAgICovXG4gIHJlYWRvbmx5IGxhbmd1YWdlOiBTY3JpcHRMYW5ndWFnZTtcblxuICAvKipcbiAgICogSW5saW5lIGNvZGUgdG8gYmUgZXhlY3V0ZWQuIFN0cmluZyB3aWxsIGJlIHVzZWQgdG8gcHJvZHVjZSBmdW5jdGlvbiBpbiB5YW1sL2pzb24uXG4gICAqIFNpbXVsYXRpb24gd2lsbCBleGVjdXRlIHRoZSBmdW5jdGlvbiBpbiB0aGlzIGNvZGUgdXNpbmcgdGhlIGxhbmd1YWdlIHNwZWNpZmllZC5cbiAgICovXG4gIHJlYWRvbmx5IGNvZGU6IFNjcmlwdENvZGU7XG5cbiAgLyoqXG4gICAqIChPcHRpb25hbCkgT3V0cHV0cyB0aGF0IHRoZSBmdW5jdGlvbiBpcyBleHBlY3RlZCB0byByZXR1cm4uXG4gICAqIEJlIHN1cmUgdG8gcHJlZml4IHRoZSBzZWxlY3RvciBmb3IgdGhlc2Ugb3V0cHV0cyB3aXRoIFwiJC5QYXlsb2FkLlwiIGZvciBleGVjdXRlU2NyaXB0IHN0ZXAgb3V0cHV0cy5cbiAgICogQGRlZmF1bHQgW11cbiAgICovXG4gIHJlYWRvbmx5IG91dHB1dHM/OiBPdXRwdXRbXTtcblxuICAvKipcbiAgICogSW5wdXRQYXlsb2FkIHRoYXQgd2lsbCBiZSBwYXNzZWQgdG8gdGhlIGZpcnN0IHBhcmFtZXRlciBvZiB0aGUgaGFuZGxlci5cbiAgICogVGhpcyBjYW4gYmUgdXNlZCB0byBwYXNzIGlucHV0IGRhdGEgdG8gdGhlIHNjcmlwdC5cbiAgICogVGhlIGtleSBvZiB0aGlzIGRpY3QgaXMgdGhlIHZhcmlhYmxlIG5hbWUgdGhhdCB3aWxsIGJlIGF2YWlsYWJsZSB0byB0aGUgY29kZS5cbiAgICogVGhlIHZhbHVlIGlzIHRoZSBWYXJpYWJsZSBvYmplY3QuXG4gICAqL1xuICByZWFkb25seSBpbnB1dFBheWxvYWQ6IHsgW25hbWU6IHN0cmluZ106IElHZW5lcmljVmFyaWFibGUgfTtcbn1cblxuLyoqXG4gKiBBdXRvbWF0aW9uU3RlcCBpbXBsZW1lbnRhdGlvbiBmb3IgYXdzOmV4ZWN1dGVTY3JpcHRcbiAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zeXN0ZW1zLW1hbmFnZXIvbGF0ZXN0L3VzZXJndWlkZS9hdXRvbWF0aW9uLWFjdGlvbi1leGVjdXRlU2NyaXB0Lmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIEV4ZWN1dGVTY3JpcHRTdGVwIGV4dGVuZHMgQXV0b21hdGlvblN0ZXAge1xuXG4gIHJlYWRvbmx5IGxhbmd1YWdlOiBTY3JpcHRMYW5ndWFnZTtcbiAgcmVhZG9ubHkgY29kZTogU2NyaXB0Q29kZTtcbiAgcmVhZG9ubHkgb3V0cHV0czogT3V0cHV0W107XG4gIHJlYWRvbmx5IGlucHV0czogeyBbbmFtZTogc3RyaW5nXTogSUdlbmVyaWNWYXJpYWJsZSB9O1xuICByZWFkb25seSBhY3Rpb246IHN0cmluZyA9ICdhd3M6ZXhlY3V0ZVNjcmlwdCc7XG5cbiAgY29uc3RydWN0b3Ioc3RhZ2U6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEV4ZWN1dGVTY3JpcHRTdGVwUHJvcHMpIHtcbiAgICBzdXBlcihzdGFnZSwgaWQsIHByb3BzKTtcbiAgICB0aGlzLmxhbmd1YWdlID0gcHJvcHMubGFuZ3VhZ2U7XG4gICAgdGhpcy5jb2RlID0gcHJvcHMuY29kZTtcbiAgICB0aGlzLm91dHB1dHMgPSBwcm9wcy5vdXRwdXRzID8/IFtdO1xuICAgIHRoaXMuaW5wdXRzID0gcHJvcHMuaW5wdXRQYXlsb2FkO1xuICAgIGNvbnN0IG5vblBheWxvYWRPdXRwdXRzID0gdGhpcy5vdXRwdXRzLmZpbHRlcihvdXQgPT4gIW91dC5zZWxlY3Rvci5zdGFydHNXaXRoKCckLlBheWxvYWQnKSk7XG4gICAgaWYgKG5vblBheWxvYWRPdXRwdXRzLmxlbmd0aCAhPSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0FsbCBzY3JpcHQgb3V0cHV0cyBtdXN0IHNwZWNpZnkgc2VsZWN0b3Igc3RhcnRpbmcgd2l0aCBcIiQuUGF5bG9hZFwiLCAnICtcbiAgICAgICAgYGJ1dCB0aGVzZSBkaWQgbm90OiAke0pTT04uc3RyaW5naWZ5KG5vblBheWxvYWRPdXRwdXRzKX1gKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgbGlzdFVzZXJPdXRwdXRzKCk6IE91dHB1dFtdIHtcbiAgICByZXR1cm4gdGhpcy5vdXRwdXRzO1xuICB9XG5cbiAgcHVibGljIGxpc3RJbnB1dHMoKTogc3RyaW5nW10ge1xuICAgIHJldHVybiBPYmplY3QudmFsdWVzKHRoaXMuaW5wdXRzKS5mbGF0TWFwKGkgPT4gaT8ucmVxdWlyZWRJbnB1dHMoKSA/PyBbXSk7XG4gIH1cblxuICBwdWJsaWMgdG9Tc21FbnRyeSgpOiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfSB7XG4gICAgcmV0dXJuIHRoaXMucHJlcGFyZVNzbUVudHJ5KHtcbiAgICAgIC4uLnRoaXMubGFuZ3VhZ2Uuc3NtSW5wdXRzKCksXG4gICAgICBTY3JpcHQ6IHRoaXMuY29kZS5jb2RlQXNTdHJpbmcoKSxcbiAgICAgIElucHV0UGF5bG9hZDogdGhpcy5pbnB1dHMsXG4gICAgfSk7XG4gIH1cblxufSJdfQ==