"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_service_1 = require("../../domain/aws-service");
const choice_1 = require("../../domain/choice");
const data_type_1 = require("../../domain/data-type");
const operation_1 = require("../../domain/operation");
const string_variable_1 = require("../../interface/variables/string-variable");
const approve_step_1 = require("../../parent-steps/automation/approve-step");
const assert_aws_resource_step_1 = require("../../parent-steps/automation/assert-aws-resource-step");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const branch_step_1 = require("../../parent-steps/automation/branch-step");
const execute_script_step_1 = require("../../parent-steps/automation/execute-script-step");
const pause_step_1 = require("../../parent-steps/automation/pause-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_resource_step_1 = require("../../parent-steps/automation/wait-for-resource-step");
const composite_step_1 = require("./composite-step");
// eslint-disable-next-line
const yaml = require('js-yaml');
/**
 * StringStep allows for including a step from an existing automation document verbatim.
 * This is useful in that it allows developers to integrate with existing document steps.
 * This step can be used just as you would use any other Step including simulation and deployment.
 */
class StringStep extends composite_step_1.CompositeAutomationStep {
    constructor(scope, id, props) {
        super(scope, id);
        const sharedProps = {};
        sharedProps.name = props.name;
        sharedProps.action = props.action;
        if (props.maxAttempts) {
            sharedProps.maxAttempts = props.maxAttempts;
        }
        if (props.timeoutSeconds) {
            sharedProps.timeoutSeconds = props.timeoutSeconds;
        }
        if (props.isEnd != undefined) {
            sharedProps.isEnd = props.isEnd;
        }
        if (props.isCritical != undefined) {
            sharedProps.isCritical = props.isCritical;
        }
        if (props.outputs != undefined) {
            sharedProps.outputs = this.toOutputs(props.outputs);
        }
        const { Service, Api, PropertySelector, DesiredValues, ...restParams } = props.inputs;
        switch (props.action) {
            case 'aws:executeAwsApi':
                const camelCased = Service
                    .replace(/-([a-z])/g, (g) => {
                    return g[1].toUpperCase();
                });
                this.automationStep = new aws_api_step_1.AwsApiStep(this, props.name, {
                    service: new aws_service_1.AwsService(Service, camelCased.charAt(0).toUpperCase() + camelCased.slice(1)),
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    outputs: sharedProps.outputs,
                    ...sharedProps,
                });
                break;
            case 'aws:waitForAwsResourceProperty':
                this.automationStep = new wait_for_resource_step_1.WaitForResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:assertAwsResourceProperty':
                this.automationStep = new assert_aws_resource_step_1.AssertAwsResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:pause':
                this.automationStep = new pause_step_1.PauseStep(this, props.name, {
                    ...sharedProps,
                });
                break;
            case 'aws:sleep':
                this.automationStep = new sleep_step_1.SleepStep(this, props.name, {
                    sleepSeconds: this.parseDurationToSeconds(restParams.Duration),
                    ...sharedProps,
                });
                break;
            case 'aws:executeScript':
                const inputs = {};
                Object.entries(restParams.InputPayload ?? {}).forEach(([key, value]) => inputs[key] = this.toVariable(value));
                const handler = restParams.Handler ? restParams.Handler.replace('function.', '') : undefined;
                this.automationStep = new execute_script_step_1.ExecuteScriptStep(this, props.name, {
                    language: execute_script_step_1.ScriptLanguage.fromRuntime(restParams.Runtime, handler),
                    inputPayload: inputs,
                    code: execute_script_step_1.ScriptCode.inline(restParams.Script),
                    ...sharedProps,
                });
                break;
            case 'aws:branch':
                this.automationStep = new branch_step_1.BranchStep(this, props.name, {
                    choices: this.toChoices(restParams.Choices),
                    defaultStepName: restParams.Default,
                    ...sharedProps,
                });
                break;
            case 'aws:approve':
                this.automationStep = new approve_step_1.ApproveStep(this, props.name, {
                    approvers: restParams.Approvers,
                    minRequiredApprovals: restParams.MinRequiredApprovals,
                    message: restParams.Message,
                    notificationArn: restParams.NotificationArn,
                    ...sharedProps,
                });
                break;
            default:
                throw new Error('Action not implemented as StringStep ' + props.action);
        }
    }
    /**
       * Builds a step object from a yaml declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromYaml(scope, inputYaml) {
        return StringStep.fromObject(scope, yaml.load(inputYaml));
    }
    /**
       * Builds a step object from a json declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromJson(scope, json) {
        return StringStep.fromObject(scope, JSON.parse(json));
    }
    /**
       * Builds a step object from an object.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromObject(scope, props) {
        return new StringStep(scope, props.name, props);
    }
    addToDocument(doc) {
        doc.addStep(this.automationStep);
    }
    toOutputs(declaredOutputs) {
        return declaredOutputs.map(declaredOutput => this.toOutput(declaredOutput));
    }
    toOutput(declaredOutput) {
        return {
            name: declaredOutput.Name,
            selector: declaredOutput.Selector,
            outputType: data_type_1.DataType.fromDataType(declaredOutput.Type).dataTypeEnum,
        };
    }
    toChoices(declaredChoices) {
        return declaredChoices.map(declaredChoice => this.toChoice(declaredChoice));
    }
    toChoice(declaredChoice) {
        const operationEntry = Object.entries(declaredChoice)
            .filter(entry => operation_1.OperationEvaluator.STRING_TO_OPERATION[entry[0]] != undefined)[0];
        return new choice_1.Choice({
            jumpToStepName: declaredChoice.NextStep,
            variable: this.toVariable(declaredChoice.Variable),
            operation: operation_1.OperationEvaluator.fromOperationName(operationEntry[0]),
            constant: operationEntry[1],
        });
    }
    toVariable(variable) {
        if (variable.includes('{{')) {
            return new string_variable_1.StringVariable(variable.replace('{{', '').replace('}}', '').trim());
        }
        return new string_variable_1.HardCodedString(variable.trim());
    }
    parseDurationToSeconds(iso8601Duration) {
        // https://stackoverflow.com/a/29153059
        const iso8601DurationRegex = /(-)?P(?:([.,\d]+)Y)?(?:([.,\d]+)M)?(?:([.,\d]+)W)?(?:([.,\d]+)D)?T(?:([.,\d]+)H)?(?:([.,\d]+)M)?(?:([.,\d]+)S)?/;
        const matches = iso8601Duration.match(iso8601DurationRegex);
        if (matches == undefined) {
            throw new Error('Could not parse Duration' + iso8601Duration);
        }
        const periods = {
            years: matches[2] === undefined ? 0 : Number.parseInt(matches[2]),
            months: matches[3] === undefined ? 0 : Number.parseInt(matches[3]),
            weeks: matches[4] === undefined ? 0 : Number.parseInt(matches[4]),
            days: matches[5] === undefined ? 0 : Number.parseInt(matches[5]),
            hours: matches[6] === undefined ? 0 : Number.parseInt(matches[6]),
            minutes: matches[7] === undefined ? 0 : Number.parseInt(matches[7]),
            seconds: matches[8] === undefined ? 0 : Number.parseInt(matches[8]),
        };
        return periods.seconds +
            periods.minutes * 60 +
            periods.hours * 60 * 60 +
            periods.days * 60 * 60 * 24 +
            periods.weeks * 60 * 60 * 24 * 7 +
            periods.months * 60 * 60 * 24 * 30.5 +
            periods.years * 60 * 60 * 24 * 365;
    }
    ;
}
exports.StringStep = StringStep;
_a = JSII_RTTI_SYMBOL_1;
StringStep[_a] = { fqn: "@cdklabs/cdk-ssm-documents.StringStep", version: "0.0.31" };
//# sourceMappingURL=data:application/json;base64,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