"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateImageSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
const aws_service_1 = require("../../domain/aws-service");
/**
 * AutomationStep implemenation for aws:createImage
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-create.html
 */
class CreateImageSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:createImage';
        this.createImageStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        console.log(`CreateImage: Creating image of ${this.createImageStep.instanceId}`);
        const imageId = this.createImage(inputs);
        console.log(`CreateImage: Waiting for ${imageId} to be available`);
        const state = this.waitForImageAvailable(imageId);
        console.log(`CreateImage: Image ${imageId} is available`);
        return {
            ImageId: imageId,
            ImageState: state,
        };
    }
    createImage(inputs) {
        const apiParamMap = {
            InstanceId: this.createImageStep.instanceId,
            Name: this.createImageStep.imageName,
            Description: this.createImageStep.imageDescription,
            NoReboot: this.createImageStep.noReboot,
            BlockDeviceMappings: this.createImageStep.blockDeviceMappings,
        };
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord(apiParamMap, x => x.resolve(inputs));
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'createImage', {
            service: aws_service_1.AwsService.EC2,
            pascalCaseApi: 'CreateImage',
            apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    name: 'ImageId',
                    selector: '$.ImageId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Create image failed for ${apiParams.InstanceId}: ${result.stackTrace}`);
        }
        return result.outputs?.['createImage.ImageId'];
    }
    waitForImageAvailable(imageId) {
        wait_for_and_assert_1.waitForAndAssertImageAvailable({
            imageId: imageId,
        }, this.props);
        return 'available';
    }
}
exports.CreateImageSimulation = CreateImageSimulation;
_a = JSII_RTTI_SYMBOL_1;
CreateImageSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.CreateImageSimulation", version: "0.0.31" };
//# sourceMappingURL=data:application/json;base64,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