"""Provides tools for using Spark SQL as an input data source."""
from typing import Optional

from pyspark.sql import DataFrame as SparkDataFrame
from pyspark.sql import SparkSession

from mlpype.base.data.data_source import DataSource
from mlpype.spark.utils import guarantee_spark


class SparkSqlSource(DataSource[SparkDataFrame]):
    """Use Spark SQL as an input data source."""

    def __init__(
        self,
        query: str,
        spark_session: Optional[SparkSession] = None,
    ) -> None:
        """Use Spark SQL as an input data source.

        Args:
            query (str): The query to run to get the data.
            spark_session (Optional[SparkSession]): The current SparkSession.
        """
        super().__init__()
        self.query = query
        self.spark_session = guarantee_spark(spark_session)

    def read(self) -> SparkDataFrame:
        """Execute the query stored in this object.

        Returns:
            SparkDataFrame: The DataFrame generated by the query.
        """
        return self.spark_session.sql(self.query)
