"""Various utilities to process NV2A PGRAPH logs formatted in the style of xemu [xemu.app].

This file is automatically generated by https://github.com/abaire/nv2a-define-collator and should not be edited.
"""

# ruff: noqa
# mypy: ignore-errors

from __future__ import annotations

import ctypes
import struct
import sys
from dataclasses import dataclass
from typing import Any, Callable, NamedTuple


ProcessorFunc = Callable[[int, int, int], str]


class StateArray(NamedTuple):
    """A multi-value entry (e.g., a vertex color)"""

    """The command ID of the first entry in the array."""
    base: int

    """Size of each element in bytes."""
    stride: int

    """Number of consecutive elements."""
    num_elements: int


class StructStateArray(NamedTuple):
    """An array of potentially multi-value elements, possibly with gaps between.

    E.g., light colors (4 elements per color x 8 lights with all parameters for each light provided consecutively
    before the next light's parameters.)
    """

    """The command ID of the first entry in the array."""
    base: int

    """Number of bytes between the first field of the first entry and the first field of the second entry."""
    struct_stride: int

    """Number of entries in the state array."""
    struct_count: int

    """Size of each field in bytes."""
    stride: int

    """Number of consecutive fields."""
    num_elements: int


_ICW_MAP_VALUES = [
    "UNSIGNED_IDENTITY",
    "UNSIGNED_INVERT",
    "EXPAND_NORMAL",
    "EXPAND_NEGATE",
    "HALFBIAS_NORMAL",
    "HALFBIAS_NEGATE",
    "SIGNED_IDENTITY",
    "SIGNED_NEGATE",
]

# From _dx_psrgbinputs_and_psalphainputs_graphics.htm
# also _dx_psrgboutputs_and_psalphaoutputs_graphics.htm
_ICW_SRC_VALUES = [
    "Zero",  # 0
    "C0",  # 1
    "C1",  # 2
    "Fog",  # 3
    "V0_Diffuse",  # 4
    "V1_Specular",  # 5
    "?6",
    "?7",
    "Tex0",  # 8
    "Tex1",  # 9
    "Tex2",  # 10
    "Tex3",  # 11
    "R0Temp",  # 12
    "R1Temp",  # 13
    "V1_R0_Sum",  # 14
    "EF_Prod",  # 15
]

# From _dx_psrgboutputs_and_psalphaoutputs_graphics.htm
_OCW_DST_VALUES = list(_ICW_SRC_VALUES)
_OCW_DST_VALUES[0] = "Discard"


def _process_combiner_control(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("COUNT", ctypes.c_uint32, 8),
            ("MUX_SELECT", ctypes.c_uint32, 4),
            ("FACTOR_0", ctypes.c_uint32, 4),
            ("FACTOR_1", ctypes.c_uint32, 16),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            elements.append(f"Count:{self.COUNT}")
            if self.MUX_SELECT:
                elements.append("Mux:MSB")
            else:
                elements.append("Mux:LSB")

            if self.FACTOR_0:
                elements.append("Factor0:EACH_STAGE")
            else:
                elements.append("Factor0:SAME_FOR_ALL")

            if self.FACTOR_1:
                elements.append("Factor1:EACH_STAGE")
            else:
                elements.append("Factor1:SAME_FOR_ALL")

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_combiner_specular_fog_cw0(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("D_SOURCE", ctypes.c_uint32, 4),
            ("D_ALPHA", ctypes.c_uint32, 1),
            ("D_INVERSE", ctypes.c_uint32, 3),
            ("C_SOURCE", ctypes.c_uint32, 4),
            ("C_ALPHA", ctypes.c_uint32, 1),
            ("C_INVERSE", ctypes.c_uint32, 3),
            ("B_SOURCE", ctypes.c_uint32, 4),
            ("B_ALPHA", ctypes.c_uint32, 1),
            ("B_INVERSE", ctypes.c_uint32, 3),
            ("A_SOURCE", ctypes.c_uint32, 4),
            ("A_ALPHA", ctypes.c_uint32, 1),
            ("A_INVERSE", ctypes.c_uint32, 3),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            for component in ["A", "B", "C", "D"]:
                src = _ICW_SRC_VALUES[getattr(self, f"{component}_SOURCE")]

                alpha = getattr(self, f"{component}_ALPHA")
                inverse = getattr(self, f"{component}_INVERSE")
                elements.append(
                    f"[{component}: %s%s%s]" % (src, " Alpha" if alpha else "", " Invert" if inverse else "")
                )

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_combiner_specular_fog_cw1(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("SPECULAR_ADD_INVERT_R12", ctypes.c_uint32, 6),
            ("SPECULAR_ADD_INVERT_R5", ctypes.c_uint32, 1),
            ("SPECULAR_CLAMP", ctypes.c_uint32, 1),
            ("G_SOURCE", ctypes.c_uint32, 4),
            ("G_ALPHA", ctypes.c_uint32, 1),
            ("G_INVERSE", ctypes.c_uint32, 3),
            ("F_SOURCE", ctypes.c_uint32, 4),
            ("F_ALPHA", ctypes.c_uint32, 1),
            ("F_INVERSE", ctypes.c_uint32, 3),
            ("E_SOURCE", ctypes.c_uint32, 4),
            ("E_ALPHA", ctypes.c_uint32, 1),
            ("E_INVERSE", ctypes.c_uint32, 3),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            for component in ["E", "F", "G"]:
                src = _ICW_SRC_VALUES[getattr(self, f"{component}_SOURCE")]
                alpha = getattr(self, f"{component}_ALPHA")
                inverse = getattr(self, f"{component}_INVERSE")
                elements.append(
                    f"[{component}: %s%s%s]" % (src, " Alpha" if alpha else "", " Invert" if inverse else "")
                )

            if self.SPECULAR_CLAMP:
                elements.append("SpecularClamp")

            if self.SPECULAR_ADD_INVERT_R5:
                elements.append("SpecularAddInvertR5")

            if self.SPECULAR_ADD_INVERT_R12 == 0x20:
                elements.append("SpecularAddInvertR12")

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_combiner_icw(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("D_SOURCE", ctypes.c_uint32, 4),
            ("D_ALPHA", ctypes.c_uint32, 1),
            ("D_MAP", ctypes.c_uint32, 3),
            ("C_SOURCE", ctypes.c_uint32, 4),
            ("C_ALPHA", ctypes.c_uint32, 1),
            ("C_MAP", ctypes.c_uint32, 3),
            ("B_SOURCE", ctypes.c_uint32, 4),
            ("B_ALPHA", ctypes.c_uint32, 1),
            ("B_MAP", ctypes.c_uint32, 3),
            ("A_SOURCE", ctypes.c_uint32, 4),
            ("A_ALPHA", ctypes.c_uint32, 1),
            ("A_MAP", ctypes.c_uint32, 3),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            for component in ["A", "B", "C", "D"]:
                src = _ICW_SRC_VALUES[getattr(self, f"{component}_SOURCE")]
                alpha = getattr(self, f"{component}_ALPHA")
                map_type = getattr(self, f"{component}_MAP")
                elements.append(
                    f"[{component}: %s %s Map:%s]" % (src, "Alpha" if alpha else "", _ICW_MAP_VALUES[map_type])
                )

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_combiner_alpha_ocw(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("CD_DST_REG", ctypes.c_uint32, 4),
            ("AB_DST_REG", ctypes.c_uint32, 4),
            ("SUM_DST_REG", ctypes.c_uint32, 4),
            ("CD_DOT", ctypes.c_uint32, 1),
            ("AB_DOT", ctypes.c_uint32, 1),
            ("MUX", ctypes.c_uint32, 1),
            ("OP", ctypes.c_uint32, 3),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            elements.append("AB_Reg:%s" % _OCW_DST_VALUES[self.AB_DST_REG])
            elements.append("CD_Reg:%s" % _OCW_DST_VALUES[self.CD_DST_REG])
            elements.append("MuxSum_Reg:%s" % _OCW_DST_VALUES[self.SUM_DST_REG])
            elements.append("AB_DOT:%s" % ("true" if self.AB_DOT else "false"))
            elements.append("CD_DOT:%s" % ("true" if self.CD_DOT else "false"))
            elements.append("MUX:%s" % ("true" if self.MUX else "false"))

            op = "!!BAD!!"
            if self.OP == 0:
                op = "NoShift"
            elif self.OP == 1:
                op = "NoShift_Bias"
            elif self.OP == 2:
                op = "ShiftLeft1"
            elif self.OP == 3:
                op = "ShiftLeft1_Bias"
            elif self.OP == 4:
                op = "ShiftLeft2"
            elif self.OP == 6:
                op = "ShiftRight1"
            elements.append("OP:%s" % op)

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_combiner_color_ocw(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("CD_DST_REG", ctypes.c_uint32, 4),
            ("AB_DST_REG", ctypes.c_uint32, 4),
            ("SUM_DST_REG", ctypes.c_uint32, 4),
            ("CD_DOT", ctypes.c_uint32, 1),
            ("AB_DOT", ctypes.c_uint32, 1),
            ("MUX", ctypes.c_uint32, 1),
            ("OP", ctypes.c_uint32, 3),
            ("CD_BLUE_TO_ALPHA", ctypes.c_uint32, 1),
            ("AB_BLUE_TO_ALPHA", ctypes.c_uint32, 13),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            elements.append("AB_Reg:%s" % _OCW_DST_VALUES[self.AB_DST_REG])
            elements.append("CD_Reg:%s" % _OCW_DST_VALUES[self.CD_DST_REG])
            elements.append("AB+CD_Reg:%s" % _OCW_DST_VALUES[self.SUM_DST_REG])
            elements.append("AB_DOT:%s" % ("true" if self.AB_DOT else "false"))
            elements.append("CD_DOT:%s" % ("true" if self.CD_DOT else "false"))
            elements.append("MUX:%s" % ("true" if self.MUX else "false"))

            op = "!!BAD!!"
            if self.OP == 0:
                op = "NoShift"
            elif self.OP == 1:
                op = "NoShift_Bias"
            elif self.OP == 2:
                op = "ShiftLeft1"
            elif self.OP == 3:
                op = "ShiftLeft1_Bias"
            elif self.OP == 4:
                op = "ShiftLeft2"
            elif self.OP == 6:
                op = "ShiftRight1"
            elements.append("OP:%s" % op)

            elements.append("AB_BlueToAlpha:%s" % ("true" if self.AB_BLUE_TO_ALPHA else "false"))
            elements.append("CD_BlueToAlpha:%s" % ("true" if self.CD_BLUE_TO_ALPHA else "false"))

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_combiner_color_factor(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("BLUE", ctypes.c_uint32, 8),
            ("GREEN", ctypes.c_uint32, 8),
            ("RED", ctypes.c_uint32, 8),
            ("ALPHA", ctypes.c_uint32, 8),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            elements.append("BLUE:%02X %f" % (self.BLUE, self.BLUE / 255.0))
            elements.append("GREEN:%02X %f" % (self.GREEN, self.GREEN / 255.0))
            elements.append("RED:%02X %f" % (self.RED, self.RED / 255.0))
            elements.append("ALPHA:%02X %f" % (self.ALPHA, self.ALPHA / 255.0))

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_control0(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("STENCIL_WRITE_ENABLE", ctypes.c_uint32, 8),
            ("_RESERVED0", ctypes.c_uint32, 4),
            ("Z_FORMAT", ctypes.c_uint32, 4),
            ("Z_PERSPECTIVE_ENABLE", ctypes.c_uint32, 4),
            ("TEXTURE_PERSPECTIVE_ENABLE", ctypes.c_uint32, 4),
            ("PREMULTIPLIED_ALPHA", ctypes.c_uint32, 4),
            ("COLOR_SPACE_CONVERT", ctypes.c_uint32, 4),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            elements.append(f"StencilWrite:{self.STENCIL_WRITE_ENABLE}")
            fmt = "float" if self.Z_FORMAT else "fixed"
            elements.append(f"ZFormat:{fmt}")

            elements.append(f"ZPerspective:{self.Z_PERSPECTIVE_ENABLE}")
            elements.append(f"TexPerspective:{self.TEXTURE_PERSPECTIVE_ENABLE}")
            elements.append(f"PremultAlpha:{self.PREMULTIPLIED_ALPHA}")

            if self.COLOR_SPACE_CONVERT == 1:
                elements.append("Convert:CRYCB=>RGB")
            elif self.COLOR_SPACE_CONVERT == 2:
                elements.append("Convert:SCRYSCB=>RGB")

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_vertex_data_array_format(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("TYPE", ctypes.c_uint32, 4),
            ("SIZE", ctypes.c_uint32, 4),
            ("STRIDE", ctypes.c_uint32, 24),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            if not self.SIZE:
                elements.append("Disabled")
            else:
                types = [
                    "UB D3D",
                    "ShortNormalize",
                    "Float",
                    "?3",
                    "UB OpenGL",
                    "Short",
                    "3ComponentPacked",
                ]
                if len(types) <= self.TYPE:
                    msg = f"Invalid vertex data array format, unknown type {self.TYPE}. 0x{_nv_op:x}(0x{nv_param:x})"
                    raise IndexError(msg)
                elements.append("Type:%s" % types[self.TYPE])

                sizes = ["Disabled", "1", "2", "3", "4", "?5", "?6", "3W"]

                elements.append("Size:%s" % sizes[self.SIZE])
                elements.append("Stride:%d (0x%X)" % (self.STRIDE, self.STRIDE))

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_draw_arrays(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("START_INDEX", ctypes.c_uint32, 24),
            ("COUNT", ctypes.c_uint32, 8),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            return "{Start:%d, Count:%d}" % (self.START_INDEX, self.COUNT)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


# See https://github.com/fgsfdsfgs/pbgl/blob/13fa676239f7de5a4189dd15a86979989adfe3fd/src/state.c#L315
def _process_set_light_control(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("SEPARATE_SPECULAR", ctypes.c_uint32, 2),
            ("RESERVED", ctypes.c_uint32, 14),
            ("LOCALEYE", ctypes.c_uint32, 1),
            ("SOUT", ctypes.c_uint32, 15),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            if self.SEPARATE_SPECULAR:
                elements.append("SeparateSpecular")

            if self.LOCALEYE:
                elements.append("LocalEye")

            if self.SOUT == 0:
                elements.append("SOut:ZeroOut")
            elif self.SOUT == 1:
                elements.append("SOut:Passthrough")
            else:
                elements.append("SOut:%d" % self.SOUT)

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_color_material(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("EMISSIVE", ctypes.c_uint32, 2),
            ("AMBIENT", ctypes.c_uint32, 2),
            ("DIFFUSE", ctypes.c_uint32, 2),
            ("SPECULAR", ctypes.c_uint32, 2),
            ("BACK_EMISSIVE", ctypes.c_uint32, 2),
            ("BACK_AMBIENT", ctypes.c_uint32, 2),
            ("BACK_DIFFUSE", ctypes.c_uint32, 2),
            ("BACK_SPECULAR", ctypes.c_uint32, 2),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            sources = [
                "Material",
                "VertexDiffuse",
                "VertexSpecular",
            ]

            for component in [field[0] for field in self._fields_]:
                source = getattr(self, f"{component}")
                if source >= len(sources):
                    msg = f"Failed to parse source {source} for component {component} of set_color_material param 0x{nv_param:x}"
                    raise ValueError(msg)
                elements.append(f"{component}:{sources[source]}")

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_light_enable_mask(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("LIGHT0", ctypes.c_uint32, 2),
            ("LIGHT1", ctypes.c_uint32, 2),
            ("LIGHT2", ctypes.c_uint32, 2),
            ("LIGHT3", ctypes.c_uint32, 2),
            ("LIGHT4", ctypes.c_uint32, 2),
            ("LIGHT5", ctypes.c_uint32, 2),
            ("LIGHT6", ctypes.c_uint32, 2),
            ("LIGHT7", ctypes.c_uint32, 2),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            light_mode = ["OFF", "INFINITE", "LOCAL", "SPOT"]

            elements.append("Light0:%s" % light_mode[self.LIGHT0])
            elements.append("Light1:%s" % light_mode[self.LIGHT1])
            elements.append("Light2:%s" % light_mode[self.LIGHT2])
            elements.append("Light3:%s" % light_mode[self.LIGHT3])
            elements.append("Light4:%s" % light_mode[self.LIGHT4])
            elements.append("Light5:%s" % light_mode[self.LIGHT5])
            elements.append("Light6:%s" % light_mode[self.LIGHT6])
            elements.append("Light7:%s" % light_mode[self.LIGHT7])

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_other_stage_input(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("STAGE1", ctypes.c_uint32, 16),
            ("STAGE2", ctypes.c_uint32, 4),
            ("STAGE3", ctypes.c_uint32, 4),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            return "{Stage1: %d, Stage2: %d, Stage3: %d}" % (
                self.STAGE1,
                self.STAGE2,
                self.STAGE3,
            )

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def process_shader_stage_program(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("STAGE_0", ctypes.c_uint32, 5),
            ("STAGE_1", ctypes.c_uint32, 5),
            ("STAGE_2", ctypes.c_uint32, 5),
            ("STAGE_3", ctypes.c_uint32, 5),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            stage_0_mode = [
                "NONE",
                "2D_PROJECTIVE",
                "3D_PROJECTIVE",
                "CUBE_MAP",
                "PASS_THROUGH",
                "CLIP_PLANE",
            ]
            elements.append(f"0:{stage_0_mode[self.STAGE_0]}")

            stage_1_mode = [
                "NONE",
                "2D_PROJECTIVE",
                "3D_PROJECTIVE",
                "CUBE_MAP",
                "PASS_THROUGH",
                "CLIP_PLANE",
                "BUMPENVMAP",
                "BUMPENVMAP_LUMINANCE",
                "?0x08",
                "?0x09",
                "?0x0A",
                "?0x0B",
                "?0x0C",
                "?0x0D",
                "?0x0E",
                "DEPENDENT_AR",
                "DEPENDENT_GB",
                "DOT_PRODUCT",
            ]
            elements.append(f"1:{stage_1_mode[self.STAGE_1]}")

            stage_2_mode = [
                "NONE",
                "2D_PROJECTIVE",
                "3D_PROJECTIVE",
                "CUBE_MAP",
                "PASS_THROUGH",
                "CLIP_PLANE",
                "BUMPENVMAP",
                "BUMPENVMAP_LUMINANCE",
                "BRDF",
                "DOT_ST",
                "DOT_ZW",
                "DOT_REFLECT_DIFFUSE",
                "?0x0C",
                "?0x0D",
                "?0x0E",
                "DEPENDENT_AR",
                "DEPENDENT_GB",
                "DOT_PRODUCT",
            ]
            elements.append(f"2:{stage_2_mode[self.STAGE_2]}")

            stage_3_mode = [
                "NONE",
                "2D_PROJECTIVE",
                "3D_PROJECTIVE",
                "CUBE_MAP",
                "PASS_THROUGH",
                "CLIP_PLANE",
                "BUMPENVMAP",
                "BUMPENVMAP_LUMINANCE",
                "BRDF",
                "DOT_ST",
                "DOT_ZW",
                "?0x0B",
                "DOT_REFLECT_SPECULAR",
                "DOT_STR_3D",
                "DOT_STR_CUBE",
                "DEPENDENT_AR",
                "DEPENDENT_GB",
                "?0x11" "DOT_REFLECT_SPECULAR_CONST",
            ]
            elements.append(f"3:{stage_3_mode[self.STAGE_3]}")

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_color_mask(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("BLUE_WRITE", ctypes.c_uint32, 8),
            ("GREEN_WRITE", ctypes.c_uint32, 8),
            ("RED_WRITE", ctypes.c_uint32, 8),
            ("ALPHA_WRITE", ctypes.c_uint32, 8),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            elements.append("Red:%s" % ("W" if self.RED_WRITE else "RO"))
            elements.append("Green:%s" % ("W" if self.GREEN_WRITE else "RO"))
            elements.append("Blue:%s" % ("W" if self.BLUE_WRITE else "RO"))
            elements.append("Alpha:%s" % ("W" if self.ALPHA_WRITE else "RO"))

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_texture_format(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class TextureFormat(ctypes.LittleEndianStructure):
        _fields_ = [
            ("CONTEXT_DMA", ctypes.c_uint32, 2),
            ("CUBEMAP_ENABLE", ctypes.c_uint32, 1),
            ("BORDER_SOURCE", ctypes.c_uint32, 1),
            ("DIMENSIONALITY", ctypes.c_uint32, 4),
            ("COLOR", ctypes.c_uint32, 8),
            ("MIPMAP_LEVELS", ctypes.c_uint32, 4),
            ("BASE_SIZE_U", ctypes.c_uint32, 4),
            ("BASE_SIZE_V", ctypes.c_uint32, 4),
            ("BASE_SIZE_P", ctypes.c_uint32, 4),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []
            if self.CONTEXT_DMA == 1:
                elements.append("DMA_A")
            if self.CONTEXT_DMA == 2:
                elements.append("DMA_B")

            if self.CUBEMAP_ENABLE:
                elements.append("ENABLE_CUBEMAP")

            if self.BORDER_SOURCE:
                elements.append("BORDER_SOURCE_COLOR")
            else:
                elements.append("BORDER_SOURCE_TEXTURE")

            if self.COLOR == 0:
                elements.append("SZ_Y8")
            elif self.COLOR == 0x00000001:
                elements.append("SZ_AY8")
            elif self.COLOR == 0x00000002:
                elements.append("SZ_A1R5G5B5")
            elif self.COLOR == 0x00000003:
                elements.append("SZ_X1R5G5B5")
            elif self.COLOR == 0x00000004:
                elements.append("SZ_A4R4G4B4")
            elif self.COLOR == 0x00000005:
                elements.append("SZ_R5G6B5")
            elif self.COLOR == 0x00000006:
                elements.append("SZ_A8R8G8B8")
            elif self.COLOR == 0x00000007:
                elements.append("SZ_X8R8G8B8")
            elif self.COLOR == 0x0000000B:
                elements.append("SZ_I8_A8R8G8B8")
            elif self.COLOR == 0x0000000C:
                elements.append("L_DXT1_A1R5G5B5")
            elif self.COLOR == 0x0000000E:
                elements.append("L_DXT23_A8R8G8B8")
            elif self.COLOR == 0x0000000F:
                elements.append("L_DXT45_A8R8G8B8")
            elif self.COLOR == 0x00000010:
                elements.append("LU_IMAGE_A1R5G5B5")
            elif self.COLOR == 0x00000011:
                elements.append("LU_IMAGE_R5G6B5")
            elif self.COLOR == 0x00000012:
                elements.append("LU_IMAGE_A8R8G8B8")
            elif self.COLOR == 0x00000013:
                elements.append("LU_IMAGE_Y8")
            elif self.COLOR == 0x00000014:
                elements.append("LU_IMAGE_SY8")
            elif self.COLOR == 0x00000015:
                elements.append("LU_IMAGE_X7SY9")
            elif self.COLOR == 0x00000016:
                elements.append("LU_IMAGE_R8B8")
            elif self.COLOR == 0x00000017:
                elements.append("LU_IMAGE_G8B8")
            elif self.COLOR == 0x00000018:
                elements.append("LU_IMAGE_SG8SB8")
            elif self.COLOR == 0x0000019:
                elements.append("SZ_A8")
            elif self.COLOR == 0x0000001A:
                elements.append("SZ_A8Y8")
            elif self.COLOR == 0x0000001B:
                elements.append("LU_IMAGE_AY8")
            elif self.COLOR == 0x0000001C:
                elements.append("LU_IMAGE_X1R5G5B5")
            elif self.COLOR == 0x0000001D:
                elements.append("LU_IMAGE_A4R4G4B4")
            elif self.COLOR == 0x0000001E:
                elements.append("LU_IMAGE_X8R8G8B8")
            elif self.COLOR == 0x0000001F:
                elements.append("LU_IMAGE_A8")
            elif self.COLOR == 0x00000020:
                elements.append("LU_IMAGE_A8Y8")
            elif self.COLOR == 0x00000024:
                elements.append("LC_IMAGE_CR8YB8CB8YA8")
            elif self.COLOR == 0x00000025:
                elements.append("LC_IMAGE_YB8CR8YA8CB8")
            elif self.COLOR == 0x00000026:
                elements.append("LU_IMAGE_A8CR8CB8Y8")
            elif self.COLOR == 0x00000027:
                elements.append("SZ_R6G5B5")
            elif self.COLOR == 0x00000028:
                elements.append("SZ_G8B8")
            elif self.COLOR == 0x00000029:
                elements.append("SZ_R8B8")
            elif self.COLOR == 0x0000002A:
                elements.append("SZ_DEPTH_X8_Y24_FIXED")
            elif self.COLOR == 0x0000002B:
                elements.append("SZ_DEPTH_X8_Y24_FLOAT")
            elif self.COLOR == 0x0000002C:
                elements.append("SZ_DEPTH_Y16_FIXED")
            elif self.COLOR == 0x0000002D:
                elements.append("SZ_DEPTH_Y16_FLOAT")
            elif self.COLOR == 0x0000002E:
                elements.append("LU_IMAGE_DEPTH_X8_Y24_FIXED")
            elif self.COLOR == 0x0000002F:
                elements.append("LU_IMAGE_DEPTH_X8_Y24_FLOAT")
            elif self.COLOR == 0x00000030:
                elements.append("LU_IMAGE_DEPTH_Y16_FIXED")
            elif self.COLOR == 0x00000031:
                elements.append("LU_IMAGE_DEPTH_Y16_FLOAT")
            elif self.COLOR == 0x00000032:
                elements.append("SZ_Y16")
            elif self.COLOR == 0x00000033:
                elements.append("SZ_YB_16_YA_16")
            elif self.COLOR == 0x00000034:
                elements.append("LC_IMAGE_A4V6YB6A4U6YA6")
            elif self.COLOR == 0x00000035:
                elements.append("LU_IMAGE_Y16")
            elif self.COLOR == 0x00000036:
                elements.append("LU_IMAGE_YB16YA16")
            elif self.COLOR == 0x00000037:
                elements.append("LU_IMAGE_R6G5B5")
            elif self.COLOR == 0x00000038:
                elements.append("SZ_R5G5B5A1")
            elif self.COLOR == 0x00000039:
                elements.append("SZ_R4G4B4A4")
            elif self.COLOR == 0x0000003A:
                elements.append("SZ_A8B8G8R8")
            elif self.COLOR == 0x0000003B:
                elements.append("SZ_B8G8R8A8")
            elif self.COLOR == 0x0000003C:
                elements.append("SZ_R8G8B8A8")
            elif self.COLOR == 0x0000003D:
                elements.append("LU_IMAGE_R5G5B5A1")
            elif self.COLOR == 0x0000003E:
                elements.append("LU_IMAGE_R4G4B4A4")
            elif self.COLOR == 0x0000003F:
                elements.append("LU_IMAGE_A8B8G8R8")
            elif self.COLOR == 0x00000040:
                elements.append("LU_IMAGE_B8G8R8A8")
            elif self.COLOR == 0x00000041:
                elements.append("LU_IMAGE_R8G8B8A8")

            elements.append("MipmapLevels:%d" % self.MIPMAP_LEVELS)
            elements.append(f"{self.DIMENSIONALITY}D")

            elements.append("BaseSizeU:%d" % (1 << self.BASE_SIZE_U))
            elements.append("BaseSizeV:%d" % (1 << self.BASE_SIZE_V))
            elements.append("BaseSizeP:%d" % (1 << self.BASE_SIZE_P))

            return "{%s}" % ", ".join(elements)

    fmt = TextureFormat(nv_param)

    return param_info + f" {fmt}"


def _process_set_texture_control(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("SEPARATE_SPECULAR", ctypes.c_uint32, 2),
            ("RESERVED", ctypes.c_uint32, 14),
            ("LOCALEYE", ctypes.c_uint32, 1),
            ("SOUT", ctypes.c_uint32, 15),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            if self.SEPARATE_SPECULAR:
                elements.append("SeparateSpecular")

            if self.LOCALEYE:
                elements.append("LocalEye")

            if self.SOUT == 0:
                elements.append("SOut:ZeroOut")
            elif self.SOUT == 1:
                elements.append("SOut:Passthrough")
            else:
                elements.append("SOut:%d" % self.SOUT)

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_texture_control1(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("RESERVED", ctypes.c_uint32, 16),
            ("IMAGE_PITCH", ctypes.c_uint32, 16),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []
            elements.append("Pitch: %d" % self.IMAGE_PITCH)

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_texture_address(_nv_class, nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _pack_ = 1
        _fields_ = [
            ("U", ctypes.c_uint32, 4),
            ("CYLWRAP_U", ctypes.c_uint32, 4),
            ("V", ctypes.c_uint32, 4),
            ("CYLWRAP_V", ctypes.c_uint32, 4),
            ("P", ctypes.c_uint32, 4),
            ("CYLWRAP_P", ctypes.c_uint32, 4),
            ("CYLWRAP_Q", ctypes.c_uint32, 4),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            border_modes = [
                "Unknown0",
                "Wrap",
                "Mirror",
                "Clamp_Edge",
                "Border",
                "Clamp_OGL",
            ]

            for component in ["U", "V", "P"]:
                border_mode = getattr(self, f"{component}")
                if border_mode >= len(border_modes):
                    msg = f"Failed to parse border mode {border_mode} for texture address op 0x{nv_op:x} param 0x{nv_param:x}"
                    raise ValueError(msg)
                cyl_wrap = getattr(self, f"CYLWRAP_{component}")

                elements.append(f"{component}:{border_modes[border_mode]}")

                if cyl_wrap:
                    elements.append(f"CylWrap_{component}")

            if self.CYLWRAP_Q:
                elements.append("CylWrap_Q")

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _lod_bias_5_8_to_float(lod_bias) -> float:
    lod_bias_5_8 = lod_bias
    signed_5_8 = lod_bias_5_8 - (1 << 13) if (lod_bias_5_8 & (1 << 12)) else lod_bias_5_8

    return float(signed_5_8) / 256.0


def _process_set_texture_control0(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _pack_ = 1
        _fields_ = [
            ("COLOR_KEY_OP", ctypes.c_uint32, 2),
            ("ALPHA_KILL_ENABLE", ctypes.c_uint32, 1),
            ("IMAGE_FIELD_ENABLE", ctypes.c_uint32, 1),
            ("MAX_ANISO", ctypes.c_uint32, 2),
            ("MAX_LOD_CLAMP", ctypes.c_uint32, 12),
            ("MIN_LOD_CLAMP", ctypes.c_uint32, 12),
            ("ENABLE", ctypes.c_uint32, 2),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            if not self.ENABLE:
                return "{Disabled}"

            elements = []

            if self.COLOR_KEY_OP == 1:
                elements.append("ColorKey:Alpha")
            elif self.COLOR_KEY_OP == 2:
                elements.append("ColorKey:RGBA")
            elif self.COLOR_KEY_OP == 3:
                elements.append("ColorKey:KILL")

            if self.ALPHA_KILL_ENABLE:
                elements.append("AlphaKillEnabled")

            if self.IMAGE_FIELD_ENABLE:
                elements.append("ImageFieldEnabled")

            elements.append("MaxAniso:%d" % (1 << self.MAX_ANISO))
            elements.append("MaxLOD:0x%X (%f)" % (self.MAX_LOD_CLAMP, _lod_bias_5_8_to_float(self.MAX_LOD_CLAMP)))
            elements.append("MinLOD:0x%X (%f)" % (self.MIN_LOD_CLAMP, _lod_bias_5_8_to_float(self.MIN_LOD_CLAMP)))

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_texture_filter(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _pack_ = 1
        _fields_ = [
            ("LOD_BIAS", ctypes.c_uint32, 13),
            ("CONVOLUTION_KERNEL", ctypes.c_uint32, 3),
            ("MIN", ctypes.c_uint32, 8),
            ("MAG", ctypes.c_uint32, 4),
            ("A_SIGNED", ctypes.c_uint32, 1),
            ("R_SIGNED", ctypes.c_uint32, 1),
            ("G_SIGNED", ctypes.c_uint32, 1),
            ("B_SIGNED", ctypes.c_uint32, 1),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            gl_lod_bias = _lod_bias_5_8_to_float(self.LOD_BIAS)
            elements.append("LODBias:0x%X (%f)" % (self.LOD_BIAS, gl_lod_bias))

            if self.CONVOLUTION_KERNEL == 1:
                elements.append("Quincunx")
            elif self.CONVOLUTION_KERNEL == 2:
                elements.append("Gaussian3")
            else:
                elements.append("UnknownKernel:%d" % self.CONVOLUTION_KERNEL)

            min_strategy = [
                "Unknown0",
                "BoxLOD0",
                "TentLOD0",
                "BoxNearestLOD",
                "TentNearestLOD",
                "BoxTentLOD",
                "TentTentLOD",
                "Convolution2dLOD0",
            ]
            elements.append("Min:%s" % min_strategy[self.MIN])

            mag_strategy = [
                "Unknown0",
                "BoxLOD0",
                "TentLOD0",
                "Unknown3",
                "Convolution2dLOD0",
            ]
            elements.append("Mag:%s" % mag_strategy[self.MAG])

            if self.A_SIGNED:
                elements.append("Signed-Alpha")
            if self.R_SIGNED:
                elements.append("Signed-Red")
            if self.G_SIGNED:
                elements.append("Signed-Green")
            if self.B_SIGNED:
                elements.append("Signed-Blue")

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_texture_palette(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _pack_ = 1
        _fields_ = [
            ("DMA", ctypes.c_uint32, 2),
            ("LENGTH", ctypes.c_uint32, 4),
            ("OFFSET", ctypes.c_uint32, 26),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            elements.append("DMA_%s" % ("B" if self.DMA else "A"))

            palette_sizes = ["256", "128", "64", "32"]
            elements.append(f"Length:{palette_sizes[self.LENGTH]}")

            elements.append("Offset:0x%08X" % self.OFFSET)

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_surface_format(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param

    class BitField(ctypes.LittleEndianStructure):
        _fields_ = [
            ("COLOR", ctypes.c_uint32, 4),
            ("ZETA", ctypes.c_uint32, 4),
            ("TYPE", ctypes.c_uint32, 4),
            ("ANTIALIASING", ctypes.c_uint32, 4),
            ("WIDTH", ctypes.c_uint32, 8),
            ("HEIGHT", ctypes.c_uint32, 8),
        ]

        def __new__(cls, *args, **kwargs):
            if args:
                return cls.from_buffer_copy(args[0].to_bytes(4, byteorder=sys.byteorder))
            return super().__new__()

        def __str__(self):
            elements = []

            if self.COLOR == 1:
                elements.append("LE_X1R5G5B5_Z1R5G5B5")
            elif self.COLOR == 2:
                elements.append("LE_X1R5G5B5_O1R5G5B5")
            elif self.COLOR == 3:
                elements.append("LE_R5G6B5")
            elif self.COLOR == 4:
                elements.append("LE_X8R8G8B8_Z8R8G8B8")
            elif self.COLOR == 5:
                elements.append("LE_X8R8G8B8_O8R8G8B8")
            elif self.COLOR == 6:
                elements.append("LE_X1A7R8G8B8_Z1A7R8G8B8")
            elif self.COLOR == 7:
                elements.append("LE_X1A7R8G8B8_O1A7R8G8B8")
            elif self.COLOR == 8:
                elements.append("LE_A8R8G8B8")
            elif self.COLOR == 9:
                elements.append("LE_B8")
            elif self.COLOR == 10:
                elements.append("LE_G8B8")

            if self.ZETA == 1:
                elements.append("Z16")
            elif self.ZETA == 2:
                elements.append("Z24S8")

            if self.TYPE == 1:
                elements.append("Type:Pitch")
            elif self.TYPE == 2:
                elements.append("Type:Swizzle")

            if self.ANTIALIASING == 0:
                elements.append("AA:Center_1")
            elif self.ANTIALIASING == 1:
                elements.append("AA:Center_Corner_2")
            elif self.ANTIALIASING == 2:
                elements.append("AA:Square_Offset_4")

            elements.append("Width:%d" % (1 << self.WIDTH))
            elements.append("Height:%d" % (1 << self.HEIGHT))

            return "{%s}" % ", ".join(elements)

    fmt = BitField(nv_param)
    return param_info + f" {fmt}"


def _process_set_texgen_rst(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param
    setting_map = {
        0: "DISABLE",
        0x00008511: "NORMAL_MAP",
        0x00008512: "REFLECTION_MAP",
        0x00002400: "EYE_LINEAR",
        0x00002401: "OBJECT_LINEAR",
        0x00002402: "SPHERE_MAP",
    }
    setting = setting_map.get(nv_param, "<<INVALID>>")
    return param_info + f" {setting}"


def _process_set_texgen_q(_nv_class, _nv_op, nv_param):
    param_info = "0x%X" % nv_param
    setting_map = {
        0: "DISABLE",
        0x00002400: "EYE_LINEAR",
        0x00002401: "OBJECT_LINEAR",
    }
    setting = setting_map.get(nv_param, "<<INVALID>>")
    return param_info + f" {setting}"


NV012_SET_BETA = 0x300
NV012_SET_OBJECT = 0x0
NV019_CONTEXT_CLIP_RECTANGLE_SET_OBJECT = 0x0
NV019_CONTEXT_CLIP_RECTANGLE_SET_POINT = 0x300
NV019_CONTEXT_CLIP_RECTANGLE_SET_SIZE = 0x304
NV042_CONTEXT_SURFACES_2D_DMA_IMAGE_DST = 0x188
NV042_CONTEXT_SURFACES_2D_DMA_IMAGE_SRC = 0x184
NV042_CONTEXT_SURFACES_2D_FORMAT = 0x300
NV042_CONTEXT_SURFACES_2D_OFFSET_DST = 0x30C
NV042_CONTEXT_SURFACES_2D_OFFSET_SRC = 0x308
NV042_CONTEXT_SURFACES_2D_PITCH = 0x304
NV044_SET_MONOCHROME_COLOR0 = 0x310
NV057_COLOR_KEY_FORMAT = 0x300
NV057_COLOR_KEY_VALUE = 0x304
NV062_SET_COLOR_FORMAT = 0x300
NV062_SET_CONTEXT_DMA_IMAGE_DESTIN = 0x188
NV062_SET_CONTEXT_DMA_IMAGE_SOURCE = 0x184
NV062_SET_OBJECT = 0x0
NV062_SET_OFFSET_DESTIN = 0x30C
NV062_SET_OFFSET_SOURCE = 0x308
NV062_SET_PITCH = 0x304
NV097_ARRAY_ELEMENT16 = 0x1800
NV097_ARRAY_ELEMENT32 = 0x1808
NV097_BACK_END_WRITE_SEMAPHORE_RELEASE = 0x1D70
NV097_BREAK_VERTEX_BUFFER_CACHE = 0x1710
NV097_CLEAR_REPORT_VALUE = 0x17C8
NV097_CLEAR_SURFACE = 0x1D94
NV097_DRAW_ARRAYS = 0x1810
NV097_FLIP_INCREMENT_WRITE = 0x12C
NV097_FLIP_STALL = 0x130
NV097_GET_REPORT = 0x17D0
NV097_INLINE_ARRAY = 0x1818
NV097_LAUNCH_TRANSFORM_PROGRAM = 0x1E90
NV097_NO_OPERATION = 0x100
NV097_SET_ALPHA_FUNC = 0x33C
NV097_SET_ALPHA_REF = 0x340
NV097_SET_ALPHA_TEST_ENABLE = 0x300
NV097_SET_ANTI_ALIASING_CONTROL = 0x1D7C
NV097_SET_BACK_LIGHT_AMBIENT_COLOR = 0xC00
NV097_SET_BACK_LIGHT_DIFFUSE_COLOR = 0xC0C
NV097_SET_BACK_LIGHT_SPECULAR_COLOR = 0xC18
NV097_SET_BACK_MATERIAL_ALPHA = 0x17AC
NV097_SET_BACK_MATERIAL_EMISSION = 0x17B0
NV097_SET_BACK_POLYGON_MODE = 0x390
NV097_SET_BACK_SCENE_AMBIENT_COLOR = 0x17A0
NV097_SET_BACK_SPECULAR_PARAMS = 0x1E28
NV097_SET_BEGIN_END = 0x17FC
NV097_SET_BLEND_COLOR = 0x34C
NV097_SET_BLEND_ENABLE = 0x304
NV097_SET_BLEND_EQUATION = 0x350
NV097_SET_BLEND_FUNC_DFACTOR = 0x348
NV097_SET_BLEND_FUNC_SFACTOR = 0x344
NV097_SET_CLEAR_RECT_HORIZONTAL = 0x1D98
NV097_SET_CLEAR_RECT_VERTICAL = 0x1D9C
NV097_SET_CLIP_MAX = 0x398
NV097_SET_CLIP_MIN = 0x394
NV097_SET_COLOR_CLEAR_VALUE = 0x1D90
NV097_SET_COLOR_KEY_COLOR = 0xAE0
NV097_SET_COLOR_MASK = 0x358
NV097_SET_COLOR_MATERIAL = 0x298
NV097_SET_COMBINER_ALPHA_ICW = 0x260
NV097_SET_COMBINER_ALPHA_OCW = 0xAA0
NV097_SET_COMBINER_COLOR_ICW = 0xAC0
NV097_SET_COMBINER_COLOR_OCW = 0x1E40
NV097_SET_COMBINER_CONTROL = 0x1E60
NV097_SET_COMBINER_FACTOR0 = 0xA60
NV097_SET_COMBINER_FACTOR1 = 0xA80
NV097_SET_COMBINER_SPECULAR_FOG_CW0 = 0x288
NV097_SET_COMBINER_SPECULAR_FOG_CW1 = 0x28C
NV097_SET_COMPOSITE_MATRIX = 0x680
NV097_SET_COMPRESS_ZBUFFER_EN = 0x1D80
NV097_SET_CONTEXT_DMA_A = 0x184
NV097_SET_CONTEXT_DMA_B = 0x188
NV097_SET_CONTEXT_DMA_COLOR = 0x194
NV097_SET_CONTEXT_DMA_NOTIFIES = 0x180
NV097_SET_CONTEXT_DMA_REPORT = 0x1A8
NV097_SET_CONTEXT_DMA_SEMAPHORE = 0x1A4
NV097_SET_CONTEXT_DMA_STATE = 0x190
NV097_SET_CONTEXT_DMA_VERTEX_A = 0x19C
NV097_SET_CONTEXT_DMA_VERTEX_B = 0x1A0
NV097_SET_CONTEXT_DMA_ZETA = 0x198
NV097_SET_CONTROL0 = 0x290
NV097_SET_CULL_FACE = 0x39C
NV097_SET_CULL_FACE_ENABLE = 0x308
NV097_SET_DEPTH_FUNC = 0x354
NV097_SET_DEPTH_MASK = 0x35C
NV097_SET_DEPTH_TEST_ENABLE = 0x30C
NV097_SET_DIFFUSE_COLOR3F = 0x1560
NV097_SET_DIFFUSE_COLOR4F = 0x1550
NV097_SET_DIFFUSE_COLOR4I = 0x156C
NV097_SET_DIFFUSE_COLOR4UB = 0x156C
NV097_SET_DITHER_ENABLE = 0x310
NV097_SET_DOT_RGBMAPPING = 0x1E74
NV097_SET_EDGE_FLAG = 0x16BC
NV097_SET_EYE_DIRECTION = 0x17E0
NV097_SET_EYE_POSITION = 0xA50
NV097_SET_EYE_VECTOR = 0x181C
NV097_SET_FLAT_SHADE_OP = 0x9FC
NV097_SET_FLIP_MODULO = 0x128
NV097_SET_FLIP_READ = 0x120
NV097_SET_FLIP_WRITE = 0x124
NV097_SET_FOG_COLOR = 0x2A8
NV097_SET_FOG_COORD = 0x1698
NV097_SET_FOG_ENABLE = 0x2A4
NV097_SET_FOG_GEN_MODE = 0x2A0
NV097_SET_FOG_MODE = 0x29C
NV097_SET_FOG_PARAMS = 0x9C0
NV097_SET_FOG_PLANE = 0x9D0
NV097_SET_FRONT_FACE = 0x3A0
NV097_SET_FRONT_POLYGON_MODE = 0x38C
NV097_SET_INVERSE_MODEL_VIEW_MATRIX = 0x580
NV097_SET_LIGHTING_ENABLE = 0x314
NV097_SET_LIGHT_AMBIENT_COLOR = 0x1000
NV097_SET_LIGHT_CONTROL = 0x294
NV097_SET_LIGHT_DIFFUSE_COLOR = 0x100C
NV097_SET_LIGHT_ENABLE_MASK = 0x3BC
NV097_SET_LIGHT_INFINITE_DIRECTION = 0x1034
NV097_SET_LIGHT_INFINITE_HALF_VECTOR = 0x1028
NV097_SET_LIGHT_LOCAL_ATTENUATION = 0x1068
NV097_SET_LIGHT_LOCAL_POSITION = 0x105C
NV097_SET_LIGHT_LOCAL_RANGE = 0x1024
NV097_SET_LIGHT_SPECULAR_COLOR = 0x1018
NV097_SET_LIGHT_SPOT_DIRECTION = 0x104C
NV097_SET_LIGHT_SPOT_FALLOFF = 0x1040
NV097_SET_LIGHT_TWO_SIDE_ENABLE = 0x17C4
NV097_SET_LINE_SMOOTH_ENABLE = 0x320
NV097_SET_LINE_WIDTH = 0x380
NV097_SET_LOGIC_OP = 0x17C0
NV097_SET_LOGIC_OP_ENABLE = 0x17BC
NV097_SET_MATERIAL_ALPHA = 0x3B4
NV097_SET_MATERIAL_EMISSION = 0x3A8
NV097_SET_MODEL_VIEW_MATRIX = 0x480
NV097_SET_NORMAL3F = 0x1530
NV097_SET_NORMAL3S = 0x1540
NV097_SET_NORMALIZATION_ENABLE = 0x3A4
NV097_SET_OBJECT = 0x0
NV097_SET_OCCLUDE_ZSTENCIL_EN = 0x1D84
NV097_SET_POINT_PARAMS = 0xA30
NV097_SET_POINT_PARAMS_ENABLE = 0x318
NV097_SET_POINT_SIZE = 0x43C
NV097_SET_POINT_SMOOTH_ENABLE = 0x31C
NV097_SET_POLYGON_OFFSET_BIAS = 0x388
NV097_SET_POLYGON_OFFSET_SCALE_FACTOR = 0x384
NV097_SET_POLY_OFFSET_FILL_ENABLE = 0x338
NV097_SET_POLY_OFFSET_LINE_ENABLE = 0x334
NV097_SET_POLY_OFFSET_POINT_ENABLE = 0x330
NV097_SET_POLY_SMOOTH_ENABLE = 0x324
NV097_SET_PROJECTION_MATRIX = 0x440
NV097_SET_SCENE_AMBIENT_COLOR = 0xA10
NV097_SET_SEMAPHORE_OFFSET = 0x1D6C
NV097_SET_SHADER_CLIP_PLANE_MODE = 0x17F8
NV097_SET_SHADER_OTHER_STAGE_INPUT = 0x1E78
NV097_SET_SHADER_STAGE_PROGRAM = 0x1E70
NV097_SET_SHADE_MODE = 0x37C
NV097_SET_SHADE_MODEL = 0x37C
NV097_SET_SHADOW_COMPARE_FUNC = 0x1E6C
NV097_SET_SHADOW_DEPTH_FUNC = 0x1E6C
NV097_SET_SHADOW_ZSLOPE_THRESHOLD = 0x1E68
NV097_SET_SKIN_MODE = 0x328
NV097_SET_SPECULAR_COLOR3F = 0x1580
NV097_SET_SPECULAR_COLOR4F = 0x1570
NV097_SET_SPECULAR_COLOR4I = 0x158C
NV097_SET_SPECULAR_COLOR4UB = 0x158C
NV097_SET_SPECULAR_ENABLE = 0x3B8
NV097_SET_SPECULAR_FOG_FACTOR = 0x1E20
NV097_SET_SPECULAR_PARAMS = 0x9E0
NV097_SET_STENCIL_FUNC = 0x364
NV097_SET_STENCIL_FUNC_MASK = 0x36C
NV097_SET_STENCIL_FUNC_REF = 0x368
NV097_SET_STENCIL_MASK = 0x360
NV097_SET_STENCIL_OP_FAIL = 0x370
NV097_SET_STENCIL_OP_ZFAIL = 0x374
NV097_SET_STENCIL_OP_ZPASS = 0x378
NV097_SET_STENCIL_TEST_ENABLE = 0x32C
NV097_SET_STIPPLE_ENABLE = 0x147C
NV097_SET_STIPPLE_PATERN_SIZE = 0x20
NV097_SET_STIPPLE_PATTERN = 0x1480
NV097_SET_SURFACE_CLIP_HORIZONTAL = 0x200
NV097_SET_SURFACE_CLIP_VERTICAL = 0x204
NV097_SET_SURFACE_COLOR_OFFSET = 0x210
NV097_SET_SURFACE_FORMAT = 0x208
NV097_SET_SURFACE_PITCH = 0x20C
NV097_SET_SURFACE_ZETA_OFFSET = 0x214
NV097_SET_SWATH_WIDTH = 0x9F8
NV097_SET_TEXCOORD0_2F = 0x1590
NV097_SET_TEXCOORD0_2S = 0x1598
NV097_SET_TEXCOORD0_4F = 0x15A0
NV097_SET_TEXCOORD0_4S = 0x15B0
NV097_SET_TEXCOORD1_2F = 0x15B8
NV097_SET_TEXCOORD1_2S = 0x15C0
NV097_SET_TEXCOORD1_4F = 0x15C8
NV097_SET_TEXCOORD1_4S = 0x15D8
NV097_SET_TEXCOORD2_2F = 0x15E0
NV097_SET_TEXCOORD2_2S = 0x15E8
NV097_SET_TEXCOORD2_4F = 0x15F0
NV097_SET_TEXCOORD2_4S = 0x1600
NV097_SET_TEXCOORD3_2F = 0x1608
NV097_SET_TEXCOORD3_2S = 0x1610
NV097_SET_TEXCOORD3_4F = 0x1620
NV097_SET_TEXCOORD3_4S = 0x1630
NV097_SET_TEXGEN_PLANE_Q = 0x870
NV097_SET_TEXGEN_PLANE_R = 0x860
NV097_SET_TEXGEN_PLANE_S = 0x840
NV097_SET_TEXGEN_PLANE_T = 0x850
NV097_SET_TEXGEN_Q = 0x3CC
NV097_SET_TEXGEN_R = 0x3C8
NV097_SET_TEXGEN_S = 0x3C0
NV097_SET_TEXGEN_T = 0x3C4
NV097_SET_TEXGEN_VIEW_MODEL = 0x9CC
NV097_SET_TEXTURE_ADDRESS = 0x1B08
NV097_SET_TEXTURE_BORDER_COLOR = 0x1B24
NV097_SET_TEXTURE_CONTROL0 = 0x1B0C
NV097_SET_TEXTURE_CONTROL1 = 0x1B10
NV097_SET_TEXTURE_FILTER = 0x1B14
NV097_SET_TEXTURE_FORMAT = 0x1B04
NV097_SET_TEXTURE_IMAGE_RECT = 0x1B1C
NV097_SET_TEXTURE_MATRIX = 0x6C0
NV097_SET_TEXTURE_MATRIX_ENABLE = 0x420
NV097_SET_TEXTURE_OFFSET = 0x1B00
NV097_SET_TEXTURE_PALETTE = 0x1B20
NV097_SET_TEXTURE_SET_BUMP_ENV_MAT = 0x1B28
NV097_SET_TEXTURE_SET_BUMP_ENV_OFFSET = 0x1B3C
NV097_SET_TEXTURE_SET_BUMP_ENV_SCALE = 0x1B38
NV097_SET_TRANSFORM_CONSTANT = 0xB80
NV097_SET_TRANSFORM_CONSTANT_LOAD = 0x1EA4
NV097_SET_TRANSFORM_DATA = 0x1E80
NV097_SET_TRANSFORM_EXECUTION_MODE = 0x1E94
NV097_SET_TRANSFORM_PROGRAM = 0xB00
NV097_SET_TRANSFORM_PROGRAM_CXT_WRITE_EN = 0x1E98
NV097_SET_TRANSFORM_PROGRAM_LOAD = 0x1E9C
NV097_SET_TRANSFORM_PROGRAM_START = 0x1EA0
NV097_SET_TWO_SIDE_LIGHT_EN = 0x17C4
NV097_SET_VERTEX3F = 0x1500
NV097_SET_VERTEX4F = 0x1518
NV097_SET_VERTEX_DATA2F_M = 0x1880
NV097_SET_VERTEX_DATA2S = 0x1900
NV097_SET_VERTEX_DATA4F_M = 0x1A00
NV097_SET_VERTEX_DATA4S_M = 0x1980
NV097_SET_VERTEX_DATA4UB = 0x1940
NV097_SET_VERTEX_DATA_ARRAY_FORMAT = 0x1760
NV097_SET_VERTEX_DATA_ARRAY_OFFSET = 0x1720
NV097_SET_VIEWPORT_OFFSET = 0xA20
NV097_SET_VIEWPORT_SCALE = 0xAF0
NV097_SET_WEIGHT1F = 0x169C
NV097_SET_WEIGHT2F = 0x16A0
NV097_SET_WEIGHT3F = 0x16B0
NV097_SET_WEIGHT4F = 0x16C0
NV097_SET_WINDOW_CLIP_HORIZONTAL = 0x2C0
NV097_SET_WINDOW_CLIP_TYPE = 0x2B4
NV097_SET_WINDOW_CLIP_VERTICAL = 0x2E0
NV097_SET_ZMIN_MAX_CONTROL = 0x1D78
NV097_SET_ZPASS_PIXEL_COUNT_ENABLE = 0x17CC
NV097_SET_ZSTENCIL_CLEAR_VALUE = 0x1D8C
NV097_WAIT_FOR_IDLE = 0x110
NV09F_CONTROL_POINT_IN = 0x300
NV09F_CONTROL_POINT_OUT = 0x304
NV09F_SET_BETA_OBJECT = 0x194
NV09F_SET_BETA_OBJECT4 = 0x198
NV09F_SET_CLIP_RECTANGLE_OBJECT = 0x188
NV09F_SET_COLOR_KEY_OBJECT = 0x184
NV09F_SET_CONTEXT_SURFACES = 0x19C
NV09F_SET_OBJECT = 0x0
NV09F_SET_OPERATION = 0x2FC
NV09F_SET_PATTERN_OBJECT = 0x18C
NV09F_SET_ROP5_OBJECT = 0x190
NV09F_SIZE = 0x308

_SPARSE_NAME_MAP = {
    (0x12, 0x0): "NV012_SET_OBJECT",
    (0x12, 0x300): "NV012_SET_BETA",
    (0x19, 0x0): "NV019_CONTEXT_CLIP_RECTANGLE_SET_OBJECT",
    (0x19, 0x300): "NV019_CONTEXT_CLIP_RECTANGLE_SET_POINT",
    (0x19, 0x304): "NV019_CONTEXT_CLIP_RECTANGLE_SET_SIZE",
    (0x42, 0x184): "NV042_CONTEXT_SURFACES_2D_DMA_IMAGE_SRC",
    (0x42, 0x188): "NV042_CONTEXT_SURFACES_2D_DMA_IMAGE_DST",
    (0x42, 0x300): "NV042_CONTEXT_SURFACES_2D_FORMAT",
    (0x42, 0x304): "NV042_CONTEXT_SURFACES_2D_PITCH",
    (0x42, 0x308): "NV042_CONTEXT_SURFACES_2D_OFFSET_SRC",
    (0x42, 0x30C): "NV042_CONTEXT_SURFACES_2D_OFFSET_DST",
    (0x44, 0x310): "NV044_SET_MONOCHROME_COLOR0",
    (0x57, 0x300): "NV057_COLOR_KEY_FORMAT",
    (0x57, 0x304): "NV057_COLOR_KEY_VALUE",
    (0x62, 0x0): "NV062_SET_OBJECT",
    (0x62, 0x184): "NV062_SET_CONTEXT_DMA_IMAGE_SOURCE",
    (0x62, 0x188): "NV062_SET_CONTEXT_DMA_IMAGE_DESTIN",
    (0x62, 0x300): "NV062_SET_COLOR_FORMAT",
    (0x62, 0x304): "NV062_SET_PITCH",
    (0x62, 0x308): "NV062_SET_OFFSET_SOURCE",
    (0x62, 0x30C): "NV062_SET_OFFSET_DESTIN",
    (0x97, 0x0): "NV097_SET_OBJECT",
    (0x97, 0x100): "NV097_NO_OPERATION",
    (0x97, 0x1000): "NV097_SET_LIGHT_AMBIENT_COLOR",
    (0x97, 0x100C): "NV097_SET_LIGHT_DIFFUSE_COLOR",
    (0x97, 0x1018): "NV097_SET_LIGHT_SPECULAR_COLOR",
    (0x97, 0x1024): "NV097_SET_LIGHT_LOCAL_RANGE",
    (0x97, 0x1028): "NV097_SET_LIGHT_INFINITE_HALF_VECTOR",
    (0x97, 0x1034): "NV097_SET_LIGHT_INFINITE_DIRECTION",
    (0x97, 0x1040): "NV097_SET_LIGHT_SPOT_FALLOFF",
    (0x97, 0x104C): "NV097_SET_LIGHT_SPOT_DIRECTION",
    (0x97, 0x105C): "NV097_SET_LIGHT_LOCAL_POSITION",
    (0x97, 0x1068): "NV097_SET_LIGHT_LOCAL_ATTENUATION",
    (0x97, 0x110): "NV097_WAIT_FOR_IDLE",
    (0x97, 0x120): "NV097_SET_FLIP_READ",
    (0x97, 0x124): "NV097_SET_FLIP_WRITE",
    (0x97, 0x128): "NV097_SET_FLIP_MODULO",
    (0x97, 0x12C): "NV097_FLIP_INCREMENT_WRITE",
    (0x97, 0x130): "NV097_FLIP_STALL",
    (0x97, 0x147C): "NV097_SET_STIPPLE_ENABLE",
    (0x97, 0x1480): "NV097_SET_STIPPLE_PATTERN",
    (0x97, 0x1500): "NV097_SET_VERTEX3F",
    (0x97, 0x1518): "NV097_SET_VERTEX4F",
    (0x97, 0x1530): "NV097_SET_NORMAL3F",
    (0x97, 0x1540): "NV097_SET_NORMAL3S",
    (0x97, 0x1550): "NV097_SET_DIFFUSE_COLOR4F",
    (0x97, 0x1560): "NV097_SET_DIFFUSE_COLOR3F",
    (0x97, 0x156C): "NV097_SET_DIFFUSE_COLOR4UB",
    (0x97, 0x1570): "NV097_SET_SPECULAR_COLOR4F",
    (0x97, 0x1580): "NV097_SET_SPECULAR_COLOR3F",
    (0x97, 0x158C): "NV097_SET_SPECULAR_COLOR4UB",
    (0x97, 0x1590): "NV097_SET_TEXCOORD0_2F",
    (0x97, 0x1598): "NV097_SET_TEXCOORD0_2S",
    (0x97, 0x15A0): "NV097_SET_TEXCOORD0_4F",
    (0x97, 0x15B0): "NV097_SET_TEXCOORD0_4S",
    (0x97, 0x15B8): "NV097_SET_TEXCOORD1_2F",
    (0x97, 0x15C0): "NV097_SET_TEXCOORD1_2S",
    (0x97, 0x15C8): "NV097_SET_TEXCOORD1_4F",
    (0x97, 0x15D8): "NV097_SET_TEXCOORD1_4S",
    (0x97, 0x15E0): "NV097_SET_TEXCOORD2_2F",
    (0x97, 0x15E8): "NV097_SET_TEXCOORD2_2S",
    (0x97, 0x15F0): "NV097_SET_TEXCOORD2_4F",
    (0x97, 0x1600): "NV097_SET_TEXCOORD2_4S",
    (0x97, 0x1608): "NV097_SET_TEXCOORD3_2F",
    (0x97, 0x1610): "NV097_SET_TEXCOORD3_2S",
    (0x97, 0x1620): "NV097_SET_TEXCOORD3_4F",
    (0x97, 0x1630): "NV097_SET_TEXCOORD3_4S",
    (0x97, 0x1698): "NV097_SET_FOG_COORD",
    (0x97, 0x169C): "NV097_SET_WEIGHT1F",
    (0x97, 0x16A0): "NV097_SET_WEIGHT2F",
    (0x97, 0x16B0): "NV097_SET_WEIGHT3F",
    (0x97, 0x16BC): "NV097_SET_EDGE_FLAG",
    (0x97, 0x16C0): "NV097_SET_WEIGHT4F",
    (0x97, 0x1710): "NV097_BREAK_VERTEX_BUFFER_CACHE",
    (0x97, 0x1720): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__POS",
    (0x97, 0x1724): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__WEIGHT",
    (0x97, 0x1728): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__NORMAL",
    (0x97, 0x172C): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__DIFFUSE",
    (0x97, 0x1730): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__SPECULAR",
    (0x97, 0x1734): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__FOG_COORD",
    (0x97, 0x1738): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__POINT_SIZE",
    (0x97, 0x173C): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__BACK_DIFFUSE",
    (0x97, 0x1740): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__BACK_SPECULAR",
    (0x97, 0x1744): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__TEX0",
    (0x97, 0x1748): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__TEX1",
    (0x97, 0x174C): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__TEX2",
    (0x97, 0x1750): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__TEX3",
    (0x97, 0x1754): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__13",
    (0x97, 0x1758): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__14",
    (0x97, 0x175C): "NV097_SET_VERTEX_DATA_ARRAY_OFFSET__15",
    (0x97, 0x1760): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__POS",
    (0x97, 0x1764): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__WEIGHT",
    (0x97, 0x1768): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__NORMAL",
    (0x97, 0x176C): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__DIFFUSE",
    (0x97, 0x1770): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__SPECULAR",
    (0x97, 0x1774): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__FOG_COORD",
    (0x97, 0x1778): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__POINT_SIZE",
    (0x97, 0x177C): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__BACK_DIFFUSE",
    (0x97, 0x1780): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__BACK_SPECULAR",
    (0x97, 0x1784): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__TEX0",
    (0x97, 0x1788): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__TEX1",
    (0x97, 0x178C): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__TEX2",
    (0x97, 0x1790): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__TEX3",
    (0x97, 0x1794): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__13",
    (0x97, 0x1798): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__14",
    (0x97, 0x179C): "NV097_SET_VERTEX_DATA_ARRAY_FORMAT__15",
    (0x97, 0x17A0): "NV097_SET_BACK_SCENE_AMBIENT_COLOR",
    (0x97, 0x17AC): "NV097_SET_BACK_MATERIAL_ALPHA",
    (0x97, 0x17B0): "NV097_SET_BACK_MATERIAL_EMISSION",
    (0x97, 0x17BC): "NV097_SET_LOGIC_OP_ENABLE",
    (0x97, 0x17C0): "NV097_SET_LOGIC_OP",
    (0x97, 0x17C4): "NV097_SET_TWO_SIDE_LIGHT_EN",
    (0x97, 0x17C8): "NV097_CLEAR_REPORT_VALUE",
    (0x97, 0x17CC): "NV097_SET_ZPASS_PIXEL_COUNT_ENABLE",
    (0x97, 0x17D0): "NV097_GET_REPORT",
    (0x97, 0x17E0): "NV097_SET_EYE_DIRECTION",
    (0x97, 0x17F8): "NV097_SET_SHADER_CLIP_PLANE_MODE",
    (0x97, 0x17FC): "NV097_SET_BEGIN_END",
    (0x97, 0x180): "NV097_SET_CONTEXT_DMA_NOTIFIES",
    (0x97, 0x1800): "NV097_ARRAY_ELEMENT16",
    (0x97, 0x1808): "NV097_ARRAY_ELEMENT32",
    (0x97, 0x1810): "NV097_DRAW_ARRAYS",
    (0x97, 0x1818): "NV097_INLINE_ARRAY",
    (0x97, 0x181C): "NV097_SET_EYE_VECTOR",
    (0x97, 0x184): "NV097_SET_CONTEXT_DMA_A",
    (0x97, 0x188): "NV097_SET_CONTEXT_DMA_B",
    (0x97, 0x1880): "NV097_SET_VERTEX_DATA2F_M[pos][x][0]",
    (0x97, 0x1884): "NV097_SET_VERTEX_DATA2F_M[pos][y][1]",
    (0x97, 0x1888): "NV097_SET_VERTEX_DATA2F_M[weights][0][2]",
    (0x97, 0x188C): "NV097_SET_VERTEX_DATA2F_M[weights][1][3]",
    (0x97, 0x1890): "NV097_SET_VERTEX_DATA2F_M[normal][x][4]",
    (0x97, 0x1894): "NV097_SET_VERTEX_DATA2F_M[normal][y][5]",
    (0x97, 0x1898): "NV097_SET_VERTEX_DATA2F_M[diffuse][r][6]",
    (0x97, 0x189C): "NV097_SET_VERTEX_DATA2F_M[diffuse][g][7]",
    (0x97, 0x18A0): "NV097_SET_VERTEX_DATA2F_M[specular][r][8]",
    (0x97, 0x18A4): "NV097_SET_VERTEX_DATA2F_M[specular][g][9]",
    (0x97, 0x18A8): "NV097_SET_VERTEX_DATA2F_M[fog_coord][0][10]",
    (0x97, 0x18AC): "NV097_SET_VERTEX_DATA2F_M[fog_coord][1][11]",
    (0x97, 0x18B0): "NV097_SET_VERTEX_DATA2F_M[point_size][0][12]",
    (0x97, 0x18B4): "NV097_SET_VERTEX_DATA2F_M[point_size][1][13]",
    (0x97, 0x18B8): "NV097_SET_VERTEX_DATA2F_M[back_diffuse][r][14]",
    (0x97, 0x18BC): "NV097_SET_VERTEX_DATA2F_M[back_diffuse][g][15]",
    (0x97, 0x18C0): "NV097_SET_VERTEX_DATA2F_M[back_specular][r][16]",
    (0x97, 0x18C4): "NV097_SET_VERTEX_DATA2F_M[back_specular][g][17]",
    (0x97, 0x18C8): "NV097_SET_VERTEX_DATA2F_M[tex0][u][18]",
    (0x97, 0x18CC): "NV097_SET_VERTEX_DATA2F_M[tex0][v][19]",
    (0x97, 0x18D0): "NV097_SET_VERTEX_DATA2F_M[tex1][u][20]",
    (0x97, 0x18D4): "NV097_SET_VERTEX_DATA2F_M[tex1][v][21]",
    (0x97, 0x18D8): "NV097_SET_VERTEX_DATA2F_M[tex2][u][22]",
    (0x97, 0x18DC): "NV097_SET_VERTEX_DATA2F_M[tex2][v][23]",
    (0x97, 0x18E0): "NV097_SET_VERTEX_DATA2F_M[tex3][u][24]",
    (0x97, 0x18E4): "NV097_SET_VERTEX_DATA2F_M[tex3][v][25]",
    (0x97, 0x18E8): "NV097_SET_VERTEX_DATA2F_M[13][0][26]",
    (0x97, 0x18EC): "NV097_SET_VERTEX_DATA2F_M[13][1][27]",
    (0x97, 0x18F0): "NV097_SET_VERTEX_DATA2F_M[14][0][28]",
    (0x97, 0x18F4): "NV097_SET_VERTEX_DATA2F_M[14][1][29]",
    (0x97, 0x18F8): "NV097_SET_VERTEX_DATA2F_M[15][0][30]",
    (0x97, 0x18FC): "NV097_SET_VERTEX_DATA2F_M[15][1][31]",
    (0x97, 0x190): "NV097_SET_CONTEXT_DMA_STATE",
    (0x97, 0x1900): "NV097_SET_VERTEX_DATA2S[pos][x|y][0]",
    (0x97, 0x1904): "NV097_SET_VERTEX_DATA2S[weights][0|1][1]",
    (0x97, 0x1908): "NV097_SET_VERTEX_DATA2S[normal][x|y][2]",
    (0x97, 0x190C): "NV097_SET_VERTEX_DATA2S[diffuse][r|g][3]",
    (0x97, 0x1910): "NV097_SET_VERTEX_DATA2S[specular][r|g][4]",
    (0x97, 0x1914): "NV097_SET_VERTEX_DATA2S[fog_coord][0|1][5]",
    (0x97, 0x1918): "NV097_SET_VERTEX_DATA2S[point_size][0|1][6]",
    (0x97, 0x191C): "NV097_SET_VERTEX_DATA2S[back_diffuse][r|g][7]",
    (0x97, 0x1920): "NV097_SET_VERTEX_DATA2S[back_specular][r|g][8]",
    (0x97, 0x1924): "NV097_SET_VERTEX_DATA2S[tex0][u|v][9]",
    (0x97, 0x1928): "NV097_SET_VERTEX_DATA2S[tex1][u|v][10]",
    (0x97, 0x192C): "NV097_SET_VERTEX_DATA2S[tex2][u|v][11]",
    (0x97, 0x1930): "NV097_SET_VERTEX_DATA2S[tex3][u|v][12]",
    (0x97, 0x1934): "NV097_SET_VERTEX_DATA2S[13][0|1][13]",
    (0x97, 0x1938): "NV097_SET_VERTEX_DATA2S[14][0|1][14]",
    (0x97, 0x193C): "NV097_SET_VERTEX_DATA2S[15][0|1][15]",
    (0x97, 0x194): "NV097_SET_CONTEXT_DMA_COLOR",
    (0x97, 0x1940): "NV097_SET_VERTEX_DATA4UB[pos]",
    (0x97, 0x1944): "NV097_SET_VERTEX_DATA4UB[weights]",
    (0x97, 0x1948): "NV097_SET_VERTEX_DATA4UB[normal]",
    (0x97, 0x194C): "NV097_SET_VERTEX_DATA4UB[diffuse]",
    (0x97, 0x1950): "NV097_SET_VERTEX_DATA4UB[specular]",
    (0x97, 0x1954): "NV097_SET_VERTEX_DATA4UB[fog_coord]",
    (0x97, 0x1958): "NV097_SET_VERTEX_DATA4UB[point_size]",
    (0x97, 0x195C): "NV097_SET_VERTEX_DATA4UB[back_diffuse]",
    (0x97, 0x1960): "NV097_SET_VERTEX_DATA4UB[back_specular]",
    (0x97, 0x1964): "NV097_SET_VERTEX_DATA4UB[tex0]",
    (0x97, 0x1968): "NV097_SET_VERTEX_DATA4UB[tex1]",
    (0x97, 0x196C): "NV097_SET_VERTEX_DATA4UB[tex2]",
    (0x97, 0x1970): "NV097_SET_VERTEX_DATA4UB[tex3]",
    (0x97, 0x1974): "NV097_SET_VERTEX_DATA4UB[13]",
    (0x97, 0x1978): "NV097_SET_VERTEX_DATA4UB[14]",
    (0x97, 0x197C): "NV097_SET_VERTEX_DATA4UB[15]",
    (0x97, 0x198): "NV097_SET_CONTEXT_DMA_ZETA",
    (0x97, 0x1980): "NV097_SET_VERTEX_DATA4S_M[pos][x|y][0]",
    (0x97, 0x1984): "NV097_SET_VERTEX_DATA4S_M[pos][z|w][1]",
    (0x97, 0x1988): "NV097_SET_VERTEX_DATA4S_M[weights][0|1][2]",
    (0x97, 0x198C): "NV097_SET_VERTEX_DATA4S_M[weights][2|3][3]",
    (0x97, 0x1990): "NV097_SET_VERTEX_DATA4S_M[normal][x|y][4]",
    (0x97, 0x1994): "NV097_SET_VERTEX_DATA4S_M[normal][z|w][5]",
    (0x97, 0x1998): "NV097_SET_VERTEX_DATA4S_M[diffuse][r|g][6]",
    (0x97, 0x199C): "NV097_SET_VERTEX_DATA4S_M[diffuse][b|a][7]",
    (0x97, 0x19A0): "NV097_SET_VERTEX_DATA4S_M[specular][r|g][8]",
    (0x97, 0x19A4): "NV097_SET_VERTEX_DATA4S_M[specular][b|a][9]",
    (0x97, 0x19A8): "NV097_SET_VERTEX_DATA4S_M[fog_coord][0|1][10]",
    (0x97, 0x19AC): "NV097_SET_VERTEX_DATA4S_M[fog_coord][2|3][11]",
    (0x97, 0x19B0): "NV097_SET_VERTEX_DATA4S_M[point_size][0|1][12]",
    (0x97, 0x19B4): "NV097_SET_VERTEX_DATA4S_M[point_size][2|3][13]",
    (0x97, 0x19B8): "NV097_SET_VERTEX_DATA4S_M[back_diffuse][r|g][14]",
    (0x97, 0x19BC): "NV097_SET_VERTEX_DATA4S_M[back_diffuse][b|a][15]",
    (0x97, 0x19C): "NV097_SET_CONTEXT_DMA_VERTEX_A",
    (0x97, 0x19C0): "NV097_SET_VERTEX_DATA4S_M[back_specular][r|g][16]",
    (0x97, 0x19C4): "NV097_SET_VERTEX_DATA4S_M[back_specular][b|a][17]",
    (0x97, 0x19C8): "NV097_SET_VERTEX_DATA4S_M[tex0][u|v][18]",
    (0x97, 0x19CC): "NV097_SET_VERTEX_DATA4S_M[tex0][2|3][19]",
    (0x97, 0x19D0): "NV097_SET_VERTEX_DATA4S_M[tex1][u|v][20]",
    (0x97, 0x19D4): "NV097_SET_VERTEX_DATA4S_M[tex1][2|3][21]",
    (0x97, 0x19D8): "NV097_SET_VERTEX_DATA4S_M[tex2][u|v][22]",
    (0x97, 0x19DC): "NV097_SET_VERTEX_DATA4S_M[tex2][2|3][23]",
    (0x97, 0x19E0): "NV097_SET_VERTEX_DATA4S_M[tex3][u|v][24]",
    (0x97, 0x19E4): "NV097_SET_VERTEX_DATA4S_M[tex3][2|3][25]",
    (0x97, 0x19E8): "NV097_SET_VERTEX_DATA4S_M[13][0|1][26]",
    (0x97, 0x19EC): "NV097_SET_VERTEX_DATA4S_M[13][2|3][27]",
    (0x97, 0x19F0): "NV097_SET_VERTEX_DATA4S_M[14][0|1][28]",
    (0x97, 0x19F4): "NV097_SET_VERTEX_DATA4S_M[14][2|3][29]",
    (0x97, 0x19F8): "NV097_SET_VERTEX_DATA4S_M[15][0|1][30]",
    (0x97, 0x19FC): "NV097_SET_VERTEX_DATA4S_M[15][2|3][31]",
    (0x97, 0x1A0): "NV097_SET_CONTEXT_DMA_VERTEX_B",
    (0x97, 0x1A00): "NV097_SET_VERTEX_DATA4F_M[pos][x][0]",
    (0x97, 0x1A04): "NV097_SET_VERTEX_DATA4F_M[pos][y][1]",
    (0x97, 0x1A08): "NV097_SET_VERTEX_DATA4F_M[pos][z][2]",
    (0x97, 0x1A0C): "NV097_SET_VERTEX_DATA4F_M[pos][w][3]",
    (0x97, 0x1A10): "NV097_SET_VERTEX_DATA4F_M[weights][0][4]",
    (0x97, 0x1A14): "NV097_SET_VERTEX_DATA4F_M[weights][1][5]",
    (0x97, 0x1A18): "NV097_SET_VERTEX_DATA4F_M[weights][2][6]",
    (0x97, 0x1A1C): "NV097_SET_VERTEX_DATA4F_M[weights][3][7]",
    (0x97, 0x1A20): "NV097_SET_VERTEX_DATA4F_M[normal][x][8]",
    (0x97, 0x1A24): "NV097_SET_VERTEX_DATA4F_M[normal][y][9]",
    (0x97, 0x1A28): "NV097_SET_VERTEX_DATA4F_M[normal][z][10]",
    (0x97, 0x1A2C): "NV097_SET_VERTEX_DATA4F_M[normal][w][11]",
    (0x97, 0x1A30): "NV097_SET_VERTEX_DATA4F_M[diffuse][r][12]",
    (0x97, 0x1A34): "NV097_SET_VERTEX_DATA4F_M[diffuse][g][13]",
    (0x97, 0x1A38): "NV097_SET_VERTEX_DATA4F_M[diffuse][b][14]",
    (0x97, 0x1A3C): "NV097_SET_VERTEX_DATA4F_M[diffuse][a][15]",
    (0x97, 0x1A4): "NV097_SET_CONTEXT_DMA_SEMAPHORE",
    (0x97, 0x1A40): "NV097_SET_VERTEX_DATA4F_M[specular][r][16]",
    (0x97, 0x1A44): "NV097_SET_VERTEX_DATA4F_M[specular][g][17]",
    (0x97, 0x1A48): "NV097_SET_VERTEX_DATA4F_M[specular][b][18]",
    (0x97, 0x1A4C): "NV097_SET_VERTEX_DATA4F_M[specular][a][19]",
    (0x97, 0x1A50): "NV097_SET_VERTEX_DATA4F_M[fog_coord][0][20]",
    (0x97, 0x1A54): "NV097_SET_VERTEX_DATA4F_M[fog_coord][1][21]",
    (0x97, 0x1A58): "NV097_SET_VERTEX_DATA4F_M[fog_coord][2][22]",
    (0x97, 0x1A5C): "NV097_SET_VERTEX_DATA4F_M[fog_coord][3][23]",
    (0x97, 0x1A60): "NV097_SET_VERTEX_DATA4F_M[point_size][0][24]",
    (0x97, 0x1A64): "NV097_SET_VERTEX_DATA4F_M[point_size][1][25]",
    (0x97, 0x1A68): "NV097_SET_VERTEX_DATA4F_M[point_size][2][26]",
    (0x97, 0x1A6C): "NV097_SET_VERTEX_DATA4F_M[point_size][3][27]",
    (0x97, 0x1A70): "NV097_SET_VERTEX_DATA4F_M[back_diffuse][r][28]",
    (0x97, 0x1A74): "NV097_SET_VERTEX_DATA4F_M[back_diffuse][g][29]",
    (0x97, 0x1A78): "NV097_SET_VERTEX_DATA4F_M[back_diffuse][b][30]",
    (0x97, 0x1A7C): "NV097_SET_VERTEX_DATA4F_M[back_diffuse][a][31]",
    (0x97, 0x1A8): "NV097_SET_CONTEXT_DMA_REPORT",
    (0x97, 0x1A80): "NV097_SET_VERTEX_DATA4F_M[back_specular][r][32]",
    (0x97, 0x1A84): "NV097_SET_VERTEX_DATA4F_M[back_specular][g][33]",
    (0x97, 0x1A88): "NV097_SET_VERTEX_DATA4F_M[back_specular][b][34]",
    (0x97, 0x1A8C): "NV097_SET_VERTEX_DATA4F_M[back_specular][a][35]",
    (0x97, 0x1A90): "NV097_SET_VERTEX_DATA4F_M[tex0][u][36]",
    (0x97, 0x1A94): "NV097_SET_VERTEX_DATA4F_M[tex0][v][37]",
    (0x97, 0x1A98): "NV097_SET_VERTEX_DATA4F_M[tex0][2][38]",
    (0x97, 0x1A9C): "NV097_SET_VERTEX_DATA4F_M[tex0][3][39]",
    (0x97, 0x1AA0): "NV097_SET_VERTEX_DATA4F_M[tex1][u][40]",
    (0x97, 0x1AA4): "NV097_SET_VERTEX_DATA4F_M[tex1][v][41]",
    (0x97, 0x1AA8): "NV097_SET_VERTEX_DATA4F_M[tex1][2][42]",
    (0x97, 0x1AAC): "NV097_SET_VERTEX_DATA4F_M[tex1][3][43]",
    (0x97, 0x1AB0): "NV097_SET_VERTEX_DATA4F_M[tex2][u][44]",
    (0x97, 0x1AB4): "NV097_SET_VERTEX_DATA4F_M[tex2][v][45]",
    (0x97, 0x1AB8): "NV097_SET_VERTEX_DATA4F_M[tex2][2][46]",
    (0x97, 0x1ABC): "NV097_SET_VERTEX_DATA4F_M[tex2][3][47]",
    (0x97, 0x1AC0): "NV097_SET_VERTEX_DATA4F_M[tex3][u][48]",
    (0x97, 0x1AC4): "NV097_SET_VERTEX_DATA4F_M[tex3][v][49]",
    (0x97, 0x1AC8): "NV097_SET_VERTEX_DATA4F_M[tex3][2][50]",
    (0x97, 0x1ACC): "NV097_SET_VERTEX_DATA4F_M[tex3][3][51]",
    (0x97, 0x1AD0): "NV097_SET_VERTEX_DATA4F_M[13][0][52]",
    (0x97, 0x1AD4): "NV097_SET_VERTEX_DATA4F_M[13][1][53]",
    (0x97, 0x1AD8): "NV097_SET_VERTEX_DATA4F_M[13][2][54]",
    (0x97, 0x1ADC): "NV097_SET_VERTEX_DATA4F_M[13][3][55]",
    (0x97, 0x1AE0): "NV097_SET_VERTEX_DATA4F_M[14][0][56]",
    (0x97, 0x1AE4): "NV097_SET_VERTEX_DATA4F_M[14][1][57]",
    (0x97, 0x1AE8): "NV097_SET_VERTEX_DATA4F_M[14][2][58]",
    (0x97, 0x1AEC): "NV097_SET_VERTEX_DATA4F_M[14][3][59]",
    (0x97, 0x1AF0): "NV097_SET_VERTEX_DATA4F_M[15][0][60]",
    (0x97, 0x1AF4): "NV097_SET_VERTEX_DATA4F_M[15][1][61]",
    (0x97, 0x1AF8): "NV097_SET_VERTEX_DATA4F_M[15][2][62]",
    (0x97, 0x1AFC): "NV097_SET_VERTEX_DATA4F_M[15][3][63]",
    (0x97, 0x1B00): "NV097_SET_TEXTURE_OFFSET",
    (0x97, 0x1B04): "NV097_SET_TEXTURE_FORMAT",
    (0x97, 0x1B08): "NV097_SET_TEXTURE_ADDRESS",
    (0x97, 0x1B0C): "NV097_SET_TEXTURE_CONTROL0",
    (0x97, 0x1B10): "NV097_SET_TEXTURE_CONTROL1",
    (0x97, 0x1B14): "NV097_SET_TEXTURE_FILTER",
    (0x97, 0x1B1C): "NV097_SET_TEXTURE_IMAGE_RECT",
    (0x97, 0x1B20): "NV097_SET_TEXTURE_PALETTE",
    (0x97, 0x1B24): "NV097_SET_TEXTURE_BORDER_COLOR",
    (0x97, 0x1B28): "NV097_SET_TEXTURE_SET_BUMP_ENV_MAT",
    (0x97, 0x1B38): "NV097_SET_TEXTURE_SET_BUMP_ENV_SCALE",
    (0x97, 0x1B3C): "NV097_SET_TEXTURE_SET_BUMP_ENV_OFFSET",
    (0x97, 0x1D6C): "NV097_SET_SEMAPHORE_OFFSET",
    (0x97, 0x1D70): "NV097_BACK_END_WRITE_SEMAPHORE_RELEASE",
    (0x97, 0x1D78): "NV097_SET_ZMIN_MAX_CONTROL",
    (0x97, 0x1D7C): "NV097_SET_ANTI_ALIASING_CONTROL",
    (0x97, 0x1D80): "NV097_SET_COMPRESS_ZBUFFER_EN",
    (0x97, 0x1D84): "NV097_SET_OCCLUDE_ZSTENCIL_EN",
    (0x97, 0x1D8C): "NV097_SET_ZSTENCIL_CLEAR_VALUE",
    (0x97, 0x1D90): "NV097_SET_COLOR_CLEAR_VALUE",
    (0x97, 0x1D94): "NV097_CLEAR_SURFACE",
    (0x97, 0x1D98): "NV097_SET_CLEAR_RECT_HORIZONTAL",
    (0x97, 0x1D9C): "NV097_SET_CLEAR_RECT_VERTICAL",
    (0x97, 0x1E20): "NV097_SET_SPECULAR_FOG_FACTOR",
    (0x97, 0x1E28): "NV097_SET_BACK_SPECULAR_PARAMS",
    (0x97, 0x1E40): "NV097_SET_COMBINER_COLOR_OCW",
    (0x97, 0x1E60): "NV097_SET_COMBINER_CONTROL",
    (0x97, 0x1E68): "NV097_SET_SHADOW_ZSLOPE_THRESHOLD",
    (0x97, 0x1E6C): "NV097_SET_SHADOW_DEPTH_FUNC",
    (0x97, 0x1E70): "NV097_SET_SHADER_STAGE_PROGRAM",
    (0x97, 0x1E74): "NV097_SET_DOT_RGBMAPPING",
    (0x97, 0x1E78): "NV097_SET_SHADER_OTHER_STAGE_INPUT",
    (0x97, 0x1E80): "NV097_SET_TRANSFORM_DATA",
    (0x97, 0x1E90): "NV097_LAUNCH_TRANSFORM_PROGRAM",
    (0x97, 0x1E94): "NV097_SET_TRANSFORM_EXECUTION_MODE",
    (0x97, 0x1E98): "NV097_SET_TRANSFORM_PROGRAM_CXT_WRITE_EN",
    (0x97, 0x1E9C): "NV097_SET_TRANSFORM_PROGRAM_LOAD",
    (0x97, 0x1EA0): "NV097_SET_TRANSFORM_PROGRAM_START",
    (0x97, 0x1EA4): "NV097_SET_TRANSFORM_CONSTANT_LOAD",
    (0x97, 0x20): "NV097_SET_STIPPLE_PATERN_SIZE",
    (0x97, 0x200): "NV097_SET_SURFACE_CLIP_HORIZONTAL",
    (0x97, 0x204): "NV097_SET_SURFACE_CLIP_VERTICAL",
    (0x97, 0x208): "NV097_SET_SURFACE_FORMAT",
    (0x97, 0x20C): "NV097_SET_SURFACE_PITCH",
    (0x97, 0x210): "NV097_SET_SURFACE_COLOR_OFFSET",
    (0x97, 0x214): "NV097_SET_SURFACE_ZETA_OFFSET",
    (0x97, 0x260): "NV097_SET_COMBINER_ALPHA_ICW",
    (0x97, 0x288): "NV097_SET_COMBINER_SPECULAR_FOG_CW0",
    (0x97, 0x28C): "NV097_SET_COMBINER_SPECULAR_FOG_CW1",
    (0x97, 0x290): "NV097_SET_CONTROL0",
    (0x97, 0x294): "NV097_SET_LIGHT_CONTROL",
    (0x97, 0x298): "NV097_SET_COLOR_MATERIAL",
    (0x97, 0x29C): "NV097_SET_FOG_MODE",
    (0x97, 0x2A0): "NV097_SET_FOG_GEN_MODE",
    (0x97, 0x2A4): "NV097_SET_FOG_ENABLE",
    (0x97, 0x2A8): "NV097_SET_FOG_COLOR",
    (0x97, 0x2B4): "NV097_SET_WINDOW_CLIP_TYPE",
    (0x97, 0x2C0): "NV097_SET_WINDOW_CLIP_HORIZONTAL",
    (0x97, 0x2E0): "NV097_SET_WINDOW_CLIP_VERTICAL",
    (0x97, 0x300): "NV097_SET_ALPHA_TEST_ENABLE",
    (0x97, 0x304): "NV097_SET_BLEND_ENABLE",
    (0x97, 0x308): "NV097_SET_CULL_FACE_ENABLE",
    (0x97, 0x30C): "NV097_SET_DEPTH_TEST_ENABLE",
    (0x97, 0x310): "NV097_SET_DITHER_ENABLE",
    (0x97, 0x314): "NV097_SET_LIGHTING_ENABLE",
    (0x97, 0x318): "NV097_SET_POINT_PARAMS_ENABLE",
    (0x97, 0x31C): "NV097_SET_POINT_SMOOTH_ENABLE",
    (0x97, 0x320): "NV097_SET_LINE_SMOOTH_ENABLE",
    (0x97, 0x324): "NV097_SET_POLY_SMOOTH_ENABLE",
    (0x97, 0x328): "NV097_SET_SKIN_MODE",
    (0x97, 0x32C): "NV097_SET_STENCIL_TEST_ENABLE",
    (0x97, 0x330): "NV097_SET_POLY_OFFSET_POINT_ENABLE",
    (0x97, 0x334): "NV097_SET_POLY_OFFSET_LINE_ENABLE",
    (0x97, 0x338): "NV097_SET_POLY_OFFSET_FILL_ENABLE",
    (0x97, 0x33C): "NV097_SET_ALPHA_FUNC",
    (0x97, 0x340): "NV097_SET_ALPHA_REF",
    (0x97, 0x344): "NV097_SET_BLEND_FUNC_SFACTOR",
    (0x97, 0x348): "NV097_SET_BLEND_FUNC_DFACTOR",
    (0x97, 0x34C): "NV097_SET_BLEND_COLOR",
    (0x97, 0x350): "NV097_SET_BLEND_EQUATION",
    (0x97, 0x354): "NV097_SET_DEPTH_FUNC",
    (0x97, 0x358): "NV097_SET_COLOR_MASK",
    (0x97, 0x35C): "NV097_SET_DEPTH_MASK",
    (0x97, 0x360): "NV097_SET_STENCIL_MASK",
    (0x97, 0x364): "NV097_SET_STENCIL_FUNC",
    (0x97, 0x368): "NV097_SET_STENCIL_FUNC_REF",
    (0x97, 0x36C): "NV097_SET_STENCIL_FUNC_MASK",
    (0x97, 0x370): "NV097_SET_STENCIL_OP_FAIL",
    (0x97, 0x374): "NV097_SET_STENCIL_OP_ZFAIL",
    (0x97, 0x378): "NV097_SET_STENCIL_OP_ZPASS",
    (0x97, 0x37C): "NV097_SET_SHADE_MODE",
    (0x97, 0x380): "NV097_SET_LINE_WIDTH",
    (0x97, 0x384): "NV097_SET_POLYGON_OFFSET_SCALE_FACTOR",
    (0x97, 0x388): "NV097_SET_POLYGON_OFFSET_BIAS",
    (0x97, 0x38C): "NV097_SET_FRONT_POLYGON_MODE",
    (0x97, 0x390): "NV097_SET_BACK_POLYGON_MODE",
    (0x97, 0x394): "NV097_SET_CLIP_MIN",
    (0x97, 0x398): "NV097_SET_CLIP_MAX",
    (0x97, 0x39C): "NV097_SET_CULL_FACE",
    (0x97, 0x3A0): "NV097_SET_FRONT_FACE",
    (0x97, 0x3A4): "NV097_SET_NORMALIZATION_ENABLE",
    (0x97, 0x3A8): "NV097_SET_MATERIAL_EMISSION",
    (0x97, 0x3B4): "NV097_SET_MATERIAL_ALPHA",
    (0x97, 0x3B8): "NV097_SET_SPECULAR_ENABLE",
    (0x97, 0x3BC): "NV097_SET_LIGHT_ENABLE_MASK",
    (0x97, 0x3C0): "NV097_SET_TEXGEN_S",
    (0x97, 0x3C4): "NV097_SET_TEXGEN_T",
    (0x97, 0x3C8): "NV097_SET_TEXGEN_R",
    (0x97, 0x3CC): "NV097_SET_TEXGEN_Q",
    (0x97, 0x420): "NV097_SET_TEXTURE_MATRIX_ENABLE",
    (0x97, 0x43C): "NV097_SET_POINT_SIZE",
    (0x97, 0x440): "NV097_SET_PROJECTION_MATRIX",
    (0x97, 0x480): "NV097_SET_MODEL_VIEW_MATRIX",
    (0x97, 0x580): "NV097_SET_INVERSE_MODEL_VIEW_MATRIX",
    (0x97, 0x680): "NV097_SET_COMPOSITE_MATRIX",
    (0x97, 0x6C0): "NV097_SET_TEXTURE_MATRIX",
    (0x97, 0x840): "NV097_SET_TEXGEN_PLANE_S",
    (0x97, 0x850): "NV097_SET_TEXGEN_PLANE_T",
    (0x97, 0x860): "NV097_SET_TEXGEN_PLANE_R",
    (0x97, 0x870): "NV097_SET_TEXGEN_PLANE_Q",
    (0x97, 0x9C0): "NV097_SET_FOG_PARAMS",
    (0x97, 0x9CC): "NV097_SET_TEXGEN_VIEW_MODEL",
    (0x97, 0x9D0): "NV097_SET_FOG_PLANE",
    (0x97, 0x9E0): "NV097_SET_SPECULAR_PARAMS",
    (0x97, 0x9F8): "NV097_SET_SWATH_WIDTH",
    (0x97, 0x9FC): "NV097_SET_FLAT_SHADE_OP",
    (0x97, 0xA10): "NV097_SET_SCENE_AMBIENT_COLOR",
    (0x97, 0xA20): "NV097_SET_VIEWPORT_OFFSET",
    (0x97, 0xA30): "NV097_SET_POINT_PARAMS",
    (0x97, 0xA50): "NV097_SET_EYE_POSITION",
    (0x97, 0xA60): "NV097_SET_COMBINER_FACTOR0",
    (0x97, 0xA80): "NV097_SET_COMBINER_FACTOR1",
    (0x97, 0xAA0): "NV097_SET_COMBINER_ALPHA_OCW",
    (0x97, 0xAC0): "NV097_SET_COMBINER_COLOR_ICW",
    (0x97, 0xAE0): "NV097_SET_COLOR_KEY_COLOR",
    (0x97, 0xAF0): "NV097_SET_VIEWPORT_SCALE",
    (0x97, 0xB00): "NV097_SET_TRANSFORM_PROGRAM",
    (0x97, 0xB80): "NV097_SET_TRANSFORM_CONSTANT",
    (0x97, 0xC00): "NV097_SET_BACK_LIGHT_AMBIENT_COLOR",
    (0x97, 0xC0C): "NV097_SET_BACK_LIGHT_DIFFUSE_COLOR",
    (0x97, 0xC18): "NV097_SET_BACK_LIGHT_SPECULAR_COLOR",
    (0x9F, 0x0): "NV09F_SET_OBJECT",
    (0x9F, 0x184): "NV09F_SET_COLOR_KEY_OBJECT",
    (0x9F, 0x188): "NV09F_SET_CLIP_RECTANGLE_OBJECT",
    (0x9F, 0x18C): "NV09F_SET_PATTERN_OBJECT",
    (0x9F, 0x190): "NV09F_SET_ROP5_OBJECT",
    (0x9F, 0x194): "NV09F_SET_BETA_OBJECT",
    (0x9F, 0x198): "NV09F_SET_BETA_OBJECT4",
    (0x9F, 0x19C): "NV09F_SET_CONTEXT_SURFACES",
    (0x9F, 0x2FC): "NV09F_SET_OPERATION",
    (0x9F, 0x300): "NV09F_CONTROL_POINT_IN",
    (0x9F, 0x304): "NV09F_CONTROL_POINT_OUT",
    (0x9F, 0x308): "NV09F_SIZE",
}


def _process_passthrough(_nv_class, _nv_op, nv_param) -> str:
    """Simply passes through the param value."""
    return f"0x{nv_param:08X} => {nv_param}"


def _process_float_param(_nv_class, _nv_op, nv_param) -> str:
    """Treats the param value as an IEEE float"""
    float_val = struct.unpack("f", nv_param.to_bytes(4, byteorder=sys.byteorder))
    return f"0x{nv_param:08X} => {float_val[0]}"


def _process_boolean_param(_nv_class, _nv_op, nv_param) -> str:
    """Treats the param value as a boolean."""
    if nv_param == 0:
        return "FALSE"
    if nv_param == 1:
        return "TRUE"

    return f"0x{nv_param:08X} => TRUE?"


def _generate_process_double_uint16(low, high):
    def _process_double_uint16(_nv_class, _nv_op, nv_param):
        low_val = nv_param & 0xFFFF
        high_val = (nv_param >> 16) & 0xFFFF
        return f"0x{nv_param:08X} {{{low}:{low_val}, {high}:{high_val}}}"

    return _process_double_uint16


def _expand_processors(
    processors: dict[int, dict[int | StateArray | StructStateArray, ProcessorFunc]],
) -> tuple[dict[tuple[int, int], ProcessorFunc], dict[tuple[int, int], str]]:
    """Flattens processor mapping into processor funcs and names."""
    flat_processors: dict[tuple[int, int], ProcessorFunc] = {}
    flat_names: dict[tuple[int, int], str] = {}

    def _insert(key: tuple[int, int], processor: ProcessorFunc, name: str):
        if key in flat_names:
            msg = f"Colliding key '{key}': old name: '{flat_names[key]}' new name: '{name}'"
            raise ValueError(msg)

        flat_processors[key] = processor
        flat_names[key] = name

    for nv_class, operations in processors.items():
        for nv_op_info, processor in operations.items():
            cmd_type = type(nv_op_info)
            base_key = (nv_class, nv_op_info)
            if cmd_type is int:
                _insert(base_key, processor, _SPARSE_NAME_MAP[base_key])
                continue

            if cmd_type is StateArray:
                base = nv_op_info.base
                name = _SPARSE_NAME_MAP[(nv_class, base)]
                stride = nv_op_info.stride
                count = nv_op_info.num_elements
                for i in range(count):
                    _insert((nv_class, base + i * stride), processor, f"{name}[{i}]")
                continue

            if cmd_type is StructStateArray:
                base = nv_op_info.base
                name = _SPARSE_NAME_MAP[(nv_class, base)]
                struct_stride = nv_op_info.struct_stride
                struct_count = nv_op_info.struct_count
                stride = nv_op_info.stride
                count = nv_op_info.num_elements

                for struct in range(struct_count):
                    for i in range(count):
                        _insert((nv_class, base + i * stride), processor, f"{name}@{struct}[{i}]")
                    base += struct_stride
                continue

            msg = f"Unsupported processor map entry '{nv_op_info}'"
            raise ValueError(msg)

    return flat_processors, flat_names


# Custom parser functions.


def ParseNv062SetColorFormat(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV062_SET_COLOR_FORMAT command."""
    _VALUES = {
        1: "LE_Y8",
        4: "LE_R5G6B5",
        6: "LE_X8R8G8B8_Z8R8G8B8",
        7: "LE_X8R8G8B8",
        10: "LE_A8R8G8B8",
        11: "LE_Y32",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097ClearReportValue(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_CLEAR_REPORT_VALUE command."""
    results: list[str] = []
    field_val = nv_param & 0xFFFFFFFF
    field_map = {
        0x1: "NV097_CLEAR_REPORT_VALUE_TYPE_ZPASS_PIXEL_CNT",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_CLEAR_REPORT_VALUE_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_CLEAR_REPORT_VALUE_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097ClearSurface(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_CLEAR_SURFACE command."""
    _VALUES = {
        1: "Z",
        2: "STENCIL",
        16: "R",
        32: "G",
        64: "B",
        128: "A",
        240: "COLOR",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097GetReport(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_GET_REPORT command."""
    results: list[str] = []
    field_val = nv_param & 0xFFFFFF
    results.append(f"NV097_GET_REPORT_:0x{field_val:X}")
    field_val = nv_param & 0xFF000000
    field_map = {
        0x1: "NV097_GET_REPORT_TYPE_ZPASS_PIXEL_CNT",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_GET_REPORT_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_GET_REPORT_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetAlphaFunc(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_ALPHA_FUNC command."""
    _VALUES = {
        512: "V_NEVER",
        513: "V_LESS",
        514: "V_EQUAL",
        515: "V_LEQUAL",
        516: "V_GREATER",
        517: "V_NOTEQUAL",
        518: "V_GEQUAL",
        519: "V_ALWAYS",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetAntiAliasingControl(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_ANTI_ALIASING_CONTROL command."""
    _VALUES = {
        1: "ENABLE",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetBeginEnd(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_BEGIN_END command."""
    _VALUES = {
        0: "OP_END",
        1: "OP_POINTS",
        2: "OP_LINES",
        3: "OP_LINE_LOOP",
        4: "OP_LINE_STRIP",
        5: "OP_TRIANGLES",
        6: "OP_TRIANGLE_STRIP",
        7: "OP_TRIANGLE_FAN",
        8: "OP_QUADS",
        9: "OP_QUAD_STRIP",
        10: "OP_POLYGON",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetBlendEquation(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_BLEND_EQUATION command."""
    results: list[str] = []
    field_val = nv_param & 0x800A
    results.append(f"NV097_SET_BLEND_EQUATION_:0x{field_val:X}")
    field_val = nv_param & 0x800B
    field_map = {
        0xF005: "NV097_SET_BLEND_EQUATION_V_FUNC_REVERSE_SUBTRACT_SIGNED",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_BLEND_EQUATION_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_BLEND_EQUATION_:0x{field_val:X}")
    field_val = nv_param & 0x8006
    field_map = {
        0xF006: "NV097_SET_BLEND_EQUATION_V_FUNC_ADD_SIGNED",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_BLEND_EQUATION_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_BLEND_EQUATION_:0x{field_val:X}")
    field_val = nv_param & 0x8007
    results.append(f"NV097_SET_BLEND_EQUATION_:0x{field_val:X}")
    field_val = nv_param & 0x8008
    results.append(f"NV097_SET_BLEND_EQUATION_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetBlendFuncDfactor(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_BLEND_FUNC_DFACTOR command."""
    results: list[str] = []
    field_val = nv_param & 0x0
    results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x1
    field_map = {
        0x301: "NV097_SET_BLEND_FUNC_DFACTOR_V_ONE_MINUS_SRC_COLOR",
        0x303: "NV097_SET_BLEND_FUNC_DFACTOR_V_ONE_MINUS_SRC_ALPHA",
        0x305: "NV097_SET_BLEND_FUNC_DFACTOR_V_ONE_MINUS_DST_ALPHA",
        0x307: "NV097_SET_BLEND_FUNC_DFACTOR_V_ONE_MINUS_DST_COLOR",
        0x8002: "NV097_SET_BLEND_FUNC_DFACTOR_V_ONE_MINUS_CONSTANT_COLOR",
        0x8004: "NV097_SET_BLEND_FUNC_DFACTOR_V_ONE_MINUS_CONSTANT_ALPHA",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_BLEND_FUNC_DFACTOR_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x300
    results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x302
    field_map = {
        0x308: "NV097_SET_BLEND_FUNC_DFACTOR_V_SRC_ALPHA_SATURATE",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_BLEND_FUNC_DFACTOR_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x304
    results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x306
    results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x8001
    results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x8003
    results.append(f"NV097_SET_BLEND_FUNC_DFACTOR_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetBlendFuncSfactor(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_BLEND_FUNC_SFACTOR command."""
    results: list[str] = []
    field_val = nv_param & 0x0
    results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x1
    field_map = {
        0x301: "NV097_SET_BLEND_FUNC_SFACTOR_V_ONE_MINUS_SRC_COLOR",
        0x303: "NV097_SET_BLEND_FUNC_SFACTOR_V_ONE_MINUS_SRC_ALPHA",
        0x305: "NV097_SET_BLEND_FUNC_SFACTOR_V_ONE_MINUS_DST_ALPHA",
        0x307: "NV097_SET_BLEND_FUNC_SFACTOR_V_ONE_MINUS_DST_COLOR",
        0x8002: "NV097_SET_BLEND_FUNC_SFACTOR_V_ONE_MINUS_CONSTANT_COLOR",
        0x8004: "NV097_SET_BLEND_FUNC_SFACTOR_V_ONE_MINUS_CONSTANT_ALPHA",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_BLEND_FUNC_SFACTOR_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x300
    results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x302
    field_map = {
        0x308: "NV097_SET_BLEND_FUNC_SFACTOR_V_SRC_ALPHA_SATURATE",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_BLEND_FUNC_SFACTOR_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x304
    results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x306
    results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x8001
    results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    field_val = nv_param & 0x8003
    results.append(f"NV097_SET_BLEND_FUNC_SFACTOR_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetCullFace(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_CULL_FACE command."""
    results: list[str] = []
    field_val = nv_param & 0x404
    field_map = {
        0x408: "NV097_SET_CULL_FACE_V_FRONT_AND_BACK",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_CULL_FACE_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_CULL_FACE_:0x{field_val:X}")
    field_val = nv_param & 0x405
    results.append(f"NV097_SET_CULL_FACE_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetDepthFunc(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_DEPTH_FUNC command."""
    _VALUES = {
        512: "V_NEVER",
        513: "V_LESS",
        514: "V_EQUAL",
        515: "V_LEQUAL",
        516: "V_GREATER",
        517: "V_NOTEQUAL",
        518: "V_GEQUAL",
        519: "V_ALWAYS",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetFlatShadeOp(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_FLAT_SHADE_OP command."""
    _VALUES = {
        0: "VERTEX_LAST",
        1: "VERTEX_FIRST",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetFogColor(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_FOG_COLOR command."""
    _VALUES = {
        255: "RED",
        65280: "GREEN",
        16711680: "BLUE",
        4278190080: "ALPHA",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetFogGenMode(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_FOG_GEN_MODE command."""
    _VALUES = {
        0: "V_SPEC_ALPHA",
        1: "V_RADIAL",
        2: "V_PLANAR",
        3: "V_ABS_PLANAR",
        6: "V_FOG_X",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetFogMode(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_FOG_MODE command."""
    results: list[str] = []
    field_val = nv_param & 0x2601
    field_map = {
        0x804: "NV097_SET_FOG_MODE_V_LINEAR_ABS",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_FOG_MODE_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_FOG_MODE_:0x{field_val:X}")
    field_val = nv_param & 0x800
    field_map = {
        0x802: "NV097_SET_FOG_MODE_V_EXP_ABS",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_FOG_MODE_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_FOG_MODE_:0x{field_val:X}")
    field_val = nv_param & 0x801
    field_map = {
        0x803: "NV097_SET_FOG_MODE_V_EXP2_ABS",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_FOG_MODE_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_FOG_MODE_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetFrontFace(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_FRONT_FACE command."""
    _VALUES = {
        2304: "V_CW",
        2305: "V_CCW",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetFrontPolygonMode(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_FRONT_POLYGON_MODE command."""
    _VALUES = {
        6912: "V_POINT",
        6913: "V_LINE",
        6914: "V_FILL",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetLineWidth(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_LINE_WIDTH command."""
    _VALUES = {
        504: "MAX_VALUE",
        511: "MASK",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetPointParams(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_POINT_PARAMS command."""
    results: list[str] = []
    field_val = nv_param & 0xA30
    results.append(f"NV097_SET_POINT_PARAMS_:0x{field_val:X}")
    field_val = nv_param & 0xA34
    results.append(f"NV097_SET_POINT_PARAMS_:0x{field_val:X}")
    field_val = nv_param & 0xA38
    results.append(f"NV097_SET_POINT_PARAMS_:0x{field_val:X}")
    field_val = nv_param & 0xA3C
    field_map = {
        0xA40: "NV097_SET_POINT_PARAMS_SIZE_RANGE_DUP_1",
        0xA44: "NV097_SET_POINT_PARAMS_SIZE_RANGE_DUP_2",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_POINT_PARAMS_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_POINT_PARAMS_:0x{field_val:X}")
    field_val = nv_param & 0xA48
    results.append(f"NV097_SET_POINT_PARAMS_:0x{field_val:X}")
    field_val = nv_param & 0xA4C
    results.append(f"NV097_SET_POINT_PARAMS_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetShadeMode(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SHADE_MODE command."""
    _VALUES = {
        7424: "V_FLAT",
        7425: "V_SMOOTH",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetShadeModel(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SHADE_MODEL command."""
    _VALUES = {
        7424: "FLAT",
        7425: "SMOOTH",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetShadowCompareFunc(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SHADOW_COMPARE_FUNC command."""
    _VALUES = {
        0: "NEVER",
        1: "GREATER",
        2: "EQUAL",
        3: "GEQUAL",
        4: "LESS",
        5: "NOTEQUAL",
        6: "LEQUAL",
        7: "ALWAYS",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetShadowDepthFunc(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SHADOW_DEPTH_FUNC command."""
    _VALUES = {
        0: "NEVER",
        1: "LESS",
        2: "EQUAL",
        3: "LEQUAL",
        4: "GREATER",
        5: "NOTEQUAL",
        6: "GEQUAL",
        7: "ALWAYS",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetSkinMode(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SKIN_MODE command."""
    _VALUES = {
        0: "OFF",
        1: "2G",
        2: "2",
        3: "3G",
        4: "3",
        5: "4G",
        6: "4",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetSpecularParams(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SPECULAR_PARAMS command."""
    _VALUES = {
        7720: "BACK",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetStencilOpFail(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_STENCIL_OP_FAIL command."""
    _VALUES = {
        0: "ZERO",
        5386: "INVERT",
        7680: "KEEP",
        7681: "REPLACE",
        7682: "INCRSAT",
        7683: "DECRSAT",
        34055: "INCR",
        34056: "DECR",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetStencilOpZfail(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_STENCIL_OP_ZFAIL command."""
    _VALUES = {
        0: "ZERO",
        5386: "INVERT",
        7680: "KEEP",
        7681: "REPLACE",
        7682: "INCRSAT",
        7683: "DECRSAT",
        34055: "INCR",
        34056: "DECR",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetStencilOpZpass(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_STENCIL_OP_ZPASS command."""
    _VALUES = {
        0: "ZERO",
        5386: "INVERT",
        7680: "KEEP",
        7681: "REPLACE",
        7682: "INCRSAT",
        7683: "DECRSAT",
        34055: "INCR",
        34056: "DECR",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetSurfaceClipHorizontal(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SURFACE_CLIP_HORIZONTAL command."""
    _VALUES = {
        65535: "X",
        4294901760: "WIDTH",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetSurfaceClipVertical(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SURFACE_CLIP_VERTICAL command."""
    _VALUES = {
        65535: "Y",
        4294901760: "HEIGHT",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetSurfacePitch(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_SURFACE_PITCH command."""
    _VALUES = {
        65535: "COLOR",
        4294901760: "ZETA",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetTexgenViewModel(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_TEXGEN_VIEW_MODEL command."""
    _VALUES = {
        0: "LOCAL_VIEWER",
        1: "INFINITE_VIEWER",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetTextureImageRect(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_TEXTURE_IMAGE_RECT command."""
    _VALUES = {
        65535: "HEIGHT",
        4294901760: "WIDTH",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv097SetTransformExecutionMode(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_TRANSFORM_EXECUTION_MODE command."""
    results: list[str] = []
    field_val = nv_param & 0x3
    field_map = {
        0x0: "NV097_SET_TRANSFORM_EXECUTION_MODE_MODE_FIXED",
        0x2: "NV097_SET_TRANSFORM_EXECUTION_MODE_MODE_PROGRAM",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_TRANSFORM_EXECUTION_MODE_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_TRANSFORM_EXECUTION_MODE_:0x{field_val:X}")
    field_val = nv_param & 0xFFFFFFFC
    field_map = {
        0x0: "NV097_SET_TRANSFORM_EXECUTION_MODE_RANGE_MODE_USER",
        0x1: "NV097_SET_TRANSFORM_EXECUTION_MODE_RANGE_MODE_PRIV",
    }
    if field_val in field_map:
        grandchild_name = field_map[field_val]
        symbolic_part = grandchild_name.replace("NV097_SET_TRANSFORM_EXECUTION_MODE_", "", 1)
        results.append(symbolic_part.replace("_", ":", 1))
    else:
        results.append(f"NV097_SET_TRANSFORM_EXECUTION_MODE_:0x{field_val:X}")
    return f'{{{', '.join(results)}}}'


def ParseNv097SetZminMaxControl(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV097_SET_ZMIN_MAX_CONTROL command."""
    _VALUES = {
        0: "ZCLAMP_CULL",
        1: "CULL_NEAR_FAR",
        16: "ZCLAMP_CLAMP",
        256: "CULL_IGNORE_W",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


def ParseNv09fSetOperation(_nv_class, _nv_op, nv_param: int) -> str:
    """Parses the components of a NV09F_SET_OPERATION command."""
    _VALUES = {
        2: "BLEND_AND",
        3: "SRCCOPY",
    }
    ret = _VALUES.get(nv_param)
    if ret:
        return ret
    return f"0x{nv_param:X}?"


# Mapping of graphics class to commands and processors.
_PROCESSORS, _NAME_MAP = _expand_processors(
    {
        0x12: {
            NV012_SET_OBJECT: _process_passthrough,
            NV012_SET_BETA: _process_passthrough,
        },
        0x19: {
            NV019_CONTEXT_CLIP_RECTANGLE_SET_OBJECT: _process_passthrough,
            NV019_CONTEXT_CLIP_RECTANGLE_SET_POINT: _generate_process_double_uint16("X", "Y"),
            NV019_CONTEXT_CLIP_RECTANGLE_SET_SIZE: _generate_process_double_uint16("W", "H"),
        },
        0x42: {
            NV042_CONTEXT_SURFACES_2D_DMA_IMAGE_SRC: _process_passthrough,
            NV042_CONTEXT_SURFACES_2D_DMA_IMAGE_DST: _process_passthrough,
            NV042_CONTEXT_SURFACES_2D_FORMAT: _process_passthrough,
            NV042_CONTEXT_SURFACES_2D_PITCH: _process_passthrough,
            NV042_CONTEXT_SURFACES_2D_OFFSET_SRC: _process_passthrough,
            NV042_CONTEXT_SURFACES_2D_OFFSET_DST: _process_passthrough,
        },
        0x44: {
            NV044_SET_MONOCHROME_COLOR0: _process_passthrough,
        },
        0x57: {
            NV057_COLOR_KEY_FORMAT: _process_passthrough,
            NV057_COLOR_KEY_VALUE: _process_passthrough,
        },
        0x62: {
            NV062_SET_OBJECT: _process_passthrough,
            NV062_SET_CONTEXT_DMA_IMAGE_SOURCE: _process_passthrough,
            NV062_SET_CONTEXT_DMA_IMAGE_DESTIN: _process_passthrough,
            NV062_SET_COLOR_FORMAT: ParseNv062SetColorFormat,
            NV062_SET_PITCH: _generate_process_double_uint16("Source", "Destination"),
            NV062_SET_OFFSET_SOURCE: _process_passthrough,
            NV062_SET_OFFSET_DESTIN: _process_passthrough,
        },
        0x97: {
            NV097_SET_OBJECT: _process_passthrough,
            NV097_SET_STIPPLE_PATERN_SIZE: _process_passthrough,
            NV097_NO_OPERATION: _process_passthrough,
            NV097_WAIT_FOR_IDLE: _process_passthrough,
            NV097_SET_FLIP_READ: _process_passthrough,
            NV097_SET_FLIP_WRITE: _process_passthrough,
            NV097_SET_FLIP_MODULO: _process_passthrough,
            NV097_FLIP_INCREMENT_WRITE: _process_passthrough,
            NV097_FLIP_STALL: _process_passthrough,
            NV097_SET_CONTEXT_DMA_NOTIFIES: _process_passthrough,
            NV097_SET_CONTEXT_DMA_A: _process_passthrough,
            NV097_SET_CONTEXT_DMA_B: _process_passthrough,
            NV097_SET_CONTEXT_DMA_STATE: _process_passthrough,
            NV097_SET_CONTEXT_DMA_COLOR: _process_passthrough,
            NV097_SET_CONTEXT_DMA_ZETA: _process_passthrough,
            NV097_SET_CONTEXT_DMA_VERTEX_A: _process_passthrough,
            NV097_SET_CONTEXT_DMA_VERTEX_B: _process_passthrough,
            NV097_SET_CONTEXT_DMA_SEMAPHORE: _process_passthrough,
            NV097_SET_CONTEXT_DMA_REPORT: _process_passthrough,
            NV097_SET_SURFACE_CLIP_HORIZONTAL: _generate_process_double_uint16("Offset", "Size"),
            NV097_SET_SURFACE_CLIP_VERTICAL: _generate_process_double_uint16("Offset", "Size"),
            NV097_SET_SURFACE_FORMAT: _process_set_surface_format,
            NV097_SET_SURFACE_PITCH: _generate_process_double_uint16("Color", "Zeta"),
            NV097_SET_SURFACE_COLOR_OFFSET: _process_passthrough,
            NV097_SET_SURFACE_ZETA_OFFSET: _process_passthrough,
            StateArray(NV097_SET_COMBINER_ALPHA_ICW, 0x4, 8): _process_combiner_icw,
            NV097_SET_COMBINER_SPECULAR_FOG_CW0: _process_combiner_specular_fog_cw0,
            NV097_SET_COMBINER_SPECULAR_FOG_CW1: _process_combiner_specular_fog_cw1,
            NV097_SET_CONTROL0: _process_set_control0,
            NV097_SET_LIGHT_CONTROL: _process_set_light_control,
            NV097_SET_COLOR_MATERIAL: _process_set_color_material,
            NV097_SET_FOG_MODE: ParseNv097SetFogMode,
            NV097_SET_FOG_GEN_MODE: ParseNv097SetFogGenMode,
            NV097_SET_FOG_ENABLE: _process_boolean_param,
            NV097_SET_FOG_COLOR: ParseNv097SetFogColor,
            NV097_SET_WINDOW_CLIP_TYPE: _process_passthrough,
            NV097_SET_WINDOW_CLIP_HORIZONTAL: _generate_process_double_uint16("Offset", "Size"),
            NV097_SET_WINDOW_CLIP_VERTICAL: _generate_process_double_uint16("Offset", "Size"),
            NV097_SET_ALPHA_TEST_ENABLE: _process_boolean_param,
            NV097_SET_BLEND_ENABLE: _process_boolean_param,
            NV097_SET_CULL_FACE_ENABLE: _process_boolean_param,
            NV097_SET_DEPTH_TEST_ENABLE: _process_boolean_param,
            NV097_SET_DITHER_ENABLE: _process_boolean_param,
            NV097_SET_LIGHTING_ENABLE: _process_boolean_param,
            NV097_SET_POINT_PARAMS_ENABLE: _process_boolean_param,
            NV097_SET_POINT_SMOOTH_ENABLE: _process_boolean_param,
            NV097_SET_LINE_SMOOTH_ENABLE: _process_boolean_param,
            NV097_SET_POLY_SMOOTH_ENABLE: _process_boolean_param,
            NV097_SET_SKIN_MODE: ParseNv097SetSkinMode,
            NV097_SET_STENCIL_TEST_ENABLE: _process_boolean_param,
            NV097_SET_POLY_OFFSET_POINT_ENABLE: _process_boolean_param,
            NV097_SET_POLY_OFFSET_LINE_ENABLE: _process_boolean_param,
            NV097_SET_POLY_OFFSET_FILL_ENABLE: _process_boolean_param,
            NV097_SET_ALPHA_FUNC: ParseNv097SetAlphaFunc,
            NV097_SET_ALPHA_REF: _process_passthrough,
            NV097_SET_BLEND_FUNC_SFACTOR: ParseNv097SetBlendFuncSfactor,
            NV097_SET_BLEND_FUNC_DFACTOR: ParseNv097SetBlendFuncDfactor,
            NV097_SET_BLEND_COLOR: _process_passthrough,
            NV097_SET_BLEND_EQUATION: ParseNv097SetBlendEquation,
            NV097_SET_DEPTH_FUNC: ParseNv097SetDepthFunc,
            NV097_SET_COLOR_MASK: _process_color_mask,
            NV097_SET_DEPTH_MASK: _process_passthrough,
            NV097_SET_STENCIL_MASK: _process_passthrough,
            NV097_SET_STENCIL_FUNC: _process_passthrough,
            NV097_SET_STENCIL_FUNC_REF: _process_passthrough,
            NV097_SET_STENCIL_FUNC_MASK: _process_passthrough,
            NV097_SET_STENCIL_OP_FAIL: ParseNv097SetStencilOpFail,
            NV097_SET_STENCIL_OP_ZFAIL: ParseNv097SetStencilOpZfail,
            NV097_SET_STENCIL_OP_ZPASS: ParseNv097SetStencilOpZpass,
            NV097_SET_SHADE_MODE: ParseNv097SetShadeMode,
            NV097_SET_LINE_WIDTH: ParseNv097SetLineWidth,
            NV097_SET_POLYGON_OFFSET_SCALE_FACTOR: _process_passthrough,
            NV097_SET_POLYGON_OFFSET_BIAS: _process_passthrough,
            NV097_SET_FRONT_POLYGON_MODE: ParseNv097SetFrontPolygonMode,
            NV097_SET_BACK_POLYGON_MODE: _process_passthrough,
            NV097_SET_CLIP_MIN: _process_float_param,
            NV097_SET_CLIP_MAX: _process_float_param,
            NV097_SET_CULL_FACE: ParseNv097SetCullFace,
            NV097_SET_FRONT_FACE: ParseNv097SetFrontFace,
            NV097_SET_NORMALIZATION_ENABLE: _process_passthrough,
            StateArray(NV097_SET_MATERIAL_EMISSION, 0x4, 3): _process_float_param,
            NV097_SET_MATERIAL_ALPHA: _process_float_param,
            NV097_SET_SPECULAR_ENABLE: _process_boolean_param,
            NV097_SET_LIGHT_ENABLE_MASK: _process_set_light_enable_mask,
            StateArray(NV097_SET_TEXGEN_S, 0x10, 4): _process_set_texgen_rst,
            StateArray(NV097_SET_TEXGEN_T, 0x10, 4): _process_set_texgen_rst,
            StateArray(NV097_SET_TEXGEN_R, 0x10, 4): _process_set_texgen_rst,
            StateArray(NV097_SET_TEXGEN_Q, 0x10, 4): _process_set_texgen_q,
            StateArray(NV097_SET_TEXTURE_MATRIX_ENABLE, 0x4, 4): _process_boolean_param,
            NV097_SET_POINT_SIZE: _process_passthrough,
            StateArray(NV097_SET_PROJECTION_MATRIX, 0x4, 16): _process_float_param,
            StructStateArray(NV097_SET_MODEL_VIEW_MATRIX, 0x40, 4, 0x4, 16): _process_float_param,
            StructStateArray(NV097_SET_INVERSE_MODEL_VIEW_MATRIX, 0x40, 4, 0x4, 16): _process_float_param,
            StateArray(NV097_SET_COMPOSITE_MATRIX, 0x4, 16): _process_float_param,
            StructStateArray(NV097_SET_TEXTURE_MATRIX, 0x40, 4, 0x4, 16): _process_float_param,
            StructStateArray(NV097_SET_TEXGEN_PLANE_S, 0x40, 4, 0x4, 4): _process_passthrough,
            StructStateArray(NV097_SET_TEXGEN_PLANE_T, 0x40, 4, 0x4, 4): _process_passthrough,
            StructStateArray(NV097_SET_TEXGEN_PLANE_R, 0x40, 4, 0x4, 4): _process_passthrough,
            StructStateArray(NV097_SET_TEXGEN_PLANE_Q, 0x40, 4, 0x4, 4): _process_passthrough,
            StateArray(NV097_SET_FOG_PARAMS, 0x4, 3): _process_float_param,
            NV097_SET_TEXGEN_VIEW_MODEL: ParseNv097SetTexgenViewModel,
            StateArray(NV097_SET_FOG_PLANE, 0x4, 4): _process_float_param,
            StateArray(NV097_SET_SPECULAR_PARAMS, 0x4, 6): _process_float_param,
            NV097_SET_SWATH_WIDTH: _process_passthrough,
            NV097_SET_FLAT_SHADE_OP: ParseNv097SetFlatShadeOp,
            StateArray(NV097_SET_SCENE_AMBIENT_COLOR, 0x4, 3): _process_float_param,
            StateArray(NV097_SET_VIEWPORT_OFFSET, 0x4, 4): _process_float_param,
            StateArray(NV097_SET_POINT_PARAMS, 0x4, 8): _process_float_param,
            StateArray(NV097_SET_EYE_POSITION, 0x4, 4): _process_float_param,
            StateArray(NV097_SET_COMBINER_FACTOR0, 0x4, 8): _process_combiner_color_factor,
            StateArray(NV097_SET_COMBINER_FACTOR1, 0x4, 8): _process_combiner_color_factor,
            StateArray(NV097_SET_COMBINER_ALPHA_OCW, 0x4, 8): _process_combiner_alpha_ocw,
            StateArray(NV097_SET_COMBINER_COLOR_ICW, 0x4, 8): _process_combiner_icw,
            StateArray(NV097_SET_COLOR_KEY_COLOR, 0x4, 4): _process_passthrough,
            StateArray(NV097_SET_VIEWPORT_SCALE, 0x4, 4): _process_float_param,
            StateArray(NV097_SET_TRANSFORM_PROGRAM, 0x4, 32): _process_passthrough,
            StateArray(NV097_SET_TRANSFORM_CONSTANT, 0x4, 32): _process_float_param,
            StructStateArray(NV097_SET_BACK_LIGHT_AMBIENT_COLOR, 0x40, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_BACK_LIGHT_DIFFUSE_COLOR, 0x40, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_BACK_LIGHT_SPECULAR_COLOR, 0x40, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_AMBIENT_COLOR, 0x80, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_DIFFUSE_COLOR, 0x80, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_SPECULAR_COLOR, 0x80, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_LOCAL_RANGE, 0x80, 8, 0x4, 1): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_INFINITE_HALF_VECTOR, 0x80, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_INFINITE_DIRECTION, 0x80, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_SPOT_FALLOFF, 0x80, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_SPOT_DIRECTION, 0x80, 8, 0x4, 4): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_LOCAL_POSITION, 0x80, 8, 0x4, 3): _process_float_param,
            StructStateArray(NV097_SET_LIGHT_LOCAL_ATTENUATION, 0x80, 8, 0x4, 3): _process_float_param,
            NV097_SET_STIPPLE_ENABLE: _process_passthrough,
            StateArray(NV097_SET_STIPPLE_PATTERN, 0x4, 32): _process_passthrough,
            StateArray(NV097_SET_VERTEX3F, 0x4, 3): _process_float_param,
            StateArray(NV097_SET_VERTEX4F, 0x4, 4): _process_float_param,
            StateArray(NV097_SET_NORMAL3F, 0x4, 3): _process_float_param,
            NV097_SET_NORMAL3S: _process_passthrough,
            NV097_SET_DIFFUSE_COLOR4F: _process_float_param,
            NV097_SET_DIFFUSE_COLOR3F: _process_float_param,
            NV097_SET_DIFFUSE_COLOR4I: _process_passthrough,
            NV097_SET_SPECULAR_COLOR4F: _process_float_param,
            NV097_SET_SPECULAR_COLOR3F: _process_float_param,
            NV097_SET_SPECULAR_COLOR4I: _process_passthrough,
            NV097_SET_TEXCOORD0_2F: _process_float_param,
            NV097_SET_TEXCOORD0_2S: _generate_process_double_uint16("U", "V"),
            NV097_SET_TEXCOORD0_4F: _process_float_param,
            NV097_SET_TEXCOORD0_4S: _process_passthrough,
            NV097_SET_TEXCOORD1_2F: _process_float_param,
            NV097_SET_TEXCOORD1_2S: _generate_process_double_uint16("U", "V"),
            NV097_SET_TEXCOORD1_4F: _process_float_param,
            NV097_SET_TEXCOORD1_4S: _process_passthrough,
            NV097_SET_TEXCOORD2_2F: _process_float_param,
            NV097_SET_TEXCOORD2_2S: _generate_process_double_uint16("U", "V"),
            NV097_SET_TEXCOORD2_4F: _process_float_param,
            NV097_SET_TEXCOORD2_4S: _process_passthrough,
            NV097_SET_TEXCOORD3_2F: _process_float_param,
            NV097_SET_TEXCOORD3_2S: _generate_process_double_uint16("U", "V"),
            NV097_SET_TEXCOORD3_4F: _process_float_param,
            NV097_SET_TEXCOORD3_4S: _process_passthrough,
            NV097_SET_FOG_COORD: _process_float_param,
            NV097_SET_WEIGHT1F: _process_float_param,
            StateArray(NV097_SET_WEIGHT2F, 0x4, 2): _process_float_param,
            StateArray(NV097_SET_WEIGHT3F, 0x4, 3): _process_float_param,
            NV097_SET_EDGE_FLAG: _process_passthrough,
            StateArray(NV097_SET_WEIGHT4F, 0x4, 4): _process_float_param,
            NV097_BREAK_VERTEX_BUFFER_CACHE: _process_passthrough,
            StateArray(NV097_SET_VERTEX_DATA_ARRAY_OFFSET, 0x4, 16): _process_passthrough,
            StateArray(NV097_SET_VERTEX_DATA_ARRAY_FORMAT, 0x4, 16): _process_vertex_data_array_format,
            StateArray(NV097_SET_BACK_SCENE_AMBIENT_COLOR, 0x4, 3): _process_float_param,
            NV097_SET_BACK_MATERIAL_ALPHA: _process_float_param,
            StateArray(NV097_SET_BACK_MATERIAL_EMISSION, 0x4, 3): _process_float_param,
            NV097_SET_LOGIC_OP_ENABLE: _process_passthrough,
            NV097_SET_LOGIC_OP: _process_passthrough,
            NV097_SET_LIGHT_TWO_SIDE_ENABLE: _process_passthrough,
            NV097_CLEAR_REPORT_VALUE: ParseNv097ClearReportValue,
            NV097_SET_ZPASS_PIXEL_COUNT_ENABLE: _process_passthrough,
            NV097_GET_REPORT: ParseNv097GetReport,
            NV097_SET_EYE_DIRECTION: _process_passthrough,
            NV097_SET_SHADER_CLIP_PLANE_MODE: _process_passthrough,
            NV097_SET_BEGIN_END: ParseNv097SetBeginEnd,
            NV097_ARRAY_ELEMENT16: _process_float_param,
            NV097_ARRAY_ELEMENT32: _process_float_param,
            NV097_DRAW_ARRAYS: _process_draw_arrays,
            NV097_INLINE_ARRAY: _process_float_param,
            StateArray(NV097_SET_EYE_VECTOR, 0x4, 3): _process_float_param,
            StateArray(NV097_SET_VERTEX_DATA2F_M, 0x4, 32): _process_float_param,
            StateArray(NV097_SET_VERTEX_DATA2S, 0x4, 16): _process_passthrough,
            StateArray(NV097_SET_VERTEX_DATA4UB, 0x4, 16): _process_passthrough,
            NV097_SET_VERTEX_DATA4S_M: _process_passthrough,
            StateArray(NV097_SET_VERTEX_DATA4F_M, 0x4, 64): _process_float_param,
            StateArray(NV097_SET_TEXTURE_OFFSET, 0x40, 4): _process_passthrough,
            StateArray(NV097_SET_TEXTURE_FORMAT, 0x40, 4): _process_set_texture_format,
            StateArray(NV097_SET_TEXTURE_ADDRESS, 0x40, 4): _process_set_texture_address,
            StateArray(NV097_SET_TEXTURE_CONTROL0, 0x40, 4): _process_set_texture_control0,
            StateArray(NV097_SET_TEXTURE_CONTROL1, 0x40, 4): _process_set_texture_control1,
            StateArray(NV097_SET_TEXTURE_FILTER, 0x40, 4): _process_set_texture_filter,
            StateArray(NV097_SET_TEXTURE_IMAGE_RECT, 0x40, 4): _generate_process_double_uint16("H", "W"),
            StateArray(NV097_SET_TEXTURE_PALETTE, 0x40, 4): _process_set_texture_palette,
            StateArray(NV097_SET_TEXTURE_BORDER_COLOR, 0x40, 4): _process_passthrough,
            StructStateArray(NV097_SET_TEXTURE_SET_BUMP_ENV_MAT, 0x40, 4, 0x4, 4): _process_passthrough,
            StateArray(NV097_SET_TEXTURE_SET_BUMP_ENV_SCALE, 0x40, 4): _process_passthrough,
            StateArray(NV097_SET_TEXTURE_SET_BUMP_ENV_OFFSET, 0x40, 4): _process_passthrough,
            NV097_SET_SEMAPHORE_OFFSET: _process_passthrough,
            NV097_BACK_END_WRITE_SEMAPHORE_RELEASE: _process_passthrough,
            NV097_SET_ZMIN_MAX_CONTROL: ParseNv097SetZminMaxControl,
            NV097_SET_ANTI_ALIASING_CONTROL: ParseNv097SetAntiAliasingControl,
            NV097_SET_COMPRESS_ZBUFFER_EN: _process_passthrough,
            NV097_SET_OCCLUDE_ZSTENCIL_EN: _process_passthrough,
            NV097_SET_ZSTENCIL_CLEAR_VALUE: _process_passthrough,
            NV097_SET_COLOR_CLEAR_VALUE: _process_passthrough,
            NV097_CLEAR_SURFACE: ParseNv097ClearSurface,
            NV097_SET_CLEAR_RECT_HORIZONTAL: _generate_process_double_uint16("Min", "Max"),
            NV097_SET_CLEAR_RECT_VERTICAL: _generate_process_double_uint16("Min", "Max"),
            StateArray(NV097_SET_SPECULAR_FOG_FACTOR, 0x4, 2): _process_combiner_color_factor,
            StateArray(NV097_SET_BACK_SPECULAR_PARAMS, 0x4, 6): _process_float_param,
            StateArray(NV097_SET_COMBINER_COLOR_OCW, 0x4, 8): _process_combiner_color_ocw,
            NV097_SET_COMBINER_CONTROL: _process_combiner_control,
            NV097_SET_SHADOW_ZSLOPE_THRESHOLD: _process_passthrough,
            NV097_SET_SHADOW_COMPARE_FUNC: ParseNv097SetShadowCompareFunc,
            NV097_SET_SHADER_STAGE_PROGRAM: process_shader_stage_program,
            NV097_SET_DOT_RGBMAPPING: _process_passthrough,
            NV097_SET_SHADER_OTHER_STAGE_INPUT: _process_set_other_stage_input,
            StateArray(NV097_SET_TRANSFORM_DATA, 0x4, 4): _process_float_param,
            NV097_LAUNCH_TRANSFORM_PROGRAM: _process_passthrough,
            NV097_SET_TRANSFORM_EXECUTION_MODE: ParseNv097SetTransformExecutionMode,
            NV097_SET_TRANSFORM_PROGRAM_CXT_WRITE_EN: _process_passthrough,
            NV097_SET_TRANSFORM_PROGRAM_LOAD: _process_passthrough,
            NV097_SET_TRANSFORM_PROGRAM_START: _process_passthrough,
            NV097_SET_TRANSFORM_CONSTANT_LOAD: _process_passthrough,
        },
        0x9F: {
            NV09F_SET_OBJECT: _process_passthrough,
            NV09F_SET_COLOR_KEY_OBJECT: _process_passthrough,
            NV09F_SET_CLIP_RECTANGLE_OBJECT: _process_passthrough,
            NV09F_SET_PATTERN_OBJECT: _process_passthrough,
            NV09F_SET_ROP5_OBJECT: _process_passthrough,
            NV09F_SET_BETA_OBJECT: _process_passthrough,
            NV09F_SET_BETA_OBJECT4: _process_passthrough,
            NV09F_SET_CONTEXT_SURFACES: _process_passthrough,
            NV09F_SET_OPERATION: ParseNv09fSetOperation,
            NV09F_CONTROL_POINT_IN: _generate_process_double_uint16("X", "Y"),
            NV09F_CONTROL_POINT_OUT: _generate_process_double_uint16("X", "Y"),
            NV09F_SIZE: _generate_process_double_uint16("W", "H"),
        },
    }
)


@dataclass
class CommandInfo:
    """Verbosely describes an nv2a command."""

    channel: int
    nv_class: int
    nv_op: int
    nv_param: int
    nv_op_name: str = ""
    nv_class_name: str = ""

    def __post_init__(self):
        self.process()

    @property
    def pretty_suffix(self) -> str:
        class_info = f"{self.nv_class_name}<0x{self.nv_class:x}>" if self.nv_class_name else f"0x{self.nv_class:x}"
        op_info = f"{self.nv_op_name}<0x{self.nv_op:x}>" if self.nv_op_name else f"0x{self.nv_op:x}"

        return f"{self.channel}: {class_info} -> {op_info} ({self.param_info})"

    def get_pretty_string(self) -> str:
        return f"nv2a_pgraph_method {self.pretty_suffix}"

    def process(self):
        processor_key = (self.nv_class, self.nv_op)
        processor = _PROCESSORS.get(processor_key)
        if processor:
            self.param_info = processor(self.nv_class, self.nv_op, self.nv_param)
        else:
            self.param_info = f"0x{self.nv_param:x}"


def get_command_info(channel: int, nv_class: int, nv_op: int, nv_param: int) -> CommandInfo:
    processor_key = (nv_class, nv_op)
    if processor_key not in _PROCESSORS:
        return CommandInfo(channel, nv_class, nv_op, nv_param)

    op_name = _NAME_MAP.get(processor_key)
    return CommandInfo(channel, nv_class, nv_op, nv_param, op_name)
