"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisStreamsToLambda = void 0;
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The KinesisStreamsToLambda class.
 */
class KinesisStreamsToLambda extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the KinesisStreamsToLambda class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis Stream
        this.kinesisStream = defaults.buildKinesisStream(this, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Add the Lambda event source mapping
        const eventSourceProps = defaults.KinesisEventSourceProps(props.kinesisEventSourceProps);
        this.lambdaFunction.addEventSource(new aws_lambda_event_sources_1.KinesisEventSource(this.kinesisStream, eventSourceProps));
        // Add permissions for the Lambda function to access Kinesis
        const policy = new iam.Policy(this, 'LambdaFunctionPolicy');
        this.kinesisStreamRole = this.lambdaFunction.role;
        policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [this.kinesisStream.streamArn],
            actions: [
                'kinesis:GetRecords',
                'kinesis:GetShardIterator',
                'kinesis:DescribeStream'
            ]
        }));
        policy.addStatements(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: ['*'],
            actions: [
                'kinesis:ListStreams',
            ]
        }));
        policy.attachToRole(this.kinesisStreamRole);
        this.kinesisStream.grantRead(this.lambdaFunction.grantPrincipal);
        // Add appropriate cfn_nag metadata
        const cfnCustomPolicy = policy.node.defaultChild;
        cfnCustomPolicy.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W12",
                        reason: "The kinesis:ListStreams action requires a wildcard resource."
                    }
                ]
            }
        };
    }
}
exports.KinesisStreamsToLambda = KinesisStreamsToLambda;
//# sourceMappingURL=data:application/json;base64,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