#!/usr/bin/env python3
"""
CloudKit Poller for WatchCode
Polls CloudKit for action responses from Apple Watch.

Setup:
1. Go to CloudKit Dashboard: https://icloud.developer.apple.com/
2. Select container: iCloud.com.watchcode.app
3. Go to API Access > Server-to-Server Keys
4. Create a new key, download the private key (.p8 file)
5. Copy the Key ID
6. Save the private key to ~/.watchcode/cloudkit_key.p8
7. Add the Key ID to ~/.watchcode/config.json

Usage:
    python3 cloudkit_poller.py --session-id <session_id> [--timeout 60]

Dependencies:
    pip3 install requests requests-cloudkit
"""

import argparse
import json
import os
import sys
import time
from pathlib import Path

# Check for required dependencies
try:
    import requests
    from requests_cloudkit import CloudKitAuth
except ImportError:
    print("Error: Required dependencies not installed.", file=sys.stderr)
    print("Run: pip3 install requests requests-cloudkit", file=sys.stderr)
    sys.exit(1)


# Configuration
CONTAINER_ID = "iCloud.com.watchcode.app"
ENVIRONMENT = "development"  # or "production"
CLOUDKIT_API_VERSION = "1"
BASE_URL = f"https://api.apple-cloudkit.com/database/{CLOUDKIT_API_VERSION}/{CONTAINER_ID}/{ENVIRONMENT}"

CONFIG_DIR = Path.home() / ".watchcode"
CONFIG_FILE = CONFIG_DIR / "config.json"
KEY_FILE = CONFIG_DIR / "cloudkit_key.p8"


def load_config() -> dict:
    """Load configuration from config.json"""
    if not CONFIG_FILE.exists():
        return {}
    try:
        with open(CONFIG_FILE, "r") as f:
            return json.load(f)
    except Exception as e:
        print(f"Warning: Could not load config: {e}", file=sys.stderr)
        return {}


def get_cloudkit_auth() -> CloudKitAuth:
    """Create CloudKit authentication handler"""
    config = load_config()

    key_id = config.get("cloudkit_key_id")
    key_file = config.get("cloudkit_key_file", str(KEY_FILE))

    if not key_id:
        print("Error: cloudkit_key_id not found in config.json", file=sys.stderr)
        print("Add your CloudKit Key ID to ~/.watchcode/config.json:", file=sys.stderr)
        print('  {"cloudkit_key_id": "YOUR_KEY_ID_HERE"}', file=sys.stderr)
        sys.exit(1)

    if not os.path.exists(key_file):
        print(f"Error: CloudKit private key not found: {key_file}", file=sys.stderr)
        print("Download your key from CloudKit Dashboard and save it there.", file=sys.stderr)
        sys.exit(1)

    # Validate file permissions (should be 600 - owner read/write only)
    key_path = Path(key_file)
    file_mode = key_path.stat().st_mode & 0o777
    if file_mode != 0o600:
        print(f"Warning: CloudKit key has insecure permissions ({oct(file_mode)})", file=sys.stderr)
        print(f"Run: chmod 600 {key_file}", file=sys.stderr)
        # Continue anyway - warning only, don't block

    return CloudKitAuth(key_id=key_id, key_file_name=key_file)


def lookup_response(session_id: str, auth: CloudKitAuth) -> dict | None:
    """
    Look up action response record by session ID.

    Args:
        session_id: The session ID to look up
        auth: CloudKit authentication handler

    Returns:
        Response record fields dict or None if not found
    """
    # Normalize to uppercase - Swift's UUID.uuidString always returns uppercase
    # and CloudKit record names are case-sensitive
    record_name = f"response_{session_id.upper()}"
    url = f"{BASE_URL}/public/records/lookup"

    payload = {
        "records": [
            {"recordName": record_name}
        ]
    }

    try:
        response = requests.post(url, json=payload, auth=auth, timeout=10)

        if response.status_code == 200:
            data = response.json()
            records = data.get("records", [])

            if records and "fields" in records[0]:
                fields = records[0]["fields"]
                # Convert CloudKit field format to simple dict
                return {
                    key: value.get("value")
                    for key, value in fields.items()
                }
        elif response.status_code == 404:
            return None
        else:
            # Check for specific CloudKit errors
            try:
                error_data = response.json()
                records = error_data.get("records", [])
                if records and records[0].get("serverErrorCode") == "NOT_FOUND":
                    return None
            except:
                pass
            print(f"CloudKit error: {response.status_code} - {response.text}", file=sys.stderr)

    except requests.RequestException as e:
        print(f"Request error: {e}", file=sys.stderr)

    return None


def delete_response(session_id: str, auth: CloudKitAuth) -> bool:
    """
    Delete a response record after processing.

    Args:
        session_id: The session ID of the record to delete
        auth: CloudKit authentication handler

    Returns:
        True if deleted successfully
    """
    # Normalize to uppercase to match Swift's UUID format
    record_name = f"response_{session_id.upper()}"
    url = f"{BASE_URL}/public/records/modify"

    payload = {
        "operations": [
            {
                "operationType": "delete",
                "record": {
                    "recordName": record_name
                }
            }
        ]
    }

    try:
        response = requests.post(url, json=payload, auth=auth, timeout=10)
        return response.status_code == 200
    except requests.RequestException as e:
        print(f"Delete error: {e}", file=sys.stderr)
        return False


def poll_for_response(session_id: str, timeout: int = 60, poll_interval: float = 2.0) -> dict | None:
    """
    Poll CloudKit for a response with the given session ID.

    Args:
        session_id: The session ID to poll for
        timeout: Maximum time to wait in seconds
        poll_interval: Time between polls in seconds

    Returns:
        Response dict with action, or None if timeout
    """
    auth = get_cloudkit_auth()
    start_time = time.time()
    attempt = 0

    print(f"[CloudKit] Polling for response: {session_id}", file=sys.stderr)
    print(f"[CloudKit] Timeout: {timeout}s, Interval: {poll_interval}s", file=sys.stderr)

    while time.time() - start_time < timeout:
        attempt += 1
        elapsed = time.time() - start_time

        response = lookup_response(session_id, auth)

        if response:
            print(f"[CloudKit] Found response after {elapsed:.1f}s (attempt {attempt})", file=sys.stderr)
            print(f"[CloudKit] Action: {response.get('action')}", file=sys.stderr)

            # Clean up the record
            if delete_response(session_id, auth):
                print(f"[CloudKit] Cleaned up record", file=sys.stderr)

            return response

        if attempt % 5 == 0:
            print(f"[CloudKit] Still waiting... ({elapsed:.0f}s elapsed)", file=sys.stderr)

        time.sleep(poll_interval)

    print(f"[CloudKit] Timeout after {timeout}s", file=sys.stderr)
    return None


def main():
    parser = argparse.ArgumentParser(
        description="Poll CloudKit for WatchCode action responses"
    )
    parser.add_argument(
        "--session-id",
        required=True,
        help="Session ID to poll for"
    )
    parser.add_argument(
        "--timeout",
        type=int,
        default=60,
        help="Timeout in seconds (default: 60)"
    )
    parser.add_argument(
        "--poll-interval",
        type=float,
        default=2.0,
        help="Poll interval in seconds (default: 2.0)"
    )
    parser.add_argument(
        "--once",
        action="store_true",
        help="Check once and exit (no polling)"
    )

    args = parser.parse_args()

    if args.once:
        auth = get_cloudkit_auth()
        response = lookup_response(args.session_id, auth)
        if response:
            print(json.dumps(response))
            sys.exit(0)
        else:
            sys.exit(1)
    else:
        response = poll_for_response(
            args.session_id,
            timeout=args.timeout,
            poll_interval=args.poll_interval
        )

        if response:
            # Output the response as JSON for the hook handler
            print(json.dumps(response))
            sys.exit(0)
        else:
            sys.exit(1)


if __name__ == "__main__":
    main()
