#!/usr/bin/env python3
"\nAuthor : Xinyuan Chen <45612704+tddschn@users.noreply.github.com>\nDate   : 2024-11-24\nPurpose: Convert Google AI Studio JSON exports to standalone HTML (Replica of Artifact UI)\n"

import argparse
import html
import json
import re
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple

import markdown

# --- Configuration & Assets ---
TAILWIND_SCRIPT = '<script src="https://cdn.tailwindcss.com"></script>'
FONTAWESOME_LINK = '<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">'
HIGHLIGHT_JS = (
    '\n<link rel="stylesheet" '
    'href="https://cdnjs.cloudflare.com/ajax/libs/highlight.js/11.9.0/styles/github-dark.min.css">\n'
    '<script src="https://cdnjs.cloudflare.com/ajax/libs/highlight.js/11.9.0/highlight.min.js"></script>\n'
    "<script>hljs.highlightAll();</script>\n"
)
# Minimal JS for folding/copying (Vanilla JS equivalent of the React logic)
INTERACTIVITY_SCRIPT = (
    "\n<script>\n"
    'document.addEventListener("DOMContentLoaded", function() {\n'
    "    // 1. Handle Copy Buttons\n"
    "    document.querySelectorAll('.copy-btn').forEach(btn => {\n"
    "        btn.addEventListener('click', () => {\n"
    "            const text = decodeURIComponent(btn.dataset.content);\n"
    "            navigator.clipboard.writeText(text);\n"
    "            const icon = btn.querySelector('i');\n"
    "            icon.className = 'fas fa-check';\n"
    "            setTimeout(() => icon.className = 'fas fa-copy', 1500);\n"
    "        });\n"
    "    });\n\n"
    "    // 2. Handle Thoughts Folding\n"
    "    // (Native <details> element handles the logic, we just style it)\n"
    "});\n"
    "</script>\n"
)
# --- Logic ported from artifact-component.tsx ---


def normalize_text(text: str) -> str:
    """Normalize whitespace for comparison."""
    return re.sub(r"\s+", " ", text).strip()


def inline_image_to_markdown(
    inline_payload: Dict[str, Any], default_label: str = "Inline Image"
) -> str:
    """Return markdown for inline images (supports multiple export shapes)."""
    if not inline_payload:
        return ""
    data = inline_payload.get("data")
    mime = inline_payload.get("mimeType")
    # Some exports wrap mime/data inside an inlineData object
    if not data and isinstance(inline_payload.get("inlineData"), dict):
        inner = inline_payload["inlineData"]
        data = inner.get("data", data)
        mime = inner.get("mimeType", mime)
    if not data:
        return ""
    mime = mime or "image/png"
    alt_text = inline_payload.get("altText") or inline_payload.get("description")
    alt_text = alt_text or default_label
    return f"\n\n![{alt_text}](data:{mime};base64,{data})\n\n"


def format_attachment_entry(
    icon: str,
    label: str,
    attachment_id: Optional[str],
    token_count: Optional[Any],
    url: Optional[str],
    link_label: str,
    kind: str,
) -> Dict[str, Any]:
    parsed_tokens: Optional[int] = None
    if isinstance(token_count, int):
        parsed_tokens = token_count
    elif isinstance(token_count, str):
        try:
            parsed_tokens = int(token_count.replace(",", ""))
        except ValueError:
            parsed_tokens = None
    return {
        "icon": icon,
        "label": label,
        "id": attachment_id or "unknown",
        "url": url or "",
        "link_label": link_label,
        "kind": kind,
        "token_count": parsed_tokens,
    }


def byte_offset_to_char_index(text: str, offset: int) -> int:
    """Map a UTF-8 byte offset from Gemini exports to a Python index."""
    if offset <= 0:
        return 0
    running = 0
    for idx, char in enumerate(text):
        if running >= offset:
            return idx
        running += len(char.encode("utf-8"))
    return len(text)


def normalize_citation_indexes(
    text: str, citations: List[Dict[str, int]]
) -> List[Dict[str, int]]:
    """Convert byte-based offsets to character indexes for slicing."""
    if not text or not citations:
        return citations
    normalized = []
    for citation in citations:
        if "index" not in citation or "footnoteNumber" not in citation:
            continue
        normalized.append(
            {
                "index": byte_offset_to_char_index(text, int(citation["index"])),
                "footnoteNumber": citation["footnoteNumber"],
            }
        )
    return normalized


def extract_grounding_metadata(
    grounding: Dict[str, Any],
) -> Tuple[List[Dict[str, int]], Dict[int, Dict[str, str]]]:
    """Return citation positions and source metadata from grounding payload."""
    if not isinstance(grounding, dict):
        return [], {}
    citations: List[Dict[str, int]] = []
    for segment in grounding.get("corroborationSegments", []):
        try:
            index = int(segment.get("index", -1))
            footnote = int(segment.get("footnoteNumber"))
        except (TypeError, ValueError):
            continue
        if index < 0:
            continue
        citations.append({"index": index, "footnoteNumber": footnote})
    sources: Dict[int, Dict[str, str]] = {}
    for source in grounding.get("groundingSources", []):
        try:
            ref_number = int(source.get("referenceNumber"))
        except (TypeError, ValueError):
            continue
        if ref_number in sources:
            continue
        sources[ref_number] = {
            "title": source.get("title") or source.get("uri") or f"Source {ref_number}",
            "uri": source.get("uri", ""),
        }
    return citations, sources


def parse_and_merge_conversation(json_data: Dict[str, Any]) -> List[Dict[str, Any]]:
    """Parses raw JSON, merges fragmented chunks, and handles 'isThought' flags."""
    merged_turns: List[Dict[str, Any]] = []
    current_turn: Optional[Dict[str, Any]] = None
    chunks: List[Dict[str, Any]] = []
    if isinstance(json_data, dict):
        chunks = json_data.get("chunkedPrompt", {}).get("chunks", [])
    elif isinstance(json_data, list):
        chunks = json_data
    for chunk in chunks:
        role = chunk.get("role", "unknown").lower()
        chunk_is_thought = chunk.get("isThought", False) or bool(
            chunk.get("thoughtSignatures")
        )
        chunk_thoughts = ""
        chunk_content = ""
        parts = chunk.get("parts", [])
        if parts:
            for part in parts:
                text = part.get("text", "")
                if "inlineImage" in part:
                    text += inline_image_to_markdown(part.get("inlineImage"))
                if "inlineData" in part and isinstance(part["inlineData"], dict):
                    text += inline_image_to_markdown(part.get("inlineData"))
                if part.get("thought", False) or chunk_is_thought:
                    chunk_thoughts += text
                else:
                    chunk_content += text
        elif "text" in chunk:
            text = chunk["text"]
            if chunk_is_thought:
                chunk_thoughts += text
            else:
                chunk_content += text
        attachment_entries: List[Dict[str, Any]] = []
        token_count = chunk.get("tokenCount")
        drive_attachment_map = (
            ("driveDocument", "📄", "Document Attachment"),
            ("driveImage", "🖼️", "Image Attachment"),
            ("driveAudio", "🎧", "Audio Attachment"),
        )
        for key, icon, label in drive_attachment_map:
            if key in chunk and isinstance(chunk[key], dict):
                attachment_id = chunk[key].get("id")
                drive_url = (
                    f"https://drive.google.com/file/d/{attachment_id}/view"
                    if attachment_id
                    else None
                )
                attachment_entries.append(
                    format_attachment_entry(
                        icon,
                        label,
                        attachment_id,
                        token_count,
                        drive_url,
                        "Open in Drive",
                        "drive",
                    )
                )
        if "youtubeVideo" in chunk and isinstance(chunk["youtubeVideo"], dict):
            video_id = chunk["youtubeVideo"].get("id")
            youtube_url = (
                f"https://www.youtube.com/watch?v={video_id}" if video_id else None
            )
            attachment_entries.append(
                format_attachment_entry(
                    "▶️",
                    "YouTube Attachment",
                    video_id,
                    token_count,
                    youtube_url,
                    "Watch on YouTube",
                    "youtube",
                )
            )
        chunk_attachments = attachment_entries
        if "inlineImage" in chunk:
            chunk_content += inline_image_to_markdown(
                chunk["inlineImage"], "Model Generated Image"
            )
        if "inlineData" in chunk and isinstance(chunk["inlineData"], dict):
            chunk_content += inline_image_to_markdown(
                chunk["inlineData"], "Model Generated Image"
            )
        if not chunk_content and not chunk_thoughts and not chunk_attachments:
            continue
        chunk_citations, chunk_sources = extract_grounding_metadata(
            chunk.get("grounding", {})
        )
        if chunk_citations:
            chunk_citations = normalize_citation_indexes(chunk_content, chunk_citations)
        should_append_content = True
        if current_turn and current_turn["role"] == role and chunk_content:
            incoming_norm = normalize_text(chunk_content)
            existing_norm = normalize_text(current_turn["content"])
            if (
                incoming_norm
                and len(incoming_norm) <= len(existing_norm)
                and incoming_norm in existing_norm
            ):
                should_append_content = False
        if current_turn and current_turn["role"] == role:
            if should_append_content and chunk_content:
                needs_spacer = current_turn["content"] and (
                    not current_turn["content"].endswith("\n")
                )
                spacer = "\n\n" if needs_spacer else ""
                chunk_start = len(current_turn["content"]) + len(spacer)
                current_turn["content"] += spacer + chunk_content
                for citation in chunk_citations:
                    current_turn.setdefault("citations", []).append(
                        {
                            "index": chunk_start + citation["index"],
                            "footnoteNumber": citation["footnoteNumber"],
                        }
                    )
            if chunk_thoughts:
                needs_spacer = current_turn["thoughts"] and (
                    not current_turn["thoughts"].endswith("\n")
                )
                current_turn["thoughts"] += (
                    "\n\n" if needs_spacer else ""
                ) + chunk_thoughts
            if chunk_sources:
                sources = current_turn.setdefault("sources", {})
                for key, value in chunk_sources.items():
                    sources.setdefault(key, value)
            if chunk_attachments:
                current_turn.setdefault("attachments", []).extend(chunk_attachments)
        else:
            if current_turn:
                merged_turns.append(current_turn)
            current_turn = {
                "role": role,
                "content": chunk_content,
                "thoughts": chunk_thoughts,
                "attachments": list(chunk_attachments),
            }
            if chunk_citations:
                current_turn["citations"] = [
                    {
                        "index": citation["index"],
                        "footnoteNumber": citation["footnoteNumber"],
                    }
                    for citation in chunk_citations
                ]
            if chunk_sources:
                current_turn["sources"] = dict(chunk_sources)
            if not chunk_attachments:
                current_turn.setdefault("attachments", [])
    if current_turn:
        merged_turns.append(current_turn)
    return merged_turns


# --- Rendering Components (Python f-strings mirroring React components) ---


def render_markdown(text: str) -> str:
    """Renders Markdown to HTML using Python's standard lib."""
    if not text:
        return ""
    return markdown.markdown(
        text, extensions=["fenced_code", "tables", "nl2br", "sane_lists"]
    )


def render_thoughts(thoughts: str, expand: bool = False) -> str:
    if not thoughts:
        return ""
    html_thoughts = render_markdown(thoughts)
    open_attr = " open" if expand else ""
    parts = [
        '\n    <div class="mb-4">\n',
        f'        <details class="group border border-gray-200 rounded-lg bg-gray-50"{open_attr}>\n',
        '            <summary class="flex items-center gap-2 p-3 font-medium cursor-pointer list-none text-gray-600 hover:bg-gray-100 rounded-t-lg select-none">\n',
        '                <i class="fas fa-brain text-purple-500"></i> Model Thoughts\n',
        "            </summary>\n",
        '            <div class="p-4 border-t border-gray-200 text-gray-600 text-sm font-mono whitespace-pre-wrap bg-white rounded-b-lg rendered-markdown prose prose-sm max-w-none">\n',
        f"                {html_thoughts}\n",
        "            </div>\n",
        "        </details>\n",
        "    </div>\n    ",
    ]
    return "".join(parts)


def format_token_count_display(token_count: Optional[int]) -> str:
    if isinstance(token_count, int):
        return f"{token_count:,}"
    return "unknown"


def render_attachments_block(
    attachments: List[Dict[str, Any]], indent: str = "        "
) -> str:
    if not attachments:
        return ""
    lines = [f'{indent}<div class="space-y-2 mb-3 attachment-block">']
    for attachment in attachments:
        kind = attachment.get("kind", "drive")
        token_display = format_token_count_display(attachment.get("token_count"))
        attachment_id = html.escape(attachment.get("id", "unknown"))
        if kind == "youtube":
            video_id = attachment.get("id", "") or ""
            embed_src = (
                f"https://www.youtube.com/embed/{html.escape(video_id, quote=True)}"
                if video_id
                else ""
            )
            video_url = (
                f"https://www.youtube.com/watch?v={video_id}" if video_id else ""
            )
            lines.append(f'{indent}    <div class="attachment-line space-y-2">')
            if embed_src:
                lines.append(
                    f'{indent}        <iframe width="560" height="315" src="{embed_src}" '
                    'title="YouTube video player" frameborder="0" '
                    'allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share" '
                    'referrerpolicy="strict-origin-when-cross-origin" allowfullscreen></iframe>'
                )
            if video_url:
                escaped_video_url = html.escape(video_url, quote=True)
                url_text = (
                    f'<a class="font-mono text-blue-700 hover:underline" '
                    f'href="{escaped_video_url}" target="_blank" rel="noopener noreferrer">'
                    f"{html.escape(video_url)}</a>"
                )
            else:
                url_text = '<span class="text-gray-400">YouTube link unavailable</span>'
            lines.append(
                f'{indent}        <div class="text-xs text-gray-700">'
                f"{url_text} - {token_display} tokens"
                "</div>"
            )
            lines.append(f"{indent}    </div>")
            continue

        icon = attachment.get("icon", "")
        label = attachment.get("label", "Attachment")
        label = label.replace("Attachment", "Attached")
        label_html = html.escape(label)
        url = attachment.get("url", "")
        link_label = f"{icon} {label_html}".strip()
        if url:
            escaped_url = html.escape(url, quote=True)
            link_html = (
                f'<a href="{escaped_url}" target="_blank" rel="noopener noreferrer">'
                f"{link_label}</a>"
            )
        else:
            link_html = f'<span class="text-gray-400">{html.escape(link_label)}</span>'
        id_html = f'<code class="font-mono">{attachment_id}</code>'
        lines.append(
            f'{indent}    <div class="attachment-line text-xs bg-gray-50 '
            'border border-gray-200 rounded px-3 py-2">'
            f"{link_html} ({token_display} tokens, {id_html})"
            "</div>"
        )
    lines.append(f"{indent}</div>")
    return "\n".join(lines)


def inject_citations(
    text: str,
    citations: List[Dict[str, int]],
    sources: Optional[Dict[int, Dict[str, str]]],
    anchor_prefix: str,
) -> str:
    """Insert inline citation links (e.g., [1]) at provided indices."""
    if not text or not citations:
        return text
    ordered = sorted(
        [c for c in citations if "index" in c and "footnoteNumber" in c],
        key=lambda item: (item["index"], item["footnoteNumber"]),
    )
    pieces: List[str] = []
    last_idx = 0
    text_len = len(text)
    sources = sources or {}
    for citation in ordered:
        idx = max(0, min(text_len, citation["index"]))
        pieces.append(text[last_idx:idx])
        footnote = citation["footnoteNumber"]
        source = sources.get(footnote, {})
        url = source.get("uri")
        if url:
            escaped = html.escape(url, quote=True)
            anchor = (
                f'<a class="citation-link" href="{escaped}" '
                'target="_blank" rel="noopener noreferrer">'
                f"[{footnote}]"
                "</a>"
            )
        else:
            anchor = (
                f'<a class="citation-link" href="#{anchor_prefix}-fn-{footnote}">'
                f"[{footnote}]"
                "</a>"
            )
        pieces.append(anchor)
        last_idx = idx
    pieces.append(text[last_idx:])
    return "".join(pieces)


def render_sources(sources: Dict[int, Dict[str, str]], anchor_prefix: str) -> str:
    if not sources:
        return ""
    items = []
    for footnote in sorted(sources):
        source = sources[footnote]
        title = html.escape(source.get("title", f"Source {footnote}"))
        uri = source.get("uri", "")
        link_attrs = (
            f' href="{html.escape(uri)}" target="_blank" rel="noopener noreferrer"'
            if uri
            else ""
        )
        items.append(
            f'<li id="{anchor_prefix}-fn-{footnote}" class="leading-relaxed">'
            f'<a class="text-blue-600 hover:underline"{link_attrs}>{title}</a>'
            "</li>"
        )
    sources_html = "".join(items)
    return (
        '\n        <div class="mt-6 sources-block">\n'
        '            <div class="flex items-center gap-2 text-xs font-semibold uppercase '
        'text-gray-500 tracking-wide">\n'
        '                <i class="fas fa-link text-gray-400"></i> Sources\n'
        "            </div>\n"
        '            <ol class="list-decimal ml-6 mt-2 space-y-1 text-sm text-gray-600">'
        f"{sources_html}"
        "            </ol>\n"
        "        </div>\n"
    )


def render_turn(
    turn: Dict[str, Any],
    no_thoughts: bool = False,
    expand_thoughts: bool = False,
    show_citations: bool = False,
    turn_index: int = 0,
) -> str:
    role = turn["role"]
    content = turn.get("content", "")
    thoughts = turn.get("thoughts", "")
    is_user = role == "user"
    attachments_html = render_attachments_block(
        turn.get("attachments", []), indent="        "
    )
    if is_user:
        header_color = "text-blue-700"
        icon = "fa-user"
        body_parts: List[str] = []
        if attachments_html:
            body_parts.append(f"\n{attachments_html}\n")
        body_parts.append(
            '\n        <div class="bg-blue-50 p-4 rounded-lg border border-blue-100 text-blue-900 '
            'font-sans text-sm overflow-x-auto relative group">\n'
            f'            <button class="copy-btn absolute top-2 right-2 text-blue-300 '
            'hover:text-blue-600 opacity-0 group-hover:opacity-100 transition-opacity" '
            f'data-content="{html.escape(content)}">\n'
            '                <i class="fas fa-copy"></i>\n'
            "            </button>\n"
            f'            <pre class="whitespace-pre-wrap font-sans">{html.escape(content)}</pre>\n'
            "        </div>\n        "
        )
        body = "".join(body_parts)
    else:
        header_color = "text-orange-600"
        icon = "fa-robot"
        thoughts_html = (
            "" if no_thoughts else render_thoughts(thoughts, expand_thoughts)
        )
        anchor_prefix = f"turn-{turn_index}"
        display_text = (
            inject_citations(
                content,
                turn.get("citations", []),
                turn.get("sources"),
                anchor_prefix,
            )
            if show_citations
            else content
        )
        body_content = render_markdown(display_text)
        sources_html = (
            render_sources(turn.get("sources", {}), anchor_prefix)
            if show_citations
            else ""
        )
        body = ""
        if thoughts_html:
            body += thoughts_html
        if attachments_html:
            body += f"\n{attachments_html}\n"
        body += (
            '\n        <div class="prose max-w-none text-gray-800 font-sans rendered-markdown '
            'relative group">\n'
            f'             <button class="copy-btn absolute -top-6 right-0 text-gray-300 '
            'hover:text-gray-600 opacity-0 group-hover:opacity-100 transition-opacity" '
            f'data-content="{html.escape(content)}">\n'
            '                <i class="fas fa-copy"></i>\n'
            "            </button>\n"
            f"            {body_content}\n"
            "        </div>"
            f"{sources_html}\n        "
        )
    return (
        '\n    <hr class="my-8 border-gray-200">\n'
        f'    <div class="{role}-content mb-4">\n'
        f'        <div class="flex items-center gap-2 mb-2 {header_color} font-bold text-sm '
        'uppercase">\n'
        f'            <i class="fas {icon}"></i> {role}\n'
        "        </div>\n"
        f"        {body}\n"
        "    </div>\n    "
    )


def generate_html(
    turns: List[Dict[str, Any]],
    title: str,
    no_thoughts: bool,
    expand_thoughts: bool,
    show_citations: bool,
) -> str:
    conversation_html = "\n".join(
        [
            render_turn(t, no_thoughts, expand_thoughts, show_citations, idx)
            for idx, t in enumerate(turns)
        ]
    )
    return (
        '<!DOCTYPE html>\n<html lang="en">\n<head>\n    <meta charset="UTF-8">\n    '
        '<meta name="viewport" content="width=device-width, initial-scale=1.0">\n    '
        f"<title>{html.escape(title)}</title>\n"
        f"    {TAILWIND_SCRIPT}\n    {FONTAWESOME_LINK}\n    {HIGHLIGHT_JS}"
        "    <style>\n"
        "        body { font-family: ui-sans-serif, system-ui, -apple-system, BlinkMacSystemFont, "
        '"Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; }\n'
        "        details > summary { list-style: none; }\n"
        "        details > summary::-webkit-details-marker { display: none; }\n"
        "        /* Markdown Specific Overrides to match React styling */\n"
        "        .rendered-markdown pre { background: #0b1120; color: #f8fafc; padding: 1rem;"
        " border-radius: 0.5rem; overflow-x: auto; }\n"
        '        .rendered-markdown code { font-family: "JetBrains Mono", monospace; '
        "font-size: 0.9em; }\n"
        "        .rendered-markdown :not(pre) > code { background: rgba(148,163,184,0.2);"
        " color: #0f172a; padding: 0.1rem 0.3rem; border-radius: 0.25rem; }\n"
        "        .rendered-markdown table { border-collapse: collapse; width: 100%; margin: 1em 0; }\n"
        "        .rendered-markdown th, .rendered-markdown td { border: 1px solid #e5e7eb; padding: 0.5rem; }\n"
        "        .rendered-markdown th { background-color: #f9fafb; }\n"
        "        .citation-link { color: #2563eb; font-weight: 600; margin-left: 0.15rem;"
        " text-decoration: none; }\n"
        "        .citation-link:hover { text-decoration: underline; }\n"
        "    </style>\n"
        '</head>\n<body class="bg-gray-100 text-gray-900">\n'
        '    <div class="max-w-screen-md mx-auto p-8 bg-white min-h-screen shadow-xl '
        'my-8 rounded-xl border border-gray-200">\n'
        f'        <h1 class="text-3xl font-extrabold mb-2 text-gray-900 tracking-tight">{html.escape(title)}</h1>\n'
        '        <div class="text-xs text-gray-400 font-mono mb-8 bg-gray-50 inline-block '
        'px-2 py-1 rounded">Generated by gai2html</div>\n\n'
        f"        {conversation_html}\n\n"
        '        <footer class="mt-16 pt-8 border-t border-gray-100 text-center text-gray-400 '
        'text-sm">\n'
        "            Exported from Google AI Studio JSON\n"
        "        </footer>\n"
        "    </div>\n"
        f"    {INTERACTIVITY_SCRIPT}\n"
        "</body>\n</html>"
    )


# --- CLI Entrypoint ---


def get_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="Convert Google AI Studio JSON export to standalone HTML (Tailwind Styled)",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
    )
    parser.add_argument("path", type=Path, help="Path to JSON file")
    parser.add_argument(
        "-o",
        "--output",
        type=Path,
        help="Output HTML path (default: same name as input)",
    )
    parser.add_argument("-t", "--title", help="Title of the HTML page", metavar="title")
    parser.add_argument(
        "--no-thoughts", action="store_true", help="Hide Model thoughts/reasoning"
    )
    parser.add_argument(
        "--expand-thoughts", action="store_true", help="Expand thoughts by default"
    )
    parser.add_argument(
        "-g",
        "--show-citations",
        action="store_true",
        help=(
            "Render inline citation links that point directly to their sources and display "
            "the consolidated sources list when grounding data is present"
        ),
    )
    return parser.parse_args()


def main() -> None:
    args = get_args()
    if not args.path.exists():
        print(f"Error: File not found: {args.path}")
        return
    try:
        with open(args.path, "r", encoding="utf-8") as f:
            data = json.load(f)
    except json.JSONDecodeError:
        print("Error: Invalid JSON file")
        return
    turns = parse_and_merge_conversation(data)
    title = args.title if args.title else args.path.stem
    html_content = generate_html(
        turns,
        title,
        args.no_thoughts,
        args.expand_thoughts,
        args.show_citations,
    )
    output_path = args.output if args.output else args.path.with_suffix(".html")
    with open(output_path, "w", encoding="utf-8") as f:
        f.write(html_content)
    print(f"Successfully converted {args.path} -> {output_path}")


if __name__ == "__main__":
    main()
