"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessPoint = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const core = require("aws-cdk-lib/core");
const aws_s3objectlambda_1 = require("aws-cdk-lib/aws-s3objectlambda");
class AccessPointBase extends core.Resource {
    /** Implement the `IAccessPoint.domainName` field. */
    get domainName() {
        const urlSuffix = this.stack.urlSuffix;
        return `${this.accessPointName}-${this.env.account}.s3-object-lambda.${urlSuffix}`;
    }
    /** Implement the `IAccessPoint.regionalDomainName` field. */
    get regionalDomainName() {
        const urlSuffix = this.stack.urlSuffix;
        const region = this.env.region;
        return `${this.accessPointName}-${this.env.account}.s3-object-lambda.${region}.${urlSuffix}`;
    }
    /** Implement the `IAccessPoint.virtualHostedUrlForObject` method. */
    virtualHostedUrlForObject(key, options) {
        const domainName = options?.regional ?? true ? this.regionalDomainName : this.domainName;
        const prefix = `https://${domainName}`;
        if (!key) {
            return prefix;
        }
        if (key.startsWith('/')) {
            key = key.slice(1);
        }
        if (key.endsWith('/')) {
            key = key.slice(0, -1);
        }
        return `${prefix}/${key}`;
    }
}
/**
 * Checks the access point name against the rules in https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-names
 * @param name The name of the access point
 */
function validateAccessPointName(name) {
    if (name.length < 3 || name.length > 50) {
        throw new Error('Access point name must be between 3 and 50 characters long');
    }
    if (name.endsWith('-s3alias')) {
        throw new Error('Access point name cannot end with the suffix -s3alias');
    }
    if (name[0] === '-' || name[name.length - 1] === '-') {
        throw new Error('Access point name cannot begin or end with a dash');
    }
    if (!/^[0-9a-z](.(?![\.A-Z_]))+[0-9a-z]$/.test(name)) {
        throw new Error('Access point name must begin with a number or lowercase letter and not contain underscores, uppercase letters, or periods');
    }
}
/**
  * An S3 object lambda access point for intercepting and
  * transforming `GetObject` requests.
  */
class AccessPoint extends AccessPointBase {
    /**
     * Reference an existing AccessPoint defined outside of the CDK code.
     */
    static fromAccessPointAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAccessPointAttributes);
            }
            throw error;
        }
        const arn = core.Arn.split(attrs.accessPointArn, core.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arn.resourceName) {
            throw new Error('Unable to parse acess point name');
        }
        const name = arn.resourceName;
        class Import extends AccessPointBase {
            constructor() {
                super(...arguments);
                this.accessPointArn = attrs.accessPointArn;
                this.accessPointCreationDate = attrs.accessPointCreationDate;
                this.accessPointName = name;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.accessPointName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AccessPoint);
            }
            throw error;
        }
        if (props.accessPointName) {
            validateAccessPointName(props.accessPointName);
        }
        const supporting = new s3.CfnAccessPoint(this, 'SupportingAccessPoint', {
            bucket: props.bucket.bucketName,
        });
        const allowedFeatures = [];
        if (props.supportsGetObjectPartNumber) {
            allowedFeatures.push('GetObject-PartNumber');
        }
        if (props.supportsGetObjectRange) {
            allowedFeatures.push('GetObject-Range');
        }
        const accessPoint = new aws_s3objectlambda_1.CfnAccessPoint(this, id, {
            name: this.physicalName,
            objectLambdaConfiguration: {
                allowedFeatures,
                cloudWatchMetricsEnabled: props.cloudWatchMetricsEnabled,
                supportingAccessPoint: supporting.attrArn,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: {
                            AwsLambda: {
                                FunctionArn: props.handler.functionArn,
                                FunctionPayload: props.payload ? JSON.stringify(props.payload) : undefined,
                            },
                        },
                    },
                ],
            },
        });
        this.accessPointName = accessPoint.ref;
        this.accessPointArn = accessPoint.attrArn;
        this.accessPointCreationDate = accessPoint.attrCreationDate;
        props.handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3-object-lambda:WriteGetObjectResponse'],
            resources: ['*'],
        }));
    }
}
exports.AccessPoint = AccessPoint;
_a = JSII_RTTI_SYMBOL_1;
AccessPoint[_a] = { fqn: "@aws-cdk/aws-s3objectlambda-alpha.AccessPoint", version: "2.117.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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