# -*- coding: utf-8 -*-
# ###
# Copyright (c) 2013, Rice University
# This software is subject to the provisions of the GNU Affero General
# Public License version 3 (AGPLv3).
# See LICENCE.txt for details.
# ###
from __future__ import unicode_literals
import sys
import base64
import io
import logging
import mimetypes
import os
import uuid
import re

from copy import deepcopy

import jinja2
import lxml.html

from lxml import etree

from .formatters import HTMLFormatter
from .models import (
    flatten_model, flatten_to_documents,
    content_to_etree, etree_to_content,
    Binder, TranslucentBinder,
    Document, DocumentPointer, CompositeDocument,
    TRANSLUCENT_BINDER_ID,
    INTERNAL_REFERENCE_TYPE,
    INLINE_REFERENCE_TYPE,
    )
from .html_parsers import (parse_metadata, parse_navigation_html_to_tree,
                           parse_resources, DocumentPointerMetadataParser,
                           HTML_DOCUMENT_NAMESPACES)

logger = logging.getLogger('cnxepub')


__all__ = (
    'get_model_extensions',
    'adapt_single_html',
    )


def get_model_extensions(binder):
    extensions = {}
    # Set model identifier file extensions.
    for model in flatten_model(binder):
        if isinstance(model, (Binder, TranslucentBinder,)):
            continue
        ext = mimetypes.guess_extension(model.media_type, strict=False)
        if ext is None:
            raise ValueError("Can't apply an extension to media-type '{}'."
                             .format(model.media_type))
        extensions[model.id] = ext
        extensions[model.ident_hash] = ext
    return extensions


def adapt_single_html(html):
    """Adapts a single html document generated by
    ``.formatters.SingleHTMLFormatter`` to a ``models.Binder``
    """
    html_root = etree.fromstring(html)

    metadata = parse_metadata(html_root.xpath('//*[@data-type="metadata"]')[0])
    id_ = metadata['cnx-archive-uri'] or 'book'

    binder = Binder(id_, metadata=metadata)
    nav_tree = parse_navigation_html_to_tree(html_root, id_)

    body = html_root.xpath('//xhtml:body', namespaces=HTML_DOCUMENT_NAMESPACES)
    _adapt_single_html_tree(binder, body[0], nav_tree, top_metadata=metadata)

    return binder


def _adapt_single_html_tree(parent, elem, nav_tree, top_metadata,
                            id_map=None, depth=0):
    title_overrides = [i.get('title') for i in nav_tree['contents']]

    # A dictionary to allow look up of a document and new id using the old html
    # element id

    if id_map is None:
        id_map = {}

    def fix_generated_ids(page, id_map):
        """Fix element ids (remove auto marker) and populate id_map."""

        content = content_to_etree(page.content)

        new_ids = set()
        suffix = 0
        for element in content.xpath('.//*[@id]'):
            id_val = element.get('id')
            if id_val.startswith('auto_'):
                # It's possible that an auto_ prefix was injected using a page
                # ID that incorporated the page_ prefix. We'll remove that
                # first if it exists so the auto prefixing fix works whether
                # it is present or not.
                new_val = re.sub(r'^auto_page_', 'auto_', id_val)

                # We max split with two to avoid breaking up ID values that
                # may have originally included '_' and only undo the auto_{id}_
                # prefixing injected by a formatter
                new_val = new_val.split('_', 2)[-1]
                # Did content from different pages w/ same original id
                # get moved to the same page?
                if new_val in new_ids:
                    while (new_val + str(suffix)) in new_ids:
                        suffix += 1
                    new_val = new_val + str(suffix)
            else:
                new_val = id_val
            new_ids.add(new_val)
            element.set('id', new_val)
            if id_val.startswith('page_'):
                # We want to map any references to the generated page ID
                # directly to the page
                id_map['#{}'.format(id_val)] = (page, '')
            else:
                id_map['#{}'.format(id_val)] = (page, new_val)

        id_map['#{}'.format(page.id)] = (page, '')
        assert not (page.id and '@' in page.id)
        id_map['#{}'.format(page.id.split('@')[0])] = (page, '')

        page.content = etree_to_content(content)

    def fix_links(page, id_map):
        """Remap all intra-book links, replace with value from id_map."""

        content = content_to_etree(page.content)
        for i in content.xpath('.//*[starts-with(@href, "#")]',
                               namespaces=HTML_DOCUMENT_NAMESPACES):
            ref_val = i.attrib['href']
            if ref_val in id_map:
                target_page, target = id_map[ref_val]
                if page == target_page:
                        i.attrib['href'] = '#{}'.format(target)
                else:
                    target_id = target_page.id.split('@')[0]
                    if not target:  # link to page
                        i.attrib['href'] = '/contents/{}'.format(target_id)
                    else:
                        i.attrib['href'] = '/contents/{}#{}'.format(
                            target_id, target)
            else:
                logger.error('Bad href: {}'.format(ref_val))

        page.content = etree_to_content(content)

    def _compute_id(p, elem, key):
        """Compute id and shortid from parent uuid and child attr"""
        p_ids = [p.id.split('@')[0]]
        if 'cnx-archive-uri' in p.metadata and p.metadata['cnx-archive-uri']:
            p_ids.insert(0, p.metadata['cnx-archive-uri'].split('@')[0])

        p_uuid = None
        for p_id in p_ids:
            try:
                p_uuid = uuid.UUID(p_id)
                break
            except ValueError:
                pass

        assert p_uuid is not None, 'Should always find a parent UUID'
        uuid_key = elem.get('data-uuid-key', elem.get('class', key))
        return str(uuid.uuid5(p_uuid, uuid_key))

    def _compute_shortid(ident_hash):
        """Compute shortId from uuid or ident_hash"""
        ver = None
        assert '@' in ident_hash
        (id_str, ver) = ident_hash.split('@')
        id_uuid = uuid.UUID(id_str)

        shortid = (base64.urlsafe_b64encode(id_uuid.bytes)[:8]).decode('utf-8')
        if ver:
            return '@'.join((shortid, ver))
        else:
            return shortid

    # Adapt each <div data-type="unit|chapter|page|composite-page"> into
    # translucent binders, documents and composite documents
    for child in elem.getchildren():
        data_type = child.attrib.get('data-type')

        if data_type in ('unit', 'chapter', 'composite-chapter',
                         'page', 'composite-page'):
            # metadata munging for all node types, in one place
            metadata = parse_metadata(
                    child.xpath('./*[@data-type="metadata"]')[0])

            # Handle version, id and uuid from metadata
            if not metadata.get('version'):
                if data_type.startswith('composite-'):
                    if top_metadata.get('version') is not None:
                        metadata['version'] = top_metadata['version']
                elif parent.metadata.get('version') is not None:
                    metadata['version'] = parent.metadata['version']

            uuid_key = child.get('data-uuid-key')
            child_id = child.attrib.get('id')
            id_ = metadata.get('cnx-archive-uri') or (child_id
                                                      if not uuid_key
                                                      else None)
            if not id_:
                id_ = _compute_id(parent, child, metadata.get('title'))
                assert metadata.get('version')
                metadata['cnx-archive-uri'] = \
                    '@'.join((id_, metadata['version']))
                metadata['cnx-archive-shortid'] = None

            if (metadata.get('cnx-archive-uri') and
                    not metadata.get('cnx-archive-shortid')):
                metadata['cnx-archive-shortid'] = \
                        _compute_shortid(metadata['cnx-archive-uri'])

            shortid = metadata.get('cnx-archive-shortid')

        if data_type in ['unit', 'chapter', 'composite-chapter']:
            # All the non-leaf node types
            title = lxml.html.HtmlElement(
                        child.xpath('*[@data-type="document-title"]',
                                    namespaces=HTML_DOCUMENT_NAMESPACES)[0]
                        ).text_content().strip()
            metadata.update({'title': title,
                             'id': id_,
                             'shortId': shortid,
                             'type': data_type})
            binder = Binder(id_, metadata=metadata)
            # Recurse
            _adapt_single_html_tree(binder, child,
                                    nav_tree['contents'].pop(0),
                                    top_metadata=top_metadata,
                                    id_map=id_map, depth=depth+1)
            parent.append(binder)
        elif data_type in ['page', 'composite-page']:
            # Leaf nodes
            nav_tree['contents'].pop(0)
            metadata_nodes = child.xpath("*[@data-type='metadata']",
                                         namespaces=HTML_DOCUMENT_NAMESPACES)
            for node in metadata_nodes:
                child.remove(node)
            for key in child.keys():
                assert key not in ('itemtype', 'itemscope'), 'Seems true'

            document_body = content_to_etree('')
            document_body.append(child)
            contents = etree.tostring(document_body)
            model = {
                'page': Document,
                'composite-page': CompositeDocument,
                }[child.attrib['data-type']]

            document = model(id_, contents, metadata=metadata)
            parent.append(document)

            fix_generated_ids(document, id_map)  # also populates id_map
        else:
            assert data_type in ['metadata', None], \
                'Unknown data-type for child node'
            # Expected non-nodal child types
            pass

    assert len(parent) == len(title_overrides), 'Nav TOC should HTML structure'

    for i, node in enumerate(parent):
        parent.set_title_for_node(node, title_overrides[i])

    # only fixup links after all pages
    # processed for whole book, to allow for foward links
    if depth == 0:
        for page in flatten_to_documents(parent):
            fix_links(page, id_map)
