import os
import sys
from typing import Any, Dict, List

from IPython import get_ipython
from IPython.core.magic import register_line_magic
from IPython.display import display, Markdown
from snippyts import to_json, to_txt, from_json, from_txt

from .llms import CHATGPT
from .logger import LOGGER
from .util import (
    get_user_home_folder,
    load_dotenv,
    set_key,
    OPENAI_API_KEY_NAME,
    PATH_HISTORY_NAME,
    remove_code_markup,
    update_env,
    validate_history_value
)




INSTRUCTIONS_FUNC_ASK_FOR_CODE = (
    "{}\n\n\nReturn Python code only, no text or comments."
)

INSTRUCTIONS_FUNC_ASK_FOR_MORE_CODE = {
    "Got the following output:\n\n\n{}\n\n\nPlease change the code as appropriate "
    "so that I can try to re-run it again. Focus on fixing any issues mentioned "
    "in the code. As before, please return only runnable Python code, no text or "
    "comments at this time."
}



def ask_for_code(
    model: Any,
    prompt: str,
    cell_idx: int = -1,
    context: List[Dict[str, str]] = []
) -> str:
    if cell_idx < 1:
        prompt = INSTRUCTIONS_FUNC_ASK_FOR_CODE.format(prompt)
    else:
        prompt = INSTRUCTIONS_FUNC_ASK_FOR_MORE_CODE.format(get_cell_content(cell_idx))
    return model(prompt, history=context)


def ask_for_answer(
    model: Any,
    prompt: str,
    context: List[Dict[str, str]] = []
) -> str:
    return model(prompt, history=context)


try:

    @register_line_magic
    def ask_and_run(model: Any, prompt: str, context: list[dict] = []) -> None:
        raw = ask_for_code(model, prompt, context=context)
        response = remove_code_markup(raw)
        if raw and not response:
            response = raw
        shell = get_ipython()
        payload = {
            'source': 'set_next_input',
            'text': response,
            'replace': False,
        }
        shell.payload_manager.write_payload(payload, single=False)
        shell.run_cell(response)
        return raw, response

    @register_line_magic
    def ask_and_write(model: Any, prompt: str, context: list[dict] = []) -> None:
        response = ask_for_answer(model, prompt, context=context)
        display(Markdown(response))
        return response

    @register_line_magic
    def ask_and_recode(model: Any, prompt: str, context: list[dict] = []) -> None:
        raw = ask_for_more_code(model, prompt, context=context)
        response = remove_code_markup(raw)
        shell = get_ipython()
        payload = {
            'source': 'set_next_input',
            'text': response,
            'replace': False,
        }
        shell.payload_manager.write_payload(payload, single=False)
        shell.run_cell(response)
        return raw, response


    @register_line_magic
    def get_cell_content(line: int) -> str:
        """
        (Generated by ChatGPT.)

        Retrieve the content of a Jupyter notebook cell by its run number.
        Usage:
            %get_cell_content 5  --> Get the contents of the cell with execution count 5.
        """
        ip = get_ipython()

        # Check if a valid execution number was provided
        try:
            execution_count = int(line.strip())
        except ValueError:
            return "Please provide a valid integer representing the execution count."

        # Access the execution history using the execution_count
        history = ip.history_manager.get_range()

        for session_id, exec_count, cell_content in history:
            if exec_count == execution_count:
                return cell_content

        return f"No cell found with execution count {execution_count}."


except AttributeError:
    LOGGER.warn(
        """if this message is displayed message means the code is being """
        """run outside of a jupyter notebook, which is not the intended """
        """use."""
    )
    pass




class Conversation:

    def __init__(
        self,
        llm: Any = CHATGPT,
        path_history=os.environ[PATH_HISTORY_NAME]
    ) -> None:
        self.history = []
        self._path_history = path_history
        if not llm:
            raise ValueError("An LLM must be specified.")
        self.llm = llm
        folder_message = f"history folder set to \"{self.path_history}\""
        LOGGER.warn(
            folder_message +
            ". Call \{CONV|Conversation\}.path_history(path: str) to modify it."
        )

    @property
    def path_history(self) -> str:
        return self._path_history

    @path_history.setter
    def path_history(self, path: str) -> None:
        self._path_history = path
        update_env(PATH_HISTORY_NAME, self._path_history)
        LOGGER.info(f"history folder updated to \"{self._path_history}\".")

    def serialize(self) -> None:
        path = self.path_history
        highest_folder = os.path.dirname(path)
        os.makedirs(highest_folder, exist_ok=True)
        to_json(self.history, path)
        LOGGER.info(f"history written to \"{self._path_history}\".")

    def deserialize(self) -> None:
        path = self.path_history
        if os.path.exists(path):
            self.history.extend(from_json(path))
            LOGGER.info(f"history read from \"{self._path_history}\".")
        LOGGER.warn(f"no history written yet at \"{self._path_history}\".")

    def clear(self) -> None:
        self.history.clear()

    def __call__(
        self,
        prompt: str,
        content: str,
        cell_idx: int = -1,
        context: int = 0
    ) -> str:

        if content == "code":

            if not context:
                raw, response = ask_and_run(self.llm, prompt, context=[])
            else:
                raw, response = ask_and_run(
                    self.llm, prompt, cell_idx, context=self.history[-context * 2:]
                )

            if response:
                self.add_question(prompt)
                self.add_response(raw)

        elif content == "text":

            if cell_idx < 1:
                if not context:
                    response = ask_and_write(self.llm, prompt)
                else:
                    response = ask_and_write(
                        self.llm, prompt, context=self.history[-context * 2:]
                    )
            elif cell_idx > 0:
                if not context:
                    response = ask_and_write(
                        self.llm, prompt, context=self.history[-cell_idx * 2:]
                    )
                else:
                    idx = max(context, cell_idx)
                    response = ask_and_write(
                        self.llm, prompt, context=self.history[-idx * 2:]
                    )

            if response:
                self.add_question(prompt)
                self.add_response(response)

        else:

            raise NotImplementedError(content)


    def add_question(self, prompt: str) -> None:
        self.history.append({
            "role": "user",
            "content": prompt
        })


    def add_response(self, response: str) -> None:
        self.history.append({
            "role": "system",
            "content": response
        })



CONV = Conversation()

def TXT(prompt: str, cell_idx: int = -1, history: int = 0) -> None:
    validate_history_value(history)
    CONV(prompt, "text", cell_idx, history)

def PY(prompt: str, cell_idx: int = -1, history: int = 0) -> None:
    validate_history_value(history)
    CONV(prompt, "code", cell_idx, history)
