# Copyright 2015 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Base classes for client used to interact with Google Cloud APIs."""

import google.auth.credentials
from google.oauth2 import service_account
import google_auth_httplib2
import six

from google.cloud._helpers import _determine_default_project
from google.cloud.credentials import get_credentials


_GOOGLE_AUTH_CREDENTIALS_HELP = (
    'This library only supports credentials from google-auth-library-python. '
    'See https://google-cloud-python.readthedocs.io/en/latest/'
    'google-cloud-auth.html for help on authentication with this library.'
)


class _ClientFactoryMixin(object):
    """Mixin to allow factories that create credentials.

    .. note::

        This class is virtual.
    """

    @classmethod
    def from_service_account_json(cls, json_credentials_path, *args, **kwargs):
        """Factory to retrieve JSON credentials while creating client.

        :type json_credentials_path: str
        :param json_credentials_path: The path to a private key file (this file
                                      was given to you when you created the
                                      service account). This file must contain
                                      a JSON object with a private key and
                                      other credentials information (downloaded
                                      from the Google APIs console).

        :type args: tuple
        :param args: Remaining positional arguments to pass to constructor.

        :type kwargs: dict
        :param kwargs: Remaining keyword arguments to pass to constructor.

        :rtype: :class:`google.cloud.pubsub.client.Client`
        :returns: The client created with the retrieved JSON credentials.
        :raises: :class:`TypeError` if there is a conflict with the kwargs
                 and the credentials created by the factory.
        """
        if 'credentials' in kwargs:
            raise TypeError('credentials must not be in keyword arguments')
        credentials = service_account.Credentials.from_service_account_file(
            json_credentials_path)
        kwargs['credentials'] = credentials
        return cls(*args, **kwargs)


class Client(_ClientFactoryMixin):
    """Client to bundle configuration needed for API requests.

    Stores ``credentials`` and ``http`` object so that subclasses
    can pass them along to a connection class.

    If no value is passed in for ``http``, a :class:`httplib2.Http` object
    will be created and authorized with the ``credentials``. If not, the
    ``credentials`` and ``http`` need not be related.

    Callers and subclasses may seek to use the private key from
    ``credentials`` to sign data.

    A custom (non-``httplib2``) HTTP object must have a ``request`` method
    which accepts the following arguments:

    * ``uri``
    * ``method``
    * ``body``
    * ``headers``

    In addition, ``redirections`` and ``connection_type`` may be used.

    A custom ``http`` object will also need to be able to add a bearer token
    to API requests and handle token refresh on 401 errors.

    :type credentials: :class:`~google.auth.credentials.Credentials`
    :param credentials: (Optional) The OAuth2 Credentials to use for this
                        client. If not passed (and if no ``http`` object is
                        passed), falls back to the default inferred from the
                        environment.

    :type http: :class:`~httplib2.Http`
    :param http: (Optional) HTTP object to make requests. Can be any object
                 that defines ``request()`` with the same interface as
                 :meth:`~httplib2.Http.request`. If not passed, an
                 ``http`` object is created that is bound to the
                 ``credentials`` for the current object.
    """

    SCOPE = None
    """The scopes required for authenticating with a service.

    Needs to be set by subclasses.
    """

    def __init__(self, credentials=None, http=None):
        if (credentials is not None and
                not isinstance(
                    credentials, google.auth.credentials.Credentials)):
            raise ValueError(_GOOGLE_AUTH_CREDENTIALS_HELP)
        if credentials is None and http is None:
            credentials = get_credentials()
        self._credentials = google.auth.credentials.with_scopes_if_required(
            credentials, self.SCOPE)
        self._http_internal = http

    @property
    def _http(self):
        """Getter for object used for HTTP transport.

        :rtype: :class:`~httplib2.Http`
        :returns: An HTTP object.
        """
        if self._http_internal is None:
            self._http_internal = google_auth_httplib2.AuthorizedHttp(
                self._credentials)
        return self._http_internal


class _ClientProjectMixin(object):
    """Mixin to allow setting the project on the client.

    :type project: str
    :param project: the project which the client acts on behalf of. If not
                    passed falls back to the default inferred from the
                    environment.

    :raises: :class:`EnvironmentError` if the project is neither passed in nor
             set in the environment. :class:`ValueError` if the project value
             is invalid.
    """

    def __init__(self, project=None):
        project = self._determine_default(project)
        if project is None:
            raise EnvironmentError('Project was not passed and could not be '
                                   'determined from the environment.')
        if isinstance(project, six.binary_type):
            project = project.decode('utf-8')
        if not isinstance(project, six.string_types):
            raise ValueError('Project must be a string.')
        self.project = project

    @staticmethod
    def _determine_default(project):
        """Helper:  use default project detection."""
        return _determine_default_project(project)


class ClientWithProject(Client, _ClientProjectMixin):
    """Client that also stores a project.

    :type project: str
    :param project: the project which the client acts on behalf of. If not
                    passed falls back to the default inferred from the
                    environment.

    :type credentials: :class:`~google.auth.credentials.Credentials`
    :param credentials: (Optional) The OAuth2 Credentials to use for this
                        client. If not passed (and if no ``http`` object is
                        passed), falls back to the default inferred from the
                        environment.

    :type http: :class:`~httplib2.Http`
    :param http: (Optional) HTTP object to make requests. Can be any object
                 that defines ``request()`` with the same interface as
                 :meth:`~httplib2.Http.request`. If not passed, an
                 ``http`` object is created that is bound to the
                 ``credentials`` for the current object.

    :raises: :class:`ValueError` if the project is neither passed in nor
             set in the environment.
    """

    def __init__(self, project=None, credentials=None, http=None):
        _ClientProjectMixin.__init__(self, project=project)
        Client.__init__(self, credentials=credentials, http=http)
