"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_codedeploy_1 = require("@aws-sdk/client-codedeploy");
const logger_1 = require("./logger");
/**
 * The lambda function called from CloudFormation for this custom resource.
 *
 * @param event
 * @returns attribues of the deployment that was created
 */
async function handler(event) {
    const logger = new logger_1.Logger();
    const codedeployClient = new client_codedeploy_1.CodeDeploy({});
    logger.appendKeys({
        stackEvent: event.RequestType,
    });
    switch (event.RequestType) {
        case 'Create':
        case 'Update': {
            // create deployment
            const props = event.ResourceProperties;
            let autoRollbackConfiguration;
            if (props.autoRollbackConfigurationEnabled === 'true') {
                autoRollbackConfiguration = {
                    enabled: true,
                    events: props.autoRollbackConfigurationEvents.split(','),
                };
            }
            else if (props.autoRollbackConfigurationEnabled === 'false') {
                autoRollbackConfiguration = {
                    enabled: false,
                };
            }
            const resp = await codedeployClient.createDeployment({
                applicationName: props.applicationName,
                deploymentConfigName: props.deploymentConfigName,
                deploymentGroupName: props.deploymentGroupName,
                autoRollbackConfiguration,
                description: props.description,
                revision: {
                    revisionType: 'AppSpecContent',
                    appSpecContent: {
                        content: props.revisionAppSpecContent,
                    },
                },
            });
            if (!resp.deploymentId) {
                throw new Error('No deploymentId received from call to CreateDeployment');
            }
            logger.appendKeys({
                deploymentId: resp.deploymentId,
            });
            logger.info('Created new deployment');
            return {
                PhysicalResourceId: resp.deploymentId,
                Data: {
                    deploymentId: resp.deploymentId,
                },
            };
        }
        case 'Delete':
            logger.appendKeys({
                deploymentId: event.PhysicalResourceId,
            });
            // cancel deployment and rollback
            try {
                const resp = await codedeployClient.stopDeployment({
                    deploymentId: event.PhysicalResourceId,
                    autoRollbackEnabled: true,
                });
                logger.info(`Stopped deployment: ${resp.status} ${resp.statusMessage}`);
            }
            catch (e) {
                logger.warn('Ignoring error', { error: e });
            }
            return {
                PhysicalResourceId: event.PhysicalResourceId,
                Data: {
                    deploymentId: event.PhysicalResourceId,
                },
            };
        default:
            logger.error('Unknown stack event');
            throw new Error(`Unknown request type: ${event.RequestType}`);
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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