from __future__ import annotations

import logging
from typing import Any, cast

from sift.calculated_channels.v2.calculated_channels_pb2 import (
    CalculatedChannelValidationResult,
    CreateCalculatedChannelResponse,
    GetCalculatedChannelRequest,
    GetCalculatedChannelResponse,
    ListCalculatedChannelsRequest,
    ListCalculatedChannelsResponse,
    ListCalculatedChannelVersionsRequest,
    ListCalculatedChannelVersionsResponse,
    UpdateCalculatedChannelRequest,
    UpdateCalculatedChannelResponse,
)
from sift.calculated_channels.v2.calculated_channels_pb2_grpc import CalculatedChannelServiceStub

from sift_client._internal.low_level_wrappers.base import LowLevelClientBase
from sift_client.sift_types.calculated_channel import (
    CalculatedChannel,
    CalculatedChannelCreate,
    CalculatedChannelUpdate,
)
from sift_client.transport import GrpcClient, WithGrpcClient

logger = logging.getLogger(__name__)


class CalculatedChannelsLowLevelClient(LowLevelClientBase, WithGrpcClient):
    """Low-level client for the CalculatedChannelsAPI.

    This class provides a thin wrapper around the autogenerated bindings for the CalculatedChannelsAPI.
    """

    def __init__(self, grpc_client: GrpcClient):
        """Initialize the CalculatedChannelsLowLevelClient.

        Args:
            grpc_client: The gRPC client to use for making API calls.
        """
        super().__init__(grpc_client)

    async def get_calculated_channel(
        self,
        *,
        calculated_channel_id: str | None = None,
        client_key: str | None = None,
        organization_id: str | None = None,
    ) -> CalculatedChannel:
        """Get a calculated channel by ID or client key.

        Args:
            calculated_channel_id: The ID of the calculated channel.
            client_key: The client key of the calculated channel.
            organization_id: The organization ID (required if using client_key and user belongs to multiple organizations).

        Returns:
            The CalculatedChannel.
        """
        request = GetCalculatedChannelRequest(
            calculated_channel_id=calculated_channel_id or "",
            client_key=client_key or "",
            organization_id=organization_id or "",
        )

        response = await self._grpc_client.get_stub(
            CalculatedChannelServiceStub
        ).GetCalculatedChannel(request)
        grpc_calculated_channel = cast("GetCalculatedChannelResponse", response).calculated_channel
        return CalculatedChannel._from_proto(grpc_calculated_channel)

    async def create_calculated_channel(
        self, *, create: CalculatedChannelCreate
    ) -> tuple[CalculatedChannel, list[Any]]:
        request = create.to_proto()
        response = await self._grpc_client.get_stub(
            CalculatedChannelServiceStub
        ).CreateCalculatedChannel(request)
        response = cast("CreateCalculatedChannelResponse", response)

        calculated_channel = CalculatedChannel._from_proto(response.calculated_channel)
        inapplicable_assets = list(response.inapplicable_assets)

        return calculated_channel, inapplicable_assets

    async def list_all_calculated_channels(
        self,
        *,
        query_filter: str | None = None,
        order_by: str | None = None,
        max_results: int | None = None,
        page_size: int | None = None,
        organization_id: str | None = None,
    ) -> list[CalculatedChannel]:
        """List all calculated channels matching the given query.

        Args:
            query_filter: The CEL query filter.
            order_by: The field to order by.
            max_results: The maximum number of results to return.
            page_size: The number of results to return per page.
            organization_id: The organization ID (required if user belongs to multiple organizations).

        Returns:
            A list of CalculatedChannels matching the given query.
        """
        return await self._handle_pagination(
            self.list_calculated_channels,
            kwargs={"organization_id": organization_id, "query_filter": query_filter},
            page_size=page_size,
            order_by=order_by,
            max_results=max_results,
        )

    async def list_calculated_channels(
        self,
        *,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
        organization_id: str | None = None,
    ) -> tuple[list[CalculatedChannel], str]:
        """List calculated channels with pagination.

        Args:
            page_size: The number of results to return per page.
            page_token: The page token for pagination.
            query_filter: The CEL query filter.
            order_by: The field to order by.
            organization_id: The organization ID.

        Returns:
            A tuple of (list of CalculatedChannels, next_page_token).
        """
        request_kwargs: dict[str, Any] = {}
        if page_size is not None:
            request_kwargs["page_size"] = page_size
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if order_by is not None:
            request_kwargs["order_by"] = order_by
        if organization_id is not None:
            request_kwargs["organization_id"] = organization_id

        request = ListCalculatedChannelsRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(
            CalculatedChannelServiceStub
        ).ListCalculatedChannels(request)
        response = cast("ListCalculatedChannelsResponse", response)

        calculated_channels = [
            CalculatedChannel._from_proto(cc) for cc in response.calculated_channels
        ]
        return calculated_channels, response.next_page_token

    async def update_calculated_channel(
        self,
        *,
        update: CalculatedChannelUpdate,
        user_notes: str | None = None,
    ) -> tuple[CalculatedChannel, list[Any]]:
        """Update a calculated channel.

        Args:
            update: The CalculatedChannelUpdate to apply.
            user_notes: User notes for the update.

        Returns:
            A tuple of (updated CalculatedChannel, list of inapplicable assets).
        """
        grpc_calculated_channel, update_mask = update.to_proto_with_mask()
        request = UpdateCalculatedChannelRequest(
            calculated_channel=grpc_calculated_channel,
            update_mask=update_mask,
            user_notes=user_notes,
        )
        response = await self._grpc_client.get_stub(
            CalculatedChannelServiceStub
        ).UpdateCalculatedChannel(request)
        response = cast("UpdateCalculatedChannelResponse", response)

        updated_calculated_channel = CalculatedChannel._from_proto(response.calculated_channel)
        inapplicable_assets = [
            cast("CalculatedChannelValidationResult", asset)
            for asset in response.inapplicable_assets
        ]

        return updated_calculated_channel, inapplicable_assets

    async def list_calculated_channel_versions(
        self,
        *,
        calculated_channel_id: str | None = None,
        client_key: str | None = None,
        organization_id: str | None = None,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
    ) -> tuple[list[CalculatedChannel], str]:
        """List versions of a calculated channel.

        Args:
            calculated_channel_id: The ID of the calculated channel.
            client_key: The client key of the calculated channel.
            organization_id: The organization ID. Required if your user belongs to multiple organizations.
            page_size: The number of results to return per page.
            page_token: The page token for pagination.
            query_filter: The CEL query filter.
            order_by: The field to order by.

        Returns:
            A tuple of (list of CalculatedChannel versions, next_page_token).

        Raises:
            ValueError: If neither calculated_channel_id nor client_key is provided.
        """
        request_kwargs = {}
        if calculated_channel_id:
            request_kwargs = {"calculated_channel_id": calculated_channel_id}
        elif client_key:
            request_kwargs = {"client_key": client_key}
        else:
            raise ValueError("Either calculated_channel_id or client_key must be provided")

        if page_size is not None:
            request_kwargs["page_size"] = str(page_size)
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if order_by is not None:
            request_kwargs["order_by"] = order_by
        if organization_id is not None:
            request_kwargs["organization_id"] = organization_id

        request = ListCalculatedChannelVersionsRequest(**request_kwargs)  # type: ignore # mypy thinks we should pass an int
        response = await self._grpc_client.get_stub(
            CalculatedChannelServiceStub
        ).ListCalculatedChannelVersions(request)
        response = cast("ListCalculatedChannelVersionsResponse", response)

        versions = [
            CalculatedChannel._from_proto(cc) for cc in response.calculated_channel_versions
        ]
        return versions, response.next_page_token

    async def list_all_calculated_channel_versions(
        self,
        *,
        calculated_channel_id: str | None = None,
        client_key: str | None = None,
        organization_id: str | None = None,
        query_filter: str | None = None,
        order_by: str | None = None,
        limit: int | None = None,
    ) -> list[CalculatedChannel]:
        """List all versions of a calculated channel."""
        return await self._handle_pagination(
            self.list_calculated_channel_versions,
            kwargs={
                "organization_id": organization_id,
                "calculated_channel_id": calculated_channel_id,
                "client_key": client_key,
                "query_filter": query_filter,
            },
            order_by=order_by,
            max_results=limit,
        )
