from __future__ import annotations

import logging
from typing import TYPE_CHECKING, Any, cast

from sift.reports.v1.reports_pb2 import (
    CancelReportRequest,
    GetReportRequest,
    GetReportResponse,
    ListReportsRequest,
    ListReportsResponse,
    RerunReportRequest,
    RerunReportResponse,
    UpdateReportRequest,
)
from sift.reports.v1.reports_pb2_grpc import ReportServiceStub

from sift_client._internal.low_level_wrappers.base import LowLevelClientBase
from sift_client.sift_types.report import Report, ReportUpdate
from sift_client.transport import WithGrpcClient

if TYPE_CHECKING:
    from sift_client.transport.grpc_transport import GrpcClient

# Configure logging
logger = logging.getLogger(__name__)


class ReportsLowLevelClient(LowLevelClientBase, WithGrpcClient):
    """Low-level client for the ReportsAPI.

    This class provides a thin wrapper around the autogenerated bindings for the ReportsAPI.
    """

    def __init__(self, grpc_client: GrpcClient):
        """Initialize the ReportsLowLevelClient.

        Args:
            grpc_client: The gRPC client to use for making API calls.
        """
        super().__init__(grpc_client)

    async def get_report(self, report_id: str) -> Report:
        """Get a report by report_id.

        Args:
            report_id: The report ID to get.

        Returns:
            The Report.

        Raises:
            ValueError: If report_id is not provided.
        """
        if not report_id:
            raise ValueError("report_id must be provided")

        request = GetReportRequest(report_id=report_id)
        response = await self._grpc_client.get_stub(ReportServiceStub).GetReport(request)
        grpc_report = cast("GetReportResponse", response).report
        return Report._from_proto(grpc_report)

    async def list_reports(
        self,
        *,
        page_size: int | None = None,
        page_token: str | None = None,
        query_filter: str | None = None,
        organization_id: str | None = None,
        order_by: str | None = None,
    ) -> tuple[list[Report], str]:
        """List reports with optional filtering and pagination.

        Args:
            page_size: The maximum number of reports to return.
            page_token: A page token for pagination.
            query_filter: A CEL filter string.
            organization_id: The organization ID to filter by.
            order_by: How to order the retrieved reports.

        Returns:
            A tuple of (reports, next_page_token).
        """
        request_kwargs: dict[str, Any] = {}
        if page_size is not None:
            request_kwargs["page_size"] = page_size
        if page_token is not None:
            request_kwargs["page_token"] = page_token
        if query_filter is not None:
            request_kwargs["filter"] = query_filter
        if organization_id is not None:
            request_kwargs["organization_id"] = organization_id
        if order_by is not None:
            request_kwargs["order_by"] = order_by

        request = ListReportsRequest(**request_kwargs)
        response = await self._grpc_client.get_stub(ReportServiceStub).ListReports(request)
        response = cast("ListReportsResponse", response)
        reports = [Report._from_proto(report) for report in response.reports]
        return reports, response.next_page_token

    async def list_all_reports(
        self,
        *,
        query_filter: str | None = None,
        organization_id: str | None = None,
        order_by: str | None = None,
        max_results: int | None = None,
    ) -> list[Report]:
        """List all reports with optional filtering.

        Args:
            query_filter: A CEL filter string.
            organization_id: The organization ID to filter by.
            order_by: How to order the retrieved reports.
            max_results: Maximum number of results to return.

        Returns:
            A list of all matching reports.
        """
        return await self._handle_pagination(
            self.list_reports,
            kwargs={
                "query_filter": query_filter,
                "organization_id": organization_id,
            },
            order_by=order_by,
            max_results=max_results,
        )

    async def rerun_report(self, report_id: str) -> tuple[str, str]:
        """Rerun a report.

        Args:
            report_id: The ID of the report to rerun.

        Returns:
            A tuple of (job_id, new_report_id).

        Raises:
            ValueError: If report_id is not provided.
        """
        if not report_id:
            raise ValueError("report_id must be provided")

        request = RerunReportRequest(report_id=report_id)
        response = await self._grpc_client.get_stub(ReportServiceStub).RerunReport(request)
        response = cast("RerunReportResponse", response)
        return response.job_id, response.report_id

    async def cancel_report(self, report_id: str) -> None:
        """Cancel a report.

        Args:
            report_id: The ID of the report to cancel.

        Raises:
            ValueError: If report_id is not provided.
        """
        if not report_id:
            raise ValueError("report_id must be provided")

        request = CancelReportRequest(report_id=report_id)
        await self._grpc_client.get_stub(ReportServiceStub).CancelReport(request)

    async def update_report(self, update: ReportUpdate) -> Report:
        """Update a report.

        Args:
            update: The updates to apply.

        Returns:
            The updated report.
        """
        report_proto, field_mask = update.to_proto_with_mask()
        request = UpdateReportRequest(report=report_proto, update_mask=field_mask)
        await self._grpc_client.get_stub(ReportServiceStub).UpdateReport(request)
        # Unfortunately, updating a report doesn't return the updated report.
        return await self.get_report(update.resource_id)
