"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LegacyStackSynthesizer = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const cfn_fn_1 = require("../cfn-fn");
const construct_compat_1 = require("../construct-compat");
const asset_parameters_1 = require("../private/asset-parameters");
const _shared_1 = require("./_shared");
const stack_synthesizer_1 = require("./stack-synthesizer");
/**
 * The well-known name for the docker image asset ECR repository. All docker
 * image assets will be pushed into this repository with an image tag based on
 * the source hash.
 */
const ASSETS_ECR_REPOSITORY_NAME = 'aws-cdk/assets';
/**
 * This allows users to work around the fact that the ECR repository is
 * (currently) not configurable by setting this context key to their desired
 * repository name. The CLI will auto-create this ECR repository if it's not
 * already created.
 */
const ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY = 'assets-ecr-repository-name';
/**
 * Use the CDK classic way of referencing assets
 *
 * This synthesizer will generate CloudFormation parameters for every referenced
 * asset, and use the CLI's current credentials to deploy the stack.
 *
 * - It does not support cross-account deployment (the CLI must have credentials
 *   to the account you are trying to deploy to).
 * - It cannot be used with **CDK Pipelines**. To deploy using CDK Pipelines,
 *   you must use the `DefaultStackSynthesizer`.
 * - Each asset will take up a CloudFormation Parameter in your template. Keep in
 *   mind that there is a maximum of 200 parameters in a CloudFormation template.
 *   To use determinstic asset locations instead, use `CliCredentialsStackSynthesizer`.
 *
 * Be aware that your CLI credentials must be valid for the duration of the
 * entire deployment. If you are using session credentials, make sure the
 * session lifetime is long enough.
 *
 * This is the only StackSynthesizer that supports customizing asset behavior
 * by overriding `Stack.addFileAsset()` and `Stack.addDockerImageAsset()`.
 */
class LegacyStackSynthesizer extends stack_synthesizer_1.StackSynthesizer {
    constructor() {
        super(...arguments);
        this.cycle = false;
        /**
         * The image ID of all the docker image assets that were already added to this
         * stack (to avoid duplication).
         */
        this.addedImageAssets = new Set();
    }
    bind(stack) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_Stack(stack);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (this.stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this.stack = stack;
    }
    addFileAsset(asset) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_FileAssetSource(asset);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addFileAsset);
            }
            throw error;
        }
        _shared_1.assertBound(this.stack);
        // Backwards compatibility hack. We have a number of conflicting goals here:
        //
        // - We want put the actual logic in this class
        // - We ALSO want to keep supporting people overriding Stack.addFileAsset (for backwards compatibility,
        // because that mechanism is currently used to make CI/CD scenarios work)
        // - We ALSO want to allow both entry points from user code (our own framework
        // code will always call stack.deploymentMechanism.addFileAsset() but existing users
        // may still be calling `stack.addFileAsset()` directly.
        //
        // Solution: delegate call to the stack, but if the stack delegates back to us again
        // then do the actual logic.
        //
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddFileAsset(asset);
        }
        this.cycle = true;
        try {
            const stack = this.stack;
            return withoutDeprecationWarnings(() => stack.addFileAsset(asset));
        }
        finally {
            this.cycle = false;
        }
    }
    addDockerImageAsset(asset) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_DockerImageAssetSource(asset);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDockerImageAsset);
            }
            throw error;
        }
        _shared_1.assertBound(this.stack);
        // See `addFileAsset` for explanation.
        // @deprecated: this can be removed for v2
        if (this.cycle) {
            return this.doAddDockerImageAsset(asset);
        }
        this.cycle = true;
        try {
            const stack = this.stack;
            return withoutDeprecationWarnings(() => stack.addDockerImageAsset(asset));
        }
        finally {
            this.cycle = false;
        }
    }
    /**
     * Synthesize the associated stack to the session
     */
    synthesize(session) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_ISynthesisSession(session);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.synthesize);
            }
            throw error;
        }
        _shared_1.assertBound(this.stack);
        this.synthesizeStackTemplate(this.stack, session);
        // Just do the default stuff, nothing special
        this.emitStackArtifact(this.stack, session);
    }
    doAddDockerImageAsset(asset) {
        var _b, _c;
        _shared_1.assertBound(this.stack);
        // check if we have an override from context
        const repositoryNameOverride = this.stack.node.tryGetContext(ASSETS_ECR_REPOSITORY_NAME_OVERRIDE_CONTEXT_KEY);
        const repositoryName = (_c = (_b = asset.repositoryName) !== null && _b !== void 0 ? _b : repositoryNameOverride) !== null && _c !== void 0 ? _c : ASSETS_ECR_REPOSITORY_NAME;
        const imageTag = asset.sourceHash;
        const assetId = asset.sourceHash;
        // only add every image (identified by source hash) once for each stack that uses it.
        if (!this.addedImageAssets.has(assetId)) {
            if (!asset.directoryName) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                repositoryName,
                imageTag,
                id: assetId,
                packaging: 'container-image',
                path: asset.directoryName,
                sourceHash: asset.sourceHash,
                buildArgs: asset.dockerBuildArgs,
                target: asset.dockerBuildTarget,
                file: asset.dockerFile,
                networkMode: asset.networkMode,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
            this.addedImageAssets.add(assetId);
        }
        return {
            imageUri: `${this.stack.account}.dkr.ecr.${this.stack.region}.${this.stack.urlSuffix}/${repositoryName}:${imageTag}`,
            repositoryName,
        };
    }
    doAddFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        let params = this.assetParameters.node.tryFindChild(asset.sourceHash);
        if (!params) {
            params = new asset_parameters_1.FileAssetParameters(this.assetParameters, asset.sourceHash);
            if (!asset.fileName || !asset.packaging) {
                throw new Error(`LegacyStackSynthesizer does not support this type of file asset: ${JSON.stringify(asset)}`);
            }
            const metadata = {
                path: asset.fileName,
                id: asset.sourceHash,
                packaging: asset.packaging,
                sourceHash: asset.sourceHash,
                s3BucketParameter: params.bucketNameParameter.logicalId,
                s3KeyParameter: params.objectKeyParameter.logicalId,
                artifactHashParameter: params.artifactHashParameter.logicalId,
            };
            this.stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.ASSET, metadata);
        }
        const bucketName = params.bucketNameParameter.valueAsString;
        // key is prefix|postfix
        const encodedKey = params.objectKeyParameter.valueAsString;
        const s3Prefix = cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const s3Filename = cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.split(cxapi.ASSET_PREFIX_SEPARATOR, encodedKey));
        const objectKey = `${s3Prefix}${s3Filename}`;
        const httpUrl = `https://s3.${this.stack.region}.${this.stack.urlSuffix}/${bucketName}/${objectKey}`;
        const s3ObjectUrl = `s3://${bucketName}/${objectKey}`;
        return { bucketName, objectKey, httpUrl, s3ObjectUrl, s3Url: httpUrl };
    }
    get assetParameters() {
        _shared_1.assertBound(this.stack);
        if (!this._assetParameters) {
            this._assetParameters = new construct_compat_1.Construct(this.stack, 'AssetParameters');
        }
        return this._assetParameters;
    }
}
exports.LegacyStackSynthesizer = LegacyStackSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
LegacyStackSynthesizer[_a] = { fqn: "@aws-cdk/core.LegacyStackSynthesizer", version: "1.153.0" };
function withoutDeprecationWarnings(block) {
    const orig = process.env.JSII_DEPRECATED;
    process.env.JSII_DEPRECATED = 'quiet';
    try {
        return block();
    }
    finally {
        process.env.JSII_DEPRECATED = orig;
    }
}
//# sourceMappingURL=data:application/json;base64,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