"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const libfs = require("../../lib/fs");
module.exports = {
    files: {
        'does not change with the file name'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const content = 'Hello, world!';
            const input1 = path.join(workdir, 'input1.txt');
            const input2 = path.join(workdir, 'input2.txt');
            const input3 = path.join(workdir, 'input3.txt');
            fs.writeFileSync(input1, content);
            fs.writeFileSync(input2, content);
            fs.writeFileSync(input3, content + '.'); // add one character, hash should be different
            // WHEN
            const hash1 = libfs.fingerprint(input1);
            const hash2 = libfs.fingerprint(input2);
            const hash3 = libfs.fingerprint(input3);
            // THEN
            test.deepEqual(hash1, hash2);
            test.notDeepEqual(hash3, hash1);
            test.done();
        },
        'works on empty files'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const input1 = path.join(workdir, 'empty');
            const input2 = path.join(workdir, 'empty');
            fs.writeFileSync(input1, '');
            fs.writeFileSync(input2, '');
            // WHEN
            const hash1 = libfs.fingerprint(input1);
            const hash2 = libfs.fingerprint(input2);
            // THEN
            test.deepEqual(hash1, hash2);
            test.done();
        },
    },
    directories: {
        'works on directories'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            libfs.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            const hashCopy = libfs.fingerprint(outdir);
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'ignores requested files'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            libfs.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            fs.writeFileSync(path.join(outdir, `${hashSrc}.ignoreme`), 'Ignore me!');
            const hashCopy = libfs.fingerprint(outdir, { exclude: ['*.ignoreme'] });
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'changes with file names'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const cpydir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            libfs.copyDirectory(srcdir, cpydir);
            // be careful not to break a symlink
            fs.renameSync(path.join(cpydir, 'normal-dir', 'file-in-subdir.txt'), path.join(cpydir, 'move-me.txt'));
            // WHEN
            const hashSrc = libfs.fingerprint(srcdir);
            const hashCopy = libfs.fingerprint(cpydir);
            // THEN
            test.notDeepEqual(hashSrc, hashCopy);
            test.done();
        },
    },
    symlinks: {
        'changes with the contents of followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = libfs.fingerprint(dir2);
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = libfs.fingerprint(dir2);
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = libfs.fingerprint(dir2);
            // THEN
            test.notDeepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
        'does not change with the contents of un-followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = libfs.fingerprint(dir2, { follow: libfs.FollowMode.NEVER });
            // THEN
            test.deepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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