'''
Return data to a postgresql server

To enable this returner the minion will need the psycopg2 installed and
the following values configured in the minion or master config::

    returner.postgres.host: 'salt'
    returner.postgres.user: 'salt'
    returner.postgres.passwd: 'salt'
    returner.postgres.db: 'salt'
    returner.postgres.port: 5432

Running the following commands as the postgres user should create the database
correctly::

    psql << EOF
    CREATE ROLE salt WITH PASSWORD 'salt';
    CREATE DATABASE salt WITH OWNER salt;
    EOF

    psql -h localhost -U salt << EOF
    --
    -- Table structure for table 'jids'
    --

    DROP TABLE IF EXISTS jids;
    CREATE TABLE jids (
      jid   bigint PRIMARY KEY,
      load  text NOT NULL
    );

    --
    -- Table structure for table 'returns'
    --

    DROP TABLE IF EXISTS returns;
    CREATE TABLE returns (
      fun       text NOT NULL,
      jid       varchar(20) NOT NULL,
      return    text NOT NULL,
      id        text NOT NULL,
      success   boolean
    );
    CREATE INDEX ON returns (id);
    CREATE INDEX ON returns (jid);
    CREATE INDEX ON returns (fun);

    DROP TABLE IF EXISTS highstate;
    -- CREATE TABLE highstate (
    --   jid       bigint PRIMARY KEY,
    --   resource  text NOT NULL,
    --   return    hstore
    -- );
    CREATE INDEX return_idx_gist
      ON highstate
      USING gist
      (return);
    EOF

Required python modules: psycopg2
'''

# Import python libs
import json

# Import third party libs
try:
    import psycopg2
    #import psycopg2.extras
    has_postgres = True
except ImportError:
    has_postgres = False


def __virtual__():
    if not has_postgres:
        return False
    return 'postgres'

def _get_conn():
    '''
    Return a postgres connection.
    '''
    return psycopg2.connect(
            host=__salt__['config.option']('returner.postgres.host'),
            user=__salt__['config.option']('returner.postgres.user'),
            password=__salt__['config.option']('returner.postgres.passwd'),
            database=__salt__['config.option']('returner.postgres.db'),
            port=__salt__['config.option']('returner.postgres.port'))

def _close_conn(conn):
    conn.commit()
    conn.close()

def returner(ret):
    '''
    Return data to a postgres server
    '''
    conn = _get_conn()
    cur = conn.cursor()
    sql = '''INSERT INTO returns
            (fun, jid, return, id, success)
            VALUES (%s, %s, %s, %s, %s)'''
    cur.execute(sql, (ret['fun'], ret['jid'], json.dumps(ret['return']), ret['id'], ret['success']))
    _close_conn(conn)

def save_load(jid, load):
    '''
    Save the load to the specified jid id
    '''
    conn = _get_conn()
    cur = conn.cursor()
    sql = '''INSERT INTO jids (jid, load) VALUES (%s, %s)'''

    cur.execute(sql, (jid, json.dumps(load)))
    _close_conn(conn)

def get_load(jid):
    '''
    Return the load data that marks a specified jid
    '''
    conn = _get_conn()
    cur = conn.cursor()
    sql = '''SELECT load FROM jids WHERE jid = %s;'''

    cur.execute(sql, (jid,))
    data = cur.fetchone()
    if data:
        return json.loads(data)
    _close_conn(conn)
    return {}

def get_jid(jid):
    '''
    Return the information returned when the specified job id was executed
    '''
    conn = _get_conn()
    cur = conn.cursor()
    sql = '''SELECT id, full_ret FROM salt_returns WHERE jid = %s'''

    cur.execute(sql, (jid,))
    data = cur.fetchall()
    ret = {}
    if data:
        for minion, full_ret in data:
            ret[minion] = json.loads(full_ret)
    _close_conn(conn)
    return ret

def get_fun(fun):
    '''
    Return a dict of the last function called for all minions
    '''
    conn = _get_conn()
    cur = conn.cursor()
    sql = '''SELECT s.id,s.jid, s.full_ret
            FROM salt_returns s
            JOIN ( SELECT MAX(jid) AS jid FROM salt_returns GROUP BY fun, id) max
            ON s.jid = max.jid
            WHERE s.fun = %s
            '''

    cur.execute(sql, (fun,))
    data = cur.fetchall()

    ret = {}
    if data:
        for minion, jid, full_ret in data:
            ret[minion] = json.loads(full_ret)
    _close_conn(conn)
    return ret

def get_jids():
    '''
    Return a list of all job ids
    '''
    conn = _get_conn()
    cur = conn.cursor()
    sql = '''SELECT jid FROM jids'''

    cur.execute(sql)
    data = cur.fetchall()
    ret = []
    for jid in data:
        ret.append(jid[0])
    _close_conn(conn)
    return ret

def get_minions():
    '''
    Return a list of minions
    '''
    conn = _get_conn()
    cur = conn.cursor()
    sql = '''SELECT DISTINCT id FROM salt_returns'''

    cur.execute(sql)
    data = cur.fetchall()
    ret = []
    for minion in data:
        ret.append(minion[0])
    _close_conn(conn)
    return ret
