"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationExportStack = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const sns = require("aws-cdk-lib/aws-sns");
const index_1 = require("./index");
class CloudFormationExportStack extends aws_cdk_lib_1.Stack {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        const noOfParameters = 5;
        const snsTopics = [];
        const conditionsDict = {};
        for (let i = 1; i <= noOfParameters; i++) {
            const snsTopicArnParam = new aws_cdk_lib_1.CfnParameter(this, `snsTopicArn${i}`, {
                type: 'String',
                description: 'SNS Topic ARN receive alarms and send Lambda errors to.',
            });
            const snsTopic = sns.Topic.fromTopicArn(this, `snsTopic${i}`, snsTopicArnParam.valueAsString);
            snsTopics.push(snsTopic);
            const condition = new cdk.CfnCondition(this, `conditionSnsTopic${i}`, {
                expression: cdk.Fn.conditionNot(cdk.Fn.conditionEquals('', snsTopic.topicArn)),
            });
            conditionsDict[snsTopicArnParam.valueAsString] = condition;
        }
        const maxNumberOfLogsParam = new aws_cdk_lib_1.CfnParameter(this, `maxNumberOfLogs`, {
            type: 'Number',
            description: 'Max number of logs to send to SNS Topic.',
        });
        const lambdaErrorSnsSender = new index_1.LambdaErrorSnsSender(this, 'lambdaErrorSnsSender', {
            snsTopics: snsTopics,
            maxNumberOfLogs: maxNumberOfLogsParam.valueAsNumber,
        });
        const subscriptions = lambdaErrorSnsSender.node
            .findAll()
            .filter((s) => s instanceof sns.CfnSubscription);
        for (const subscription of subscriptions) {
            if (conditionsDict[subscription.topicArn]) {
                subscription.cfnOptions.condition =
                    conditionsDict[subscription.topicArn];
            }
        }
        const snsErrorFunc = lambdaErrorSnsSender.node.findChild('lambdaSnsError');
        for (const lambdaPermission of snsErrorFunc.permissionsNode.children) {
            if (lambdaPermission instanceof lambda.CfnPermission) {
                if (lambdaPermission.sourceArn &&
                    conditionsDict[lambdaPermission.sourceArn]) {
                    lambdaPermission.cfnOptions.condition =
                        conditionsDict[lambdaPermission.sourceArn];
                }
            }
        }
        const lambdaPolicy = snsErrorFunc.node
            .findAll()
            .find((c) => c instanceof iam.CfnPolicy);
        const newPolicyStatements = [];
        for (const statement of (lambdaPolicy.policyDocument?.statements ??
            [])) {
            let addCondition = false;
            let condition;
            for (const resource of statement.resources) {
                if (conditionsDict[resource]) {
                    addCondition = true;
                    condition = conditionsDict[resource];
                    break;
                }
            }
            if (addCondition && condition) {
                const newStatement = {
                    'Fn::If': [
                        condition.logicalId,
                        statement.toJSON(),
                        ,
                        { Ref: 'AWS::NoValue' },
                    ],
                };
                newPolicyStatements.push(newStatement);
            }
            else {
                newPolicyStatements.push(statement.toJSON());
            }
        }
        lambdaPolicy.policyDocument = {
            Statement: newPolicyStatements,
        };
        setNodeNames(this.node.children);
    }
}
exports.CloudFormationExportStack = CloudFormationExportStack;
const app = new aws_cdk_lib_1.App();
new CloudFormationExportStack(app, 'lambda-error-sns-sender-cf', {
    synthesizer: new aws_cdk_lib_1.DefaultStackSynthesizer({
        generateBootstrapVersionRule: false,
    }),
});
app.synth();
function setNodeNames(constructs, parentName) {
    let i = 0;
    for (const construct of constructs) {
        let newParentName = parentName;
        if (construct.node.defaultChild instanceof cdk.CfnElement) {
            newParentName = parentName
                ? `${parentName}${construct.node.id}`
                : construct.node.id;
            construct.node.defaultChild.overrideLogicalId(newParentName);
        }
        if (construct instanceof lambda.CfnPermission) {
            construct.overrideLogicalId(`${newParentName}Permission${i}`);
            i++;
        }
        setNodeNames(construct.node.children, newParentName);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWRGb3JtYXRpb25FeHBvcnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvY2xvdWRGb3JtYXRpb25FeHBvcnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkNBTXFCO0FBQ3JCLG1DQUFtQztBQUNuQywyQ0FBMkM7QUFDM0MsaURBQWlEO0FBQ2pELDJDQUEyQztBQUUzQyxtQ0FBK0M7QUFFL0MsTUFBYSx5QkFBMEIsU0FBUSxtQkFBSztJQUNsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQW9CLEVBQUU7UUFDOUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFeEIsTUFBTSxjQUFjLEdBQUcsQ0FBQyxDQUFDO1FBQ3pCLE1BQU0sU0FBUyxHQUFnQixFQUFFLENBQUM7UUFDbEMsTUFBTSxjQUFjLEdBQXFDLEVBQUUsQ0FBQztRQUU1RCxLQUFLLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLElBQUksY0FBYyxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQ3hDLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSwwQkFBWSxDQUFDLElBQUksRUFBRSxjQUFjLENBQUMsRUFBRSxFQUFFO2dCQUNqRSxJQUFJLEVBQUUsUUFBUTtnQkFDZCxXQUFXLEVBQUUseURBQXlEO2FBQ3ZFLENBQUMsQ0FBQztZQUVILE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUNyQyxJQUFJLEVBQ0osV0FBVyxDQUFDLEVBQUUsRUFDZCxnQkFBZ0IsQ0FBQyxhQUFhLENBQy9CLENBQUM7WUFDRixTQUFTLENBQUMsSUFBSSxDQUFDLFFBQXFCLENBQUMsQ0FBQztZQUV0QyxNQUFNLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLG9CQUFvQixDQUFDLEVBQUUsRUFBRTtnQkFDcEUsVUFBVSxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUM3QixHQUFHLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxFQUFFLEVBQUUsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUM5QzthQUNGLENBQUMsQ0FBQztZQUVILGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsR0FBRyxTQUFTLENBQUM7U0FDNUQ7UUFFRCxNQUFNLG9CQUFvQixHQUFHLElBQUksMEJBQVksQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDckUsSUFBSSxFQUFFLFFBQVE7WUFDZCxXQUFXLEVBQUUsMENBQTBDO1NBQ3hELENBQUMsQ0FBQztRQUVILE1BQU0sb0JBQW9CLEdBQUcsSUFBSSw0QkFBb0IsQ0FDbkQsSUFBSSxFQUNKLHNCQUFzQixFQUN0QjtZQUNFLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLGVBQWUsRUFBRSxvQkFBb0IsQ0FBQyxhQUFhO1NBQ3BELENBQ0YsQ0FBQztRQUVGLE1BQU0sYUFBYSxHQUFHLG9CQUFvQixDQUFDLElBQUk7YUFDNUMsT0FBTyxFQUFFO2FBQ1QsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLFlBQVksR0FBRyxDQUFDLGVBQWUsQ0FBMEIsQ0FBQztRQUU1RSxLQUFLLE1BQU0sWUFBWSxJQUFJLGFBQWEsRUFBRTtZQUN4QyxJQUFJLGNBQWMsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLEVBQUU7Z0JBQ3pDLFlBQVksQ0FBQyxVQUFVLENBQUMsU0FBUztvQkFDL0IsY0FBYyxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsQ0FBQzthQUN6QztTQUNGO1FBRUQsTUFBTSxZQUFZLEdBQUcsb0JBQW9CLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDdEQsZ0JBQWdCLENBQ0UsQ0FBQztRQUVyQixLQUFLLE1BQU0sZ0JBQWdCLElBQUksWUFBWSxDQUFDLGVBQWUsQ0FBQyxRQUFRLEVBQUU7WUFDcEUsSUFBSSxnQkFBZ0IsWUFBWSxNQUFNLENBQUMsYUFBYSxFQUFFO2dCQUNwRCxJQUNFLGdCQUFnQixDQUFDLFNBQVM7b0JBQzFCLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsRUFDMUM7b0JBQ0EsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLFNBQVM7d0JBQ25DLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztpQkFDOUM7YUFDRjtTQUNGO1FBRUQsTUFBTSxZQUFZLEdBQUcsWUFBWSxDQUFDLElBQUk7YUFDbkMsT0FBTyxFQUFFO2FBQ1QsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLFlBQVksR0FBRyxDQUFDLFNBQVMsQ0FBa0IsQ0FBQztRQUU1RCxNQUFNLG1CQUFtQixHQUFHLEVBQUUsQ0FBQztRQUMvQixLQUFLLE1BQU0sU0FBUyxJQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsRUFBRSxVQUFVO1lBQzlELEVBQUUsQ0FBMEIsRUFBRTtZQUM5QixJQUFJLFlBQVksR0FBRyxLQUFLLENBQUM7WUFDekIsSUFBSSxTQUF1QyxDQUFDO1lBRTVDLEtBQUssTUFBTSxRQUFRLElBQUksU0FBUyxDQUFDLFNBQVMsRUFBRTtnQkFDMUMsSUFBSSxjQUFjLENBQUMsUUFBUSxDQUFDLEVBQUU7b0JBQzVCLFlBQVksR0FBRyxJQUFJLENBQUM7b0JBQ3BCLFNBQVMsR0FBRyxjQUFjLENBQUMsUUFBUSxDQUFDLENBQUM7b0JBQ3JDLE1BQU07aUJBQ1A7YUFDRjtZQUVELElBQUksWUFBWSxJQUFJLFNBQVMsRUFBRTtnQkFDN0IsTUFBTSxZQUFZLEdBQUc7b0JBQ25CLFFBQVEsRUFBRTt3QkFDUixTQUFTLENBQUMsU0FBUzt3QkFDbkIsU0FBUyxDQUFDLE1BQU0sRUFBRTt3QkFDbEIsQUFEbUI7d0JBRW5CLEVBQUUsR0FBRyxFQUFFLGNBQWMsRUFBRTtxQkFDeEI7aUJBQ0YsQ0FBQztnQkFFRixtQkFBbUIsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7YUFDeEM7aUJBQU07Z0JBQ0wsbUJBQW1CLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO2FBQzlDO1NBQ0Y7UUFFRCxZQUFZLENBQUMsY0FBYyxHQUFHO1lBQzVCLFNBQVMsRUFBRSxtQkFBbUI7U0FDL0IsQ0FBQztRQUVGLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ25DLENBQUM7Q0FDRjtBQS9HRCw4REErR0M7QUFFRCxNQUFNLEdBQUcsR0FBRyxJQUFJLGlCQUFHLEVBQUUsQ0FBQztBQUV0QixJQUFJLHlCQUF5QixDQUFDLEdBQUcsRUFBRSw0QkFBNEIsRUFBRTtJQUMvRCxXQUFXLEVBQUUsSUFBSSxxQ0FBdUIsQ0FBQztRQUN2Qyw0QkFBNEIsRUFBRSxLQUFLO0tBQ3BDLENBQUM7Q0FDSCxDQUFDLENBQUM7QUFFSCxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7QUFFWixTQUFTLFlBQVksQ0FBQyxVQUF3QixFQUFFLFVBQW1CO0lBQ2pFLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNWLEtBQUssTUFBTSxTQUFTLElBQUksVUFBVSxFQUFFO1FBQ2xDLElBQUksYUFBYSxHQUFHLFVBQVUsQ0FBQztRQUUvQixJQUFJLFNBQVMsQ0FBQyxJQUFJLENBQUMsWUFBWSxZQUFZLEdBQUcsQ0FBQyxVQUFVLEVBQUU7WUFDekQsYUFBYSxHQUFHLFVBQVU7Z0JBQ3hCLENBQUMsQ0FBQyxHQUFHLFVBQVUsR0FBRyxTQUFTLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRTtnQkFDckMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ3RCLFNBQVMsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGlCQUFpQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzlEO1FBRUQsSUFBSSxTQUFTLFlBQVksTUFBTSxDQUFDLGFBQWEsRUFBRTtZQUM3QyxTQUFTLENBQUMsaUJBQWlCLENBQUMsR0FBRyxhQUFhLGFBQWEsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUM5RCxDQUFDLEVBQUUsQ0FBQztTQUNMO1FBRUQsWUFBWSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLGFBQWEsQ0FBQyxDQUFDO0tBQ3REO0FBQ0gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7XG4gIEFwcCxcbiAgU3RhY2ssXG4gIFN0YWNrUHJvcHMsXG4gIENmblBhcmFtZXRlcixcbiAgRGVmYXVsdFN0YWNrU3ludGhlc2l6ZXIsXG59IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBzbnMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXNucyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IExhbWJkYUVycm9yU25zU2VuZGVyIH0gZnJvbSAnLi9pbmRleCc7XG5cbmV4cG9ydCBjbGFzcyBDbG91ZEZvcm1hdGlvbkV4cG9ydFN0YWNrIGV4dGVuZHMgU3RhY2sge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU3RhY2tQcm9wcyA9IHt9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICBjb25zdCBub09mUGFyYW1ldGVycyA9IDU7XG4gICAgY29uc3Qgc25zVG9waWNzOiBzbnMuVG9waWNbXSA9IFtdO1xuICAgIGNvbnN0IGNvbmRpdGlvbnNEaWN0OiBSZWNvcmQ8c3RyaW5nLCBjZGsuQ2ZuQ29uZGl0aW9uPiA9IHt9O1xuXG4gICAgZm9yIChsZXQgaSA9IDE7IGkgPD0gbm9PZlBhcmFtZXRlcnM7IGkrKykge1xuICAgICAgY29uc3Qgc25zVG9waWNBcm5QYXJhbSA9IG5ldyBDZm5QYXJhbWV0ZXIodGhpcywgYHNuc1RvcGljQXJuJHtpfWAsIHtcbiAgICAgICAgdHlwZTogJ1N0cmluZycsXG4gICAgICAgIGRlc2NyaXB0aW9uOiAnU05TIFRvcGljIEFSTiByZWNlaXZlIGFsYXJtcyBhbmQgc2VuZCBMYW1iZGEgZXJyb3JzIHRvLicsXG4gICAgICB9KTtcblxuICAgICAgY29uc3Qgc25zVG9waWMgPSBzbnMuVG9waWMuZnJvbVRvcGljQXJuKFxuICAgICAgICB0aGlzLFxuICAgICAgICBgc25zVG9waWMke2l9YCxcbiAgICAgICAgc25zVG9waWNBcm5QYXJhbS52YWx1ZUFzU3RyaW5nLFxuICAgICAgKTtcbiAgICAgIHNuc1RvcGljcy5wdXNoKHNuc1RvcGljIGFzIHNucy5Ub3BpYyk7XG5cbiAgICAgIGNvbnN0IGNvbmRpdGlvbiA9IG5ldyBjZGsuQ2ZuQ29uZGl0aW9uKHRoaXMsIGBjb25kaXRpb25TbnNUb3BpYyR7aX1gLCB7XG4gICAgICAgIGV4cHJlc3Npb246IGNkay5Gbi5jb25kaXRpb25Ob3QoXG4gICAgICAgICAgY2RrLkZuLmNvbmRpdGlvbkVxdWFscygnJywgc25zVG9waWMudG9waWNBcm4pLFxuICAgICAgICApLFxuICAgICAgfSk7XG5cbiAgICAgIGNvbmRpdGlvbnNEaWN0W3Nuc1RvcGljQXJuUGFyYW0udmFsdWVBc1N0cmluZ10gPSBjb25kaXRpb247XG4gICAgfVxuXG4gICAgY29uc3QgbWF4TnVtYmVyT2ZMb2dzUGFyYW0gPSBuZXcgQ2ZuUGFyYW1ldGVyKHRoaXMsIGBtYXhOdW1iZXJPZkxvZ3NgLCB7XG4gICAgICB0eXBlOiAnTnVtYmVyJyxcbiAgICAgIGRlc2NyaXB0aW9uOiAnTWF4IG51bWJlciBvZiBsb2dzIHRvIHNlbmQgdG8gU05TIFRvcGljLicsXG4gICAgfSk7XG5cbiAgICBjb25zdCBsYW1iZGFFcnJvclNuc1NlbmRlciA9IG5ldyBMYW1iZGFFcnJvclNuc1NlbmRlcihcbiAgICAgIHRoaXMsXG4gICAgICAnbGFtYmRhRXJyb3JTbnNTZW5kZXInLFxuICAgICAge1xuICAgICAgICBzbnNUb3BpY3M6IHNuc1RvcGljcyxcbiAgICAgICAgbWF4TnVtYmVyT2ZMb2dzOiBtYXhOdW1iZXJPZkxvZ3NQYXJhbS52YWx1ZUFzTnVtYmVyLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgY29uc3Qgc3Vic2NyaXB0aW9ucyA9IGxhbWJkYUVycm9yU25zU2VuZGVyLm5vZGVcbiAgICAgIC5maW5kQWxsKClcbiAgICAgIC5maWx0ZXIoKHMpID0+IHMgaW5zdGFuY2VvZiBzbnMuQ2ZuU3Vic2NyaXB0aW9uKSBhcyBzbnMuQ2ZuU3Vic2NyaXB0aW9uW107XG5cbiAgICBmb3IgKGNvbnN0IHN1YnNjcmlwdGlvbiBvZiBzdWJzY3JpcHRpb25zKSB7XG4gICAgICBpZiAoY29uZGl0aW9uc0RpY3Rbc3Vic2NyaXB0aW9uLnRvcGljQXJuXSkge1xuICAgICAgICBzdWJzY3JpcHRpb24uY2ZuT3B0aW9ucy5jb25kaXRpb24gPVxuICAgICAgICAgIGNvbmRpdGlvbnNEaWN0W3N1YnNjcmlwdGlvbi50b3BpY0Fybl07XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3Qgc25zRXJyb3JGdW5jID0gbGFtYmRhRXJyb3JTbnNTZW5kZXIubm9kZS5maW5kQ2hpbGQoXG4gICAgICAnbGFtYmRhU25zRXJyb3InLFxuICAgICkgYXMgbGFtYmRhLkZ1bmN0aW9uO1xuXG4gICAgZm9yIChjb25zdCBsYW1iZGFQZXJtaXNzaW9uIG9mIHNuc0Vycm9yRnVuYy5wZXJtaXNzaW9uc05vZGUuY2hpbGRyZW4pIHtcbiAgICAgIGlmIChsYW1iZGFQZXJtaXNzaW9uIGluc3RhbmNlb2YgbGFtYmRhLkNmblBlcm1pc3Npb24pIHtcbiAgICAgICAgaWYgKFxuICAgICAgICAgIGxhbWJkYVBlcm1pc3Npb24uc291cmNlQXJuICYmXG4gICAgICAgICAgY29uZGl0aW9uc0RpY3RbbGFtYmRhUGVybWlzc2lvbi5zb3VyY2VBcm5dXG4gICAgICAgICkge1xuICAgICAgICAgIGxhbWJkYVBlcm1pc3Npb24uY2ZuT3B0aW9ucy5jb25kaXRpb24gPVxuICAgICAgICAgICAgY29uZGl0aW9uc0RpY3RbbGFtYmRhUGVybWlzc2lvbi5zb3VyY2VBcm5dO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgbGFtYmRhUG9saWN5ID0gc25zRXJyb3JGdW5jLm5vZGVcbiAgICAgIC5maW5kQWxsKClcbiAgICAgIC5maW5kKChjKSA9PiBjIGluc3RhbmNlb2YgaWFtLkNmblBvbGljeSkgYXMgaWFtLkNmblBvbGljeTtcblxuICAgIGNvbnN0IG5ld1BvbGljeVN0YXRlbWVudHMgPSBbXTtcbiAgICBmb3IgKGNvbnN0IHN0YXRlbWVudCBvZiAobGFtYmRhUG9saWN5LnBvbGljeURvY3VtZW50Py5zdGF0ZW1lbnRzID8/XG4gICAgICBbXSkgYXMgaWFtLlBvbGljeVN0YXRlbWVudFtdKSB7XG4gICAgICBsZXQgYWRkQ29uZGl0aW9uID0gZmFsc2U7XG4gICAgICBsZXQgY29uZGl0aW9uOiBjZGsuQ2ZuQ29uZGl0aW9uIHwgdW5kZWZpbmVkO1xuXG4gICAgICBmb3IgKGNvbnN0IHJlc291cmNlIG9mIHN0YXRlbWVudC5yZXNvdXJjZXMpIHtcbiAgICAgICAgaWYgKGNvbmRpdGlvbnNEaWN0W3Jlc291cmNlXSkge1xuICAgICAgICAgIGFkZENvbmRpdGlvbiA9IHRydWU7XG4gICAgICAgICAgY29uZGl0aW9uID0gY29uZGl0aW9uc0RpY3RbcmVzb3VyY2VdO1xuICAgICAgICAgIGJyZWFrO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIGlmIChhZGRDb25kaXRpb24gJiYgY29uZGl0aW9uKSB7XG4gICAgICAgIGNvbnN0IG5ld1N0YXRlbWVudCA9IHtcbiAgICAgICAgICAnRm46OklmJzogW1xuICAgICAgICAgICAgY29uZGl0aW9uLmxvZ2ljYWxJZCxcbiAgICAgICAgICAgIHN0YXRlbWVudC50b0pTT04oKSxcbiAgICAgICAgICAgICxcbiAgICAgICAgICAgIHsgUmVmOiAnQVdTOjpOb1ZhbHVlJyB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIH07XG5cbiAgICAgICAgbmV3UG9saWN5U3RhdGVtZW50cy5wdXNoKG5ld1N0YXRlbWVudCk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBuZXdQb2xpY3lTdGF0ZW1lbnRzLnB1c2goc3RhdGVtZW50LnRvSlNPTigpKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBsYW1iZGFQb2xpY3kucG9saWN5RG9jdW1lbnQgPSB7XG4gICAgICBTdGF0ZW1lbnQ6IG5ld1BvbGljeVN0YXRlbWVudHMsXG4gICAgfTtcblxuICAgIHNldE5vZGVOYW1lcyh0aGlzLm5vZGUuY2hpbGRyZW4pO1xuICB9XG59XG5cbmNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcblxubmV3IENsb3VkRm9ybWF0aW9uRXhwb3J0U3RhY2soYXBwLCAnbGFtYmRhLWVycm9yLXNucy1zZW5kZXItY2YnLCB7XG4gIHN5bnRoZXNpemVyOiBuZXcgRGVmYXVsdFN0YWNrU3ludGhlc2l6ZXIoe1xuICAgIGdlbmVyYXRlQm9vdHN0cmFwVmVyc2lvblJ1bGU6IGZhbHNlLFxuICB9KSxcbn0pO1xuXG5hcHAuc3ludGgoKTtcblxuZnVuY3Rpb24gc2V0Tm9kZU5hbWVzKGNvbnN0cnVjdHM6IElDb25zdHJ1Y3RbXSwgcGFyZW50TmFtZT86IHN0cmluZykge1xuICBsZXQgaSA9IDA7XG4gIGZvciAoY29uc3QgY29uc3RydWN0IG9mIGNvbnN0cnVjdHMpIHtcbiAgICBsZXQgbmV3UGFyZW50TmFtZSA9IHBhcmVudE5hbWU7XG5cbiAgICBpZiAoY29uc3RydWN0Lm5vZGUuZGVmYXVsdENoaWxkIGluc3RhbmNlb2YgY2RrLkNmbkVsZW1lbnQpIHtcbiAgICAgIG5ld1BhcmVudE5hbWUgPSBwYXJlbnROYW1lXG4gICAgICAgID8gYCR7cGFyZW50TmFtZX0ke2NvbnN0cnVjdC5ub2RlLmlkfWBcbiAgICAgICAgOiBjb25zdHJ1Y3Qubm9kZS5pZDtcbiAgICAgIGNvbnN0cnVjdC5ub2RlLmRlZmF1bHRDaGlsZC5vdmVycmlkZUxvZ2ljYWxJZChuZXdQYXJlbnROYW1lKTtcbiAgICB9XG5cbiAgICBpZiAoY29uc3RydWN0IGluc3RhbmNlb2YgbGFtYmRhLkNmblBlcm1pc3Npb24pIHtcbiAgICAgIGNvbnN0cnVjdC5vdmVycmlkZUxvZ2ljYWxJZChgJHtuZXdQYXJlbnROYW1lfVBlcm1pc3Npb24ke2l9YCk7XG4gICAgICBpKys7XG4gICAgfVxuXG4gICAgc2V0Tm9kZU5hbWVzKGNvbnN0cnVjdC5ub2RlLmNoaWxkcmVuLCBuZXdQYXJlbnROYW1lKTtcbiAgfVxufVxuIl19