"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
class KeyBase extends cdk_1.Resource {
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias) {
        return new alias_1.Alias(this, 'Alias', { name: alias, targetKey: this });
    }
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = cdk_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return;
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatements(statement);
    }
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalAndResource({
            grantee,
            actions,
            resourceArns: [this.keyArn],
            resource: this,
            resourceSelfArns: ['*']
        });
    }
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt');
    }
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor() {
                super(...arguments);
                this.keyArn = keyArn;
                this.policy = undefined;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.policy) {
            this.policy = props.policy;
        }
        else {
            this.policy = new aws_iam_1.PolicyDocument();
            this.allowAccountToAdmin();
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
        });
        this.keyArn = resource.attrArn;
        resource.applyRemovalPolicy(props.removalPolicy);
    }
    /**
     * Let users from this account admin this key.
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    allowAccountToAdmin() {
        const actions = [
            "kms:Create*",
            "kms:Describe*",
            "kms:Enable*",
            "kms:List*",
            "kms:Put*",
            "kms:Update*",
            "kms:Revoke*",
            "kms:Disable*",
            "kms:Get*",
            "kms:Delete*",
            "kms:ScheduleKeyDeletion",
            "kms:CancelKeyDeletion"
        ];
        this.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions,
            principals: [new iam.AccountRootPrincipal()]
        }));
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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