import { IResource, Resource, aws_elasticache, aws_ec2, aws_iam } from 'aws-cdk-lib';
import { Metric, MetricOptions } from 'aws-cdk-lib/aws-cloudwatch';
import { IKey } from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
import { DailySnapshotTime } from './daily-snapshot-time';
import { IUserGroup } from './user-group';
import { Engine } from './util';
/**
 * Storage unit for data storage in ElastiCache Serverless.
 */
export declare enum StorageUnit {
    /**
     * Gigabytes
     */
    GB = "GB"
}
/**
 * Interface for configuring data storage limits.
 */
export interface DataStorageOptions {
    /**
     * The lower limit for data storage the cache is set to use.
     *
     * @default - no lower limit
     */
    readonly minimum?: number;
    /**
     * The upper limit for data storage the cache is set to use.
     *
     * @default - no upper limit
     */
    readonly maximum?: number;
}
/**
 * The data storage limit.
 */
export declare class DataStorage {
    /**
     * Creates data storage settings with gigabytes as the unit.
     * @param options The configuration options containing min and/or max values.
     */
    static gb(options: DataStorageOptions): DataStorage;
    /**
     * The lower limit for data storage the cache is set to use.
     */
    readonly minimum?: number;
    /**
     * The upper limit for data storage the cache is set to use.
     */
    readonly maximum?: number;
    /**
     * The unit of the storage sizes.
     */
    readonly unit: StorageUnit;
    private constructor();
    /**
     * Render dataStorage property.
     *
     * @internal
     */
    _render(): aws_elasticache.CfnServerlessCache.DataStorageProperty;
    private validate;
}
/**
 * Interface for configuring ECPU per second limits.
 */
export interface ECPUPerSecondOptions {
    /**
     * The configuration for the minimum number of ECPUs the cache should be able consume per second.
     *
     * @default - no minimum configuration
     */
    readonly minimum?: number;
    /**
     * The configuration for the maximum number of ECPUs the cache can consume per second.
     *
     * @default - no maximum configuration
     */
    readonly maximum?: number;
}
/**
 * The configuration for the number of ElastiCache Processing Units (ECPU) the cache can consume per second.
 */
export declare class ECPUPerSecond {
    /**
     * Creates ECPU per second settings.
     *
     * @param options The configuration options containing min and/or max values.
     */
    static of(options: ECPUPerSecondOptions): ECPUPerSecond;
    /**
     * The configuration for the minimum number of ECPUs the cache should be able consume per second.
     */
    readonly minimum?: number;
    /**
     * The configuration for the maximum number of ECPUs the cache can consume per second.
     */
    readonly maximum?: number;
    private constructor();
    /**
     * Render ecpuPerSecond property.
     *
     * @internal
     */
    _render(): aws_elasticache.CfnServerlessCache.ECPUPerSecondProperty;
    private validate;
}
/**
 * Interface for an ElastiCache Serverless Cache
 */
export interface IServerlessCache extends IResource, aws_ec2.IConnectable {
    /**
     * The serverless cache ARN.
     *
     * @attribute
     */
    readonly serverlessCacheArn: string;
    /**
     * The serverless cache name.
     */
    readonly serverlessCacheName: string;
    /**
     * The DNS hostname of the cache node.
     *
     * @attribute
     */
    readonly endpointAddress: string;
    /**
     * The port number that the cache engine is listening on.
     *
     * @attribute
     */
    readonly endpointPort: number;
    /**
     * Grant the given identity the specified actions.
     */
    grant(grantee: aws_iam.IGrantable, ...actions: string[]): aws_iam.Grant;
    /**
     * Grant the given identity connection access to the cache.
     */
    grantConnect(grantee: aws_iam.IGrantable): aws_iam.Grant;
    /**
     * Create a CloudWatch metric.
     */
    metric(metricName: string, props?: MetricOptions): Metric;
}
/**
 * The version number of the engine the serverless cache is compatible with.
 */
export declare enum MajorVersion {
    /**
     * Version 7
     */
    VER_7 = "7",
    /**
     * Version 8
     */
    VER_8 = "8"
}
/**
 * Properties for defining an ElastiCache Serverless Cache.
 */
export interface ServerlessCacheProps {
    /**
     * The engine the serverless cache is compatible with.
     */
    readonly engine: Engine;
    /**
     * The unique identifier of the serverless cache.
     * The name can have up to 40 characters, and must not contain spaces.
     *
     * @default - auto generate
     */
    readonly serverlessCacheName?: string;
    /**
     * The usage limits for storage and ElastiCache Processing Units for the cache.
     *
     * @default - no limits.
     */
    readonly cacheUsageLimits?: CacheUsageLimits;
    /**
     * The daily time when a cache snapshot will be created.
     * This property must be set along with `snapshotRetentionLimit`.
     *
     * @default - ElastiCache automatically assigns the backup window if \`snapshotRetentionLimit\` is set. Otherwise, no snapshots are taken.
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/backups-automatic.html
     */
    readonly dailySnapshotTime?: DailySnapshotTime;
    /**
     * A description of the serverless cache.
     * The description can have up to 255 characters and must not contain < and > characters.
     *
     * @default - no description
     */
    readonly description?: string;
    /**
     * The name of the final snapshot taken of a cache before the cache is deleted.
     *
     * @default - no final snapshot taken
     */
    readonly finalSnapshotName?: string;
    /**
     * The Customer Managed Key that is used to encrypt data at rest in the serverless cache.
     *
     * @default - use AWS managed key
     */
    readonly kmsKey?: IKey;
    /**
     * The version number of the engine the serverless cache is compatible with.
     */
    readonly majorEngineVersion: MajorVersion;
    /**
     * The security groups to associate with the serverless cache.
     *
     * @default - a new security group is created
     */
    readonly securityGroups?: aws_ec2.ISecurityGroup[];
    /**
     * The ARN of the snapshot from which to restore data into the new cache.
     *
     * @default - not restored
     */
    readonly snapshotArnsToRestore?: string[];
    /**
     * The number of serverless cache snapshots the system will retain.
     * To enable automatic backups, this property must be set.
     *
     * \`snapshotRetentionLimit\` must be between 1 and 35.
     *
     * @default - no automatic backups
     */
    readonly snapshotRetentionLimit?: number;
    /**
     * The VPC to place the serverless cache in.
     */
    readonly vpc: aws_ec2.IVpc;
    /**
     * Where to place the serverless cache within the VPC.
     *
     * @default - private subnets
     */
    readonly vpcSubnets?: aws_ec2.SubnetSelection;
    /**
     * The user group associated with the serverless cache. Available for Valkey and Redis OSS only.
     *
     * @default - no user group associated
     */
    readonly userGroup?: IUserGroup;
}
/**
 * Attributes for importing an ElastiCache Serverless Cache.
 */
export interface ServerlessCacheAttributes {
    /**
     * The serverless cache name.
     */
    readonly serverlessCacheName: string;
    /**
     * The DNS hostname of the cache node.
     */
    readonly endpointAddress: string;
    /**
     * The port number that the cache engine is listening on.
     */
    readonly endpointPort: number;
    /**
     * The security groups to associate with the serverless cache.
     */
    readonly securityGroups: aws_ec2.ISecurityGroup[];
}
/**
 * The usage limits for storage and ElastiCache Processing Units for the cache.
 */
export interface CacheUsageLimits {
    /**
     * The data storage limit.
     *
     * @default - no limits
     */
    readonly dataStorage?: DataStorage;
    /**
     * The configuration for the number of ElastiCache Processing Units (ECPU) the cache can consume per second.
     *
     * @default - no limits
     */
    readonly ecpuPerSecond?: ECPUPerSecond;
}
/**
 * A new or imported serverless cache.
 */
declare abstract class ServerlessCacheBase extends Resource implements IServerlessCache {
    /**
     * Imports an existing ServerlessCache from attributes
     */
    static fromServerlessCacheAttributes(scope: Construct, id: string, attrs: ServerlessCacheAttributes): IServerlessCache;
    /**
     * The serverless cache ARN.
     */
    abstract readonly serverlessCacheArn: string;
    /**
     * The serverless cache name.
     */
    abstract readonly serverlessCacheName: string;
    /**
     * The DNS hostname of the cache node.
     */
    abstract readonly endpointAddress: string;
    /**
     * The port number that the cache engine is listening on.
     */
    abstract readonly endpointPort: number;
    /**
     * The connection object associated with the ElastiCache Serverless Cache.
     */
    abstract readonly connections: aws_ec2.Connections;
    /**
     * Grant the given identity the specified actions.
     * @param grantee the identity to be granted the actions
     * @param actions the data-access actions
     *
     * @see https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonelasticache.html
     */
    grant(grantee: aws_iam.IGrantable, ...actions: string[]): aws_iam.Grant;
    /**
     * Permits an IAM principal to perform connect to the serverless cache.
     *
     * Actions: Connect
     *
     * @param grantee The principal to grant access to.
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/auth-iam.html
     */
    grantConnect(grantee: aws_iam.IGrantable): aws_iam.Grant;
    /**
     * Create a CloudWatch metric for serverless cache.
     *
     * @param metricName name of the metric.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/serverless-metrics-events-redis.html
     * @see https://docs.aws.amazon.com/AmazonElastiCache/latest/dg/serverless-metrics-events.memcached.html
     */
    metric(metricName: string, props?: MetricOptions): Metric;
    /**
     * Metric for the total number of bytes used by the data stored in your cache.
     *
     * @default - average over 5 minutes
     */
    metricBytesUsedForCache(props?: MetricOptions): Metric;
    /**
     * Metric for the total number of ElastiCacheProcessingUnits (ECPUs) consumed by the requests executed on your cache.
     *
     * @default - average over 5 minutes
     */
    metricElastiCacheProcessingUnits(props?: MetricOptions): Metric;
}
/**
 * Represents an ElastiCache Serverless Cache construct in AWS CDK.
 *
 * @example
 * declare const vpc: aws_ec2.IVpc;
 *
 * const serverlessCache = new ServerlessCache(
 *   stack,
 *   'ServerlessCache',
 *   {
 *     serverlessCacheName: 'my-serverlessCache',
 *     engine: Engine.VALKEY,
 *     vpc,
 *   },
 * );
 */
export declare class ServerlessCache extends ServerlessCacheBase {
    /**
     * The serverless cache ARN.
     */
    readonly serverlessCacheArn: string;
    /**
     * The serverless cache name.
     */
    readonly serverlessCacheName: string;
    /**
     * The DNS hostname of the cache node.
     */
    readonly endpointAddress: string;
    /**
     * The port number that the cache engine is listening on.
     */
    readonly endpointPort: number;
    /**
     * The connection object associated with the ElastiCache Serverless Cache.
     */
    readonly connections: aws_ec2.Connections;
    private readonly props;
    private readonly securityGroups;
    private readonly vpcSubnets;
    constructor(scope: Construct, id: string, props: ServerlessCacheProps);
    protected createResource(scope: Construct, id: string, props: aws_elasticache.CfnServerlessCacheProps): aws_elasticache.CfnServerlessCache;
    protected createSecurityGroup(scope: Construct, id: string, props: aws_ec2.SecurityGroupProps): aws_ec2.SecurityGroup;
    private renderCacheUsageLimits;
    /**
     * Validates a serverless cache name.
     */
    private validateServerlessCacheName;
    /**
     * Validates a description.
     */
    private validateDescription;
    /**
     * Validates an automatic backup settings.
     */
    private validateAutomaticBackupSettings;
    /**
     * Validates final snapshot name.
     */
    private validateFinalSnapshotName;
    /**
     * Validates an engine and a user group combination.
     */
    private validateUserGroup;
}
export {};
