import { IResource, Resource, aws_elasticache } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IUser } from './user';
/**
 * Interface for a User Group
 */
export interface IUserGroup extends IResource {
    /**
     * The ARN of the user group.
     *
     * @attribute
     */
    readonly userGroupArn: string;
    /**
     * The ID of the user group.
     */
    readonly userGroupId: string;
}
/**
 * Properties for defining a User Group.
 */
export interface UserGroupProps {
    /**
     * The ID of the user group.
     *
     * \`userGroupId\` can have up to 40 characters.
     *
     * \`userGroupId\` must consist only of alphanumeric characters or hyphens,
     * with the first character as a letter, and it can't end with a hyphen or contain two consecutive hyphens.
     *
     * @default - auto generate
     */
    readonly userGroupId?: string;
    /**
     * The list of User that belong to the user group.
     *
     * A user with the username `default` must be included in `users`.
     */
    readonly users: IUser[];
}
/**
 * Attributes for importing a User Group.
 */
export interface UserGroupAttributes {
    /**
     * The ID of the user group.
     */
    readonly userGroupId: string;
}
/**
 * Represents a user group construct in AWS CDK.
 *
 * @example
 * declare const user: User;
 *
 * const userGroup = new UserGroup(
 *   stack,
 *   'UserGroup',
 *   {
 *      users: [user],
 *   },
 * );
 */
export declare class UserGroup extends Resource implements IUserGroup {
    /**
     * Imports an existing user group from attributes
     */
    static fromUserGroupId(scope: Construct, id: string, userGroupId: string): IUserGroup;
    /**
     * The ARN of the user group.
     */
    readonly userGroupArn: string;
    /**
     * The ID of the user group.
     */
    readonly userGroupId: string;
    private readonly props;
    private readonly users;
    constructor(scope: Construct, id: string, props: UserGroupProps);
    protected createResource(scope: Construct, id: string, props: aws_elasticache.CfnUserGroupProps): aws_elasticache.CfnUserGroup;
    /**
     * Validates user group id.
     */
    private validateUserGroupId;
    /**
     * Validates default user.
     */
    private validateDefaultUser;
    /**
     * Validates that there are no duplicate usernames in the user group
     */
    private validateDuplicateUsernames;
    /**
     * Adds a user to the user group
     *
     * @param user the user to add
     */
    addUser(user: IUser): void;
}
