import { IResource, Resource, SecretValue, aws_redshiftserverless } from 'aws-cdk-lib';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { IKey } from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
/**
 * A Redshift Serverless Namespace
 */
export interface INamespace extends IResource {
    /**
     * The namespace ARN
     *
     * @attribute
     */
    readonly namespaceArn: string;
    /**
     * The namespace name
     *
     * @attribute
     */
    readonly namespaceName: string;
    /**
     * The namespace id
     *
     * @attribute
     */
    readonly namespaceId: string;
}
/**
 * Properties for defining a Redshift Serverless Namespace.
 */
export interface NamespaceProps {
    /**
     * The username of the administrator for the primary database created in the namespace.
     *
     * You must specify both `adminUsername` and `adminUserPassword`, or neither.
     *
     * @default - no admin user
     */
    readonly adminUsername?: string;
    /**
     * The password of the administrator for the primary database created in the namespace.
     *
     * You must specify both `adminUsername` and `adminUserPassword`, or neither.
     *
     * @default - no admin user
     */
    readonly adminUserPassword?: SecretValue;
    /**
     * The name of the primary database created in the namespace.
     *
     * @default - 'dev'
     */
    readonly dbName?: string;
    /**
     * The IAM role to set as a default in the namespace.
     *
     * `defaultIamRole` must be included in `iamRoles`.
     *
     * @default - no default IAM role
     */
    readonly defaultIamRole?: IRole;
    /**
     * The name of the snapshot to be created before the namespace is deleted.
     *
     * If not specified, the final snapshot will not be taken.
     *
     * @default - no final snapshot
     */
    readonly finalSnapshotName?: string;
    /**
     * How long days to retain the final snapshot.
     *
     * You must set `finalSnapshotName` when you specify `finalSnapshotRetentionPeriod`.
     *
     * @default - Retained indefinitely if `finalSnapshotName` is specified, otherwise no final snapshot
     */
    readonly finalSnapshotRetentionPeriod?: number;
    /**
     * A list of IAM roles to associate with the namespace.
     *
     * @default - no IAM role associated
     */
    readonly iamRoles?: IRole[];
    /**
     * A Customer Managed Key used to encrypt your data.
     *
     * @default - use AWS managed key
     */
    readonly kmsKey?: IKey;
    /**
     * The types of logs the namespace can export.
     *
     * @default - no logs export
     */
    readonly logExports?: LogExport[];
    /**
     * The namespace name.
     *
     * @default - auto generate
     */
    readonly namespaceName?: string;
}
/**
 * The types of logs the namespace can export.
 */
export declare enum LogExport {
    /**
     * User log
     */
    USER_LOG = "userlog",
    /**
     * Connection log
     */
    CONNECTION_LOG = "connectionlog",
    /**
     * User activity log
     */
    USER_ACTIVITY_LOG = "useractivitylog"
}
/**
 * Attributes for importing a Redshift Serverless Namespace.
 */
export interface NamespaceAttributes {
    /**
     * The namespace name
     */
    readonly namespaceName: string;
    /**
     * The namespace id
     */
    readonly namespaceId: string;
}
/**
 * Represents a Redshift Serverless Namespace construct in AWS CDK.
 *
 * @example
 *
 * const nameSpace = new Namespace(
 *   stack,
 *   'Namespace',
 *   {
 *     namespaceName: 'my-namespace',
 *   },
 * );
 */
export declare class Namespace extends Resource implements INamespace {
    /**
     * Imports an existing Namespace from attributes
     */
    static fromNamespaceAttributes(scope: Construct, id: string, attrs: NamespaceAttributes): INamespace;
    /**
     * The namespace Arn
     */
    readonly namespaceArn: string;
    /**
     * The namespace name
     */
    readonly namespaceName: string;
    /**
     * The namespace id
     */
    readonly namespaceId: string;
    private readonly props;
    private readonly iamRoles;
    constructor(scope: Construct, id: string, props: NamespaceProps);
    protected createResource(scope: Construct, id: string, props: aws_redshiftserverless.CfnNamespaceProps): aws_redshiftserverless.CfnNamespace;
    /**
     * Validates admin settings.
     */
    private validateAdmin;
    /**
     * Validates a database name.
     */
    private validateDbName;
    /**
     * Validates final snapshot settings.
     */
    private validateFinalSnapshot;
    /**
     * Validates role settings.
     */
    private validateDefaultIamRole;
    /**
     * Validates a namespace name.
     */
    private validateNamespaceName;
    /**
     * Adds a role to the namespace
     *
     * @param role the role to add
     */
    addIamRole(role: IRole): void;
}
