# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, wildcard-import, unused-wildcard-import
"""
Version 1 of the H-Land model is designed to agree with the HBV96
configuration of the HBV model used by the German Federal Institute
of Hydrology (BfG) but offers more flexibility in some regards (e.g. in
parameterization).  It can briefly be summarized as follows:

 * Calculates the *actual* potential evapotranspiration from *reference*
   potential evaporation values.
 * Applies separate correction factors on the liquid and the frozen
   amount of precipitation.
 * Implements interception with simple "buckets".
 * Uses the degree-day method for calculating snowmelt.
 * Considers both the melting of ice and the (re)freezing of water
   within the snow layer.
 * Assumes a linear relationship between soil evaporation and relative soil
   moisture (as long as a maximum evaporation value is not exceeded).
 * Assumes a saturation excess mechanism for the generation of direct runoff.
 * Provides an optional "response area" option, which modifies the usual
   calculation of direct runoff and percolation.
 * Distinguishes between an upper zone layer related to direct runoff and a
   lower zone layer related to base flow.
 * Allows for percolation from the upper to the lower zone layer and allows
   for a capillary rise from upper zone layer to the soils layer.
 * Considers water areas as "groundwater lakes" being connected with
   the lower zone layer.
 * In contrast to the original HBV96 implementation, both the upper and
   the lower zone layer can be handled as nonlinear storages.
 * Conceptualizes the melting of glacial ice with an additional application
   of the degree-day method.

The following figure shows the general structure of H-Land Version 1.
Note that zones of type |FIELD| and |FOREST| are based on the same set of
process equations:

.. image:: HydPy-H-Land_Version-1.png

Integration tests:

    The following integration tests are based on the data used for testing
    application model |lland_v2|.  Hopefully, this eases drawing comparisons
    between both models.

    All integration tests are performed over a period of five days with
    a simulation step of one hour:

    >>> from hydpy import pub
    >>> pub.timegrids = '01.01.2000', '05.01.2000', '1h'

    Prepare the model instance and build the connections to element `land`
    and node `outlet`:

    >>> from hydpy.models.hland_v1 import *
    >>> parameterstep('1h')
    >>> from hydpy import Node, Element
    >>> outlet = Node('outlet')
    >>> land = Element('land', outlets=outlet)
    >>> land.model = model

    All tests shall be performed using a single zone with a size of one
    square kilometre at an altitude of 100 meter:

    >>> nmbzones(1)
    >>> area(1.0)
    >>> zonearea(1.0)
    >>> zonez(1.0)

    The reference elevation levels for precipitation (|ZRelP|), temperature
    (|ZRelT|), and evaporation (|ZRelE|) are all set to 200 m:

    >>> zrelp(2.0)
    >>> zrelt(2.0)
    >>> zrele(2.0)

    Initialize a test function object, which prepares and runs the tests
    and prints their results for the given sequences:

    >>> from hydpy import IntegrationTest
    >>> IntegrationTest.plotting_options.height = 900
    >>> IntegrationTest.plotting_options.activated=(
    ...     inputs.p, inputs.t, fluxes.qt)
    >>> test = IntegrationTest(land)
    >>> test.dateformat = '%d.%m. %H:00'

    .. _hland_v1_ex1:

    **Example 1**

    In the first example, |FIELD| is selected as the only zone type
    (note that the results for type |FOREST| would be the same):

    >>> zonetype(FIELD)

    The following set of control parameter values tries to configure
    application model |hland_v1| in a manner that allows to retrace
    the influence of the different implemented methods on the shown
    results:

    >>> pcorr(1.2)
    >>> pcalt(0.1)
    >>> rfcf(1.1)
    >>> sfcf(1.3)
    >>> tcalt(0.6)
    >>> ecorr(0.8)
    >>> ecalt(-0.1)
    >>> epf(0.1)
    >>> etf(0.1)
    >>> ered(0.5)
    >>> icmax(2.0)
    >>> tt(0.0)
    >>> ttint(2.0)
    >>> dttm(1.0)
    >>> cfmax(0.5)
    >>> gmelt(1.0)
    >>> cfr(0.1)
    >>> whc(0.4)
    >>> fc(200.0)
    >>> lp(0.8)
    >>> beta(2.0)
    >>> percmax(0.5)
    >>> cflux(0.1)
    >>> alpha(2.0)
    >>> k(0.001)
    >>> k4(0.005)
    >>> gamma(0.0)
    >>> maxbaz(3)
    >>> abstr(0.01)

    Initially, relative soil moisture is 50 % and the lower zone layer
    contains only 10 mm.  All other storages are empty:

    >>> test.inits = ((states.ic, 0.0),
    ...               (states.sp, 0.0),
    ...               (states.wc, 0.0),
    ...               (states.sm, 100.0),
    ...               (states.uz, 0.0),
    ...               (states.lz, 10.0),
    ...               (logs.quh, 0.05))

    As mentioned above, the values of the input sequences |P|,
    |hland_inputs.T|, and |EPN| are taken from :ref:`here <lland_v2_ex1>`.
    For educational purposes, unrealistically high values of |EPN| are
    used again.  For the sake of simplicity, the values of |TN| are assumed
    to be constantly 1 °C below the values of |hland_inputs.T|:

    >>> inputs.p.series = (
    ...     0.0, 0.0,  0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    ...     0.0, 0.0, 0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0,
    ...     2.1, 10.4, 3.5, 3.4, 1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1,
    ...     20.7, 37.9, 8.2, 3.6, 7.5, 18.5, 15.4, 6.3, 1.9, 4.9, 2.7, 0.5,
    ...     0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6, 0.7, 0.3, 0.3, 0.0,
    ...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.3, 0.0,
    ...     0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
    ...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
    >>> inputs.t.series = (
    ...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2,
    ...     35.2, 37.1, 31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8,
    ...     19.6, 19.2, 19.2, 19.2, 18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8,
    ...     19.0, 19.2, 19.3, 19.0, 18.8, 18.7, 17.8, 17.4, 17.3, 16.8, 16.5,
    ...     16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3, 14.9, 15.7, 16.0,
    ...     16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3, 18.1,
    ...     16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1,
    ...     11.9, 12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8,
    ...     22.2, 20.1, 17.8, 15.2, 14.5, 12.4, 11.7, 11.9)
    >>> inputs.tn.series = inputs.t.series-1.0
    >>> inputs.epn.series = (
    ...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761,
    ...     0.291908, 1.932875, 4.369536, 7.317556, 8.264362, 9.369867,
    ...     5.126178, 6.62503, 7.397619, 2.39151, 1.829834, 1.136569,
    ...     0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
    ...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599,
    ...     0.187298, 1.264612, 3.045538, 1.930758, 2.461001, 6.215945,
    ...     3.374783, 8.821555, 4.046025, 2.110757, 2.239257, 2.877848,
    ...     1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
    ...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593,
    ...     0.154861, 0.470369, 1.173726, 4.202296, 4.359715, 5.305753,
    ...     5.376027, 4.658915, 7.789594, 4.851567, 5.30692, 3.286036,
    ...     1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
    ...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229,
    ...     1.391958, 3.195876, 5.191651, 7.155036, 8.391432, 8.391286,
    ...     10.715238, 9.383394, 7.861915, 6.298329, 2.948416, 1.309232,
    ...     0.32955, 0.089508, 0.085771, 0.0845, 0.084864)

    In the first example, the |RespArea| option is disabled and a
    relatively large value for the accuracy related parameter |RecStep|
    is set:

    >>> resparea(False)
    >>> recstep(100)

    The following results show the response of application model |hland_v1|
    to the given extreme precipitation event, which is strongly
    attenuated by the large evaporation values.  One striking difference
    to other models like |lland_v2| is the block-like appearance of
    percolation (|Perc|), which is one reason for the unusual transitions
    between event periods (with relevant amounts of both runoff components
    |Q0| and |Q1|) and the subsequent pure base flow periods (with relevant
    amounts of |Q1| only):

    >>> test('hland_v1_ex1')
    |         date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |      pc |        ep |      epc |       ei |        tf | glmelt | melt | refr |       in_ |         r |       ea | cfpot |       cf |     perc | contriarea |      inuz |        q0 |  el |       q1 |      inuh |     outuh |        qt |       ic |  sp |  wc |         sm |        uz |        lz |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.052569 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.051111 |      0.0 | 0.0 | 0.0 |  99.947431 |       0.0 |      9.95 | 0.014198 |
    | 01.01. 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.051025 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.078833 |      0.0 | 0.0 | 0.0 |  99.896406 |       0.0 |   9.90025 | 0.021898 |
    | 01.01. 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.050572 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.03975 |      0.0 | 0.0 | 0.0 |  99.845834 |       0.0 |  9.850749 | 0.011042 |
    | 01.01. 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.05003 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.039502 |      0.0 | 0.0 | 0.0 |  99.795804 |       0.0 |  9.801495 | 0.010973 |
    | 01.01. 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.050521 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.039254 |      0.0 | 0.0 | 0.0 |  99.745284 |       0.0 |  9.752488 | 0.010904 |
    | 01.01. 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.053505 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.039008 |      0.0 | 0.0 | 0.0 |  99.691779 |       0.0 |  9.703725 | 0.010835 |
    | 01.01. 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.151906 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.038763 |      0.0 | 0.0 | 0.0 |  99.539873 |       0.0 |  9.655206 | 0.010767 |
    | 01.01. 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.004318 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.038519 |      0.0 | 0.0 | 0.0 |  98.535555 |       0.0 |   9.60693 |   0.0107 |
    | 01.01. 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.247495 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.038276 |      0.0 | 0.0 | 0.0 |  96.288059 |       0.0 |  9.558896 | 0.010632 |
    | 01.01. 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.677977 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.038035 |      0.0 | 0.0 | 0.0 |  92.610082 |       0.0 |  9.511101 | 0.010565 |
    | 01.01. 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.995196 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.037795 |      0.0 | 0.0 | 0.0 |  88.614886 |       0.0 |  9.463546 | 0.010499 |
    | 01.01. 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.334217 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.037556 |      0.0 | 0.0 | 0.0 |   84.28067 |       0.0 |  9.416228 | 0.010432 |
    | 01.01. 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.255237 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.037318 |      0.0 | 0.0 | 0.0 |  82.025433 |       0.0 |  9.369147 | 0.010366 |
    | 01.01. 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.836657 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.037081 |      0.0 | 0.0 | 0.0 |  79.188775 |       0.0 |  9.322301 |   0.0103 |
    | 01.01. 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  3.05792 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.036846 |      0.0 | 0.0 | 0.0 |  76.130856 |       0.0 |   9.27569 | 0.010235 |
    | 01.01. 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.928078 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.036612 |      0.0 | 0.0 | 0.0 |  75.202777 |       0.0 |  9.229311 |  0.01017 |
    | 01.01. 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.718317 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.036379 |      0.0 | 0.0 | 0.0 |  74.484461 |       0.0 |  9.183165 | 0.010105 |
    | 01.01. 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.441908 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.036147 |      0.0 | 0.0 | 0.0 |  74.042552 |       0.0 |  9.137249 | 0.010041 |
    | 01.01. 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.12436 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.035916 | 1.006935 | 0.0 | 0.0 |  73.918192 |       0.0 |  9.091563 | 0.009977 |
    | 01.01. 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.023694 |   nan | 0.060598 |      0.5 |        1.0 |  0.712508 |  0.000002 | 0.0 | 0.047958 |   0.04796 |  0.046243 |  0.036243 | 1.903859 | 0.0 | 0.0 |  78.841725 |  0.212506 |  9.543605 | 0.010067 |
    | 01.01. 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.015028 |   nan | 0.059165 |      0.5 |        1.0 |   0.46128 |  0.000007 | 0.0 | 0.050218 |  0.050225 |  0.047958 |  0.037958 | 1.941161 | 0.0 | 0.0 |  81.714476 |  0.173779 |  9.993387 | 0.010544 |
    | 01.01. 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.012429 |   nan | 0.056743 |      0.5 |        1.0 |  0.905176 |   0.00007 | 0.0 | 0.052467 |  0.052537 |  0.050236 |  0.040236 | 1.954058 | 0.0 | 0.0 |  86.559232 |  0.578886 |  10.44092 | 0.011177 |
    | 01.01. 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.007038 |   nan | 0.051622 |      0.5 |        1.0 |  2.298561 |  0.004477 | 0.0 | 0.054705 |  0.059182 |    0.0535 |    0.0435 | 1.976736 | 0.0 | 0.0 |  96.800491 |   2.37297 | 10.886215 | 0.012083 |
    | 01.01. 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.024367 |   nan | 0.051563 |      0.5 |        1.0 | -0.029183 |   0.00942 | 0.0 | 0.056931 |  0.066351 |  0.059298 |  0.049298 | 1.919551 | 0.0 | 0.0 |  96.900842 |  1.834367 | 11.329284 | 0.013694 |
    | 02.01. 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.022832 |   nan | 0.051262 |      0.5 |        1.0 |  0.125066 |  0.004491 | 0.0 | 0.059146 |  0.063637 |  0.064155 |  0.054155 | 1.925086 | 0.0 | 0.0 |  97.504096 |  1.454943 | 11.770138 | 0.015043 |
    | 02.01. 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.017851 |   nan | 0.049918 |      0.5 |        1.0 |  0.779355 |  0.004082 | 0.0 | 0.061351 |  0.065432 |  0.064639 |  0.054639 | 1.942997 | 0.0 | 0.0 | 100.195975 |  1.730216 | 12.208787 | 0.015178 |
    | 02.01. 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.020132 |   nan | 0.048989 |      0.5 |        1.0 |  0.562851 |  0.005447 | 0.0 | 0.063544 |  0.068991 |  0.065824 |  0.055824 | 1.936885 | 0.0 | 0.0 | 102.050789 |   1.78762 | 12.645243 | 0.015507 |
    | 02.01. 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.008153 |   nan | 0.044429 |      0.5 |        1.0 |  3.155927 |  0.035664 | 0.0 | 0.065726 |   0.10139 |    0.0754 |    0.0654 | 1.976535 | 0.0 | 0.0 | 111.178794 |  4.407884 | 13.079517 | 0.018167 |
    | 02.01. 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.018923 |   nan | 0.042982 |      0.5 |        1.0 |  1.234664 |  0.106278 | 0.0 | 0.067898 |  0.174175 |  0.110365 |  0.100365 |  1.94692 | 0.0 | 0.0 | 114.059742 |   5.03627 | 13.511619 | 0.027879 |
    | 02.01. 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.019537 |   nan | 0.041625 |      0.5 |        1.0 |  1.254823 |  0.153262 | 0.0 | 0.070058 |   0.22332 |  0.168922 |  0.158922 |  1.94647 | 0.0 | 0.0 | 116.771503 |  5.637831 | 13.941561 | 0.044145 |
    | 02.01. 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 |   0.0499 |   nan | 0.041162 |      0.5 |        1.0 |  0.426562 |  0.167893 | 0.0 | 0.072208 |  0.240101 |  0.216128 |  0.206128 | 1.864353 | 0.0 | 0.0 |  117.66711 |    5.3965 | 14.369353 | 0.057258 |
    | 02.01. 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.383924 |   nan | 0.041166 |      0.5 |        1.0 | -0.041166 |  0.129717 | 0.0 | 0.074347 |  0.204064 |  0.228364 |  0.218364 | 0.939422 | 0.0 | 0.0 | 117.324353 |  4.725616 | 14.795007 | 0.060657 |
    | 02.01. 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.735028 |   nan | 0.041338 |      0.5 |        1.0 | -0.041338 |  0.086003 | 0.0 | 0.076475 |  0.162478 |  0.202831 |  0.192831 |      0.0 | 0.0 | 0.0 | 115.630662 |  4.098276 | 15.218532 | 0.053564 |
    | 02.01. 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.165815 |   nan | 0.042185 |      0.5 |        1.0 | -0.042185 |  0.054998 | 0.0 | 0.078593 |  0.133591 |    0.1653 |    0.1553 |      0.0 | 0.0 | 0.0 | 114.507032 |  3.501093 | 15.639939 | 0.043139 |
    | 02.01. 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.403262 |   nan | 0.042746 |      0.5 |        1.0 | -0.042746 |  0.033312 | 0.0 |   0.0807 |  0.114012 |  0.135659 |  0.125659 |      0.0 | 0.0 | 0.0 | 113.146516 |  2.925034 | 16.059239 | 0.034905 |
    | 02.01. 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.629327 |   nan | 0.043427 |      0.5 |        1.0 | -0.043427 |  0.018617 | 0.0 | 0.082796 |  0.101413 |  0.115563 |  0.105563 |      0.0 | 0.0 | 0.0 | 109.560616 |  2.362991 | 16.476443 | 0.029323 |
    | 02.01. 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.638626 |   nan | 0.044423 |      0.5 |        1.0 |  0.638818 |  0.014291 | 0.0 | 0.084882 |  0.099173 |  0.103715 |  0.093715 | 0.162204 | 0.0 | 0.0 | 110.559972 |  2.487517 | 16.891561 | 0.026032 |
    | 02.01. 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 2.132116 |   nan | 0.042924 |      0.5 |        1.0 |  1.537393 |  0.027763 | 0.0 | 0.086958 |  0.114721 |  0.103126 |  0.093126 |      0.0 | 0.0 | 0.0 | 112.061866 |  3.497147 | 17.304603 | 0.025868 |
    | 02.01. 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.868067 |   nan | 0.043969 |      0.5 |        1.0 | -0.043969 |  0.033174 | 0.0 | 0.089023 |  0.122197 |  0.112927 |  0.102927 |      0.0 | 0.0 | 0.0 | 110.237768 |  2.920004 |  17.71558 | 0.028591 |
    | 02.01. 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.059355 |   nan | 0.037017 |      0.5 |        1.0 |  6.826517 |  0.275437 | 0.0 | 0.091078 |  0.366515 |  0.174829 |  0.164829 |  1.84926 | 0.0 | 0.0 | 125.943496 |  8.971084 | 18.124502 | 0.045786 |
    | 02.01. 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.009912 |   nan | 0.023488 |      0.5 |        1.0 | 17.771214 |  4.821721 | 0.0 | 0.093123 |  4.914843 |  1.322962 |  1.312962 | 1.979276 | 0.0 | 0.0 |  153.03683 | 21.420577 |  18.53138 | 0.364712 |
    | 02.01. 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.445428 |   nan | 0.021467 |      0.5 |        1.0 |  5.670172 |  7.878038 | 0.0 | 0.095157 |  7.973195 |  4.583738 |  4.573738 | 1.092627 | 0.0 | 0.0 | 156.642106 |  18.71271 | 18.936223 | 1.270483 |
    | 02.01. 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.427817 |   nan | 0.021028 |      0.5 |        1.0 |  2.045842 |  4.840645 | 0.0 | 0.097181 |  4.937827 |  6.619035 |  6.609035 | 1.133348 | 0.0 | 0.0 | 157.537874 | 15.417906 | 19.339042 | 1.835843 |
    | 02.01. 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.049957 |   nan | 0.019705 |      0.5 |        1.0 |  4.970827 |    3.9218 | 0.0 | 0.099195 |  4.020995 |  5.408613 |  5.398613 | 1.900086 | 0.0 | 0.0 | 160.560437 | 15.966933 | 19.739846 | 1.499615 |
    | 02.01. 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.004295 |   nan | 0.015831 |      0.5 |        1.0 | 14.084406 |  7.677383 | 0.0 | 0.101199 |  7.778582 |  5.059755 |  5.049755 |  1.99141 | 0.0 | 0.0 | 168.349822 | 21.873957 | 20.138647 |  1.40271 |
    | 02.01. 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.006196 |   nan |  0.01316 |      0.5 |        1.0 | 12.943663 | 11.510418 | 0.0 | 0.103193 | 11.613611 |  7.795791 |  7.785791 | 1.987608 | 0.0 | 0.0 | 173.686572 | 22.807202 | 20.535454 |  2.16272 |
    | 02.01. 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.018028 |   nan | 0.012238 |      0.5 |        1.0 |  5.622968 |  8.843986 | 0.0 | 0.105177 |  8.949163 | 10.169283 | 10.159283 | 1.963945 | 0.0 | 0.0 | 175.517585 | 19.086184 | 20.930277 | 2.822023 |
    | 02.01. 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.029884 |   nan | 0.011986 |      0.5 |        1.0 |  1.698652 |  4.947934 | 0.0 | 0.107151 |  5.055086 |  8.675912 |  8.665912 | 1.940232 | 0.0 | 0.0 | 176.010193 | 15.336902 | 21.323125 | 2.407198 |
    | 03.01. 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.020966 |   nan | 0.011345 |      0.5 |        1.0 |  4.450824 |  3.725205 | 0.0 | 0.109116 |   3.83432 |  5.649155 |  5.639155 | 1.958069 | 0.0 | 0.0 | 177.299835 | 15.562521 |  21.71401 | 1.566432 |
    | 03.01. 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.027175 |   nan | 0.011011 |      0.5 |        1.0 |  2.476827 |  3.288302 | 0.0 |  0.11107 |  3.399372 |  4.008946 |  3.998946 |  1.94565 | 0.0 | 0.0 | 177.961502 | 14.251047 |  22.10294 | 1.110818 |
    | 03.01. 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.034946 |   nan | 0.010963 |      0.5 |        1.0 |  0.416309 |  2.220294 | 0.0 | 0.113015 |  2.333308 |  3.259124 |  3.249124 | 1.930107 | 0.0 | 0.0 | 178.049897 | 11.947062 | 22.489925 | 0.902534 |
    | 03.01. 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.036358 |   nan | 0.010958 |      0.5 |        1.0 |  0.121958 |  1.351723 | 0.0 |  0.11495 |  1.466673 |  2.377626 |  2.367626 | 1.927284 | 0.0 | 0.0 | 178.059289 | 10.217297 | 22.874975 | 0.657674 |
    | 03.01. 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.035498 |   nan | 0.010916 |      0.5 |        1.0 |  0.402268 |  0.912964 | 0.0 | 0.116875 |  1.029839 |  1.562184 |  1.552184 | 1.929004 | 0.0 | 0.0 | 178.142807 |  9.206601 |   23.2581 | 0.431162 |
    | 03.01. 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.028761 |   nan | 0.010641 |      0.5 |        1.0 |  2.195093 |  0.896634 | 0.0 | 0.118791 |  1.015425 |   1.12371 |   1.11371 | 1.942479 | 0.0 | 0.0 | 178.699157 |  10.00506 |  23.63931 | 0.309364 |
    | 03.01. 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.061669 |   nan | 0.010608 |      0.5 |        1.0 |  0.322839 |  0.851988 | 0.0 | 0.120697 |  0.972684 |   1.00913 |   0.99913 | 1.876663 | 0.0 | 0.0 | 178.732328 |  8.975912 | 24.018613 | 0.277536 |
    | 03.01. 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.191814 |   nan | 0.010622 |      0.5 |        1.0 |  0.080631 |  0.604807 | 0.0 | 0.122593 |    0.7274 |  0.927674 |  0.917674 | 1.616372 | 0.0 | 0.0 | 178.574145 |  7.951736 |  24.39602 |  0.25491 |
    | 03.01. 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.490148 |   nan | 0.010713 |      0.5 |        1.0 | -0.010713 |  0.419019 | 0.0 |  0.12448 |  0.543499 |  0.741041 |  0.731041 | 0.636076 | 0.0 | 0.0 |  178.09471 |  7.022004 |  24.77154 | 0.203067 |
    | 03.01. 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  3.19172 |   nan | 0.010953 |      0.5 |        1.0 | -0.010953 |  0.290078 | 0.0 | 0.126358 |  0.416436 |   0.55613 |   0.54613 |      0.0 | 0.0 | 0.0 | 174.913943 |  6.220974 | 25.145182 | 0.151703 |
    | 03.01. 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.335546 |   nan | 0.012543 |      0.5 |        1.0 | -0.012543 |  0.201588 | 0.0 | 0.128226 |  0.329814 |  0.425423 |  0.415423 |      0.0 | 0.0 | 0.0 | 171.590941 |  5.506843 | 25.516957 | 0.115395 |
    | 03.01. 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 2.253813 |   nan | 0.014061 |      0.5 |        1.0 |   0.78739 |  0.172588 | 0.0 | 0.130085 |  0.302673 |  0.343032 |  0.333032 |      0.0 | 0.0 | 0.0 | 169.638537 |  5.621644 | 25.886872 | 0.092509 |
    | 03.01. 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.715969 |   nan | 0.015181 |      0.5 |        1.0 | -0.015181 |  0.148076 | 0.0 | 0.131934 |   0.28001 |  0.303668 |  0.293668 |      0.0 | 0.0 | 0.0 | 165.937749 |  4.958388 | 26.254937 | 0.081574 |
    | 03.01. 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.576688 |   nan | 0.017031 |      0.5 |        1.0 | -0.017031 |  0.100609 | 0.0 | 0.133775 |  0.234383 |  0.274907 |  0.264907 |      0.0 | 0.0 | 0.0 | 162.378092 |  4.340748 | 26.621163 | 0.073585 |
    | 03.01. 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 6.099883 |   nan | 0.018811 |      0.5 |        1.0 | -0.018811 |  0.066455 | 0.0 | 0.135606 |  0.202061 |   0.23734 |   0.22734 |      0.0 | 0.0 | 0.0 |  156.29702 |  3.755482 | 26.985557 |  0.06315 |
    | 03.01. 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.958804 |   nan | 0.021851 |      0.5 |        1.0 | -0.021851 |  0.042037 | 0.0 | 0.137428 |  0.179465 |  0.204222 |  0.194222 |      0.0 | 0.0 | 0.0 | 152.360068 |  3.191593 | 27.348129 | 0.053951 |
    | 03.01. 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.221357 |   nan |  0.02382 |      0.5 |        1.0 |  -0.02382 |  0.024952 | 0.0 | 0.139241 |  0.164192 |  0.181093 |  0.171093 |      0.0 | 0.0 | 0.0 | 148.162531 |  2.642821 | 27.708889 | 0.047526 |
    | 03.01. 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.541892 |   nan | 0.025919 |      0.5 |        1.0 | -0.025919 |  0.013481 | 0.0 | 0.141044 |  0.154525 |  0.165438 |  0.155438 |      0.0 | 0.0 | 0.0 | 145.646557 |  2.103421 | 28.067844 | 0.043177 |
    | 03.01. 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.145352 |   nan | 0.027177 |      0.5 |        1.0 | -0.027177 |  0.006295 | 0.0 | 0.142839 |  0.149135 |  0.155476 |  0.145476 |      0.0 | 0.0 | 0.0 | 144.528382 |  1.569949 | 28.425005 |  0.04041 |
    | 03.01. 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.207331 |   nan | 0.027736 |      0.5 |        1.0 | -0.027736 |  0.002299 | 0.0 | 0.144625 |  0.146924 |  0.149841 |  0.139841 |      0.0 | 0.0 | 0.0 | 144.348787 |  1.039914 |  28.78038 | 0.038845 |
    | 03.01. 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.065993 |   nan | 0.027826 |      0.5 |        1.0 | -0.027826 |  0.000516 | 0.0 | 0.146402 |  0.146918 |  0.147414 |  0.137414 |      0.0 | 0.0 | 0.0 |  144.31062 |  0.511573 | 29.133978 | 0.038171 |
    | 03.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.064624 |   nan | 0.027845 | 0.483697 |        1.0 | -0.027845 |  0.000032 | 0.0 | 0.148088 |   0.14812 |  0.147186 |  0.137186 |      0.0 | 0.0 | 0.0 |  144.27384 |       0.0 | 29.469586 | 0.038107 |
    | 03.01. 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |   0.0635 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.147348 |  0.147348 |  0.147681 |  0.137681 |      0.0 | 0.0 | 0.0 |  144.21034 |       0.0 | 29.322238 | 0.038245 |
    | 03.01. 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.062642 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.146611 |  0.146611 |  0.147356 |  0.137356 |      0.0 | 0.0 | 0.0 | 144.147698 |       0.0 | 29.175627 | 0.038154 |
    | 04.01. 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.061918 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.145878 |  0.145878 |  0.146612 |  0.136612 |      0.0 | 0.0 | 0.0 |  144.08578 |       0.0 | 29.029749 | 0.037948 |
    | 04.01. 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.063555 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.145149 |  0.145149 |  0.145879 |  0.135879 |      0.0 | 0.0 | 0.0 | 144.022225 |       0.0 |   28.8846 | 0.037744 |
    | 04.01. 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |   0.0638 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.144423 |  0.144423 |   0.14515 |   0.13515 |      0.0 | 0.0 | 0.0 | 143.958424 |       0.0 | 28.740177 | 0.037542 |
    | 04.01. 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.026911 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.143701 |  0.143701 |  0.144424 |  0.134424 | 1.484581 | 0.0 | 0.0 | 143.931513 |       0.0 | 28.596476 |  0.03734 |
    | 04.01. 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.03133 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.142982 |  0.142982 |  0.143702 |  0.133702 | 1.414925 | 0.0 | 0.0 | 143.900183 |       0.0 | 28.453494 | 0.037139 |
    | 04.01. 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.031873 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.142267 |  0.142267 |  0.142983 |  0.132983 | 1.344047 | 0.0 | 0.0 |  143.86831 |       0.0 | 28.311226 |  0.03694 |
    | 04.01. 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.11649 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.141556 |  0.141556 |  0.142268 |  0.132268 | 1.084943 | 0.0 | 0.0 |  143.75182 |       0.0 |  28.16967 | 0.036741 |
    | 04.01. 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.480578 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.140848 |  0.140848 |  0.141557 |  0.131557 | 0.846748 | 0.0 | 0.0 | 143.271242 |       0.0 | 28.028822 | 0.036544 |
    | 04.01. 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.751285 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.140144 |  0.140144 |  0.140849 |  0.130849 |      0.0 | 0.0 | 0.0 | 141.519957 |       0.0 | 27.888678 | 0.036347 |
    | 04.01. 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.789957 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.139443 |  0.139443 |  0.140145 |  0.130145 |      0.0 | 0.0 | 0.0 |     137.73 |       0.0 | 27.749234 | 0.036151 |
    | 04.01. 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.905386 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.138746 |  0.138746 |  0.139444 |  0.129444 |      0.0 | 0.0 | 0.0 | 132.824614 |       0.0 | 27.610488 | 0.035957 |
    | 04.01. 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.818153 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.138052 |  0.138052 |  0.138747 |  0.128747 |      0.0 | 0.0 | 0.0 | 127.006461 |       0.0 | 27.472436 | 0.035763 |
    | 04.01. 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.563202 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.137362 |  0.137362 |  0.138053 |  0.128053 |      0.0 | 0.0 | 0.0 | 121.443259 |       0.0 | 27.335074 |  0.03557 |
    | 04.01. 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 6.792752 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.136675 |  0.136675 |  0.137363 |  0.127363 |      0.0 | 0.0 | 0.0 | 114.650507 |       0.0 | 27.198398 | 0.035379 |
    | 04.01. 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.615733 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.135992 |  0.135992 |  0.136676 |  0.126676 |      0.0 | 0.0 | 0.0 | 109.034774 |       0.0 | 27.062406 | 0.035188 |
    | 04.01. 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |   4.4747 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.135312 |  0.135312 |  0.135993 |  0.125993 |      0.0 | 0.0 | 0.0 | 104.560075 |       0.0 | 26.927094 | 0.034998 |
    | 04.01. 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.437651 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.134635 |  0.134635 |  0.135313 |  0.125313 |      0.0 | 0.0 | 0.0 | 101.122424 |       0.0 | 26.792459 | 0.034809 |
    | 04.01. 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.556348 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.133962 |  0.133962 |  0.134636 |  0.124636 |      0.0 | 0.0 | 0.0 |  99.566076 |       0.0 | 26.658496 | 0.034621 |
    | 04.01. 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.680454 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.133292 |  0.133292 |  0.133963 |  0.123963 |      0.0 | 0.0 | 0.0 |  98.885622 |       0.0 | 26.525204 | 0.034434 |
    | 04.01. 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.170108 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.132626 |  0.132626 |  0.133293 |  0.123293 |      0.0 | 0.0 | 0.0 |  98.715514 |       0.0 | 26.392578 | 0.034248 |
    | 04.01. 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.046123 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.131963 |  0.131963 |  0.132627 |  0.122627 |      0.0 | 0.0 | 0.0 |  98.669391 |       0.0 | 26.260615 | 0.034063 |
    | 04.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.044177 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.131303 |  0.131303 |  0.131964 |  0.121964 |      0.0 | 0.0 | 0.0 |  98.625215 |       0.0 | 26.129312 | 0.033879 |
    | 04.01. 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.043503 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.130647 |  0.130647 |  0.131304 |  0.121304 |      0.0 | 0.0 | 0.0 |  98.581712 |       0.0 | 25.998665 | 0.033696 |
    | 04.01. 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.043671 |   nan |      0.0 |      0.0 |        1.0 |       0.0 |       0.0 | 0.0 | 0.129993 |  0.129993 |  0.130647 |  0.120647 |      0.0 | 0.0 | 0.0 |  98.538041 |       0.0 | 25.868672 | 0.033513 |

    .. raw:: html

        <iframe
            src="hland_v1_ex1.html"
            width="100%"
            height="930px"
            frameborder=0
        ></iframe>

    .. _hland_v1_ex2:

    **Example 2**

    The functionality of |hland_v1| can be changed substantially by
    enabling its |RespArea| option, which decreases |Perc| but increases
    |Q0| in dry periods (regarding relative soil moisture).  Hence the graph
    of |Perc| appears less blocky and reaches its maximum at the same time
    as the graph of |SM|, whereas |Q0| shows more pronounced peaks in the
    initial subperiod when the soil is not saturated yet:

    >>> resparea(True)
    >>> test('hland_v1_ex2')
    |         date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |      pc |        ep |      epc |       ei |        tf | glmelt | melt | refr |       in_ |         r |       ea | cfpot |       cf |     perc | contriarea |      inuz |        q0 |  el |       q1 |      inuh |    outuh |       qt |       ic |  sp |  wc |         sm |        uz |        lz |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.052569 |   nan |      0.0 |      0.0 |   0.249737 |       0.0 |       0.0 | 0.0 |     0.05 |      0.05 | 0.061111 | 0.051111 |      0.0 | 0.0 | 0.0 |  99.947431 |       0.0 |      9.95 | 0.014198 |
    | 01.01. 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.051025 |   nan |      0.0 |      0.0 |   0.249482 |       0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 | 0.088833 | 0.078833 |      0.0 | 0.0 | 0.0 |  99.896406 |       0.0 |   9.90025 | 0.021898 |
    | 01.01. 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.050572 |   nan |      0.0 |      0.0 |    0.24923 |       0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |  0.04975 |  0.03975 |      0.0 | 0.0 | 0.0 |  99.845834 |       0.0 |  9.850749 | 0.011042 |
    | 01.01. 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.05003 |   nan |      0.0 |      0.0 |    0.24898 |       0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 | 0.049502 | 0.039502 |      0.0 | 0.0 | 0.0 |  99.795804 |       0.0 |  9.801495 | 0.010973 |
    | 01.01. 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.050521 |   nan |      0.0 |      0.0 |   0.248728 |       0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 | 0.049254 | 0.039254 |      0.0 | 0.0 | 0.0 |  99.745284 |       0.0 |  9.752488 | 0.010904 |
    | 01.01. 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.053505 |   nan |      0.0 |      0.0 |   0.248461 |       0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 | 0.049008 | 0.039008 |      0.0 | 0.0 | 0.0 |  99.691779 |       0.0 |  9.703725 | 0.010835 |
    | 01.01. 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.151906 |   nan |      0.0 |      0.0 |   0.247705 |       0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 | 0.048763 | 0.038763 |      0.0 | 0.0 | 0.0 |  99.539873 |       0.0 |  9.655206 | 0.010767 |
    | 01.01. 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.004318 |   nan |      0.0 |      0.0 |   0.242731 |       0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 | 0.048519 | 0.038519 |      0.0 | 0.0 | 0.0 |  98.535555 |       0.0 |   9.60693 |   0.0107 |
    | 01.01. 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.247495 |   nan |      0.0 |      0.0 |   0.231785 |       0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 | 0.048276 | 0.038276 |      0.0 | 0.0 | 0.0 |  96.288059 |       0.0 |  9.558896 | 0.010632 |
    | 01.01. 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.677977 |   nan |      0.0 |      0.0 |   0.214416 |       0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 | 0.048035 | 0.038035 |      0.0 | 0.0 | 0.0 |  92.610082 |       0.0 |  9.511101 | 0.010565 |
    | 01.01. 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.995196 |   nan |      0.0 |      0.0 |   0.196315 |       0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 | 0.047795 | 0.037795 |      0.0 | 0.0 | 0.0 |  88.614886 |       0.0 |  9.463546 | 0.010499 |
    | 01.01. 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.334217 |   nan |      0.0 |      0.0 |   0.177581 |       0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 | 0.047556 | 0.037556 |      0.0 | 0.0 | 0.0 |   84.28067 |       0.0 |  9.416228 | 0.010432 |
    | 01.01. 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.255237 |   nan |      0.0 |      0.0 |   0.168204 |       0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 | 0.047318 | 0.037318 |      0.0 | 0.0 | 0.0 |  82.025433 |       0.0 |  9.369147 | 0.010366 |
    | 01.01. 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.836657 |   nan |      0.0 |      0.0 |   0.156772 |       0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 | 0.047081 | 0.037081 |      0.0 | 0.0 | 0.0 |  79.188775 |       0.0 |  9.322301 |   0.0103 |
    | 01.01. 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  3.05792 |   nan |      0.0 |      0.0 |   0.144898 |       0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 | 0.046846 | 0.036846 |      0.0 | 0.0 | 0.0 |  76.130856 |       0.0 |   9.27569 | 0.010235 |
    | 01.01. 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.928078 |   nan |      0.0 |      0.0 |   0.141386 |       0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 | 0.046612 | 0.036612 |      0.0 | 0.0 | 0.0 |  75.202777 |       0.0 |  9.229311 |  0.01017 |
    | 01.01. 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.718317 |   nan |      0.0 |      0.0 |   0.138698 |       0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 | 0.046379 | 0.036379 |      0.0 | 0.0 | 0.0 |  74.484461 |       0.0 |  9.183165 | 0.010105 |
    | 01.01. 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.441908 |   nan |      0.0 |      0.0 |   0.137057 |       0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 | 0.046147 | 0.036147 |      0.0 | 0.0 | 0.0 |  74.042552 |       0.0 |  9.137249 | 0.010041 |
    | 01.01. 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.12436 |   nan |      0.0 |      0.0 |   0.136597 |       0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 | 0.045916 | 0.035916 | 1.006935 | 0.0 | 0.0 |  73.918192 |       0.0 |  9.091563 | 0.009977 |
    | 01.01. 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.023694 |   nan | 0.060598 |   0.0777 |     0.1554 |  0.712508 |  0.016629 | 0.0 | 0.045846 |  0.062475 | 0.049468 | 0.039468 | 1.903859 | 0.0 | 0.0 |  78.841725 |  0.618179 |  9.123417 | 0.010963 |
    | 01.01. 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.015028 |   nan | 0.059165 | 0.083466 |   0.166931 |   0.46128 |  0.100569 | 0.0 | 0.046034 |  0.146603 | 0.077439 | 0.067439 | 1.941161 | 0.0 | 0.0 |  81.714476 |  0.895425 |  9.160848 | 0.018733 |
    | 01.01. 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.012429 |   nan | 0.056743 | 0.093656 |   0.187313 |  0.905176 |  0.275044 | 0.0 | 0.046273 |  0.321317 | 0.166733 | 0.156733 | 1.954058 | 0.0 | 0.0 |  86.559232 |  1.431901 |  9.208232 | 0.043537 |
    | 01.01. 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.007038 |   nan | 0.051622 | 0.117129 |   0.234258 |  2.298561 |  0.903885 | 0.0 | 0.046627 |  0.950512 | 0.422313 | 0.412313 | 1.976736 | 0.0 | 0.0 |  96.800491 |  2.709448 |  9.278734 | 0.114531 |
    | 01.01. 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.024367 |   nan | 0.051563 | 0.117372 |   0.234744 | -0.029183 |  0.805699 | 0.0 | 0.046981 |  0.852679 |  0.78895 |  0.77895 | 1.919551 | 0.0 | 0.0 |  96.900842 |  1.757194 |  9.349126 | 0.216375 |
    | 02.01. 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.022832 |   nan | 0.051262 | 0.118838 |   0.237676 |  0.125066 |   0.30497 | 0.0 |  0.04734 |   0.35231 | 0.763227 | 0.753227 | 1.925086 | 0.0 | 0.0 |  97.504096 |  1.458453 |  9.420624 |  0.20923 |
    | 02.01. 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.017851 |   nan | 0.049918 |  0.12549 |   0.250981 |  0.779355 |  0.292533 | 0.0 | 0.047731 |  0.340263 | 0.460826 | 0.450826 | 1.942997 | 0.0 | 0.0 | 100.195975 |  1.819784 |  9.498384 | 0.125229 |
    | 02.01. 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.020132 |   nan | 0.048989 |  0.13018 |   0.260359 |  0.562851 |  0.365199 | 0.0 | 0.048143 |  0.413342 |  0.35918 |  0.34918 | 1.936885 | 0.0 | 0.0 | 102.050789 |  1.887257 |   9.58042 | 0.096994 |
    | 02.01. 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.008153 |   nan | 0.044429 | 0.154509 |   0.309018 |  3.155927 |  1.047457 | 0.0 | 0.048675 |  1.096132 | 0.548833 | 0.538833 | 1.976535 | 0.0 | 0.0 | 111.178794 |  3.841219 |  9.686255 | 0.149676 |
    | 02.01. 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.018923 |   nan | 0.042982 |  0.16262 |   0.325241 |  1.234664 |  1.415468 | 0.0 | 0.049244 |  1.464713 | 1.026307 | 1.016307 |  1.94692 | 0.0 | 0.0 | 114.059742 |  3.497794 |  9.799631 | 0.282308 |
    | 02.01. 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.019537 |   nan | 0.041625 | 0.170445 |    0.34089 |  1.254823 |  1.088288 | 0.0 |  0.04985 |  1.138139 | 1.310234 | 1.300234 |  1.94647 | 0.0 | 0.0 | 116.771503 |  3.493884 |  9.920225 | 0.361176 |
    | 02.01. 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 |   0.0499 |   nan | 0.041162 | 0.173069 |   0.346139 |  0.426562 |  0.792891 | 0.0 | 0.050466 |  0.843358 | 1.145204 | 1.135204 | 1.864353 | 0.0 | 0.0 |  117.66711 |  2.954486 | 10.042828 | 0.315334 |
    | 02.01. 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.383924 |   nan | 0.041166 | 0.172063 |   0.344125 | -0.041166 |  0.439147 | 0.0 | 0.051074 |  0.490221 |  0.83039 |  0.82039 | 0.939422 | 0.0 | 0.0 | 117.324353 |   2.30211 | 10.163816 | 0.227886 |
    | 02.01. 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.735028 |   nan | 0.041338 | 0.167131 |   0.334261 | -0.041338 |  0.238811 | 0.0 | 0.051655 |  0.290465 | 0.524306 | 0.514306 |      0.0 | 0.0 | 0.0 | 115.630662 |   1.85483 | 10.279292 | 0.142863 |
    | 02.01. 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.165815 |   nan | 0.042185 | 0.163898 |   0.327797 | -0.042185 |  0.135278 | 0.0 | 0.052216 |  0.187494 | 0.311973 | 0.301973 |      0.0 | 0.0 | 0.0 | 114.507032 |   1.51347 | 10.390974 | 0.083881 |
    | 02.01. 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.403262 |   nan | 0.042746 | 0.160027 |   0.320053 | -0.042746 |  0.078912 | 0.0 | 0.052755 |  0.131667 |  0.19797 |  0.18797 |      0.0 | 0.0 | 0.0 | 113.146516 |  1.231784 | 10.498246 | 0.052214 |
    | 02.01. 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.629327 |   nan | 0.043427 | 0.150044 |   0.300088 | -0.043427 |  0.050702 | 0.0 | 0.053241 |  0.103943 | 0.137912 | 0.127912 |      0.0 | 0.0 | 0.0 | 109.560616 |  0.987612 | 10.595049 | 0.035531 |
    | 02.01. 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.638626 |   nan | 0.044423 | 0.152794 |   0.305588 |  0.638818 |  0.063517 | 0.0 | 0.053739 |  0.117256 | 0.113063 | 0.103063 | 0.162204 | 0.0 | 0.0 | 110.559972 |  1.410119 | 10.694103 | 0.028628 |
    | 02.01. 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 2.132116 |   nan | 0.042924 | 0.156973 |   0.313947 |  1.537393 |  0.280085 | 0.0 | 0.054255 |  0.334341 | 0.162539 | 0.152539 |      0.0 | 0.0 | 0.0 | 112.061866 |  2.510453 | 10.796821 | 0.042372 |
    | 02.01. 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.868067 |   nan | 0.043969 | 0.151905 |   0.303809 | -0.043969 |  0.384322 | 0.0 | 0.054744 |  0.439066 | 0.309372 | 0.299372 |      0.0 | 0.0 | 0.0 | 110.237768 |  1.930258 | 10.893982 | 0.083159 |
    | 02.01. 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.059355 |   nan | 0.037017 | 0.198272 |   0.396544 |  6.826517 |  2.037496 | 0.0 | 0.055461 |  2.092957 | 0.783325 | 0.773325 |  1.84926 | 0.0 | 0.0 | 125.943496 |  6.521007 | 11.036793 | 0.214812 |
    | 02.01. 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.009912 |   nan | 0.023488 | 0.292753 |   0.585507 | 17.771214 |  9.477037 | 0.0 | 0.056648 |  9.533685 | 3.378921 | 3.368921 | 1.979276 | 0.0 | 0.0 |  153.03683 | 14.522431 | 11.272899 | 0.935811 |
    | 02.01. 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.445428 |   nan | 0.021467 | 0.306709 |   0.613419 |  5.670172 |  8.550769 | 0.0 | 0.057898 |  8.608667 |  7.67463 |  7.66463 | 1.092627 | 0.0 | 0.0 | 156.642106 | 11.335124 |  11.52171 | 2.129064 |
    | 02.01. 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.427817 |   nan | 0.021028 | 0.310227 |   0.620455 |  2.045842 |  4.140784 | 0.0 |  0.05916 |  4.199943 |  7.83451 |  7.82451 | 1.133348 | 0.0 | 0.0 | 157.537874 |  8.929955 | 11.772778 | 2.173475 |
    | 02.01. 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.049957 |   nan | 0.019705 | 0.322246 |   0.644491 |  4.970827 |  3.415715 | 0.0 | 0.060475 |   3.47619 | 5.018826 | 5.008826 | 1.900086 | 0.0 | 0.0 | 160.560437 | 10.162822 | 12.034548 |  1.39134 |
    | 02.01. 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.004295 |   nan | 0.015831 | 0.354271 |   0.708542 | 14.084406 |  7.923725 | 0.0 | 0.061944 |  7.985669 |  4.63913 |  4.62913 |  1.99141 | 0.0 | 0.0 | 168.349822 | 15.969232 | 12.326875 |  1.28587 |
    | 02.01. 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.006196 |   nan |  0.01316 | 0.377088 |   0.754176 | 12.943663 | 11.304212 | 0.0 |  0.06352 | 11.367732 | 7.735132 | 7.725132 | 1.987608 | 0.0 | 0.0 | 173.686572 | 17.231595 | 12.640443 |  2.14587 |
    | 02.01. 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.018028 |   nan | 0.012238 |  0.38508 |   0.770161 |  5.622968 |  8.161988 | 0.0 | 0.065128 |  8.227115 | 9.918248 | 9.908248 | 1.963945 | 0.0 | 0.0 | 175.517585 | 14.307494 | 12.960395 | 2.752291 |
    | 02.01. 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.029884 |   nan | 0.011986 | 0.387245 |    0.77449 |  1.698652 |  4.333944 | 0.0 | 0.066738 |  4.400682 | 8.074712 | 8.064712 | 1.940232 | 0.0 | 0.0 | 176.010193 | 11.284958 | 13.280902 | 2.240198 |
    | 03.01. 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.020966 |   nan | 0.011345 |  0.39294 |   0.785881 |  4.450824 |  3.332947 | 0.0 | 0.068369 |  3.401316 | 5.028919 | 5.018919 | 1.958069 | 0.0 | 0.0 | 177.299835 | 12.009895 | 13.605473 | 1.394144 |
    | 03.01. 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.027175 |   nan | 0.011011 | 0.395879 |   0.791757 |  2.476827 |   3.05638 | 0.0 | 0.070007 |  3.126387 | 3.562302 | 3.552302 |  1.94565 | 0.0 | 0.0 | 177.961502 | 11.034464 | 13.931345 |  0.98675 |
    | 03.01. 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.034946 |   nan | 0.010963 | 0.396272 |   0.792544 |  0.416309 |  2.001337 | 0.0 | 0.071638 |  2.072975 | 2.953391 | 2.943391 | 1.930107 | 0.0 | 0.0 | 178.049897 |  9.053164 | 14.255979 | 0.817608 |
    | 03.01. 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.036358 |   nan | 0.010958 | 0.396314 |   0.792628 |  0.121958 |  1.154152 | 0.0 | 0.073261 |  1.227413 | 2.119164 | 2.109164 | 1.927284 | 0.0 | 0.0 | 178.059289 |  7.624656 | 14.579032 | 0.585879 |
    | 03.01. 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.035498 |   nan | 0.010916 | 0.396686 |   0.793371 |  0.402268 |  0.759428 | 0.0 | 0.074879 |  0.834307 | 1.327959 | 1.317959 | 1.929004 | 0.0 | 0.0 | 178.142807 |  6.870811 | 14.900839 |   0.3661 |
    | 03.01. 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.028761 |   nan | 0.010641 | 0.399167 |   0.798335 |  2.195093 |   0.80239 | 0.0 |   0.0765 |   0.87889 | 0.931571 | 0.921571 | 1.942479 | 0.0 | 0.0 | 178.699157 |  7.864347 | 15.223506 | 0.255992 |
    | 03.01. 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.061669 |   nan | 0.010608 | 0.399316 |   0.798631 |  0.322839 |  0.800248 | 0.0 | 0.078114 |  0.878362 | 0.868865 | 0.858865 | 1.876663 | 0.0 | 0.0 | 178.732328 |  6.987622 | 15.544708 | 0.238574 |
    | 03.01. 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.191814 |   nan | 0.010622 | 0.398609 |   0.797218 |  0.080631 |  0.553214 | 0.0 | 0.079717 |   0.63293 | 0.823939 | 0.813939 | 1.616372 | 0.0 | 0.0 | 178.574145 |  6.116431 |   15.8636 | 0.226094 |
    | 03.01. 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.490148 |   nan | 0.010713 | 0.396472 |   0.792943 | -0.010713 |  0.375567 | 0.0 |   0.0813 |  0.456868 | 0.648346 | 0.638346 | 0.636076 | 0.0 | 0.0 |  178.09471 |  5.333679 | 16.178771 | 0.177318 |
    | 03.01. 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  3.19172 |   nan | 0.010953 | 0.382436 |   0.764872 | -0.010953 |  0.278561 | 0.0 | 0.082806 |  0.361367 |  0.47477 |  0.46477 |      0.0 | 0.0 | 0.0 | 174.913943 |   4.66173 | 16.478401 | 0.129103 |
    | 03.01. 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.335546 |   nan | 0.012543 | 0.368043 |   0.736086 | -0.012543 |  0.208651 | 0.0 | 0.084232 |  0.292883 | 0.367371 | 0.357371 |      0.0 | 0.0 | 0.0 | 171.590941 |  4.072493 | 16.762212 |  0.09927 |
    | 03.01. 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 2.253813 |   nan | 0.014061 | 0.359715 |   0.719431 |   0.78739 |  0.198136 | 0.0 |  0.08561 |  0.283746 | 0.306071 | 0.296071 |      0.0 | 0.0 | 0.0 | 169.638537 |  4.302031 | 17.036318 | 0.082242 |
    | 03.01. 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.715969 |   nan | 0.015181 | 0.344192 |   0.688383 | -0.015181 |  0.199364 | 0.0 | 0.086903 |  0.286266 | 0.286337 | 0.276337 |      0.0 | 0.0 | 0.0 | 165.937749 |  3.743294 | 17.293607 |  0.07676 |
    | 03.01. 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.576688 |   nan | 0.017031 | 0.329583 |   0.659166 | -0.017031 |  0.149065 | 0.0 | 0.088116 |  0.237181 | 0.274798 | 0.264798 |      0.0 | 0.0 | 0.0 | 162.378092 |  3.247615 | 17.535074 | 0.073555 |
    | 03.01. 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 6.099883 |   nan | 0.018811 | 0.305359 |   0.610719 | -0.018811 |  0.121494 | 0.0 | 0.089202 |  0.210696 | 0.242203 | 0.232203 |      0.0 | 0.0 | 0.0 |  156.29702 |  2.801951 | 17.751232 | 0.064501 |
    | 03.01. 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.958804 |   nan | 0.021851 |  0.29017 |    0.58034 | -0.021851 |  0.090079 | 0.0 | 0.090207 |  0.180286 | 0.209824 | 0.199824 |      0.0 | 0.0 | 0.0 | 152.360068 |   2.39985 | 17.951194 | 0.055507 |
    | 03.01. 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.221357 |   nan |  0.02382 | 0.274402 |   0.548803 |  -0.02382 |  0.066087 | 0.0 | 0.091128 |  0.157215 | 0.181917 | 0.171917 |      0.0 | 0.0 | 0.0 | 148.162531 |  2.035542 | 18.134468 | 0.047755 |
    | 03.01. 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.541892 |   nan | 0.025919 | 0.265161 |   0.530323 | -0.025919 |  0.043836 | 0.0 | 0.091998 |  0.135834 |  0.15759 |  0.14759 |      0.0 | 0.0 | 0.0 | 145.646557 |  1.700626 | 18.307631 | 0.040997 |
    | 03.01. 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.145352 |   nan | 0.027177 | 0.261106 |   0.522211 | -0.027177 |  0.025957 | 0.0 | 0.092844 |  0.118801 |   0.1368 |   0.1268 |      0.0 | 0.0 | 0.0 | 144.528382 |  1.386387 | 18.475893 | 0.035222 |
    | 03.01. 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.207331 |   nan | 0.027736 | 0.260457 |   0.520914 | -0.027736 |  0.013472 | 0.0 | 0.093682 |  0.107154 | 0.119998 | 0.109998 |      0.0 | 0.0 | 0.0 | 144.348787 |  1.084722 | 18.642669 | 0.030555 |
    | 03.01. 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.065993 |   nan | 0.027826 | 0.260319 |   0.520639 | -0.027826 |   0.00595 | 0.0 | 0.094515 |  0.100465 | 0.108256 | 0.098256 |      0.0 | 0.0 | 0.0 |  144.31062 |  0.790627 | 18.808473 | 0.027293 |
    | 03.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.064624 |   nan | 0.027845 | 0.260187 |   0.520374 | -0.027845 |  0.001991 | 0.0 | 0.095343 |  0.097335 | 0.101256 | 0.091256 |      0.0 | 0.0 | 0.0 |  144.27384 |  0.500604 | 18.973317 | 0.025349 |
    | 03.01. 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.063512 |   nan | 0.027863 | 0.260058 |   0.520116 | -0.027863 |   0.00037 | 0.0 | 0.096167 |  0.096537 | 0.097853 | 0.087853 |      0.0 | 0.0 | 0.0 | 144.238191 |  0.212312 | 19.137208 | 0.024404 |
    | 03.01. 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.062667 |   nan | 0.027881 | 0.184419 |   0.519866 | -0.027881 |  0.000012 | 0.0 | 0.096608 |   0.09662 | 0.096733 | 0.086733 |      0.0 | 0.0 | 0.0 | 144.203406 |       0.0 | 19.225019 | 0.024092 |
    | 04.01. 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.061942 |   nan |      0.0 |      0.0 |   0.519419 |       0.0 |       0.0 | 0.0 | 0.096125 |  0.096125 | 0.096492 | 0.086492 |      0.0 | 0.0 | 0.0 | 144.141463 |       0.0 | 19.128894 | 0.024026 |
    | 04.01. 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.063579 |   nan |      0.0 |      0.0 |   0.518961 |       0.0 |       0.0 | 0.0 | 0.095644 |  0.095644 | 0.096128 | 0.086128 |      0.0 | 0.0 | 0.0 | 144.077884 |       0.0 |  19.03325 | 0.023925 |
    | 04.01. 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.063825 |   nan |      0.0 |      0.0 |   0.518501 |       0.0 |       0.0 | 0.0 | 0.095166 |  0.095166 | 0.095645 | 0.085645 |      0.0 | 0.0 | 0.0 | 144.014059 |       0.0 | 18.938083 |  0.02379 |
    | 04.01. 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.026921 |   nan |      0.0 |      0.0 |   0.518307 |       0.0 |       0.0 | 0.0 |  0.09469 |   0.09469 | 0.095167 | 0.085167 | 1.484581 | 0.0 | 0.0 | 143.987137 |       0.0 | 18.843393 | 0.023657 |
    | 04.01. 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.031342 |   nan |      0.0 |      0.0 |   0.518082 |       0.0 |       0.0 | 0.0 | 0.094217 |  0.094217 | 0.094691 | 0.084691 | 1.414925 | 0.0 | 0.0 | 143.955795 |       0.0 | 18.749176 | 0.023525 |
    | 04.01. 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.031885 |   nan |      0.0 |      0.0 |   0.517852 |       0.0 |       0.0 | 0.0 | 0.093746 |  0.093746 | 0.094217 | 0.084217 | 1.344047 | 0.0 | 0.0 |  143.92391 |       0.0 |  18.65543 | 0.023394 |
    | 04.01. 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.116535 |   nan |      0.0 |      0.0 |   0.517014 |       0.0 |       0.0 | 0.0 | 0.093277 |  0.093277 | 0.093746 | 0.083746 | 1.084943 | 0.0 | 0.0 | 143.807375 |       0.0 | 18.562153 | 0.023263 |
    | 04.01. 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.480764 |   nan |      0.0 |      0.0 |   0.513563 |       0.0 |       0.0 | 0.0 | 0.092811 |  0.092811 | 0.093278 | 0.083278 | 0.846748 | 0.0 | 0.0 | 143.326611 |       0.0 | 18.469342 | 0.023133 |
    | 04.01. 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.751726 |   nan |      0.0 |      0.0 |   0.501086 |       0.0 |       0.0 | 0.0 | 0.092347 |  0.092347 | 0.092811 | 0.082811 |      0.0 | 0.0 | 0.0 | 141.574885 |       0.0 | 18.376995 | 0.023003 |
    | 04.01. 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.791428 |   nan |      0.0 |      0.0 |   0.474607 |       0.0 |       0.0 | 0.0 | 0.091885 |  0.091885 | 0.092347 | 0.082347 |      0.0 | 0.0 | 0.0 | 137.783457 |       0.0 |  18.28511 | 0.022874 |
    | 04.01. 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  4.90729 |   nan |      0.0 |      0.0 |   0.441402 |       0.0 |       0.0 | 0.0 | 0.091426 |  0.091426 | 0.091885 | 0.081885 |      0.0 | 0.0 | 0.0 | 132.876167 |       0.0 | 18.193685 | 0.022746 |
    | 04.01. 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.820411 |   nan |      0.0 |      0.0 |   0.403579 |       0.0 |       0.0 | 0.0 | 0.090968 |  0.090968 | 0.091426 | 0.081426 |      0.0 | 0.0 | 0.0 | 127.055756 |       0.0 | 18.102717 | 0.022618 |
    | 04.01. 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.565361 |   nan |      0.0 |      0.0 |   0.368998 |       0.0 |       0.0 | 0.0 | 0.090514 |  0.090514 | 0.090969 | 0.080969 |      0.0 | 0.0 | 0.0 | 121.490395 |       0.0 | 18.012203 | 0.022491 |
    | 04.01. 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 6.795388 |   nan |      0.0 |      0.0 |   0.328874 |       0.0 |       0.0 | 0.0 | 0.090061 |  0.090061 | 0.090514 | 0.080514 |      0.0 | 0.0 | 0.0 | 114.695007 |       0.0 | 17.922142 | 0.022365 |
    | 04.01. 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.617912 |   nan |      0.0 |      0.0 |   0.297445 |       0.0 |       0.0 | 0.0 | 0.089611 |  0.089611 | 0.090062 | 0.080062 |      0.0 | 0.0 | 0.0 | 109.077094 |       0.0 | 17.832531 | 0.022239 |
    | 04.01. 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.476436 |   nan |      0.0 |      0.0 |   0.273532 |       0.0 |       0.0 | 0.0 | 0.089163 |  0.089163 | 0.089611 | 0.079611 |      0.0 | 0.0 | 0.0 | 104.600658 |       0.0 | 17.743369 | 0.022114 |
    | 04.01. 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.438985 |   nan |      0.0 |      0.0 |   0.255842 |       0.0 |       0.0 | 0.0 | 0.088717 |  0.088717 | 0.089163 | 0.079163 |      0.0 | 0.0 | 0.0 | 101.161673 |       0.0 | 17.654652 |  0.02199 |
    | 04.01. 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.556952 |   nan |      0.0 |      0.0 |   0.248028 |       0.0 |       0.0 | 0.0 | 0.088273 |  0.088273 | 0.088717 | 0.078717 |      0.0 | 0.0 | 0.0 |  99.604721 |       0.0 | 17.566378 | 0.021866 |
    | 04.01. 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.680718 |   nan |      0.0 |      0.0 |   0.244649 |       0.0 |       0.0 | 0.0 | 0.087832 |  0.087832 | 0.088274 | 0.078274 |      0.0 | 0.0 | 0.0 |  98.924003 |       0.0 | 17.478547 | 0.021743 |
    | 04.01. 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.170174 |   nan |      0.0 |      0.0 |   0.243808 |       0.0 |       0.0 | 0.0 | 0.087393 |  0.087393 | 0.087832 | 0.077832 |      0.0 | 0.0 | 0.0 |  98.753829 |       0.0 | 17.391154 |  0.02162 |
    | 04.01. 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.046141 |   nan |      0.0 |      0.0 |    0.24358 |       0.0 |       0.0 | 0.0 | 0.086956 |  0.086956 | 0.087393 | 0.077393 |      0.0 | 0.0 | 0.0 |  98.707688 |       0.0 | 17.304198 | 0.021498 |
    | 04.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.044194 |   nan |      0.0 |      0.0 |   0.243362 |       0.0 |       0.0 | 0.0 | 0.086521 |  0.086521 | 0.086956 | 0.076956 |      0.0 | 0.0 | 0.0 |  98.663494 |       0.0 | 17.217677 | 0.021377 |
    | 04.01. 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.043519 |   nan |      0.0 |      0.0 |   0.243147 |       0.0 |       0.0 | 0.0 | 0.086088 |  0.086088 | 0.086521 | 0.076521 |      0.0 | 0.0 | 0.0 |  98.619975 |       0.0 | 17.131589 | 0.021256 |
    | 04.01. 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.043688 |   nan |      0.0 |      0.0 |   0.242932 |       0.0 |       0.0 | 0.0 | 0.085658 |  0.085658 | 0.086089 | 0.076089 |      0.0 | 0.0 | 0.0 |  98.576287 |       0.0 | 17.045931 | 0.021136 |

    .. raw:: html

        <iframe
            src="hland_v1_ex2.html"
            width="100%"
            height="930px"
            frameborder=0
        ></iframe>

    .. _hland_v1_ex3:

    **Example 3**

    The next example is supposed to demonstrate that one should be
    careful when reducing the value of |RecStep| in order to save
    computation times.  Setting |RecStep| to one, which is the lowest
    possible value, results in low accuracies in the calculation of |Q0|.
    This can easily be seen when comparing the |Q0| graph of the last example
    and the |Q0| graph of this example.  Obviously, the |Q0| graph of
    this example is (much more) wrong, as its maximum peak is largely
    above the maximum peak of |R|, which is physically impossible:

    >>> recstep(1)
    >>> test('hland_v1_ex3')
    |         date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |      pc |        ep |      epc |       ei |        tf | glmelt | melt | refr |       in_ |         r |       ea | cfpot |       cf |     perc | contriarea |      inuz |        q0 |  el |       q1 |      inuh |     outuh |        qt |       ic |  sp |  wc |         sm |       uz |        lz |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.052569 |   nan |      0.0 |      0.0 |   0.249737 |       0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.051111 |      0.0 | 0.0 | 0.0 |  99.947431 |      0.0 |      9.95 | 0.014198 |
    | 01.01. 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.051025 |   nan |      0.0 |      0.0 |   0.249482 |       0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.078833 |      0.0 | 0.0 | 0.0 |  99.896406 |      0.0 |   9.90025 | 0.021898 |
    | 01.01. 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.050572 |   nan |      0.0 |      0.0 |    0.24923 |       0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.03975 |      0.0 | 0.0 | 0.0 |  99.845834 |      0.0 |  9.850749 | 0.011042 |
    | 01.01. 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.05003 |   nan |      0.0 |      0.0 |    0.24898 |       0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.039502 |      0.0 | 0.0 | 0.0 |  99.795804 |      0.0 |  9.801495 | 0.010973 |
    | 01.01. 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.050521 |   nan |      0.0 |      0.0 |   0.248728 |       0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.039254 |      0.0 | 0.0 | 0.0 |  99.745284 |      0.0 |  9.752488 | 0.010904 |
    | 01.01. 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.053505 |   nan |      0.0 |      0.0 |   0.248461 |       0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.039008 |      0.0 | 0.0 | 0.0 |  99.691779 |      0.0 |  9.703725 | 0.010835 |
    | 01.01. 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.151906 |   nan |      0.0 |      0.0 |   0.247705 |       0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.038763 |      0.0 | 0.0 | 0.0 |  99.539873 |      0.0 |  9.655206 | 0.010767 |
    | 01.01. 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.004318 |   nan |      0.0 |      0.0 |   0.242731 |       0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.038519 |      0.0 | 0.0 | 0.0 |  98.535555 |      0.0 |   9.60693 |   0.0107 |
    | 01.01. 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.247495 |   nan |      0.0 |      0.0 |   0.231785 |       0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.038276 |      0.0 | 0.0 | 0.0 |  96.288059 |      0.0 |  9.558896 | 0.010632 |
    | 01.01. 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.677977 |   nan |      0.0 |      0.0 |   0.214416 |       0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.038035 |      0.0 | 0.0 | 0.0 |  92.610082 |      0.0 |  9.511101 | 0.010565 |
    | 01.01. 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.995196 |   nan |      0.0 |      0.0 |   0.196315 |       0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.037795 |      0.0 | 0.0 | 0.0 |  88.614886 |      0.0 |  9.463546 | 0.010499 |
    | 01.01. 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.334217 |   nan |      0.0 |      0.0 |   0.177581 |       0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.037556 |      0.0 | 0.0 | 0.0 |   84.28067 |      0.0 |  9.416228 | 0.010432 |
    | 01.01. 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.255237 |   nan |      0.0 |      0.0 |   0.168204 |       0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.037318 |      0.0 | 0.0 | 0.0 |  82.025433 |      0.0 |  9.369147 | 0.010366 |
    | 01.01. 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.836657 |   nan |      0.0 |      0.0 |   0.156772 |       0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.037081 |      0.0 | 0.0 | 0.0 |  79.188775 |      0.0 |  9.322301 |   0.0103 |
    | 01.01. 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  3.05792 |   nan |      0.0 |      0.0 |   0.144898 |       0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.036846 |      0.0 | 0.0 | 0.0 |  76.130856 |      0.0 |   9.27569 | 0.010235 |
    | 01.01. 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.928078 |   nan |      0.0 |      0.0 |   0.141386 |       0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.036612 |      0.0 | 0.0 | 0.0 |  75.202777 |      0.0 |  9.229311 |  0.01017 |
    | 01.01. 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.718317 |   nan |      0.0 |      0.0 |   0.138698 |       0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.036379 |      0.0 | 0.0 | 0.0 |  74.484461 |      0.0 |  9.183165 | 0.010105 |
    | 01.01. 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.441908 |   nan |      0.0 |      0.0 |   0.137057 |       0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.036147 |      0.0 | 0.0 | 0.0 |  74.042552 |      0.0 |  9.137249 | 0.010041 |
    | 01.01. 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  0.12436 |   nan |      0.0 |      0.0 |   0.136597 |       0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.035916 | 1.006935 | 0.0 | 0.0 |  73.918192 |      0.0 |  9.091563 | 0.009977 |
    | 01.01. 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.023694 |   nan | 0.060598 |   0.0777 |     0.1554 |  0.712508 |  0.068166 | 0.0 | 0.045846 |  0.114013 |  0.060921 |  0.050921 | 1.903859 | 0.0 | 0.0 |  78.841725 | 0.566641 |  9.123417 | 0.014145 |
    | 01.01. 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.015028 |   nan | 0.059165 | 0.083466 |   0.166931 |   0.46128 |  0.181105 | 0.0 | 0.046034 |   0.22714 |  0.123968 |  0.113968 | 1.941161 | 0.0 | 0.0 |  81.714476 | 0.763351 |  9.160848 | 0.031658 |
    | 01.01. 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.012429 |   nan | 0.056743 | 0.093656 |   0.187313 |  0.905176 |  0.594339 | 0.0 | 0.046273 |  0.640612 |  0.293883 |  0.283883 | 1.954058 | 0.0 | 0.0 |  86.559232 | 0.980532 |  9.208232 | 0.078856 |
    | 01.01. 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.007038 |   nan | 0.051622 | 0.117129 |   0.234258 |  2.298561 |  2.459152 | 0.0 | 0.046627 |  2.505778 |  0.963211 |  0.953211 | 1.976736 | 0.0 | 0.0 |  96.800491 | 0.702812 |  9.278734 | 0.264781 |
    | 01.01. 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.024367 |   nan | 0.051563 | 0.117372 |   0.234744 | -0.029183 |  0.013306 | 0.0 | 0.046981 |  0.060286 |  1.547854 |  1.537854 | 1.919551 | 0.0 | 0.0 |  96.900842 | 0.542951 |  9.349126 | 0.427182 |
    | 02.01. 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.022832 |   nan | 0.051262 | 0.118838 |   0.237676 |  0.125066 |  0.012336 | 0.0 |  0.04734 |  0.059676 |  0.603593 |  0.593593 | 1.925086 | 0.0 | 0.0 |  97.504096 | 0.536843 |  9.420624 | 0.164887 |
    | 02.01. 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.017851 |   nan | 0.049918 |  0.12549 |   0.250981 |  0.779355 |  0.106781 | 0.0 | 0.047731 |  0.154511 |  0.080886 |  0.070886 | 1.942997 | 0.0 | 0.0 | 100.195975 | 1.083926 |  9.498384 | 0.019691 |
    | 02.01. 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.020132 |   nan | 0.048989 |  0.13018 |   0.260359 |  0.562851 |  0.197648 | 0.0 | 0.048143 |  0.245791 |  0.153721 |  0.143721 | 1.936885 | 0.0 | 0.0 | 102.050789 |  1.31895 |   9.58042 | 0.039923 |
    | 02.01. 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.008153 |   nan | 0.044429 | 0.154509 |   0.309018 |  3.155927 |  2.732817 | 0.0 | 0.048675 |  2.781491 |  0.788996 |  0.778996 | 1.976535 | 0.0 | 0.0 | 111.178794 | 1.587552 |  9.686255 | 0.216388 |
    | 02.01. 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.018923 |   nan | 0.042982 |  0.16262 |   0.325241 |  1.234664 |  0.546805 | 0.0 | 0.049244 |  0.596049 |  1.732349 |  1.722349 |  1.94692 | 0.0 | 0.0 | 114.059742 |  2.11279 |  9.799631 |  0.47843 |
    | 02.01. 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.019537 |   nan | 0.041625 | 0.170445 |    0.34089 |  1.254823 |  0.825003 | 0.0 |  0.04985 |  0.874853 |  1.143659 |  1.133659 |  1.94647 | 0.0 | 0.0 | 116.771503 | 2.372166 |  9.920225 | 0.314905 |
    | 02.01. 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 |   0.0499 |   nan | 0.041162 | 0.173069 |   0.346139 |  0.426562 |   0.43648 | 0.0 | 0.050466 |  0.486946 |  0.726695 |  0.716695 | 1.864353 | 0.0 | 0.0 |  117.66711 | 2.189179 | 10.042828 | 0.199082 |
    | 02.01. 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.383924 |   nan | 0.041166 | 0.172063 |   0.344125 | -0.041166 |  0.189312 | 0.0 | 0.051074 |  0.240387 |  0.518357 |  0.508357 | 0.939422 | 0.0 | 0.0 | 117.324353 | 1.786638 | 10.163816 |  0.14121 |
    | 02.01. 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.735028 |   nan | 0.041338 | 0.167131 |   0.334261 | -0.041338 |  0.105245 | 0.0 | 0.051655 |    0.1569 |  0.276625 |  0.266625 |      0.0 | 0.0 | 0.0 | 115.630662 | 1.472924 | 10.279292 | 0.074063 |
    | 02.01. 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.165815 |   nan | 0.042185 | 0.163898 |   0.327797 | -0.042185 |  0.057724 | 0.0 | 0.052216 |   0.10994 |  0.165017 |  0.155017 |      0.0 | 0.0 | 0.0 | 114.507032 | 1.209117 | 10.390974 |  0.04306 |
    | 02.01. 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.403262 |   nan | 0.042746 | 0.160027 |   0.320053 | -0.042746 |  0.031087 | 0.0 | 0.052755 |  0.083842 |  0.114576 |  0.104576 |      0.0 | 0.0 | 0.0 | 113.146516 | 0.975258 | 10.498246 | 0.029049 |
    | 02.01. 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.629327 |   nan | 0.043427 | 0.150044 |   0.300088 | -0.043427 |  0.017681 | 0.0 | 0.053241 |  0.070923 |   0.08677 |   0.07677 |      0.0 | 0.0 | 0.0 | 109.560616 | 0.764105 | 10.595049 | 0.021325 |
    | 02.01. 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.638626 |   nan | 0.044423 | 0.152794 |   0.305588 |  0.638818 |  0.068463 | 0.0 | 0.053739 |  0.122202 |  0.085189 |  0.075189 | 0.162204 | 0.0 | 0.0 | 110.559972 | 1.181666 | 10.694103 | 0.020886 |
    | 02.01. 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 2.132116 |   nan | 0.042924 | 0.156973 |   0.313947 |  1.537393 |  0.543518 | 0.0 | 0.054255 |  0.597773 |  0.216489 |  0.206489 |      0.0 | 0.0 | 0.0 | 112.061866 | 2.018568 | 10.796821 | 0.057358 |
    | 02.01. 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.868067 |   nan | 0.043969 | 0.151905 |   0.303809 | -0.043969 |  0.215943 | 0.0 | 0.054744 |  0.270687 |  0.419405 |  0.409405 |      0.0 | 0.0 | 0.0 | 110.237768 | 1.606751 | 10.893982 | 0.113724 |
    | 02.01. 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.059355 |   nan | 0.037017 | 0.198272 |   0.396544 |  6.826517 |  8.234997 | 0.0 | 0.055461 |  8.290458 |  2.125544 |  2.115544 |  1.84926 | 0.0 | 0.0 | 125.943496 |      0.0 | 11.036793 | 0.587651 |
    | 02.01. 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.009912 |   nan | 0.023488 | 0.292753 |   0.585507 | 17.771214 | 17.478461 | 0.0 | 0.056648 | 17.535108 |  8.562653 |  8.552653 | 1.979276 | 0.0 | 0.0 |  153.03683 |      0.0 | 11.272899 | 2.375737 |
    | 02.01. 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.445428 |   nan | 0.021467 | 0.306709 |   0.613419 |  5.670172 |  0.668443 | 0.0 | 0.057898 |  0.726341 |  11.74546 |  11.73546 | 1.092627 | 0.0 | 0.0 | 156.642106 | 4.695019 |  11.52171 |  3.25985 |
    | 02.01. 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.427817 |   nan | 0.021028 | 0.310227 |   0.620455 |  2.045842 |  1.113349 | 0.0 |  0.05916 |  1.172509 |  4.560771 |  4.550771 | 1.133348 | 0.0 | 0.0 | 157.537874 | 5.317284 | 11.772778 | 1.264103 |
    | 02.01. 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.049957 |   nan | 0.019705 | 0.322246 |   0.644491 |  4.970827 |  3.697378 | 0.0 | 0.060475 |  3.757853 |  1.647881 |  1.637881 | 1.900086 | 0.0 | 0.0 | 160.560437 | 6.268488 | 12.034548 | 0.454967 |
    | 02.01. 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.004295 |   nan | 0.015831 | 0.354271 |   0.708542 | 14.084406 | 19.998623 | 0.0 | 0.061944 | 20.060568 |  6.806157 |  6.796157 |  1.99141 | 0.0 | 0.0 | 168.349822 |      0.0 | 12.326875 | 1.887821 |
    | 02.01. 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.006196 |   nan |  0.01316 | 0.377088 |   0.754176 | 12.943663 |  4.626295 | 0.0 |  0.06352 |  4.689814 | 13.022019 | 13.012019 | 1.987608 | 0.0 | 0.0 | 173.686572 |  7.94028 | 12.640443 |  3.61445 |
    | 02.01. 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.018028 |   nan | 0.012238 |  0.38508 |   0.770161 |  5.622968 |   5.00981 | 0.0 | 0.065128 |  5.074938 |   8.19112 |   8.18112 | 1.963945 | 0.0 | 0.0 | 175.517585 | 8.168358 | 12.960395 | 2.272533 |
    | 02.01. 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.029884 |   nan | 0.011986 | 0.387245 |    0.77449 |  1.698652 |  1.833774 | 0.0 | 0.066738 |  1.900512 |  4.283927 |  4.273927 | 1.940232 | 0.0 | 0.0 | 176.010193 | 7.645991 | 13.280902 | 1.187202 |
    | 03.01. 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.020966 |   nan | 0.011345 |  0.39294 |   0.785881 |  4.450824 |  3.303079 | 0.0 | 0.068369 |  3.371448 |  2.932815 |  2.922815 | 1.958069 | 0.0 | 0.0 | 177.299835 | 8.400796 | 13.605473 | 0.811893 |
    | 03.01. 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.027175 |   nan | 0.011011 | 0.395879 |   0.791757 |  2.476827 |  2.320194 | 0.0 | 0.070007 |  2.390201 |  2.826519 |  2.816519 |  1.94565 | 0.0 | 0.0 | 177.961502 |  8.16155 | 13.931345 | 0.782366 |
    | 03.01. 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.034946 |   nan | 0.010963 | 0.396272 |   0.792544 |  0.416309 |  1.100126 | 0.0 | 0.071638 |  1.171764 |  2.337492 |  2.327492 | 1.930107 | 0.0 | 0.0 | 178.049897 | 7.081462 | 14.255979 | 0.646526 |
    | 03.01. 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.036358 |   nan | 0.010958 | 0.396314 |   0.792628 |  0.121958 |  0.633402 | 0.0 | 0.073261 |  0.706664 |  1.339172 |  1.329172 | 1.927284 | 0.0 | 0.0 | 178.059289 | 6.173703 | 14.579032 | 0.369214 |
    | 03.01. 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.035498 |   nan | 0.010916 | 0.396686 |   0.793371 |  0.402268 |  0.472482 | 0.0 | 0.074879 |   0.54736 |  0.774619 |  0.764619 | 1.929004 | 0.0 | 0.0 | 178.142807 | 5.706804 | 14.900839 | 0.212394 |
    | 03.01. 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.028761 |   nan | 0.010641 | 0.399167 |   0.798335 |  2.195093 |  0.830047 | 0.0 |   0.0765 |  0.906547 |   0.66258 |   0.65258 | 1.942479 | 0.0 | 0.0 | 178.699157 | 6.672683 | 15.223506 | 0.181272 |
    | 03.01. 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.061669 |   nan | 0.010608 | 0.399316 |   0.798631 |  0.322839 |  0.563435 | 0.0 | 0.078114 |  0.641549 |   0.76784 |   0.75784 | 1.876663 | 0.0 | 0.0 | 178.732328 | 6.032771 | 15.544708 | 0.210511 |
    | 03.01. 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.191814 |   nan | 0.010622 | 0.398609 |   0.797218 |  0.080631 |  0.368358 | 0.0 | 0.079717 |  0.448075 |  0.657443 |  0.647443 | 1.616372 | 0.0 | 0.0 | 178.574145 | 5.346435 |   15.8636 | 0.179845 |
    | 03.01. 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.490148 |   nan | 0.010713 | 0.396472 |   0.792943 | -0.010713 |  0.241689 | 0.0 |   0.0813 |  0.322989 |  0.463272 |  0.453272 | 0.636076 | 0.0 | 0.0 |  178.09471 | 4.697562 | 16.178771 | 0.125909 |
    | 03.01. 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  3.19172 |   nan | 0.010953 | 0.382436 |   0.764872 | -0.010953 |  0.178198 | 0.0 | 0.082806 |  0.261004 |  0.337012 |  0.327012 |      0.0 | 0.0 | 0.0 | 174.913943 | 4.125975 | 16.478401 | 0.090837 |
    | 03.01. 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.335546 |   nan | 0.012543 | 0.368043 |   0.736086 | -0.012543 |  0.131736 | 0.0 | 0.084232 |  0.215968 |  0.264771 |  0.254771 |      0.0 | 0.0 | 0.0 | 171.590941 | 3.613653 | 16.762212 |  0.07077 |
    | 03.01. 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 2.253813 |   nan | 0.014061 | 0.359715 |   0.719431 |   0.78739 |  0.177258 | 0.0 |  0.08561 |  0.262867 |  0.236398 |  0.226398 |      0.0 | 0.0 | 0.0 | 169.638537 |  3.86407 | 17.036318 | 0.062888 |
    | 03.01. 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.715969 |   nan | 0.015181 | 0.344192 |   0.688383 | -0.015181 |  0.131965 | 0.0 | 0.086903 |  0.218868 |  0.242668 |  0.232668 |      0.0 | 0.0 | 0.0 | 165.937749 | 3.372732 | 17.293607 |  0.06463 |
    | 03.01. 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.576688 |   nan | 0.017031 | 0.329583 |   0.659166 | -0.017031 |  0.096755 | 0.0 | 0.088116 |  0.184871 |  0.221091 |  0.211091 |      0.0 | 0.0 | 0.0 | 162.378092 | 2.929363 | 17.535074 | 0.058636 |
    | 03.01. 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 6.099883 |   nan | 0.018811 | 0.305359 |   0.610719 | -0.018811 |  0.077624 | 0.0 | 0.089202 |  0.166826 |  0.188416 |  0.178416 |      0.0 | 0.0 | 0.0 |  156.29702 | 2.527568 | 17.751232 |  0.04956 |
    | 03.01. 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.958804 |   nan | 0.021851 |  0.29017 |    0.58034 | -0.021851 |  0.055641 | 0.0 | 0.090207 |  0.145848 |  0.166174 |  0.156174 |      0.0 | 0.0 | 0.0 | 152.360068 | 2.159906 | 17.951194 | 0.043382 |
    | 03.01. 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.221357 |   nan |  0.02382 | 0.274402 |   0.548803 |  -0.02382 |  0.039036 | 0.0 | 0.091128 |  0.130164 |  0.147025 |  0.137025 |      0.0 | 0.0 | 0.0 | 148.162531 | 1.822648 | 18.134468 | 0.038062 |
    | 03.01. 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 2.541892 |   nan | 0.025919 | 0.265161 |   0.530323 | -0.025919 |  0.024087 | 0.0 | 0.091998 |  0.116085 |  0.130521 |  0.120521 |      0.0 | 0.0 | 0.0 | 145.646557 | 1.507481 | 18.307631 | 0.033478 |
    | 03.01. 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.145352 |   nan | 0.027177 | 0.261106 |   0.522211 | -0.027177 |  0.012726 | 0.0 | 0.092844 |  0.105569 |  0.116877 |  0.106877 |      0.0 | 0.0 | 0.0 | 144.528382 | 1.206472 | 18.475893 | 0.029688 |
    | 03.01. 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.207331 |   nan | 0.027736 | 0.260457 |   0.520914 | -0.027736 |  0.005478 | 0.0 | 0.093682 |   0.09916 |  0.106482 |  0.096482 |      0.0 | 0.0 | 0.0 | 144.348787 | 0.912801 | 18.642669 | 0.026801 |
    | 03.01. 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.065993 |   nan | 0.027826 | 0.260319 |   0.520639 | -0.027826 |  0.001727 | 0.0 | 0.094515 |  0.096242 |  0.099936 |  0.089936 |      0.0 | 0.0 | 0.0 |  144.31062 | 0.622929 | 18.808473 | 0.024982 |
    | 03.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.064624 |   nan | 0.027845 | 0.260187 |   0.520374 | -0.027845 |  0.000267 | 0.0 | 0.095343 |   0.09561 |   0.09675 |   0.08675 |      0.0 | 0.0 | 0.0 |  144.27384 | 0.334631 | 18.973317 | 0.024097 |
    | 03.01. 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.063512 |   nan | 0.027863 | 0.260058 |   0.520116 | -0.027863 |  0.000001 | 0.0 | 0.096167 |  0.096168 |  0.095874 |  0.085874 |      0.0 | 0.0 | 0.0 | 144.238191 | 0.046709 | 19.137208 | 0.023854 |
    | 03.01. 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.062667 |   nan | 0.027881 | 0.018828 |   0.519866 | -0.027881 |       0.0 | 0.0 |  0.09578 |   0.09578 |  0.095958 |  0.085958 |      0.0 | 0.0 | 0.0 | 144.203406 |      0.0 | 19.060256 | 0.023877 |
    | 04.01. 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.061942 |   nan |      0.0 |      0.0 |   0.519419 |       0.0 |       0.0 | 0.0 | 0.095301 |  0.095301 |   0.09576 |   0.08576 |      0.0 | 0.0 | 0.0 | 144.141463 |      0.0 | 18.964955 | 0.023822 |
    | 04.01. 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.063579 |   nan |      0.0 |      0.0 |   0.518961 |       0.0 |       0.0 | 0.0 | 0.094825 |  0.094825 |  0.095302 |  0.085302 |      0.0 | 0.0 | 0.0 | 144.077884 |      0.0 |  18.87013 | 0.023695 |
    | 04.01. 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.063825 |   nan |      0.0 |      0.0 |   0.518501 |       0.0 |       0.0 | 0.0 | 0.094351 |  0.094351 |  0.094825 |  0.084825 |      0.0 | 0.0 | 0.0 | 144.014059 |      0.0 |  18.77578 | 0.023563 |
    | 04.01. 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.026921 |   nan |      0.0 |      0.0 |   0.518307 |       0.0 |       0.0 | 0.0 | 0.093879 |  0.093879 |  0.094351 |  0.084351 | 1.484581 | 0.0 | 0.0 | 143.987137 |      0.0 | 18.681901 | 0.023431 |
    | 04.01. 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.031342 |   nan |      0.0 |      0.0 |   0.518082 |       0.0 |       0.0 | 0.0 |  0.09341 |   0.09341 |  0.093879 |  0.083879 | 1.414925 | 0.0 | 0.0 | 143.955795 |      0.0 | 18.588491 |   0.0233 |
    | 04.01. 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.031885 |   nan |      0.0 |      0.0 |   0.517852 |       0.0 |       0.0 | 0.0 | 0.092942 |  0.092942 |   0.09341 |   0.08341 | 1.344047 | 0.0 | 0.0 |  143.92391 |      0.0 | 18.495549 | 0.023169 |
    | 04.01. 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.116535 |   nan |      0.0 |      0.0 |   0.517014 |       0.0 |       0.0 | 0.0 | 0.092478 |  0.092478 |  0.092943 |  0.082943 | 1.084943 | 0.0 | 0.0 | 143.807375 |      0.0 | 18.403071 |  0.02304 |
    | 04.01. 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.480764 |   nan |      0.0 |      0.0 |   0.513563 |       0.0 |       0.0 | 0.0 | 0.092015 |  0.092015 |  0.092478 |  0.082478 | 0.846748 | 0.0 | 0.0 | 143.326611 |      0.0 | 18.311056 | 0.022911 |
    | 04.01. 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.751726 |   nan |      0.0 |      0.0 |   0.501086 |       0.0 |       0.0 | 0.0 | 0.091555 |  0.091555 |  0.092016 |  0.082016 |      0.0 | 0.0 | 0.0 | 141.574885 |      0.0 |   18.2195 | 0.022782 |
    | 04.01. 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.791428 |   nan |      0.0 |      0.0 |   0.474607 |       0.0 |       0.0 | 0.0 | 0.091098 |  0.091098 |  0.091556 |  0.081556 |      0.0 | 0.0 | 0.0 | 137.783457 |      0.0 | 18.128403 | 0.022654 |
    | 04.01. 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |  4.90729 |   nan |      0.0 |      0.0 |   0.441402 |       0.0 |       0.0 | 0.0 | 0.090642 |  0.090642 |  0.091098 |  0.081098 |      0.0 | 0.0 | 0.0 | 132.876167 |      0.0 | 18.037761 | 0.022527 |
    | 04.01. 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.820411 |   nan |      0.0 |      0.0 |   0.403579 |       0.0 |       0.0 | 0.0 | 0.090189 |  0.090189 |  0.090643 |  0.080643 |      0.0 | 0.0 | 0.0 | 127.055756 |      0.0 | 17.947572 | 0.022401 |
    | 04.01. 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.565361 |   nan |      0.0 |      0.0 |   0.368998 |       0.0 |       0.0 | 0.0 | 0.089738 |  0.089738 |  0.090189 |  0.080189 |      0.0 | 0.0 | 0.0 | 121.490395 |      0.0 | 17.857834 | 0.022275 |
    | 04.01. 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 6.795388 |   nan |      0.0 |      0.0 |   0.328874 |       0.0 |       0.0 | 0.0 | 0.089289 |  0.089289 |  0.089738 |  0.079738 |      0.0 | 0.0 | 0.0 | 114.695007 |      0.0 | 17.768545 |  0.02215 |
    | 04.01. 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 5.617912 |   nan |      0.0 |      0.0 |   0.297445 |       0.0 |       0.0 | 0.0 | 0.088843 |  0.088843 |   0.08929 |   0.07929 |      0.0 | 0.0 | 0.0 | 109.077094 |      0.0 | 17.679702 | 0.022025 |
    | 04.01. 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 4.476436 |   nan |      0.0 |      0.0 |   0.273532 |       0.0 |       0.0 | 0.0 | 0.088399 |  0.088399 |  0.088843 |  0.078843 |      0.0 | 0.0 | 0.0 | 104.600658 |      0.0 | 17.591304 | 0.021901 |
    | 04.01. 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 3.438985 |   nan |      0.0 |      0.0 |   0.255842 |       0.0 |       0.0 | 0.0 | 0.087957 |  0.087957 |  0.088399 |  0.078399 |      0.0 | 0.0 | 0.0 | 101.161673 |      0.0 | 17.503347 | 0.021778 |
    | 04.01. 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 1.556952 |   nan |      0.0 |      0.0 |   0.248028 |       0.0 |       0.0 | 0.0 | 0.087517 |  0.087517 |  0.087957 |  0.077957 |      0.0 | 0.0 | 0.0 |  99.604721 |      0.0 | 17.415831 | 0.021655 |
    | 04.01. 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.680718 |   nan |      0.0 |      0.0 |   0.244649 |       0.0 |       0.0 | 0.0 | 0.087079 |  0.087079 |  0.087517 |  0.077517 |      0.0 | 0.0 | 0.0 |  98.924003 |      0.0 | 17.328751 | 0.021533 |
    | 04.01. 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.170174 |   nan |      0.0 |      0.0 |   0.243808 |       0.0 |       0.0 | 0.0 | 0.086644 |  0.086644 |   0.08708 |   0.07708 |      0.0 | 0.0 | 0.0 |  98.753829 |      0.0 | 17.242108 | 0.021411 |
    | 04.01. 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.046141 |   nan |      0.0 |      0.0 |    0.24358 |       0.0 |       0.0 | 0.0 | 0.086211 |  0.086211 |  0.086644 |  0.076644 |      0.0 | 0.0 | 0.0 |  98.707688 |      0.0 | 17.155897 |  0.02129 |
    | 04.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.044194 |   nan |      0.0 |      0.0 |   0.243362 |       0.0 |       0.0 | 0.0 | 0.085779 |  0.085779 |  0.086211 |  0.076211 |      0.0 | 0.0 | 0.0 |  98.663494 |      0.0 | 17.070118 |  0.02117 |
    | 04.01. 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.043519 |   nan |      0.0 |      0.0 |   0.243147 |       0.0 |       0.0 | 0.0 | 0.085351 |  0.085351 |   0.08578 |   0.07578 |      0.0 | 0.0 | 0.0 |  98.619975 |      0.0 | 16.984767 |  0.02105 |
    | 04.01. 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.043688 |   nan |      0.0 |      0.0 |   0.242932 |       0.0 |       0.0 | 0.0 | 0.084924 |  0.084924 |  0.085351 |  0.075351 |      0.0 | 0.0 | 0.0 |  98.576287 |      0.0 | 16.899843 | 0.020931 |

    .. raw:: html

        <iframe
            src="hland_v1_ex3.html"
            width="100%"
            height="930px"
            frameborder=0
        ></iframe>

    .. _hland_v1_ex4:

    **Example 4**

    In the fourth example, the functionality of zones of type |ILAKE| is
    demonstrated.  For these "internal lakes" only the lower zone storage
    (|LZ|) is relevant (all other storage values are zero).  Precipitation
    (|PC|) is directly added to |LZ| and evaporation (|EPC|) is directly
    subtracted from |LZ|.  The latter occurs even when |LZ| is completely
    empty, possibly resulting in negative storage values in drought periods.
    The only case for which lake evaporation (|EL|) is prevented is when
    the actual temperature (|TC|) is below the threshold temperature
    for the occurrence of lake ice (|TTIce|).  In this example, the value
    of |TTIce| is set to the unrealistic value of 13°C, resulting in a
    deviation between the graphs of |EPC| and |EL| for the last day of
    the simulation period:

    >>> recstep(100)
    >>> zonetype(ILAKE)
    >>> ttice(13.0)
    >>> parameters.update()
    >>> test('hland_v1_ex4')
    |         date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |      pc |        ep |      epc |  ei |      tf | glmelt | melt | refr |     in_ |       r |  ea | cfpot |  cf | perc | contriarea | inuz |  q0 |       el |       q1 |     inuh |    outuh |       qt |  ic |  sp |  wc |  sm |  uz |         lz |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |     0.0 |   0.11682 |  0.08411 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  0.08411 | 0.049579 | 0.049579 | 0.061018 | 0.051018 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    9.86631 | 0.014172 |
    | 01.01. 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.113449 | 0.081683 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.081683 | 0.048923 | 0.048923 | 0.088416 | 0.078416 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.735704 | 0.021782 |
    | 01.01. 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.080999 | 0.048274 | 0.048274 | 0.048925 | 0.038925 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.606431 | 0.010812 |
    | 01.01. 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  0.111348 | 0.080171 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.080171 | 0.047631 | 0.047631 | 0.048275 | 0.038275 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.478629 | 0.010632 |
    | 01.01. 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.080999 | 0.046988 | 0.046988 | 0.047631 | 0.037631 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.350643 | 0.010453 |
    | 01.01. 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.119203 | 0.085826 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.085826 | 0.046324 | 0.046324 | 0.046984 | 0.036984 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.218493 | 0.010273 |
    | 01.01. 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.338613 | 0.243802 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.243802 | 0.044873 | 0.044873 | 0.046149 | 0.036149 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.929818 | 0.010041 |
    | 01.01. 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 |     0.0 |  2.242135 | 1.614337 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.614337 | 0.036577 | 0.036577 | 0.043352 | 0.033352 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.278903 | 0.009265 |
    | 01.01. 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 |     0.0 |  5.068662 | 3.649436 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 3.649436 | 0.018147 | 0.018147 | 0.034325 | 0.024325 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.611319 | 0.006757 |
    | 01.01. 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 |     0.0 |  8.488365 | 6.111623 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 6.111623 |      0.0 |      0.0 |  0.01821 |  0.00821 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -2.500304 | 0.002281 |
    | 01.01. 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 |     0.0 |   9.58666 | 6.902395 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 6.902395 |      0.0 |      0.0 | 0.004033 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -9.402699 |      0.0 |
    | 01.01. 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 |     0.0 | 10.869046 | 7.825713 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 7.825713 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -17.228412 |      0.0 |
    | 01.01. 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 |     0.0 |  5.946366 | 4.281384 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 4.281384 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -21.509796 |      0.0 |
    | 01.01. 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.685035 | 5.533225 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 5.533225 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -27.043021 |      0.0 |
    | 01.01. 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 |     0.0 |  8.581238 | 6.178491 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 6.178491 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -33.221512 |      0.0 |
    | 01.01. 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 |  0.2376 |  2.774152 | 1.950491 | 0.0 |  0.2376 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.950491 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -34.934403 |      0.0 |
    | 01.01. 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 |     0.0 |  2.122607 | 1.528277 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.528277 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -36.46268 |      0.0 |
    | 01.01. 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 |     0.0 |   1.31842 | 0.949262 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.949262 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -37.411942 |      0.0 |
    | 01.01. 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.871144 | 0.537465 | 0.0 |  1.5444 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.537465 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -36.405007 |      0.0 |
    | 01.01. 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |  6.6528 |  0.259718 | 0.096141 | 0.0 |  6.6528 |    0.0 |  0.0 |  0.0 |  6.6528 |  6.6528 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.096141 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -29.848348 |      0.0 |
    | 01.01. 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 |  3.4452 |  0.115333 | 0.058839 | 0.0 |  3.4452 |    0.0 |  0.0 |  0.0 |  3.4452 |  3.4452 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.058839 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -26.461986 |      0.0 |
    | 01.01. 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.114207 | 0.045942 | 0.0 |  5.8212 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.045942 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -20.686729 |      0.0 |
    | 01.01. 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 12.5928 |  0.113828 | 0.023264 | 0.0 | 12.5928 |    0.0 |  0.0 |  0.0 | 12.5928 | 12.5928 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.023264 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -8.117193 |      0.0 |
    | 01.01. 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.1188 |   0.11307 | 0.080449 | 0.0 |  0.1188 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.080449 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -8.078842 |      0.0 |
    | 02.01. 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.8316 |   0.11307 | 0.074914 | 0.0 |  0.8316 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.074914 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -7.322156 |      0.0 |
    | 02.01. 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   3.564 |   0.11307 | 0.057003 | 0.0 |   3.564 |    0.0 |  0.0 |  0.0 |   3.564 |   3.564 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.057003 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -3.815158 |      0.0 |
    | 02.01. 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |  2.4948 |  0.112498 | 0.063115 | 0.0 |  2.4948 |    0.0 |  0.0 |  0.0 |  2.4948 |  2.4948 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.063115 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -1.383473 |      0.0 |
    | 02.01. 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 12.3552 |  0.112116 | 0.023465 | 0.0 | 12.3552 |    0.0 |  0.0 |  0.0 | 12.3552 | 12.3552 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.023465 | 0.054741 | 0.054741 | 0.012165 | 0.002165 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  10.893521 | 0.000601 |
    | 02.01. 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   4.158 |  0.111732 |  0.05308 | 0.0 |   4.158 |    0.0 |  0.0 |  0.0 |   4.158 |   4.158 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  0.05308 | 0.074992 | 0.074992 | 0.047077 | 0.037077 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.923448 | 0.010299 |
    | 02.01. 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |  4.0392 |  0.111348 |  0.05353 | 0.0 |  4.0392 |    0.0 |  0.0 |  0.0 |  4.0392 |  4.0392 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  0.05353 | 0.094546 | 0.094546 | 0.074837 | 0.064837 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.814573 |  0.01801 |
    | 02.01. 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |  1.4256 |  0.217266 | 0.135647 | 0.0 |  1.4256 |    0.0 |  0.0 |  0.0 |  1.4256 |  1.4256 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.135647 | 0.100523 | 0.100523 | 0.091529 | 0.081529 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  20.004003 | 0.022647 |
    | 02.01. 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  0.1188 |   1.46695 |  1.04373 | 0.0 |  0.1188 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  1.04373 | 0.095395 | 0.095395 | 0.098055 | 0.088055 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.983677 |  0.02446 |
    | 02.01. 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |     0.0 |  3.532824 | 2.543633 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 2.543633 |   0.0822 |   0.0822 | 0.093603 | 0.083603 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  16.357844 | 0.023223 |
    | 02.01. 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |     0.0 |  2.239679 | 1.612569 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.612569 | 0.073726 | 0.073726 | 0.083249 | 0.073249 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.671548 | 0.020347 |
    | 02.01. 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |  0.4752 |  2.854761 | 1.960038 | 0.0 |  0.4752 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.960038 | 0.065934 | 0.065934 | 0.073878 | 0.063878 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.120776 | 0.017744 |
    | 02.01. 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 |  0.1188 |  7.210496 | 5.130246 | 0.0 |  0.1188 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 5.130246 | 0.040547 | 0.040547 | 0.062024 | 0.052024 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.068783 | 0.014451 |
    | 02.01. 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |  4.2768 |  3.914748 | 1.837796 | 0.0 |  4.2768 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.837796 | 0.052539 | 0.052539 | 0.048853 | 0.038853 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  10.455248 | 0.010793 |
    | 02.01. 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |  7.0092 | 10.233004 | 3.655358 | 0.0 |  7.0092 |    0.0 |  0.0 |  0.0 |  7.0092 |  7.0092 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 3.655358 | 0.069045 | 0.069045 | 0.053542 | 0.043542 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.740045 | 0.012095 |
    | 02.01. 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |  1.3068 |  4.693389 | 2.965278 | 0.0 |  1.3068 |    0.0 |  0.0 |  0.0 |  1.3068 |  1.3068 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 2.965278 | 0.060408 | 0.060408 | 0.063458 | 0.053458 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  12.021159 | 0.014849 |
    | 02.01. 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 24.5916 |  2.448478 |  0.15074 | 0.0 | 24.5916 |    0.0 |  0.0 |  0.0 | 24.5916 | 24.5916 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  0.15074 |  0.18231 |  0.18231 | 0.089417 | 0.079417 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  36.279708 |  0.02206 |
    | 02.01. 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 45.0252 |  2.597538 | 0.020724 | 0.0 | 45.0252 |    0.0 |  0.0 |  0.0 | 45.0252 | 45.0252 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.020724 | 0.406421 | 0.406421 | 0.205023 | 0.195023 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.877763 | 0.054173 |
    | 02.01. 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 |  9.7416 |  3.338304 | 0.907373 | 0.0 |  9.7416 |    0.0 |  0.0 |  0.0 |  9.7416 |  9.7416 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.907373 |  0.44856 |  0.44856 | 0.365983 | 0.355983 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   89.26343 | 0.098884 |
    | 02.01. 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 |  4.2768 |  1.846084 | 0.866652 | 0.0 |  4.2768 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.866652 | 0.463368 | 0.463368 | 0.442486 | 0.432486 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   92.21021 | 0.120135 |
    | 02.01. 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |    8.91 |  0.338261 | 0.099914 | 0.0 |    8.91 |    0.0 |  0.0 |  0.0 |    8.91 |    8.91 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.099914 | 0.505101 | 0.505101 | 0.469351 | 0.459351 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 100.515194 | 0.127598 |
    | 02.01. 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  21.978 |  0.107442 |  0.00859 | 0.0 |  21.978 |    0.0 |  0.0 |  0.0 |  21.978 |  21.978 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  0.00859 | 0.612423 | 0.612423 | 0.519677 | 0.509677 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  121.87218 | 0.141577 |
    | 02.01. 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 18.2952 |  0.107243 | 0.012392 | 0.0 | 18.2952 |    0.0 |  0.0 |  0.0 | 18.2952 | 18.2952 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.012392 | 0.700775 | 0.700775 | 0.608208 | 0.598208 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 139.454213 | 0.166169 |
    | 02.01. 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 |  7.4844 |  0.105848 | 0.036055 | 0.0 |  7.4844 |    0.0 |  0.0 |  0.0 |  7.4844 |  7.4844 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.036055 | 0.734513 | 0.734513 | 0.688638 | 0.678638 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 146.168045 | 0.188511 |
    | 02.01. 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  2.2572 |  0.104032 | 0.059768 | 0.0 |  2.2572 |    0.0 |  0.0 |  0.0 |  2.2572 |  2.2572 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.059768 | 0.741827 | 0.741827 | 0.728641 | 0.718641 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 147.623649 | 0.199622 |
    | 03.01. 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.104235 | 0.041931 | 0.0 |  5.8212 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.041931 | 0.767015 | 0.767015 | 0.745799 | 0.735799 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 152.635904 | 0.204389 |
    | 03.01. 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |  3.2076 |  0.104032 |  0.05435 | 0.0 |  3.2076 |    0.0 |  0.0 |  0.0 |  3.2076 |  3.2076 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  0.05435 | 0.778946 | 0.778946 | 0.764069 | 0.754069 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.010208 | 0.209464 |
    | 03.01. 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 |   0.594 |  0.103014 | 0.069893 | 0.0 |   0.594 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.069893 | 0.777672 | 0.777672 | 0.776011 | 0.766011 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.756644 | 0.212781 |
    | 03.01. 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.103422 | 0.072716 | 0.0 |  0.2376 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.072716 | 0.774608 | 0.774608 | 0.777274 | 0.767274 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.146921 | 0.213132 |
    | 03.01. 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 |   0.594 |   0.10464 | 0.070996 | 0.0 |   0.594 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.070996 |  0.77335 |  0.77335 | 0.775009 | 0.765009 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 153.896575 | 0.212502 |
    | 03.01. 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 |  2.8512 |  0.106248 | 0.057521 | 0.0 |  2.8512 |    0.0 |  0.0 |  0.0 |  2.8512 |  2.8512 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.057521 | 0.783451 | 0.783451 | 0.775874 | 0.765874 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.906803 | 0.212743 |
    | 03.01. 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  0.179639 | 0.123337 | 0.0 |  0.4752 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.123337 | 0.781293 | 0.781293 | 0.780727 | 0.770727 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.477372 | 0.214091 |
    | 03.01. 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.545628 | 0.383628 | 0.0 |  0.2376 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.383628 | 0.776657 | 0.776657 | 0.780743 | 0.770743 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.554688 | 0.214095 |
    | 03.01. 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  1.361522 | 0.980296 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.980296 | 0.767872 | 0.767872 | 0.775735 | 0.765735 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  152.80652 | 0.212704 |
    | 03.01. 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |     0.0 |  4.874663 | 3.509758 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 3.509758 | 0.746484 | 0.746484 | 0.765071 | 0.755071 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 148.550278 | 0.209742 |
    | 03.01. 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.057269 | 3.513746 | 0.0 |  0.3564 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 3.513746 | 0.726965 | 0.726965 | 0.746899 | 0.736899 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 144.665968 | 0.204694 |
    | 03.01. 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  3.0888 |  6.154673 | 3.253813 | 0.0 |  3.0888 |    0.0 |  0.0 |  0.0 |  3.0888 |  3.0888 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 3.253813 | 0.722505 | 0.722505 | 0.730311 | 0.720311 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  143.77845 | 0.200086 |
    | 03.01. 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |  0.8316 |  6.236191 | 4.131769 | 0.0 |  0.8316 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 4.131769 | 0.702391 | 0.702391 | 0.719026 | 0.709026 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  139.77589 | 0.196952 |
    | 03.01. 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 |  0.3564 |  5.404341 | 3.754888 | 0.0 |  0.3564 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 3.754888 | 0.681887 | 0.681887 | 0.702305 | 0.692305 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 135.695514 | 0.192307 |
    | 03.01. 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |  0.3564 |  9.035929 | 6.278083 | 0.0 |  0.3564 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 6.278083 | 0.648869 | 0.648869 | 0.679106 | 0.669106 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 129.124962 | 0.185863 |
    | 03.01. 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 |     0.0 |  5.627818 | 4.052029 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 4.052029 | 0.625365 | 0.625365 | 0.650983 | 0.640983 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 124.447569 | 0.178051 |
    | 03.01. 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |     0.0 |  6.156027 |  4.43234 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  4.43234 | 0.600076 | 0.600076 | 0.624968 | 0.614968 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.415153 | 0.170825 |
    | 03.01. 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.811802 | 2.744497 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 2.744497 | 0.583353 | 0.583353 |  0.60198 |  0.59198 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 116.087303 | 0.164439 |
    | 03.01. 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |     0.0 |  1.747211 | 1.257992 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.257992 | 0.574147 | 0.574147 | 0.585024 | 0.575024 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 114.255164 | 0.159729 |
    | 03.01. 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.318724 | 0.229481 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.229481 | 0.570128 | 0.570128 |   0.5753 |   0.5653 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 113.455555 | 0.157028 |
    | 03.01. 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.101575 | 0.073134 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.073134 | 0.566912 | 0.566912 | 0.570307 | 0.560307 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 112.815508 | 0.155641 |
    | 03.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.564078 | 0.564078 | 0.566997 | 0.556997 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 112.251431 | 0.154721 |
    | 03.01. 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |     0.0 |  0.097808 | 0.070422 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.561257 | 0.561257 | 0.564081 | 0.554081 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 111.690174 | 0.153911 |
    | 03.01. 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096529 | 0.069501 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.558451 | 0.558451 |  0.56126 |  0.55126 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 111.131723 | 0.153128 |
    | 04.01. 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.095455 | 0.068728 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.555659 | 0.555659 | 0.558454 | 0.548454 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.576064 | 0.152348 |
    | 04.01. 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 |  0.55288 |  0.55288 | 0.555662 | 0.545662 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.023184 | 0.151573 |
    | 04.01. 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.550116 | 0.550116 | 0.552883 | 0.542883 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.473068 | 0.150801 |
    | 04.01. 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.096957 | 0.059819 | 0.0 |  1.5444 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.555087 | 0.555087 | 0.551835 | 0.541835 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.462381 |  0.15051 |
    | 04.01. 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 |     0.0 |  0.096744 | 0.069656 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.552312 | 0.552312 | 0.553366 | 0.543366 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.910069 | 0.150935 |
    | 04.01. 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 |  0.54955 |  0.54955 | 0.552315 | 0.542315 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.360518 | 0.150643 |
    | 04.01. 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.359866 | 0.259103 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.546803 | 0.546803 | 0.549553 | 0.539553 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.813716 | 0.149876 |
    | 04.01. 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 |  0.8316 |  1.614671 | 1.069795 | 0.0 |  0.8316 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.548227 | 0.548227 |  0.54773 |  0.53773 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.097089 | 0.149369 |
    | 04.01. 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 |  0.4752 |  3.707216 | 2.545322 | 0.0 |  0.4752 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.547861 | 0.547861 | 0.547829 | 0.537829 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.024428 | 0.149397 |
    | 04.01. 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |  0.1188 |  6.022315 | 4.284859 | 0.0 |  0.1188 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.545716 | 0.545716 | 0.547466 | 0.537466 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.597512 | 0.149296 |
    | 04.01. 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  8.299842 | 5.698554 | 0.0 |  0.4752 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 5.698554 | 0.516871 | 0.516871 | 0.539783 | 0.529783 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 102.857287 | 0.147162 |
    | 04.01. 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |     0.0 |  9.734061 | 7.008524 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 7.008524 | 0.479244 | 0.479244 | 0.514919 | 0.504919 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  95.369519 | 0.140255 |
    | 04.01. 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 |     0.0 |  9.733892 | 7.008402 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 7.008402 | 0.441806 | 0.441806 | 0.479286 | 0.469286 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  87.919312 | 0.130357 |
    | 04.01. 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 |     0.0 | 12.429676 | 8.949367 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 8.949367 |  0.39485 |  0.39485 | 0.439691 | 0.429691 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.575095 | 0.119358 |
    | 04.01. 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 |     0.0 | 10.884737 | 7.837011 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 7.837011 |  0.35369 |  0.35369 | 0.396138 | 0.386138 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  70.384394 | 0.107261 |
    | 04.01. 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 |     0.0 |  9.119821 | 6.566271 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 6.566271 | 0.319091 | 0.319091 | 0.355148 | 0.345148 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  63.499032 | 0.095874 |
    | 04.01. 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 |     0.0 |  7.306062 | 5.260364 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 5.260364 | 0.291193 | 0.291193 |  0.32058 |  0.31058 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  57.947474 | 0.086272 |
    | 04.01. 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 |     0.0 |  3.420163 | 2.462517 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 2.462517 | 0.277425 | 0.277425 | 0.294333 | 0.284333 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  55.207533 | 0.078981 |
    | 04.01. 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |     0.0 |  1.518709 | 1.093471 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 1.093471 |  0.27057 |  0.27057 | 0.278961 | 0.268961 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  53.843492 | 0.074711 |
    | 04.01. 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |     0.0 |  0.382278 |  0.27524 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |  0.27524 | 0.267841 | 0.267841 | 0.271487 | 0.261487 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   53.30041 | 0.072635 |
    | 04.01. 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 |     0.0 |  0.103829 | 0.074757 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 | 0.074757 | 0.266128 | 0.266128 | 0.268067 | 0.258067 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.959525 | 0.071685 |
    | 04.01. 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.264798 | 0.264798 | 0.266213 | 0.256213 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.694727 |  0.07117 |
    | 04.01. 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.263474 | 0.263474 | 0.264799 | 0.254799 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.431254 | 0.070778 |
    | 04.01. 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |  0.0 | 0.0 |      0.0 | 0.262156 | 0.262156 | 0.263475 | 0.253475 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.169097 |  0.07041 |

    .. raw:: html

        <iframe
            src="hland_v1_ex4.html"
            width="100%"
            height="930px"
            frameborder=0
        ></iframe>

    .. _hland_v1_ex5:

    **Example 5**

    This example demonstrates the functionality of zones of type |GLACIER|.
    |GLACIER| zones are similar to zones of type |FIELD| or |FOREST|, but
    possess neither an interception storage nor a soil storage.  Instead,
    all precipitation or melting water is passed to the upper zone storage
    (|UZ|) directly. The snow routines of |GLACIER|, |FIELD|, and |FOREST|
    zones are identical.  Additional glacier melt can only occur if the
    glacier is not covered by any snow.  In the following test run,
    the simulation time period can be distinguished into three subperiods.
    On the first two days (-20°C), the snow layer builds up.  On the third
    day (+20°C and |SP| > 0), the snow melts and (with some time delay)
    the melted water is released. On the fourth day (+20°C and |SP| = 0),
    an increased amount of water is passed to |UZ|, as |GMelt| is set to
    a larger value than |CFMax|:

    >>> zonetype(GLACIER)
    >>> parameters.update()
    >>> inputs.t.series[:48] = -20.0
    >>> inputs.t.series[48:] = 20.0
    >>> inputs.tn.series = inputs.t.series
    >>> test('hland_v1_ex5')
    |         date |    p |     t |    tn |       epn | tmean |    tc | fracrain | rfc | sfc |      pc |        ep |      epc |  ei |      tf | glmelt |   melt | refr |      in_ |        r |  ea | cfpot |  cf | perc | contriarea |     inuz |        q0 |  el |       q1 |      inuh |     outuh |        qt |  ic |       sp |       wc |  sm |        uz |        lz |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01.01. 00:00 |  0.0 | -20.0 | -20.0 |  0.100707 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  0.106749 |  0.07686 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.061111 |  0.051111 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |      9.95 | 0.014198 |
    | 01.01. 01:00 |  0.0 | -20.0 | -20.0 |  0.097801 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  0.103669 | 0.074642 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.088833 |  0.078833 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |   9.90025 | 0.021898 |
    | 01.01. 02:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |   0.04975 |   0.03975 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.850749 | 0.011042 |
    | 01.01. 03:00 |  0.0 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  0.101749 |  0.07326 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049502 |  0.039502 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.801495 | 0.010973 |
    | 01.01. 04:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049254 |  0.039254 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.752488 | 0.010904 |
    | 01.01. 05:00 |  0.0 | -20.0 | -20.0 |  0.102761 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  0.108927 | 0.078427 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049008 |  0.039008 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.703725 | 0.010835 |
    | 01.01. 06:00 |  0.0 | -20.0 | -20.0 |  0.291908 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  0.309422 | 0.222784 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048763 |  0.038763 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.655206 | 0.010767 |
    | 01.01. 07:00 |  0.0 | -20.0 | -20.0 |  1.932875 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  2.048847 |  1.47517 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048519 |  0.038519 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |   9.60693 |   0.0107 |
    | 01.01. 08:00 |  0.0 | -20.0 | -20.0 |  4.369536 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  4.631708 |  3.33483 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048276 |  0.038276 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.558896 | 0.010632 |
    | 01.01. 09:00 |  0.0 | -20.0 | -20.0 |  7.317556 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  7.756609 | 5.584759 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048035 |  0.038035 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.511101 | 0.010565 |
    | 01.01. 10:00 |  0.0 | -20.0 | -20.0 |  8.264362 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  8.760224 | 6.307361 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047795 |  0.037795 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.463546 | 0.010499 |
    | 01.01. 11:00 |  0.0 | -20.0 | -20.0 |  9.369867 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  9.932059 | 7.151082 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047556 |  0.037556 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.416228 | 0.010432 |
    | 01.01. 12:00 |  0.0 | -20.0 | -20.0 |  5.126178 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  5.433749 | 3.912299 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047318 |  0.037318 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.369147 | 0.010366 |
    | 01.01. 13:00 |  0.0 | -20.0 | -20.0 |   6.62503 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  7.022532 | 5.056223 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047081 |  0.037081 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |  9.322301 |   0.0103 |
    | 01.01. 14:00 |  0.0 | -20.0 | -20.0 |  7.397619 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  7.841476 | 5.645863 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046846 |  0.036846 | 0.0 |      0.0 |      0.0 | 0.0 |       0.0 |   9.27569 | 0.010235 |
    | 01.01. 15:00 |  0.2 | -20.0 | -20.0 |   2.39151 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  0.2808 |  2.535001 | 1.774662 | 0.0 |  0.2808 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046612 |  0.036612 | 0.0 |   0.2808 |      0.0 | 0.0 |       0.0 |  9.229311 |  0.01017 |
    | 01.01. 16:00 |  0.0 | -20.0 | -20.0 |  1.829834 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  1.939624 | 1.396529 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046379 |  0.036379 | 0.0 |   0.2808 |      0.0 | 0.0 |       0.0 |  9.183165 | 0.010105 |
    | 01.01. 17:00 |  0.0 | -20.0 | -20.0 |  1.136569 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  1.204763 | 0.867429 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046147 |  0.036147 | 0.0 |   0.2808 |      0.0 | 0.0 |       0.0 |  9.137249 | 0.010041 |
    | 01.01. 18:00 |  1.3 | -20.0 | -20.0 |  0.750986 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  1.8252 |  0.796045 | 0.477532 | 0.0 |  1.8252 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045916 |  0.035916 | 0.0 |    2.106 |      0.0 | 0.0 |       0.0 |  9.091563 | 0.009977 |
    | 01.01. 19:00 |  5.6 | -20.0 | -20.0 |  0.223895 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  7.8624 |  0.237329 | 0.077844 | 0.0 |  7.8624 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.045458 |  0.045458 |  0.045686 |  0.035686 | 0.0 |   9.9684 |      0.0 | 0.0 |       0.0 |  9.046105 | 0.009913 |
    | 01.01. 20:00 |  2.9 | -20.0 | -20.0 |  0.099425 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  4.0716 |   0.10539 | 0.050502 | 0.0 |  4.0716 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.045231 |  0.045231 |  0.045458 |  0.035458 | 0.0 |    14.04 |      0.0 | 0.0 |       0.0 |  9.000874 | 0.009849 |
    | 01.01. 21:00 |  4.9 | -20.0 | -20.0 |  0.098454 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  6.8796 |  0.104361 | 0.037765 | 0.0 |  6.8796 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.045004 |  0.045004 |  0.045231 |  0.035231 | 0.0 |  20.9196 |      0.0 | 0.0 |       0.0 |   8.95587 | 0.009786 |
    | 01.01. 22:00 | 10.6 | -20.0 | -20.0 |  0.098128 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 14.8824 |  0.104016 | 0.016908 | 0.0 | 14.8824 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.044779 |  0.044779 |  0.045005 |  0.035005 | 0.0 |   35.802 |      0.0 | 0.0 |       0.0 |  8.911091 | 0.009724 |
    | 01.01. 23:00 |  0.1 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  0.1404 |  0.103322 | 0.073355 | 0.0 |  0.1404 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.044555 |  0.044555 |   0.04478 |   0.03478 | 0.0 |  35.9424 |      0.0 | 0.0 |       0.0 |  8.866535 | 0.009661 |
    | 02.01. 00:00 |  0.7 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  0.9828 |  0.103322 | 0.067429 | 0.0 |  0.9828 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.044333 |  0.044333 |  0.044556 |  0.034556 | 0.0 |  36.9252 |      0.0 | 0.0 |       0.0 |  8.822202 | 0.009599 |
    | 02.01. 01:00 |  3.0 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |   4.212 |  0.103322 | 0.048821 | 0.0 |   4.212 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.044111 |  0.044111 |  0.044333 |  0.034333 | 0.0 |  41.1372 |      0.0 | 0.0 |       0.0 |  8.778091 | 0.009537 |
    | 02.01. 02:00 |  2.1 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  2.9484 |    0.1028 | 0.055116 | 0.0 |  2.9484 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 |  0.04389 |   0.04389 |  0.044111 |  0.034111 | 0.0 |  44.0856 |      0.0 | 0.0 |       0.0 |  8.734201 | 0.009475 |
    | 02.01. 03:00 | 10.4 | -20.0 | -20.0 |  0.096652 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 14.6016 |  0.102451 | 0.017128 | 0.0 | 14.6016 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.043671 |  0.043671 |  0.043891 |  0.033891 | 0.0 |  58.6872 |      0.0 | 0.0 |       0.0 |   8.69053 | 0.009414 |
    | 02.01. 04:00 |  3.5 | -20.0 | -20.0 |  0.096321 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |   4.914 |    0.1021 | 0.044973 | 0.0 |   4.914 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.043453 |  0.043453 |  0.043671 |  0.033671 | 0.0 |  63.6012 |      0.0 | 0.0 |       0.0 |  8.647077 | 0.009353 |
    | 02.01. 05:00 |  3.4 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  4.7736 |  0.101749 | 0.045452 | 0.0 |  4.7736 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.043235 |  0.043235 |  0.043453 |  0.033453 | 0.0 |  68.3748 |      0.0 | 0.0 |       0.0 |  8.603842 | 0.009292 |
    | 02.01. 06:00 |  1.2 | -20.0 | -20.0 |  0.187298 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  1.6848 |  0.198536 | 0.120782 | 0.0 |  1.6848 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.043019 |  0.043019 |  0.043236 |  0.033236 | 0.0 |  70.0596 |      0.0 | 0.0 |       0.0 |  8.560823 | 0.009232 |
    | 02.01. 07:00 |  0.1 | -20.0 | -20.0 |  1.264612 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  0.1404 |  1.340489 | 0.951696 | 0.0 |  0.1404 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.042804 |  0.042804 |  0.043019 |  0.033019 | 0.0 |     70.2 |      0.0 | 0.0 |       0.0 |  8.518019 | 0.009172 |
    | 02.01. 08:00 |  0.0 | -20.0 | -20.0 |  3.045538 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |   3.22827 | 2.324355 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 |  0.04259 |   0.04259 |  0.042804 |  0.032804 | 0.0 |     70.2 |      0.0 | 0.0 |       0.0 |  8.475429 | 0.009112 |
    | 02.01. 09:00 |  0.0 | -20.0 | -20.0 |  1.930758 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |     0.0 |  2.046603 | 1.473555 | 0.0 |     0.0 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.042377 |  0.042377 |   0.04259 |   0.03259 | 0.0 |     70.2 |      0.0 | 0.0 |       0.0 |  8.433051 | 0.009053 |
    | 02.01. 10:00 |  0.4 | -20.0 | -20.0 |  2.461001 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  0.5616 |  2.608661 | 1.775661 | 0.0 |  0.5616 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.042165 |  0.042165 |  0.042377 |  0.032377 | 0.0 |  70.7616 |      0.0 | 0.0 |       0.0 |  8.390886 | 0.008994 |
    | 02.01. 11:00 |  0.1 | -20.0 | -20.0 |  6.215945 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  0.1404 |  6.588902 | 4.677869 | 0.0 |  0.1404 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.041954 |  0.041954 |  0.042165 |  0.032165 | 0.0 |   70.902 |      0.0 | 0.0 |       0.0 |  8.348932 | 0.008935 |
    | 02.01. 12:00 |  3.6 | -20.0 | -20.0 |  3.374783 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  5.0544 |   3.57727 | 1.553726 | 0.0 |  5.0544 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.041745 |  0.041745 |  0.041955 |  0.031955 | 0.0 |  75.9564 |      0.0 | 0.0 |       0.0 |  8.307187 | 0.008876 |
    | 02.01. 13:00 |  5.9 | -20.0 | -20.0 |  8.821555 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  8.2836 |  9.350848 | 2.940569 | 0.0 |  8.2836 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.041536 |  0.041536 |  0.041745 |  0.031745 | 0.0 |    84.24 |      0.0 | 0.0 |       0.0 |  8.265651 | 0.008818 |
    | 02.01. 14:00 |  1.1 | -20.0 | -20.0 |  4.046025 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  1.5444 |  4.288787 | 2.646028 | 0.0 |  1.5444 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.041328 |  0.041328 |  0.041536 |  0.031536 | 0.0 |  85.7844 |      0.0 | 0.0 |       0.0 |  8.224323 |  0.00876 |
    | 02.01. 15:00 | 20.7 | -20.0 | -20.0 |  2.110757 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 29.0628 |  2.237402 | 0.088084 | 0.0 | 29.0628 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.041122 |  0.041122 |  0.041328 |  0.031328 | 0.0 | 114.8472 |      0.0 | 0.0 |       0.0 |  8.183201 | 0.008702 |
    | 02.01. 16:00 | 37.9 | -20.0 | -20.0 |  2.239257 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 53.2116 |  2.373612 | 0.008352 | 0.0 | 53.2116 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.040916 |  0.040916 |  0.041122 |  0.031122 | 0.0 | 168.0588 |      0.0 | 0.0 |       0.0 |  8.142285 | 0.008645 |
    | 02.01. 17:00 |  8.2 | -20.0 | -20.0 |  2.877848 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 11.5128 |  3.050519 | 0.694563 | 0.0 | 11.5128 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.040711 |  0.040711 |  0.040916 |  0.030916 | 0.0 | 179.5716 |      0.0 | 0.0 |       0.0 |  8.101574 | 0.008588 |
    | 02.01. 18:00 |  3.6 | -20.0 | -20.0 |  1.591452 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  5.0544 |  1.686939 | 0.732693 | 0.0 |  5.0544 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.040508 |  0.040508 |  0.040712 |  0.030712 | 0.0 |  184.626 |      0.0 | 0.0 |       0.0 |  8.061066 | 0.008531 |
    | 02.01. 19:00 |  7.5 | -20.0 | -20.0 |  0.291604 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |   10.53 |    0.3091 | 0.077646 | 0.0 |   10.53 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.040305 |  0.040305 |  0.040508 |  0.030508 | 0.0 |  195.156 |      0.0 | 0.0 |       0.0 |  8.020761 | 0.008474 |
    | 02.01. 20:00 | 18.5 | -20.0 | -20.0 |  0.092622 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  25.974 |  0.098179 | 0.005264 | 0.0 |  25.974 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.040104 |  0.040104 |  0.040306 |  0.030306 | 0.0 |   221.13 |      0.0 | 0.0 |       0.0 |  7.980657 | 0.008418 |
    | 02.01. 21:00 | 15.4 | -20.0 | -20.0 |  0.092451 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 21.6216 |  0.097998 |  0.00812 | 0.0 | 21.6216 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.039903 |  0.039903 |  0.040104 |  0.030104 | 0.0 | 242.7516 |      0.0 | 0.0 |       0.0 |  7.940753 | 0.008362 |
    | 02.01. 22:00 |  6.3 | -20.0 | -20.0 |  0.091248 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  8.8452 |  0.096723 | 0.028755 | 0.0 |  8.8452 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.039704 |  0.039704 |  0.039904 |  0.029904 | 0.0 | 251.5968 |      0.0 | 0.0 |       0.0 |   7.90105 | 0.008307 |
    | 02.01. 23:00 |  1.9 | -20.0 | -20.0 |  0.089683 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 |  2.6676 |  0.095064 |  0.05242 | 0.0 |  2.6676 |    0.0 |    0.0 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.039505 |  0.039505 |  0.039704 |  0.029704 | 0.0 | 254.2644 |      0.0 | 0.0 |       0.0 |  7.861544 | 0.008251 |
    | 03.01. 00:00 |  4.9 |  20.0 |  20.0 |  0.089858 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  5.8212 |  0.095249 | 0.038316 | 0.0 |  5.8212 |    0.0 |    9.8 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.039308 |  0.039308 |  0.039505 |  0.029505 | 0.0 | 244.4644 |  15.6212 | 0.0 |       0.0 |  7.822237 | 0.008196 |
    | 03.01. 01:00 |  2.7 |  20.0 |  20.0 |  0.089683 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  3.2076 |  0.095064 | 0.049664 | 0.0 |  3.2076 |    0.0 |    9.8 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.039111 |  0.039111 |  0.039308 |  0.029308 | 0.0 | 234.6644 |  28.6288 | 0.0 |       0.0 |  7.783126 | 0.008141 |
    | 03.01. 02:00 |  0.5 |  20.0 |  20.0 |  0.088805 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |   0.594 |  0.094133 | 0.063867 | 0.0 |   0.594 |    0.0 |    9.8 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.038916 |  0.038916 |  0.039111 |  0.029111 | 0.0 | 224.8644 |  39.0228 | 0.0 |       0.0 |   7.74421 | 0.008087 |
    | 03.01. 03:00 |  0.2 |  20.0 |  20.0 |  0.089157 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.094506 | 0.066447 | 0.0 |  0.2376 |    0.0 |    9.8 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.038721 |  0.038721 |  0.038916 |  0.028916 | 0.0 | 215.0644 |  49.0604 | 0.0 |       0.0 |  7.705489 | 0.008032 |
    | 03.01. 04:00 |  0.5 |  20.0 |  20.0 |  0.090207 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |   0.594 |  0.095619 | 0.064876 | 0.0 |   0.594 |    0.0 |    9.8 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.038527 |  0.038527 |  0.038721 |  0.028721 | 0.0 | 205.2644 |  59.4544 | 0.0 |       0.0 |  7.666961 | 0.007978 |
    | 03.01. 05:00 |  2.4 |  20.0 |  20.0 |  0.091593 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  2.8512 |  0.097089 | 0.052562 | 0.0 |  2.8512 |    0.0 |    9.8 |  0.0 |      0.0 |      0.0 | 0.0 |   nan | 0.0 |  0.0 |        1.0 |      0.0 |       0.0 | 0.0 | 0.038335 |  0.038335 |  0.038528 |  0.028528 | 0.0 | 195.4644 |  72.1056 | 0.0 |       0.0 |  7.628627 | 0.007924 |
    | 03.01. 06:00 |  0.4 |  20.0 |  20.0 |  0.154861 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  0.164153 | 0.112705 | 0.0 |  0.4752 |    0.0 |    9.8 |  0.0 |  8.11504 |  8.11504 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  8.11504 |  0.109914 | 0.0 | 0.040643 |  0.150557 |  0.063316 |  0.053316 | 0.0 | 185.6644 | 74.26576 | 0.0 |  7.505126 |  8.087984 |  0.01481 |
    | 03.01. 07:00 |  0.2 |  20.0 |  20.0 |  0.470369 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.2376 |  0.498591 | 0.350557 | 0.0 |  0.2376 |    0.0 |    9.8 |  0.0 |  13.9576 |  13.9576 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  13.9576 |  2.794687 | 0.0 |  0.04294 |  2.837627 |  0.722745 |  0.712745 | 0.0 | 175.8644 | 70.34576 | 0.0 | 18.168039 |  8.545044 | 0.197985 |
    | 03.01. 08:00 |  0.0 |  20.0 |  20.0 |  1.173726 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |   1.24415 | 0.895788 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 |  9.073105 | 0.0 | 0.045225 |  9.118331 |  3.636212 |  3.626212 | 0.0 | 166.0644 | 66.42576 | 0.0 | 22.314934 |  8.999818 | 1.007281 |
    | 03.01. 09:00 |  0.0 |  20.0 |  20.0 |  4.202296 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  4.454434 | 3.207192 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 12.252207 | 0.0 | 0.047499 | 12.299706 |  8.429591 |  8.419591 | 0.0 | 156.2644 | 62.50576 | 0.0 | 23.282727 |  9.452319 | 2.338775 |
    | 03.01. 10:00 |  0.3 |  20.0 |  20.0 |  4.359715 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.3564 |  4.621298 | 3.210837 | 0.0 |  0.3564 |    0.0 |    9.8 |  0.0 |  14.0764 |  14.0764 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  14.0764 | 13.218409 | 0.0 | 0.049762 | 13.268171 | 11.807948 | 11.797948 | 0.0 | 146.4644 | 58.58576 | 0.0 | 23.640718 |  9.902558 | 3.277208 |
    | 03.01. 11:00 |  2.6 |  20.0 |  20.0 |  5.305753 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  3.0888 |  5.624098 | 2.973312 | 0.0 |  3.0888 |    0.0 |    9.8 |  0.0 |  16.8088 |  16.8088 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  16.8088 | 14.987247 | 0.0 | 0.052013 | 15.039259 | 13.446532 | 13.436532 | 0.0 | 136.6644 | 54.66576 | 0.0 | 24.962271 | 10.350545 |  3.73237 |
    | 03.01. 12:00 |  0.7 |  20.0 |  20.0 |  5.376027 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.8316 |  5.698589 | 3.775582 | 0.0 |  0.8316 |    0.0 |    9.8 |  0.0 |  14.5516 |  14.5516 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  14.5516 | 14.862113 | 0.0 | 0.054253 | 14.916366 | 14.618374 | 14.608374 | 0.0 | 126.8644 | 50.74576 | 0.0 | 24.151758 | 10.796292 | 4.057882 |
    | 03.01. 13:00 |  0.3 |  20.0 |  20.0 |  4.658915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.3564 |   4.93845 | 3.431191 | 0.0 |  0.3564 |    0.0 |    9.8 |  0.0 |  14.0764 |  14.0764 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  14.0764 | 13.932087 | 0.0 | 0.056481 | 13.988568 | 14.737498 | 14.727498 | 0.0 | 117.0644 | 46.82576 | 0.0 | 23.796072 | 11.239811 | 4.090972 |
    | 03.01. 14:00 |  0.3 |  20.0 |  20.0 |  7.789594 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.3564 |   8.25697 | 5.736869 | 0.0 |  0.3564 |    0.0 |    9.8 |  0.0 |  14.0764 |  14.0764 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  14.0764 | 13.638844 | 0.0 | 0.058699 | 13.697543 | 14.130073 | 14.120073 | 0.0 | 107.2644 | 42.90576 | 0.0 | 23.733628 | 11.681112 | 3.922243 |
    | 03.01. 15:00 |  0.0 |  20.0 |  20.0 |  4.851567 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  5.142661 | 3.702716 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.400255 | 0.0 | 0.060906 | 13.461161 | 13.709686 | 13.699686 | 0.0 |  97.4644 | 38.98576 | 0.0 | 23.553372 | 12.120206 | 3.805468 |
    | 03.01. 16:00 |  0.0 |  20.0 |  20.0 |   5.30692 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  5.625335 | 4.050241 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.252908 | 0.0 | 0.063101 | 13.316009 | 13.481434 | 13.471434 | 0.0 |  87.6644 | 35.06576 | 0.0 | 23.520464 | 12.557105 | 3.742065 |
    | 03.01. 17:00 |  0.0 |  20.0 |  20.0 |  3.286036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  3.483198 | 2.507903 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.226053 | 0.0 | 0.065286 | 13.291338 | 13.342783 | 13.332783 | 0.0 |  77.8644 | 31.14576 | 0.0 | 23.514412 |  12.99182 | 3.703551 |
    | 03.01. 18:00 |  0.0 |  20.0 |  20.0 |  1.506216 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  1.596589 | 1.149544 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.221115 | 0.0 | 0.067459 | 13.288574 | 13.296206 | 13.286206 | 0.0 |  68.0644 | 27.22576 | 0.0 | 23.513297 |  13.42436 | 3.690613 |
    | 03.01. 19:00 |  0.0 |  20.0 |  20.0 |  0.274762 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.291248 | 0.209698 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.220205 | 0.0 | 0.069622 | 13.289827 | 13.289467 | 13.279467 | 0.0 |  58.2644 | 23.30576 | 0.0 | 23.513091 | 13.854739 | 3.688741 |
    | 03.01. 20:00 |  0.0 |  20.0 |  20.0 |  0.087565 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.092819 |  0.06683 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.220038 | 0.0 | 0.071774 | 13.291812 |  13.28999 |  13.27999 | 0.0 |  48.4644 | 19.38576 | 0.0 | 23.513054 | 14.282965 | 3.688886 |
    | 03.01. 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.220007 | 0.0 | 0.073915 | 13.293922 |  13.29184 |  13.28184 | 0.0 |  38.6644 | 15.46576 | 0.0 | 23.513047 |  14.70905 |   3.6894 |
    | 03.01. 22:00 |  0.0 |  20.0 |  20.0 |  0.084317 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.089376 | 0.064351 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 | 13.220001 | 0.0 | 0.076045 | 13.296047 | 13.293925 | 13.283925 | 0.0 |  28.8644 | 11.54576 | 0.0 | 23.513045 | 15.133005 | 3.689979 |
    | 03.01. 23:00 |  0.0 |  20.0 |  20.0 |  0.083215 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.088208 |  0.06351 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 |     13.22 | 0.0 | 0.078165 | 13.298165 | 13.296045 | 13.286045 | 0.0 |  19.0644 |  7.62576 | 0.0 | 23.513045 |  15.55484 | 3.690568 |
    | 04.01. 00:00 |  0.0 |  20.0 |  20.0 |  0.082289 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.087226 | 0.062803 | 0.0 |     0.0 |    0.0 |    9.8 |  0.0 |    13.72 |    13.72 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |    13.72 |     13.22 | 0.0 | 0.080274 | 13.300274 | 13.298163 | 13.288163 | 0.0 |   9.2644 |  3.70576 | 0.0 | 23.513045 | 15.974566 | 3.691156 |
    | 04.01. 01:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 |   19.6 | 9.2644 |  0.0 | 32.57016 | 32.57016 | 0.0 |   nan | 0.0 |  0.5 |        1.0 | 32.57016 | 24.610649 | 0.0 | 0.082373 | 24.693022 | 15.831527 | 15.821527 | 0.0 |      0.0 |      0.0 | 0.0 | 30.972556 | 16.392193 | 4.394869 |
    | 04.01. 02:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 23.093168 | 0.0 | 0.084461 | 23.177629 | 21.824546 | 21.814546 | 0.0 |      0.0 |      0.0 | 0.0 | 26.979388 | 16.807732 | 6.059596 |
    | 04.01. 03:00 |  1.3 |  20.0 |  20.0 |  0.083584 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  1.5444 |  0.088599 | 0.054662 | 0.0 |  1.5444 |   19.6 |    0.0 |  0.0 |  21.1444 |  21.1444 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  21.1444 | 20.422967 | 0.0 | 0.086539 | 20.509506 | 22.921467 | 22.911467 | 0.0 |      0.0 |      0.0 | 0.0 | 27.200821 | 17.221193 | 6.364296 |
    | 04.01. 04:00 |  0.0 |  20.0 |  20.0 |    0.0834 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.088404 | 0.063651 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.687024 | 0.0 | 0.088606 |  19.77563 | 20.939339 | 20.929339 | 0.0 |      0.0 |      0.0 | 0.0 | 26.613797 | 17.632587 | 5.813705 |
    | 04.01. 05:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.165589 | 0.0 | 0.090663 | 19.256252 | 19.823296 | 19.813296 | 0.0 |      0.0 |      0.0 | 0.0 | 26.548208 | 18.041924 | 5.503693 |
    | 04.01. 06:00 |  0.0 |  20.0 |  20.0 |  0.310229 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.328843 | 0.236767 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.107491 | 0.0 |  0.09271 | 19.200201 | 19.359213 | 19.349213 | 0.0 |      0.0 |      0.0 | 0.0 | 26.540717 | 18.449215 | 5.374781 |
    | 04.01. 07:00 |  0.7 |  20.0 |  20.0 |  1.391958 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.8316 |  1.475475 | 0.977572 | 0.0 |  0.8316 |   19.6 |    0.0 |  0.0 |  20.4316 |  20.4316 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  20.4316 | 19.599069 | 0.0 | 0.094746 | 19.693815 | 19.322348 | 19.312348 | 0.0 |      0.0 |      0.0 | 0.0 | 26.873248 | 18.854469 | 5.364541 |
    | 04.01. 08:00 |  0.4 |  20.0 |  20.0 |  3.195876 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  3.387629 | 2.325898 | 0.0 |  0.4752 |   19.6 |    0.0 |  0.0 |  20.0752 |  20.0752 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  20.0752 | 19.680557 | 0.0 | 0.096772 |  19.77733 | 19.602682 | 19.592682 | 0.0 |      0.0 |      0.0 | 0.0 | 26.767891 | 19.257696 | 5.442412 |
    | 04.01. 09:00 |  0.1 |  20.0 |  20.0 |  5.191651 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.1188 |   5.50315 | 3.915475 | 0.0 |  0.1188 |   19.6 |    0.0 |  0.0 |  19.7188 |  19.7188 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  19.7188 | 19.373189 | 0.0 | 0.098788 | 19.471977 | 19.690915 | 19.680915 | 0.0 |      0.0 |      0.0 | 0.0 | 26.613502 | 19.658908 | 5.466921 |
    | 04.01. 10:00 |  0.4 |  20.0 |  20.0 |  7.155036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |  0.4752 |  7.584338 | 5.207299 | 0.0 |  0.4752 |   19.6 |    0.0 |  0.0 |  20.0752 |  20.0752 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |  20.0752 | 19.449526 | 0.0 | 0.100795 |  19.55032 | 19.557243 | 19.547243 | 0.0 |      0.0 |      0.0 | 0.0 | 26.739177 | 20.058113 |  5.42979 |
    | 04.01. 11:00 |  0.0 |  20.0 |  20.0 |  8.391432 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  8.894918 | 6.404341 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.276741 | 0.0 | 0.102791 | 19.379531 | 19.494958 | 19.484958 | 0.0 |      0.0 |      0.0 | 0.0 | 26.562436 | 20.455323 | 5.412488 |
    | 04.01. 12:00 |  0.0 |  20.0 |  20.0 |  8.391286 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  8.894763 | 6.404229 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.120091 | 0.0 | 0.104777 | 19.224868 | 19.383115 | 19.373115 | 0.0 |      0.0 |      0.0 | 0.0 | 26.542345 | 20.850546 | 5.381421 |
    | 04.01. 13:00 |  0.0 |  20.0 |  20.0 | 10.715238 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 | 11.358152 |  8.17787 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.102299 | 0.0 | 0.106753 | 19.209052 | 19.255723 | 19.245723 | 0.0 |      0.0 |      0.0 | 0.0 | 26.540046 | 21.243793 | 5.346034 |
    | 04.01. 14:00 |  0.0 |  20.0 |  20.0 |  9.383394 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  9.946398 | 7.161406 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.100263 | 0.0 | 0.108719 | 19.208982 | 19.212551 | 19.202551 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539783 | 21.635074 | 5.334042 |
    | 04.01. 15:00 |  0.0 |  20.0 |  20.0 |  7.861915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |   8.33363 | 6.000214 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |  19.10003 | 0.0 | 0.110675 | 19.210706 | 19.209381 | 19.199381 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539753 | 22.024399 | 5.333161 |
    | 04.01. 16:00 |  0.0 |  20.0 |  20.0 |  6.298329 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  6.676229 | 4.806885 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 | 19.100003 | 0.0 | 0.112622 | 19.212625 | 19.210749 | 19.200749 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 22.411777 | 5.333541 |
    | 04.01. 17:00 |  0.0 |  20.0 |  20.0 |  2.948416 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  3.125321 | 2.250231 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |      19.1 | 0.0 | 0.114559 | 19.214559 | 19.212629 | 19.202629 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 22.797218 | 5.334063 |
    | 04.01. 18:00 |  0.0 |  20.0 |  20.0 |  1.309232 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  1.387786 | 0.999206 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |      19.1 | 0.0 | 0.116486 | 19.216486 | 19.214558 | 19.204558 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 23.180732 | 5.334599 |
    | 04.01. 19:00 |  0.0 |  20.0 |  20.0 |   0.32955 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.349323 | 0.251513 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |      19.1 | 0.0 | 0.118404 | 19.218404 | 19.216484 | 19.206484 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 23.562328 | 5.335134 |
    | 04.01. 20:00 |  0.0 |  20.0 |  20.0 |  0.089508 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.094878 | 0.068313 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |      19.1 | 0.0 | 0.120312 | 19.220312 | 19.218402 | 19.208402 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 23.942017 | 5.335667 |
    | 04.01. 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |      19.1 | 0.0 |  0.12221 |  19.22221 |  19.22031 |  19.21031 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 24.319807 | 5.336197 |
    | 04.01. 22:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |      19.1 | 0.0 | 0.124099 | 19.224099 | 19.222208 | 19.212208 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 24.695708 | 5.336724 |
    | 04.01. 23:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 |   19.6 |    0.0 |  0.0 |     19.6 |     19.6 | 0.0 |   nan | 0.0 |  0.5 |        1.0 |     19.6 |      19.1 | 0.0 | 0.125979 | 19.225979 | 19.224097 | 19.214097 | 0.0 |      0.0 |      0.0 | 0.0 | 26.539749 | 25.069729 | 5.337249 |

    .. raw:: html

        <iframe
            src="hland_v1_ex5.html"
            width="100%"
            height="930px"
            frameborder=0
        ></iframe>
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core import parametertools
from hydpy.core import sequencetools
# ...from hland
from hydpy.models.hland import hland_model
from hydpy.models.hland import hland_control
from hydpy.models.hland import hland_derived
from hydpy.models.hland import hland_inputs
from hydpy.models.hland import hland_fluxes
from hydpy.models.hland import hland_states
from hydpy.models.hland import hland_logs
from hydpy.models.hland import hland_outlets
from hydpy.models.hland import hland_masks
from hydpy.models.hland.hland_constants import *


class Model(modeltools.AdHocModel):
    """HBV96 version of HydPy-H-Land (|hland_v1|)."""
    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (hland_model.calc_tc_v1,
                   hland_model.calc_tmean_v1,
                   hland_model.calc_fracrain_v1,
                   hland_model.calc_rfc_sfc_v1,
                   hland_model.calc_pc_v1,
                   hland_model.calc_ep_v1,
                   hland_model.calc_epc_v1,
                   hland_model.calc_tf_ic_v1,
                   hland_model.calc_ei_ic_v1,
                   hland_model.calc_sp_wc_v1,
                   hland_model.calc_melt_sp_wc_v1,
                   hland_model.calc_refr_sp_wc_v1,
                   hland_model.calc_in_wc_v1,
                   hland_model.calc_glmelt_in_v1,
                   hland_model.calc_r_sm_v1,
                   hland_model.calc_cf_sm_v1,
                   hland_model.calc_ea_sm_v1,
                   hland_model.calc_inuz_v1,
                   hland_model.calc_contriarea_v1,
                   hland_model.calc_q0_perc_uz_v1,
                   hland_model.calc_lz_v1,
                   hland_model.calc_el_lz_v1,
                   hland_model.calc_q1_lz_v1,
                   hland_model.calc_inuh_v1,
                   hland_model.calc_outuh_quh_v1,
                   hland_model.calc_qt_v1)
    ADD_METHODS = ()
    OUTLET_METHODS = (hland_model.update_q_v1,)
    SENDER_METHODS = ()

class ControlParameters(parametertools.SubParameters):
    """Control parameters of |hland_v1|, directly defined by the user."""
    CLASSES = (hland_control.Area,
               hland_control.NmbZones,
               hland_control.ZoneType,
               hland_control.ZoneArea,
               hland_control.ZoneZ,
               hland_control.ZRelP,
               hland_control.ZRelT,
               hland_control.ZRelE,
               hland_control.PCorr,
               hland_control.PCAlt,
               hland_control.RfCF,
               hland_control.SfCF,
               hland_control.TCAlt,
               hland_control.ECorr,
               hland_control.ECAlt,
               hland_control.EPF,
               hland_control.ETF,
               hland_control.ERed,
               hland_control.TTIce,
               hland_control.IcMax,
               hland_control.TT,
               hland_control.TTInt,
               hland_control.DTTM,
               hland_control.CFMax,
               hland_control.GMelt,
               hland_control.CFR,
               hland_control.WHC,
               hland_control.FC,
               hland_control.LP,
               hland_control.Beta,
               hland_control.PercMax,
               hland_control.CFlux,
               hland_control.RespArea,
               hland_control.RecStep,
               hland_control.Alpha,
               hland_control.K,
               hland_control.K4,
               hland_control.Gamma,
               hland_control.MaxBaz,
               hland_control.Abstr)


class DerivedParameters(parametertools.SubParameters):
    """Derived parameters of |hland_v1|, indirectly defined by the user."""
    CLASSES = (hland_derived.RelZoneArea,
               hland_derived.RelSoilArea,
               hland_derived.RelSoilZoneArea,
               hland_derived.RelLandZoneArea,
               hland_derived.RelLandArea,
               hland_derived.TTM,
               hland_derived.DT,
               hland_derived.UH,
               hland_derived.QFactor)


class InputSequences(sequencetools.InputSequences):
    """Input sequences of |hland_v1|."""
    CLASSES = (hland_inputs.P,
               hland_inputs.T,
               hland_inputs.TN,
               hland_inputs.EPN)


class FluxSequences(sequencetools.FluxSequences):
    """Flux sequences of |hland_v1|."""
    CLASSES = (hland_fluxes.TMean,
               hland_fluxes.TC,
               hland_fluxes.FracRain,
               hland_fluxes.RfC,
               hland_fluxes.SfC,
               hland_fluxes.PC,
               hland_fluxes.EP,
               hland_fluxes.EPC,
               hland_fluxes.EI,
               hland_fluxes.TF,
               hland_fluxes.GlMelt,
               hland_fluxes.Melt,
               hland_fluxes.Refr,
               hland_fluxes.In_,
               hland_fluxes.R,
               hland_fluxes.EA,
               hland_fluxes.CFPot,
               hland_fluxes.CF,
               hland_fluxes.Perc,
               hland_fluxes.ContriArea,
               hland_fluxes.InUZ,
               hland_fluxes.Q0,
               hland_fluxes.EL,
               hland_fluxes.Q1,
               hland_fluxes.InUH,
               hland_fluxes.OutUH,
               hland_fluxes.QT)


class StateSequences(sequencetools.StateSequences):
    """State sequences of |hland_v1|."""
    CLASSES = (hland_states.Ic,
               hland_states.SP,
               hland_states.WC,
               hland_states.SM,
               hland_states.UZ,
               hland_states.LZ)


class LogSequences(sequencetools.LogSequences):
    """Aide sequences of |hland_v1|."""
    CLASSES = (hland_logs.QUH,)


class OutletSequences(sequencetools.LinkSequences):
    """Downstream link sequences of |hland_v1|."""
    CLASSES = (hland_outlets.Q,)


class Masks(masktools.Masks):
    """Masks applicable to |hland_v1|."""
    CLASSES = hland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
cythonizer.finalise()
